 /* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains low-level objects to support bitmaps, pixmaps, and
   stippled line and filling styles.

*/

extern "C" {
#include <X11/Xlib.h>
#include <malloc.h>
}

#include <stdio.h>
#include <iostream.h>
#include <am_inc.h>

#include GDEFS__H
#include GEM__H
#include "gemX.h"
//#include "gemX_gif.h"
#include "gem_GIF.h"

// // // // // // // // // // // // // // // // // // // // // // // // // //
// bitmap data for standard stipples
// // // // // // // // // // // // // // // // // // // // // // // // // //

static char stipples[][4] = {
  {0x00, 0x00, 0x00, 0x00},
  {0x01, 0x00, 0x00, 0x00},
  {0x01, 0x00, 0x04, 0x00},
  {0x05, 0x00, 0x04, 0x00},
  {0x05, 0x00, 0x05, 0x00},
  {0x05, 0x02, 0x05, 0x00},
  {0x05, 0x02, 0x05, 0x08},
  {0x05, 0x0a, 0x05, 0x08},
  {0x05, 0x0a, 0x05, 0x0a},
  {0x05, 0x0a, 0x05, 0x0e},
  {0x05, 0x0b, 0x05, 0x0e},
  {0x05, 0x0b, 0x05, 0x0f},
  {0x05, 0x0f, 0x05, 0x0f},
  {0x05, 0x0f, 0x0d, 0x0f},
  {0x07, 0x0f, 0x0b, 0x0f},
  {0x07, 0x0f, 0x0f, 0x0f},
  {0x0f, 0x0f, 0x0f, 0x0f}};



// // // // // // // // // // // // // // // // // // // //
// Am_Image_Array
// // // // // // // // // // // // // // // // // // // //

Am_WRAPPER_IMPL (Am_Image_Array)

/////
// Am_Image_Array constructors.
/////

Am_Image_Array::Am_Image_Array ()
{
  data = NULL;
}

Am_Image_Array::Am_Image_Array (const char* file_name)
{
   data = new Am_Image_Array_Data(file_name);
}


Am_Image_Array::Am_Image_Array (unsigned int width, 
				unsigned int height, int depth,
					// default color 0
				Am_Style /*intial_color*/)
{
  data = new Am_Image_Array_Data();
  
  unsigned char* temp = new unsigned char[width * height];
  data->image_ = new Am_Generic_Image(temp, width, height, depth);
  
  for (int i = 0; i < (height * width); i++)
    temp[i] = 0;
}


Am_Image_Array::Am_Image_Array (int percent)
{
 
  if ((percent < 0) || (percent > 100)) {
    fprintf(stderr, "%d is not a valid percent.  Please use a number"
	    "between 0 and 100\n", percent);
    fprintf(stderr, "Creating a null image instead.\n");
    return;
  }
  int nbits = (int)((float) percent / 5.88);
  data = new Am_Image_Array_Data(stipples[nbits], 4, 4);
}

Am_Image_Array::Am_Image_Array(char *bit_data, int height, int width) 
{
  data = new Am_Image_Array_Data(bit_data, height, width);
}

Am_Image_Array Am_No_Image;

////
// Am_Image_Array_Data constructors
////

Am_WRAPPER_DATA_IMPL (Am_Image_Array, (this))

Am_Image_Array_Data* Am_Image_Array_Data::list = NULL;

Am_Image_Array_Data::Am_Image_Array_Data (const char* file_name) 
{
   x_hot = -1;
   y_hot = -1;
   main_bitmap = 0;
   colors = NULL;
   num_colors = 0;
   image_ = NULL;
   name = new char [strlen(file_name) + 1];
   strcpy (name, file_name);
   main_display = NULL;
   head = NULL;
   next = list;
   list = this;
   refs = 1;
}

Am_Image_Array_Data::Am_Image_Array_Data (const char* bit_data, int h, int w)
					  
{
   x_hot = -1;
   y_hot = -1;
   main_bitmap = 0;
   colors = NULL;
   num_colors = 0;
   int nbytes = (((w + 7) / 8) * h);
   unsigned char* temp = new unsigned char [nbytes];
   memcpy(temp, bit_data, nbytes);
   image_ = new Am_Generic_Image ((unsigned char*)temp, w, h, 1);
   name = NULL;
   main_display = NULL;
   head = NULL;
   next = list;
   list = this;
   refs = 1;
}


Am_Image_Array_Data::Am_Image_Array_Data () 
{
   x_hot = -1;
   y_hot = -1;
   colors = NULL;
   num_colors = 0;
   main_bitmap = 0;
   image_ = NULL;
   name = NULL;
   main_display = NULL;
   head = NULL;
   next = list;
   list = this;
   refs = 1;
}

Am_Image_Array_Data::Am_Image_Array_Data (Am_Image_Array_Data* proto)
{
  // should I copy or use pointers to cursor, main_bitmap, etc?
  cursor = proto->cursor;
  main_bitmap = proto->main_bitmap;
  x_hot = proto->x_hot;
  y_hot = proto->y_hot;
  image_ = new Am_Generic_Image(proto->image_);
  name = new char [strlen(proto->name) + 1];
  strcpy (name, proto->name);
  main_display = proto->main_display;
  // BUGGY: should copy entire bitmap list!
  // (broken for multiple displays)
  head = proto->head;
  next = list;
  list = this;
  refs = 1;
  // Should copy color list as well
  colors = proto->colors;
  num_colors = proto->num_colors;
}


/////
// Am_Image_Array_Data destructor
/////

Am_Image_Array_Data::~Am_Image_Array_Data ()
{
//   /*  if (image)
//     {
//       /*
//       if (image->data) delete[] image->data;
//       image->data = NULL;
//       delete image;
//       */
//       XDestroyImage(image);
//       image = NULL;
//     }*/
  if (name) delete[] name; 
  if (image_) {
    delete image_;
    image_ = NULL;
  }
  if (colors) {
    int screen_num = DefaultScreen(main_display);
    Colormap c = XDefaultColormap(main_display, screen_num);
    for (int i = 0; i < num_colors; i++)
      XFreeColors(main_display, c, &(colors[i].pixel), 1, 0);
    delete[] colors;
    colors = NULL;
  }
  if (main_display && main_bitmap)
    XFreePixmap(main_display, main_bitmap);
  Bitmap_Item* current = head;
  Bitmap_Item* next = NULL;
  while (current) {
    next = current->next;
    current->next = NULL;
    delete current; // destructor takes care of freeing colors, pixmap, etc.
    current = next;
  }
  head = NULL;
  remove (this);
}

void Am_Image_Array_Data::remove (Am_Image_Array_Data* image)
{
  Am_Image_Array_Data* prev = NULL;
  Am_Image_Array_Data* curr = list;
  while (curr) {
    if (curr == image) {
      if (prev)
        prev->next = curr->next;
      else
        list = curr->next;
      return;
    }
    prev = curr;
    curr = curr->next;
  }
}

void Am_Image_Array_Data::remove (Display* display)
{
  Am_Image_Array_Data* curr;
  for (curr = list; curr; curr = curr->next) {
    if (curr->main_display == display) {
      if (curr->colors) {
        int screen_num = DefaultScreen (curr->main_display);
        Colormap c = XDefaultColormap (curr->main_display, screen_num);
        for (int i = 0; i < curr->num_colors; i++)
          XFreeColors (curr->main_display, c, &(curr->colors[i].pixel), 1, 0);
        delete[] curr->colors;
        curr->colors = NULL;
      }
      if (curr->main_display && curr->main_bitmap)
        XFreePixmap (curr->main_display, curr->main_bitmap);
      curr->main_display = NULL;
    }
    Bitmap_Item* prev = NULL;
    Bitmap_Item* curr_index = curr->head;
    while (curr_index) {
      if (curr_index->display == display) {
        if (prev)
          prev->next = curr_index->next;
        else
          curr->head = curr_index->next;
        delete curr_index;
        break;
      }
      prev = curr_index;
      curr_index = curr_index->next;
    }
  }
}

/////
// Am_Image_Array setters and getters
/////

// NDY: implementations of getbit, setbit, write_to_file
int Am_Image_Array::Get_Bit(int /*x*/, int /*y*/)
{
  if (data)
    return 0; // return something useful
  else return 0;
}
void Am_Image_Array::Set_Bit(int /*x*/, int /*y*/, int /*val*/)
{
  if (data)
    data = Am_Image_Array_Data::Narrow(data->Make_Unique());
    // then do something useful
}
int Am_Image_Array::Write_To_File (const char* /*file_name*/,
					Am_Image_File_Format /*form*/)
{
  if (data)
    // do something useful
    return 0;
  else return 0;
}
// end of NDY stuff.



void Am_Drawonable_Impl::Get_Image_Size (const Am_Image_Array& image,
					 int& ret_width, int& ret_height)
{
  Am_Image_Array_Data* data = (Am_Image_Array_Data::Narrow(image));
  
  if (data) {
    // Make sure the image has been created for this display
    data->Get_X_Pixmap (this);
    image.Get_Size(ret_width, ret_height);
    data->Release ();
  }
}

// This function is not guaranteed to return the expected width and height
// of the bitmap, since it may not have been created on the display yet.
// You should call Am_Drawonable::Get_Image_Size() instead.
void Am_Image_Array::Get_Size (int& ret_width, int& ret_height) const
{
  if (data && data->image_) {
    short unsigned int w, h; // using temps explicitly gets rid of a warning
    data->image_->Get_Size(w, h);
    ret_width = w; 
    ret_height = h;
  }
}

/*
void Am_Image_Array::Get_Hot_Spot (int& x, int& y) const
{
  if (data) {
    x = data->x_hot;
    y = data->y_hot;
  }
}

void Am_Image_Array::Set_Hot_Spot (int x, int y)
{
  if (data) {
    data = Am_Image_Array_Data::Narrow(data->Make_Unique());
    
    data->x_hot = x;
    data->y_hot = y;
  }
}
*/

/////////////
// Utility function
/////////////

XImage *Am_Image_Array_Data::Make_XImage_From_Generic_Image
(Am_Drawonable_Impl* draw, XColor* &cols, int &n_cols) {
  // Make a copy of the image data, and adjust colors in the image 
  // to the ones we have in the X colormap.
  Am_RGB_Value* ctbl;
  n_cols = image_->Get_Color_Map(ctbl);
  cols = new XColor[n_cols];

  int i;
  // first, allocate all the closest colors.  Could be optimized a lot!
  for (i = 0; i < n_cols; i++)
    {
      cols[i].red = ctbl[i].red * 256;
      cols[i].green = ctbl[i].green * 256;
      cols[i].blue = ctbl[i].blue * 256;
      //      cerr << cols[i].red << " " << cols[i].green <<
      //      	" " << cols[i].blue;
      draw->Allocate_Closest_Color(cols[i]);
      //      cerr << " ... " << cols[i].red << " " << cols[i].green <<
      //      	" " << cols[i].blue << endl;
    }
  
  // Next, set the pixels in the image correctly.
  unsigned short width, height;
  image_->Get_Size(width, height);
  int depth = image_->Get_Depth();
  unsigned char* gif_image = image_->Get_Data();
  // X uses free() so we must use malloc.  (so says sentinel)
  unsigned char* x_image = 
    (unsigned char*)malloc(width * height * sizeof(unsigned char));
  
  for (i = 0; i < (width * height); i++)
    x_image[i] = (unsigned char)(cols[gif_image[i]].pixel);
  
  // With a friend like X, who needs enemies?
  Visual *V = DefaultVisual(draw->screen->display, 
			    draw->screen->screen_number);
  VisualID vid = XVisualIDFromVisual(V);
  XVisualInfo vit; // template
  vit.visualid = vid;
  int n;
  XVisualInfo *vi = XGetVisualInfo(draw->screen->display, VisualIDMask, 
				   &vit, &n);
  if (n == 0) {
    cerr << "** Gem error: X failed to allocate Visual\n";
    return NULL;
  }
  else
    depth = vi->depth;
  image_->Set_Depth(depth);
  
  XFree(vi);
  // Why so much crap just to get the depth?  It might not work anyway.
  
  int type;
  type = ZPixmap;
  
  XImage *image;
  image = XCreateImage(draw->screen->display, V, depth, type, 
		       0 /* offset */, (char *)x_image, width, 
		       height, 8 /* bitmap_pad */, 
		       width /* bytes per line */);
  
  image->byte_order= LSBFirst; // shouldn't matter, only one byte per byte
  image->bitmap_unit = 8; // bits per unit in scan line: bytes
  image->bitmap_bit_order = LSBFirst; // shouldn't matter in a ZPixmap
  image->bits_per_pixel = depth;
  
  return image;
}



/////
// Am_Image_Array_Data setters/getters
////

Pixmap Am_Image_Array_Data::Get_X_Pixmap (Am_Drawonable_Impl* draw)
{
  Display* disp = draw->screen->display;
  if (disp == main_display)
    return main_bitmap;

  Pixmap bitmap;
  int tmp;

  if (Get_Bitmap (disp, bitmap))
    return bitmap;

  XColor *cols = NULL;
  int n_cols = 0;
  XImage *im;
  
  /* then we need to read it in or make it */
  if (name) {
    unsigned int w, h;
    /* then we need to read it in from a file */
    // cheap test if it's a gif
    int l = strlen(name);
    if (!strcmp(&(name[l - 3]), "gif")) // then assume it's gif
      {
	// only load the image if it's not loaded already.
	if (image_ == NULL) image_ = Am_GIF_Image::Create(name);
	if (image_ == NULL) return NULL; // error
	
	im = Make_XImage_From_Generic_Image(draw, cols, n_cols);
	
	bitmap = XCreatePixmap(disp, draw->xlib_drawable, im->width, 
			       im->height, im->depth);
	draw->set_gc_using_fill(Am_No_Style, Am_DRAW_COPY);
	// Clear the clip region; restore it later.
	XSetClipMask (draw->screen->display, draw->screen->gc, None);
	XPutImage(disp, bitmap, draw->screen->gc, im, 0, 0, 0, 0,
		  im->width, im->height);
	draw->Set_Clip(draw->screen->clip_region);
      }
    else { // it's an xbm
      tmp = XReadBitmapFile (disp, draw->xlib_drawable, name, 
			     &w, &h, &bitmap, &x_hot, &y_hot);
      // fill an empty structure to store width, height in.
      image_ = new Am_Generic_Image (NULL, w, h, 1); 
      if (tmp == BitmapOpenFailed) {
	cerr << "** Bitmap Error: " << name << " is not a valid file name"
	     << endl;
	Am_Error ();
      }
      if (tmp == BitmapFileInvalid) {
	cerr << "** Bitmap Error: " << name
	     << " does not contain valid bitmap data" << endl;
	Am_Error ();
      }
      if (tmp == BitmapNoMemory) {
	cerr << "** Bitmap Error: Out of memory error when reading in bitmap "
	  "from " << name << endl;
	Am_Error ();
      }
      if (tmp == BadAlloc)
	Am_Error ("** Bitmap Error: Bad Alloc error");
      if (tmp == BadDrawable)
	Am_Error ("** Bitmap Error: Bad Drawable error");
      if (tmp != BitmapSuccess) {
	cerr << "** Error in reading in bitmap from file " << name
	     << endl;
	Am_Error ();
      }
    } // else bmp format
  }
  else {
    // then we need to make it from data
    // Assume that it's in XYBitmap format in the generic image item.
    unsigned short int width, height;
    image_->Get_Size(width, height);
    bitmap = XCreateBitmapFromData (disp, draw->xlib_drawable,
		                    (char*)image_->Get_Data(), width, height);
    
    if (bitmap == BadAlloc) {
      cerr << "Bad Alloc error" << endl;
      Am_Error ();
    }
    if (bitmap == BadDrawable) {
      cerr << "Bad Drawable error" << endl;
      Am_Error ();
    }
  }

  if (main_display)
    Add_Bitmap (disp, bitmap, cols, n_cols);
  else {
    main_display = disp;
    main_bitmap = bitmap;
    colors = cols;
    num_colors = n_cols;
  }
  return bitmap;
}

void Am_Image_Array_Data::Add_Bitmap (Display* display, Pixmap bitmap, 
				      XColor *colors, int n_colors)
{
  Bitmap_Item* new_node 
    = new Bitmap_Item (display, bitmap, colors, n_colors);
  new_node->next = head;
  head = new_node;
}

bool Am_Image_Array_Data::Get_Bitmap (Display* display, Pixmap& bitmap)
{
  Bitmap_Item* current;
  for (current = head; current != NULL; current = current->next)
    if (current->display == display) {
      bitmap = current->bitmap;
      return true;
    }
  return false;
}


/////
// Am_Image_Array drawing function.
/////


// Defaults:
//   i_left = 0,
//   i_top = 0,
//   ls = Am_No_Style,
//   fs = Am_No_Style,
//   f = Am_DRAW_COPY
// if width is -1, then it ignores them and uses the ones set in the image 
// class
//
// What the hell is i_left and i_top for anwyay?
void Am_Drawonable_Impl::Draw_Image (int left, int top, int width, int height,
			     const Am_Image_Array& image, 
			     int /*i_left*/, int /*i_top*/,
			     const Am_Style& ls, //color of 'on' bits
			     const Am_Style& fs, //background behind image
			     Am_Draw_Function f)
{
  int i_width, i_height;
    
  Am_Image_Array_Data* image_data = Am_Image_Array_Data::Narrow(image);
  Pixmap the_pixmap =  image_data->Get_X_Pixmap (this);
  int depth = image_data->Get_Depth();
  if (image_data)
    image_data->Release ();

  if (width < 0) {
    Get_Image_Size(image, i_width, i_height);
  }
  else {
    i_width = width;
    i_height = height;
  }

  if (depth > 1)
    {
      // make it do _something_ with the gc, at least...
      set_gc_using_fill (fs, f);
      // multicolor pixmap: simply blit it.
      XCopyArea(screen->display, the_pixmap, xlib_drawable, screen->gc, 
		0, 0, i_width, i_height, left, top);
    }
  else // bitmap, might be transparent.
    {
      // The idea of transparent and opaque bitmaps is implemented here, but it
      // is not through the fill-flag of any style object.  The way to make a
      // bitmap transparent is to just not pass any fill-style.  If a fill-style
      // is passed, the bitmap will be opaque (i.e., the 'off' bits will be drawn)
      // with the color of the fill-style.
      
      if (fs.Valid()) {
	set_gc_using_fill (fs, f);
	XFillRectangle (screen->display, xlib_drawable, screen->gc,
			left, top, i_width, i_height);
	
      }
      
      if (ls.Valid()) {
	set_gc_using_fill (ls, f, image);
	XSetTSOrigin(screen->display, screen->gc, left, top);
	XFillRectangle (screen->display, xlib_drawable, screen->gc,
			left, top, i_width, i_height);
      }
    }
}




