// misc.cc
// This contains misc. machine dependant utility routines
// used throughout amulet.
//
// This should be broken into 3 machine-dependant files instead of just one
// file with lots of #ifdefs.

#include <am_inc.h>

#include MISC__H
#include AM_IO__H
#include <string.h>

// We need a better test for UNIX machines
#if !defined(_WINDOWS) & !defined(_MACINTOSH)

#include <signal.h>
#include <stdlib.h>
//#ifdef NEED_UNISTD
#include <unistd.h>
//#endif
#include <sys/time.h>

#ifdef NEED_SELECT
#include <sys/select.h>
#endif

#ifdef GCC

extern "C" {
int select();
}
#endif



void Am_Break_Into_Debugger ()
{
  cerr << 
    "** Breaking into the debugger:  Expect a SIGINT interrupt signal.\n";
  kill(getpid(), SIGINT);
}

void Am_Wait (int milliseconds)
{
  timeval tim;
  tim.tv_sec = milliseconds/1000;
  tim.tv_usec = (milliseconds % 1000) * 1000;
  // Use select to hack a subsecond pause on UNIX.
  // There's no machine independant way to pause for a fraction of a second.
  // select usually does something, but NULLs say, "do nothing."
  // tim tells it how long to do nothing.
  select (NULL, NULL, NULL, NULL, &tim);
}


// Provide an installation default for AMULET_DIR, so users may not
// set this variable every time. One can redefine this default from
// the Makefile.vars.* to adapt to local directory structures.
#ifndef DEFAULT_AMULET_DIR
#define DEFAULT_AMULET_DIR "/usr/local/lib/amulet"
#endif

char* Am_Get_Amulet_Pathname () {
  char* amulet_dir = getenv("AMULET_DIR");
  if (amulet_dir != 0)
    return amulet_dir;
  return DEFAULT_AMULET_DIR;
}


char *Am_Merge_Pathname(char *name)
{
  char * am_dir = Am_Get_Amulet_Pathname();
  if (!am_dir) am_dir = "..";
  char *ret_val = (char *) new char [(strlen(am_dir) + strlen(name) + 2)];

  strcpy(ret_val, am_dir);
  strcat(ret_val, "/");
  strcat(ret_val, name);
  return ret_val;
}

#elif (_WINDOWS)

// the Windows specific code
 
#include <windows.h>

void Am_Break_Into_Debugger ()
{
  cerr << "Breaking into the debugger not implemented yet for this machine.\n";
}

void Am_Wait (int milliseconds)
{
  // this probably doesn't work.
  Sleep(milliseconds);
}

char *Am_Merge_Pathname(char *name)
{
  char * am_dir = Am_Get_Amulet_Pathname();
  if (!am_dir) am_dir = "..";
  char *ret_val = (char *) new char [(strlen(am_dir) + strlen(name) + 2)];

  strcpy(ret_val, am_dir);
  strcat(ret_val, "/");
  strcat(ret_val, name);
  return ret_val;
}

#elif (_MACINTOSH)
#include AM_IO__H
#include TYPES__H

// the Mac specific code
void Am_Wait (int milliseconds)
{
  //  cerr << "Give up." << endl;
}

void Am_Break_Into_Debugger ()
{
  cerr << "Breaking into the debugger not implemented yet for this machine.\n";
}

char* Am_Get_Amulet_Pathname ()
{
  static char* pathname = NULL;
  static unsigned char amulet_env[] = "\pamulet.env";
  
  OSErr errCode;
  short foundVRefNum;
  long foundDirID;
  FSSpec fileSpec;
  short refNum = 0;
  long fileLength;
    
  if (pathname) return pathname;
  
  errCode = FindFolder(kOnSystemDisk, kPreferencesFolderType, false, &foundVRefNum, &foundDirID);
  if (errCode) goto exit;
  
  errCode = FSMakeFSSpec(foundVRefNum, foundDirID, amulet_env, &fileSpec);
  if (errCode == fnfErr) {
    Am_Error( "Unable to find amulet.env in Preferences Folder\nThis text file must contain the full pathname to the amulet\ndirectory in form volume_name:folder\nThere should be no : at the end of the pathname ");
  } else if (errCode) goto exit;

  errCode = FSpOpenDF(&fileSpec, fsRdPerm, &refNum);
  if (errCode) goto exit;
  
  errCode = GetEOF(refNum, &fileLength);
  if (errCode) goto exit;
  
  pathname = new char [fileLength + 1];
  errCode = FSRead(refNum, &fileLength, pathname);
  if (pathname) {
    while (pathname[fileLength-1] == '\r') {
      fileLength --;
    }
  }
  pathname[fileLength] = '\0';
  
exit:
  if (refNum)  FSClose(refNum);
  return pathname;
}

char *Am_Merge_Pathname(char *name)
{
  char * am_dir = Am_Get_Amulet_Pathname();
  if (!am_dir) am_dir = "";
  char *ret_val = (char *) new char [(strlen(am_dir) + strlen(name) + 2)];
  
  char* p = name;
  char	ch;
  while ((ch = *p) != '\0') {
  	if (ch == '/') *p = ':';
  	p++;
  } 
  
  strcpy(ret_val, am_dir);
  strcat(ret_val, ":");
  strcat(ret_val, name);
  return ret_val;
}

#else
// Unsupported platform

#include AM_IO__H
void Am_Break_Into_Debugger ()
{
  cerr << "Breaking into the debugger not implemented yet for this machine.\n";
}
#endif
