/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains the selection_widget
   
   Designed and implemented by Brad Myers
*/

#include <am_inc.h>

#include AM_IO__H

#include WIDGETS_ADVANCED__H
#include STANDARD_SLOTS__H
#include VALUE_LIST__H
#include OPAL_ADVANCED__H  // for Am_DRAWONABLE, Am_Window_Coordinate
#include INTER_ADVANCED__H //needed for Am_Inter_Tracing

#include WIDGETS__H
#include GEM__H
#include INTER__H
#include OPAL__H
#include INTER__H
#include REGISTRY__H


#define MOVE_THRESHHOLD 3 //number of pixels before allowed to move an object

inline int IABS (int a) { return (a < 0)? -a : a; }
inline int IMIN (int a, int b) { return (a < b)? a : b; }
inline int IMAX (int a, int b) { return (a > b)? a : b; }

///////////////////////////////////////////////////////////////////////////
// Handles
///////////////////////////////////////////////////////////////////////////

#define HANDLE_SIZE 9  //should be odd
#define HANDLE_SIZE_D2 4  // HANDLE_SIZE / 2

void draw_handle (int left, int top, bool filled, Am_Drawonable* draw,
		  Am_Style style) {
  if (filled) {
    draw->Draw_Rectangle (Am_No_Style, style, left, top,
			  HANDLE_SIZE, HANDLE_SIZE);
    //outline in white
    draw->Draw_Rectangle (Am_White, Am_No_Style, left, top,
			  HANDLE_SIZE, HANDLE_SIZE);
  }
  else draw->Draw_Rectangle (style, Am_No_Style, left, top,
			  HANDLE_SIZE, HANDLE_SIZE);
}

void draw_8_handles_for_rect(int left, int top, int width, int height,
			     bool filled, Am_Drawonable* draw,
			     Am_Style style) {
  int width_d2 = width / 2;
  int height_d2 = height / 2;
  
  
  // left-top
  draw_handle(left, top, filled, draw, style);
  //left-middle
  draw_handle(left, top + height_d2 - HANDLE_SIZE_D2, filled, draw, style);
  //left-bottom
  draw_handle(left, top+height-HANDLE_SIZE, filled, draw, style);

  // middle-top
  draw_handle(left + width_d2 - HANDLE_SIZE_D2, top, filled, draw, style);
  //middle-bottom
  draw_handle(left + width_d2 - HANDLE_SIZE_D2, top+height-HANDLE_SIZE,
	      filled, draw, style);

  // right-top
  draw_handle(left + width - HANDLE_SIZE, top, filled, draw, style);
  // right-middle
  draw_handle(left + width - HANDLE_SIZE, top + height_d2 - HANDLE_SIZE_D2,
	      filled, draw, style);
  // right-bottom
  draw_handle(left + width - HANDLE_SIZE, top + height - HANDLE_SIZE,
	      filled, draw, style);

}

	 
void draw_2_handles_for_line(int left, int top, int width, int height,
			     bool use_lt_rb, bool filled, Am_Drawonable* draw,
			     Am_Style style) {
  if (use_lt_rb) {
    // left-top
    draw_handle(left, top, filled, draw, style);
    // right-bottom
    draw_handle(left + width - HANDLE_SIZE, top + height - HANDLE_SIZE,
		filled, draw, style);
  }
  else {
    //left-bottom
    draw_handle(left, top+height-HANDLE_SIZE, filled, draw, style);
    // right-top
    draw_handle(left + width - HANDLE_SIZE, top, filled, draw, style);
  }
}

//Figure out which end of a line is up.
//self is the selections handles widget
Am_Define_Formula (bool, selections_handles_use_lt_rb) {
  Am_Object for_obj;
  for_obj = self.GV(Am_ITEM);
  if (for_obj.Valid()) {
    Am_Value value;
    for_obj.GVM(Am_AS_LINE, value);
    bool as_line = value.Valid();
    if (as_line) {
      int x1 = for_obj.GV(Am_X1);
      int y1 = for_obj.GV(Am_Y1);
      int x2 = for_obj.GV(Am_X2);
      int y2 = for_obj.GV(Am_Y2);
      if (x1 < x2) {
	if (y1 < y2) return true;
	else return false;
      }
      else {
	if (y2 < y1) return true;
	else return false;
      }
    }
  }
  return true; //if get here, not a valid line, so don't care
}


//Set into the top-level widget to validate all objects in the values list
Am_Define_Formula(int, remove_from_value_if_invalid) {
  self.Make_Unique(Am_VALUE);
  Am_Value_List values;
  values = self.GV(Am_VALUE);
  Am_Object obj, o;
  Am_Value v;
  bool changed = false;
  for(values.Start(); !values.Last(); values.Next()) {
    obj = values.Get();
    if (!Am_Object_And_Owners_Valid_And_Visible(obj)) {
      values.Delete(false);
      changed = true;
    }
    else {
      //set up dependencies so this formula will be re-evaluated if
      //the object is modified
      obj.GVM(Am_LEFT, v);
      obj.GVM(Am_VISIBLE, v);
      o = obj.GV_Owner();
    }
  }
  if (changed) {
    self.Note_Changed(Am_VALUE);
  }
  return 0;
}

/*
//This one formula sets all the handle's sizes.  It goes in the Left slot.
//self is one selection handle object
Am_Define_Formula (int, selections_handles_ltwh) {
  Am_Object for_obj, sel_group;
  int left, top, width, height;
  for_obj = self.GV(Am_ITEM);
  sel_group = self.GV_Owner(); 
  bool valid;
  if (Am_Object_And_Owners_Valid_And_Visible(for_obj)) {
    left = - HANDLE_SIZE_D2;
    top = - HANDLE_SIZE_D2;
    if (sel_group.Valid())
      valid=Am_Translate_Coordinates(for_obj, left, top,
				     sel_group, left, top, cc);
    width = (int)for_obj.GV(Am_WIDTH) + HANDLE_SIZE - 1;
    height = (int)for_obj.GV(Am_HEIGHT) + HANDLE_SIZE - 1;
    // cout << "<>For " << self << " calculating l,t,w,h (" << left << ","
    //	  << top << "," << width << "," << height << ") for_obj "
    //	  << for_obj << " owner " << owner
    //	  << " valid " << valid << endl << flush;
  }
  else {
    left = 0; top = 0; width = 0; height = 0;
  }
  self.Set(Am_TOP, top);
  self.Set(Am_WIDTH, width);  
  self.Set(Am_HEIGHT, height);  
  return(left);
}
*/

// Set selection handles' left and top. 
// Put it in the Am_LEFT slot.

Am_Define_Formula (int, selections_handles_lt) {
  Am_Object for_obj, sel_group;
  int left, top;
  for_obj = self.GV(Am_ITEM);
  sel_group = self.GV_Owner(); 
  bool valid;
  if (Am_Object_And_Owners_Valid_And_Visible(for_obj)) {
    left = - HANDLE_SIZE_D2;
    top = - HANDLE_SIZE_D2;
    if (sel_group.Valid())
      valid=Am_Translate_Coordinates(for_obj, left, top,
				     sel_group, left, top, cc);
    // cout << "<>For " << self << " calculating l,t (" << left << ","
    //	  << top << ") for_obj "
    //	  << for_obj << " owner " << owner
    //	  << " valid " << valid << endl << flush;
  }
  else {
    left = 0; top = 0; 
  }
  self.Set(Am_TOP, top);
  return(left);
}

// Set selection handles width, height.
// Put this in the Am_Width slot.
Am_Define_Formula (int, selections_handles_wh) {
  Am_Object for_obj, sel_group;
  int width, height;
  for_obj = self.GV(Am_ITEM);
  sel_group = self.GV_Owner(); 
  if (Am_Object_And_Owners_Valid_And_Visible(for_obj)) {
    if (sel_group.Valid())
    width = (int)for_obj.GV(Am_WIDTH) + HANDLE_SIZE - 1;
    height = (int)for_obj.GV(Am_HEIGHT) + HANDLE_SIZE - 1;
    // cout << "<>For " << self << " calculating w,h (" 
    // << width << "," << height << ") for_obj "
    //	  << for_obj << " owner " << owner
    //	  << " valid " << valid << endl << flush;
  }
  else {
    width = 0; height = 0;
  }
  self.Set(Am_HEIGHT, height);  
  return(width);
}

Am_Define_Method(Am_Draw_Method, void, selection_handles_draw,
		 (Am_Object self, Am_Drawonable* draw,
		  int x_offset, int y_offset)) {
  Am_Object for_obj;
  for_obj = self.Get(Am_ITEM);

  if (for_obj.Valid()) { //otherwise don't draw anything	   
    Am_Value value;
    for_obj.Get(Am_AS_LINE, value);
    bool as_line = value.Valid();

	int left = (int)self.Get (Am_LEFT) + x_offset;
    int top = (int)self.Get (Am_TOP) + y_offset;
    int width = self.Get (Am_WIDTH);
    int height = self.Get (Am_HEIGHT);		 
     
    Am_Style style;
    style = self.Get(Am_FILL_STYLE);
    if (!style.Valid()) {
      //get from owner's owner so don't need a constraint
      //owner must be valid if I am being drawn
      style = self.Get_Owner().Get_Owner().Get(Am_FILL_STYLE);
    }
    //cout << "--Drawing " << self << " for obj " << for_obj << " as line "
    //	 << as_line << " l,t,w,h (" << left << ","
    //	 << top << "," << width << "," << height << ")\n" << flush;
    if (as_line) {
      bool use_lt_rb = self.Get(Am_SELECTIONS_HANDLES_USE_LT_RB);
      draw_2_handles_for_line(left, top, width, height, use_lt_rb,
                              true, draw, style);
    }
    else {
      draw_8_handles_for_rect(left, top, width, height, true, draw, style);
    }
  }
}

///////////////////////////////////////////////////////////////////////////
// For top-level widget
///////////////////////////////////////////////////////////////////////////


Am_Define_Formula (int, width_from_owner_or_zero) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    if (owner.Is_Instance_Of(Am_Scrolling_Group))
      return owner.GV(Am_INNER_WIDTH);
    else return owner.GV(Am_WIDTH);
  }
  else return 0;
}
Am_Define_Formula (int, height_from_owner_or_zero) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    if (owner.Is_Instance_Of(Am_Scrolling_Group))
      return owner.GV(Am_INNER_HEIGHT);
    else return owner.GV(Am_HEIGHT);
  }
  else return 0;
}

Am_Define_Formula (int, set_command_and_move_old_owner) {
  Am_Object cmd;
  cmd = self.GV(Am_COMMAND);
  if (cmd.Valid()) 
    cmd.Set(Am_SAVED_OLD_OWNER, self);
  cmd = self.GV(Am_MOVE_GROW_COMMAND);
  if (cmd.Valid()) 
    cmd.Set(Am_SAVED_OLD_OWNER, self);
  return 1;
}


///////////////////////////////////////////////////////////////////////////
// For become selected
///////////////////////////////////////////////////////////////////////////

//Where method, for become selected.
//  if inside me, test owner's where function and return what it
//returns, but if it returns nothing, then return me, to act as the "none" case
Am_Define_Method(Am_Where_Method, Am_Object, owner_start_where_or_none,
		 (Am_Object inter, Am_Object object, Am_Object event_window,
		  Am_Input_Char ic, int x, int y)) {
  //object is the selection handles widget
  Am_Object operates_on;
  operates_on = object.Get(Am_OPERATES_ON);
  // cout  << "Testing sel start_where for " << object << endl << flush;
  if (operates_on.Valid()) {
    if (Am_Point_In_All_Owners(operates_on, x, y, event_window) &&
	Am_Point_In_Obj (operates_on, x, y, event_window).Valid()) {
      Am_Object part;
      //next function call will typically turn into something like
      //      part = Am_Point_In_Part(operates_on, x, y, event_window);
      Am_Where_Method method;
      method = object.Get(Am_START_WHERE_TEST);
      part = method.Call(inter, operates_on, event_window, ic, x, y);
      if (part.Valid()) {
	// cout << " click in part " << part << endl << flush;
	return part;
      }
      else {
	// cout << " click in " << operates_on << " but not part\n" <<flush;
	return object;
      }
    }
  }
  //otherwise not in object at all
  return Am_No_Object;   
}

void set_commands_for_sel(Am_Object inter, Am_Object widget,
			  Am_Object clicked_obj,
			  Am_Value new_value) {
  Am_Object inter_command, widget_command;
  Am_Value old_value;
  inter_command = inter.Get_Part(Am_COMMAND);
  if (inter_command.Valid()) {
    inter_command.Get(Am_VALUE, old_value);
    inter_command.Set(Am_OLD_VALUE, old_value);
    inter_command.Set(Am_VALUE, new_value);
    inter_command.Set(Am_OBJECT_MODIFIED, clicked_obj);
  }
  widget_command = widget.Get_Part(Am_COMMAND);
  if (widget_command.Valid()) {
    widget_command.Get(Am_VALUE, old_value);
    widget_command.Set(Am_OLD_VALUE, old_value);
    widget_command.Set(Am_VALUE, new_value);
    widget_command.Set(Am_OBJECT_MODIFIED, clicked_obj);
  }
}
  
//called to select the object.
//object to be operated on is in interim
//depending on state of shift, add to select set or make this be the only
//object.
// **** Should use destructive modification for lists ***
Am_Define_Method(Am_Mouse_Event_Method, void, sel_object,
		 (Am_Object inter, int /* mouse_x */, int /* mouse_y */,
		  Am_Object /* ref_obj */, Am_Input_Char ic)) {
  Am_Object new_object;
  if (inter.Valid()) {
    new_object = inter.Get(Am_START_OBJECT);
    if(new_object.Valid ()) {
      Am_Value new_value;
      Am_Object widget = inter.Get_Owner();
      bool toggle_in_set = ic.shift;
      if (new_object == widget) { // then clicked in the background
	if (toggle_in_set) {   //  don't do anything
	  Am_Abort_Interactor(inter); //make sure not queued for undo
	  return;
	}
	else {  // select nothing
	  if (Am_Inter_Tracing(widget))
	     cout << "Selection handle setting empty for "
		  << widget << endl << flush;
	  new_object = NULL; //so object_modified will be null
	  new_value = Am_Value_List();
	}
      }
      else { // over a specific object
	Am_Value_List list;
	list = widget.Get(Am_VALUE);
	list.Start();
	if (toggle_in_set) {
	  if(list.Member(new_object)) {
	    if (Am_Inter_Tracing(widget))
	      cout << "Selection handle removing " << new_object << " from "
		   << widget << endl << flush;
	    list.Delete();
	  }
	  else { // not a member, add it
	    if (Am_Inter_Tracing(widget))
	      cout << "Selection handle adding " << new_object << " to "
		   << widget << endl << flush;
	    list.Add(new_object);
	  }
	}
	else { //if object is selected, do nothing, otherwise,
	       // make new_object be the only selection
	  if(list.Member(new_object)) { //nothing to do if already selected
	    Am_Abort_Interactor(inter); //make sure not queued for undo
	    return;
	  }
	  else {
	    if (Am_Inter_Tracing(widget))
	      cout << "Selection handle setting " << widget
		   << " to contain only " << new_object << endl << flush;
	    list.Make_Empty();
	    list.Add(new_object);
	  }
	}
	new_value = list;
      }
      widget.Set(Am_VALUE, new_value);
      set_commands_for_sel(inter, widget, new_object, new_value);
    }
  }
}

///////////////////////////////////////////////////////////////////////////
// For Moving
///////////////////////////////////////////////////////////////////////////


//goes into the move-grow interactor in the selection widget
Am_Define_Object_Formula (Am_Compute_MG_Feedback_Object) {
  if ((bool)self.GV(Am_AS_LINE))
    return self.GV_Owner().GV_Part(Am_LINE_FEEDBACK_OBJECT);
  else return self.GV_Owner().GV_Part(Am_RECT_FEEDBACK_OBJECT);
}

//get owner's start_when, and remove the "any" from the front.  Used as the
//start_when of the moving or growing (so shift doesn't cause moving)
Am_Define_Formula (long, compute_mg_start_when) {
  Am_Object widget = self.GV_Owner();
  if (widget.Valid()) {
    Am_Input_Char parent_char;
    Am_Value value;
    widget.GVM(Am_START_WHEN, value);
    if (value.type == Am_STRING) {
      Am_String sval;
      sval = value;
      parent_char = Am_Input_Char(sval);
    }
    else if (value.type == Am_INT || value.type == Am_LONG)
      parent_char = Am_Input_Char::Narrow(value.value.long_value);
    else Am_Error("Am_START_WHEN slot has wrong type");
    parent_char.any_modifier = false;
    parent_char.shift = false;
    if (parent_char.button_down == Am_ANY_DOWN_UP)
      parent_char.button_down = Am_BUTTON_DOWN;
    if (parent_char.click_count == Am_ANY_CLICK)
      parent_char.click_count = Am_SINGLE_CLICK;
    // cout << "computed start when " << parent_char << " from widget " <<
    //  widget << endl << flush;
    return (long)parent_char;
  }
  else return Am_Input_Char("LEFT_DOWN");
}

void calculate_group_size(Am_Value_List list, int &min_left, int &min_top, 
			  int &max_right, int &max_bottom,
			  Am_Object &owner) {
  min_left = 29999;
  min_top  = 29999;
  max_right = -29999;
  max_bottom = -29999;
  int cur_left, cur_top;
  Am_Object obj, cur_owner;
  for(list.Start(); !list.Last(); list.Next()) {
    obj = list.Get();
    cur_owner = obj.Get_Owner();
    if (cur_owner.Valid()) {
      if (!owner.Valid()) owner = cur_owner;
      else if (owner != cur_owner)
	Am_Error("Moving objects from different groups.");
    }
    cur_left = obj.Get(Am_LEFT);
    cur_top = obj.Get(Am_TOP);
    min_left = IMIN(min_left, cur_left);
    min_top = IMIN(min_top, cur_top);
    max_right = IMAX(max_right, cur_left + (int)obj.Get(Am_WIDTH));
    max_bottom = IMAX(max_bottom, cur_top + (int)obj.Get(Am_HEIGHT));
  }
}

//figure out the bounds of all the selected objects, and assign it into
//fake_group
void calculate_fake_group_size_and_set(Am_Object fake_group,
				       Am_Value_List list) {
  int min_left = 29999;
  int min_top  = 29999;
  int max_right = -29999;
  int max_bottom = -29999;
  int left_offset, top_offset;
  Am_Object owner;
  calculate_group_size(list, min_left, min_top, max_right, max_bottom, owner);
  Am_Translate_Coordinates(owner, 0, 0, fake_group.Get_Owner(),
			   left_offset, top_offset);
  fake_group.Set(Am_LEFT, min_left + left_offset);
  fake_group.Set(Am_TOP, min_top + top_offset);
  fake_group.Set(Am_WIDTH, max_right - min_left);
  fake_group.Set(Am_HEIGHT, max_bottom - min_top);
  // cout << "Calculate fake group " << fake_group << " size = " << min_left
  // << "," << min_top << "," << max_right - min_left << ","
  // << max_bottom - min_top << endl << flush;
}

Am_Define_Method(Am_Object_Method, void, copy_values_for_grow,
		 (Am_Object cmd)) {
  //copy from me to widget's command
  Am_Object widget, widget_command;
  widget = cmd.Get_Owner().Get_Owner();
  Am_Value value;
  if (widget.Valid()) {
    widget_command = widget.Get_Part(Am_MOVE_GROW_COMMAND);
    if (widget_command.Valid()) {
      cmd.Get(Am_OLD_VALUE, value);
      widget_command.Set(Am_OLD_VALUE, value);
      cmd.Get(Am_VALUE, value);
      widget_command.Set(Am_VALUE, value);
      cmd.Get(Am_OBJECT_MODIFIED, value);
      widget_command.Set(Am_OBJECT_MODIFIED, value);

      widget_command.Set(Am_GROWING, true);
    }
  }
}

void set_commands_for_move(Am_Object widget, Am_Object inter,
			   Am_Value object_modified, Am_Value old_value,
			   Am_Value new_value, int final_left, int final_top) {
  Am_Object inter_command, widget_command;
  inter_command = inter.Get_Part(Am_COMMAND);
  if (inter_command.Valid()) {
    inter_command.Set(Am_OLD_VALUE, old_value);
    inter_command.Set(Am_VALUE, new_value);
    inter_command.Set(Am_OBJECT_MODIFIED, object_modified);
    inter_command.Set(Am_LEFT, final_left);
    inter_command.Set(Am_TOP, final_top);
  }
  widget_command = widget.Get_Part(Am_MOVE_GROW_COMMAND);
  if (widget_command.Valid()) {
    widget_command.Set(Am_OLD_VALUE, old_value);
    widget_command.Set(Am_VALUE, new_value);
    widget_command.Set(Am_OBJECT_MODIFIED, object_modified);
    widget_command.Set(Am_GROWING, false);
  }
}


//move all the selected objects by the offsets.
void adjust_all_objects_position(Am_Object widget, Am_Object inter,
				 Am_Value_List list,
				 int left_offset, int top_offset,
				 int final_left, int final_top) {
  // cout << "<>adjusting offsets of sel objects by " << left_offset << ","
  //   << top_offset << endl << flush;
  Am_Object obj;
  Am_Value_List new_values, old_values;
  Am_Object owner;
  for(list.Start(); !list.Last(); list.Next()) {
    obj = list.Get();
    owner = obj.Get_Owner();
    int old_left = obj.Get(Am_LEFT);
    int old_top = obj.Get(Am_TOP);
    int new_left = old_left + left_offset;
    int new_top = old_top + top_offset;
    old_values.Add(Am_Inter_Location(false, owner, old_left, old_top, 0,0));
    new_values.Add(Am_Inter_Location(false, owner, new_left, new_top, 0,0));
    if (Am_Inter_Tracing(widget))
      cout << "Selection handle widget " << widget
	   << " moving  " << obj << " to " << new_left << "," << new_top
	   << endl << flush;
    obj.Set(Am_LEFT, new_left);
    obj.Set(Am_TOP, new_top);
  }
  Am_Value objs_value, old_values_value, new_values_value;
  objs_value = list;
  old_values_value = old_values;
  new_values_value = new_values;
  set_commands_for_move(widget, inter, objs_value, old_values_value,
			new_values_value, final_left, final_top);
}

//Where function: For moving.  Same as owner_start_where_or_none but won't
//return object (if click in background).  Also, if multiple objects selected,
//returns the fake group, but first sets its size based on the current
//selection.
Am_Define_Method(Am_Where_Method, Am_Object, owner_start_where_or_fake,
		 (Am_Object inter, Am_Object object, Am_Object event_window,
		  Am_Input_Char ic, int x, int y)) {
  // call the procedure defined by owner_start_where_or_none method above
  Am_Object ret = owner_start_where_or_none_proc(inter, object,
						    event_window, ic, x, y);
  if (ret == object)
    ret = Am_No_Object;
  else { //have a valid selection
    Am_Value_List list;
    list = object.Get(Am_VALUE);
    list.Start();
    if (!list.Member(ret))
      ret = Am_No_Object;  //click caused the object to now not be selected
    else if (list.Length() > 1) { // otherwise, ret is OK
      Am_Object fake_group = object.Get_Part(Am_FAKE_GROUP);
      calculate_fake_group_size_and_set(fake_group, list);
      ret = fake_group;
    }
  }
  // cout << "  Move where returning " << ret << endl << flush;
  return ret;
}

Am_Define_Method(Am_Current_Location_Method, void, sel_move_start_do,
		 (Am_Object command_obj, Am_Object /* object_modified */,
		  Am_Inter_Location data)) {
  Am_Object inter, feedback;
  inter = command_obj.Get_Owner();
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  if (feedback.Valid()) feedback.Set(Am_VISIBLE, false);
  inter.Set (Am_MOVED_ENOUGH, false);
  int first_x, first_y, w, h;
  data.Get_Points(first_x, first_y, w, h);
  inter.Set (Am_LEFT, first_x);
  inter.Set (Am_TOP, first_y);
}

Am_Define_Method(Am_Current_Location_Method, void, sel_move_inter_interim_do,
		 (Am_Object inter, Am_Object object_modified,
		  Am_Inter_Location data)) {
  bool moved_enough = inter.Get (Am_MOVED_ENOUGH);
  // cout << "sel interim do, moved " << moved_enough << endl << flush;
  Am_Object feedback;
  feedback = inter.Get(Am_FEEDBACK_OBJECT);
  if (!moved_enough) {
    // see if moved enough yet
    int first_x, first_y, current_x, current_y, w, h;
    data.Get_Points(current_x, current_y, w, h);
    first_x = inter.Get (Am_LEFT);
    first_y = inter.Get (Am_TOP);
    // cout << "Selection move interim, cur_x,y = " << current_x << ","
    //  << current_x << endl << flush;
    if (IABS(first_x - current_x) >= MOVE_THRESHHOLD ||
	IABS(first_y - current_y) >= MOVE_THRESHHOLD) {
      // if moving a selected object, move ALL the selected objects.
      // if moving an object which is NOT selected, select it first, then move
      // it.
      
      Am_Object sel_inter;
      sel_inter = inter.Get_Sibling(Am_INTERACTOR);
      // cout << "<>Big enough, starting.  Feedback " << feedback
      //   << " aborting inter " << sel_inter << endl << flush;
      if (feedback.Valid()) feedback.Set(Am_VISIBLE, true);
      inter.Set (Am_MOVED_ENOUGH, true);
      Am_Abort_Interactor(sel_inter);
    }
    else return; //still not big enough
  }
  //if get here, then moved enough, so move the feedback 
  if (!feedback.Valid()) feedback = object_modified; //this shouldn't happen
  Am_Modify_Object_Pos(feedback, data, false, inter);
}

Am_Define_Method(Am_Current_Location_Method, void, sel_move_do,
		 (Am_Object inter, Am_Object object_modified,
		  Am_Inter_Location data)) {
  //if not moved far enough yet, don't call the do method
  bool moved_enough = inter.Get (Am_MOVED_ENOUGH);
  // cout << "Selection move stop, moved_enough = " << moved_enough
  //    << endl << flush;
  if (moved_enough) {
    //if moving fake-rect, then need to move the individual objects
    int orig_left, orig_top, final_left, final_top;
    Am_Value_List list;
    Am_Object fake_group, widget;
    widget = inter.Get_Owner();
    list = widget.Get(Am_VALUE);
    if (list.Length() > 1) {
      fake_group = widget.Get_Part(Am_FAKE_GROUP);
      orig_left = fake_group.Get(Am_LEFT);
      orig_top = fake_group.Get(Am_TOP);
    }
    // move real object or fake rect
    // this code copied from Am_Move_Grow_Do
    Am_Object feedback;
    feedback = inter.Get(Am_FEEDBACK_OBJECT);
    if (feedback.Valid ()) 
      feedback.Set(Am_VISIBLE, false);
    if (object_modified.Valid ()) {
      bool growing = inter.Get(Am_GROWING);
      Am_Modify_Object_Pos(object_modified, data, growing, inter);
    }
    if (list.Length() > 1) {
      final_left = fake_group.Get(Am_LEFT);
      final_top = fake_group.Get(Am_TOP);
      adjust_all_objects_position(widget, inter, list, final_left - orig_left,
				  final_top - orig_top, final_left, final_top);
    }
    else { // have moved the object, but still have to set up the commands
      Am_Inter_Location old_data;
      old_data = inter.Get(Am_OLD_VALUE);
      Am_Value objs_value, old_values_value, new_values_value;
      objs_value = object_modified;
      old_values_value = old_data;
      new_values_value = data;
      final_left = object_modified.Get(Am_LEFT);
      final_top = object_modified.Get(Am_TOP);
      set_commands_for_move(widget, inter, objs_value, old_values_value,
			    new_values_value, final_left, final_top);
    }
  }
  else Am_Abort_Interactor(inter); //make sure not queued
}


///////////////////////////////////////////////////////////////////////////
// For Undo Move/Grow
///////////////////////////////////////////////////////////////////////////

// goes in the Am_IMPLEMENTATION_PARENT slot of the command in the
// interactor to get the Am_MOVE_GROW_COMMAND of the widget
Am_Define_Object_Formula (get_owners_move_grow_command) {
  Am_Object command = 0;
  Am_Object inter = self.GV_Owner(); // owner will be interactor
  if (inter.Valid ()) {
    Am_Object widget = inter.GV_Owner(); // widget the interactor is in
    if (widget.Valid ()) {
      command = widget.GV_Part(Am_MOVE_GROW_COMMAND);
      if (command.Valid ()) {
	if(!command.Is_Instance_Of(Am_Command))
	  command = 0;// then command slot just contains a regular object
      }
    }
  }
  return command;
}

// set new_data_value based on the position(s) of obj(s)
void update_data_from_objs(Am_Value objs_value, Am_Value &new_data_value) {
  Am_Object obj, owner;
  int left, top;
  if (Am_Value_List::Test(objs_value)) {
    Am_Value_List objs_list, new_data_list;
    objs_list = objs_value;
    for(objs_list.Start(); !objs_list.Last(); objs_list.Next()) {
      obj = objs_list.Get();
      owner = obj.Get_Owner();
      left = obj.Get(Am_LEFT);
      top = obj.Get(Am_TOP);
      new_data_list.Add(Am_Inter_Location(false, owner, left, top, 0, 0));
    }
    new_data_value = new_data_list;
  }
  else if (Am_Object::Test(objs_value)) {
    obj = objs_value;
    owner = obj.Get_Owner();
    left = obj.Get(Am_LEFT);
    top = obj.Get(Am_TOP);
    Am_Inter_Location new_data(false, owner, left, top, 0, 0);
    new_data_value = new_data;
  }
  else Am_Error("objs not a list or object");
}

void update_objs_from_value(Am_Value objs_value, Am_Value new_data_value,
			    bool growing, Am_Object command_obj) {
  int min_left = 29999;
  int min_top  = 29999;
  Am_Object obj;
  Am_Inter_Location new_data;
  if (Am_Value_List::Test(objs_value)) {
    Am_Value_List objs_list, new_data_list;
    if (!Am_Value_List::Test(new_data_value))
      Am_Error("list of objs but not list of values");
  cout << "Length = " << (int)objs_list.Length() << endl << flush;
    objs_list = objs_value;
    new_data_list = new_data_value;
    if (objs_list.Length() != new_data_list.Length())
      Am_Error("Lists have different lengths");
    int cur_left, cur_top;
    for(objs_list.Start(), new_data_list.Start(); !objs_list.Last();
	objs_list.Next(), new_data_list.Next()) {
      obj = objs_list.Get();
      new_data = new_data_list.Get();
      Am_Modify_Object_Pos(obj, new_data, growing, command_obj);
      cur_left = obj.Get(Am_LEFT);
      cur_top = obj.Get(Am_TOP);
      min_left = IMIN(min_left, cur_left);
      min_top = IMIN(min_top, cur_top);
    }
  }
  else if (Am_Object::Test(objs_value)) {
    obj = objs_value;
    new_data = new_data_value;
    Am_Modify_Object_Pos(obj, new_data, growing, command_obj);
    min_left = obj.Get(Am_LEFT);
    min_top = obj.Get(Am_TOP);
  }
  else Am_Error("objs not a list or object");
  command_obj.Set(Am_LEFT, min_left);
  command_obj.Set(Am_TOP, min_top);
}

void adjust_all_objects_position_for_undo(Am_Value_List list, int left_offset,
					  int top_offset,
					  Am_Value_List &new_list) {
  Am_Object obj;
  Am_Object owner;
  new_list.Make_Empty();
  for(list.Start(); !list.Last(); list.Next()) {
    obj = list.Get();
    owner = obj.Get_Owner();
    int old_left = obj.Get(Am_LEFT);
    int old_top = obj.Get(Am_TOP);
    int new_left = old_left + left_offset;
    int new_top = old_top + top_offset;
    if (Am_Inter_Tracing(Am_INTER_TRACE_SETTING))
      cout << "++Setting " << obj << " left " << new_left << " top " << new_top
	   << endl << flush;
    
    obj.Set(Am_LEFT, new_left);
    obj.Set(Am_TOP, new_top);
    new_list.Add(Am_Inter_Location(false, owner, new_left, new_top, 0, 0));
  }
}

void update_new_objs_from_pos(Am_Value objs_value, int left, int top,
			      Am_Value &new_data_value,
			      bool growing, Am_Object command_obj) {
  if (growing) // then must be a single object, so new_data_value will be OK
    update_objs_from_value(objs_value, new_data_value, growing, command_obj);
  else {
    Am_Object obj;
    if (Am_Value_List::Test(objs_value)) {
      Am_Value_List objs_list, new_data_value_list;
      objs_list = objs_value;
      // calculate group offset and move
      int min_left, min_top, max_right, max_bottom;
      Am_Object owner;
      calculate_group_size(objs_list, min_left, min_top, max_right, max_bottom,
			   owner);
      adjust_all_objects_position_for_undo(objs_list, left - min_left,
					   top - min_top, new_data_value_list);
      new_data_value = new_data_value_list;
      command_obj.Set(Am_LEFT, min_left);
      command_obj.Set(Am_TOP, min_top);
    }
    else if (objs_value.type == Am_OBJECT) {
      obj = objs_value;
      obj.Set(Am_LEFT, left);
      obj.Set(Am_TOP, top);
      command_obj.Set(Am_LEFT, left);
      command_obj.Set(Am_TOP, top);
    }
  }
}
    

void sel_move_grow_general_undo_redo(Am_Object command_obj, bool undo,
				     bool selective, bool reload_data,
				     Am_Value objs_value) {
  Am_Object inter;
  inter = command_obj.Get(Am_SAVED_OLD_OWNER);
  
  if (reload_data) {
    command_obj.Set(Am_OBJECT_MODIFIED, objs_value);
    Am_Object parent;
    parent = command_obj.Get(Am_IMPLEMENTATION_PARENT);
    if (parent.Valid())
      parent.Set(Am_OBJECT_MODIFIED, objs_value);
  }
  else command_obj.Get(Am_OBJECT_MODIFIED, objs_value);
  
  if (inter.Valid () && Am_Inter_Tracing(inter)) {
    if (selective) cout << "Selective ";
    if (undo) cout << "Undo"; else cout << "repeat";
    cout << " command " << command_obj << " on " << objs_value <<endl << flush;
  }
  if (objs_value.Valid ()) {
    Am_Value old_data_value, new_data_value;

    command_obj.Get(Am_OLD_VALUE, old_data_value);
    if (!reload_data) command_obj.Get(Am_VALUE, new_data_value);
    bool growing = command_obj.Get(Am_GROWING);
    if (selective) {
      if (undo) update_data_from_objs(objs_value, new_data_value);
      else      update_data_from_objs(objs_value, old_data_value);
    }
    if (undo) {
      update_objs_from_value(objs_value, old_data_value, growing, command_obj);
      // swap current and old values, in case undo or undo-the-undo again
      command_obj.Set(Am_OLD_VALUE, new_data_value);
      command_obj.Set(Am_VALUE, old_data_value);
    }
    else {
      if (reload_data) {
	int left, top;
	left = command_obj.Get(Am_LEFT);
	top = command_obj.Get(Am_TOP);
	update_new_objs_from_pos(objs_value, left, top, new_data_value,
				 growing, command_obj);
	command_obj.Set(Am_LEFT, left);
	command_obj.Set(Am_TOP, top);
      }
      else update_objs_from_value(objs_value, new_data_value,
				  growing, command_obj);
      if (selective) command_obj.Set(Am_OLD_VALUE, old_data_value);
      if (reload_data) command_obj.Set(Am_VALUE, new_data_value);
    }
  }
}

Am_Define_Method(Am_Object_Method, void, sel_move_undo,
		 (Am_Object command_obj)) {
  sel_move_grow_general_undo_redo(command_obj, true, false, false, 0);
}
Am_Define_Method(Am_Object_Method, void, sel_move_selective_undo,
		 (Am_Object command_obj)){
  sel_move_grow_general_undo_redo(command_obj, true, true, false, 0);
}
Am_Define_Method(Am_Object_Method, void, sel_move_selective_repeat,
		 (Am_Object command_obj)){
  sel_move_grow_general_undo_redo(command_obj, false, true, false, 0);
}
Am_Define_Method(Am_Selective_Repeat_New_Method, void,
		 sel_move_selective_repeat_new,
		 (Am_Object command_obj, Am_Value new_sel)){
  sel_move_grow_general_undo_redo(command_obj, false, true, true, new_sel);
}
 

///////////////////////////////////////////////////////////////////////////
// For Growing
///////////////////////////////////////////////////////////////////////////

//x,y is w.r.t me
bool check_in_8_handles(Am_Object sel_handles, int x, int y) {
  int width = sel_handles.Get(Am_WIDTH);
  int height = sel_handles.Get(Am_HEIGHT);
  
  int width_d2 = width / 2;
  int height_d2 = height / 2;
  
  if (x <= HANDLE_SIZE) {
    // left-top
    if (y <= HANDLE_SIZE) return true;
    // left-middle
    if (y >= height_d2 - HANDLE_SIZE_D2 && y <= height_d2 + HANDLE_SIZE_D2)
      return true;
    // left-bottom
    if (y > height - HANDLE_SIZE) return true;
  }
  else if (x >= width_d2 - HANDLE_SIZE_D2 && x <= width_d2 + HANDLE_SIZE_D2) {
    // middle-top
    if (y <= HANDLE_SIZE) return true;
    // middle-bottom
    if (y > height - HANDLE_SIZE) return true;
  }
  else if (x > width - HANDLE_SIZE) {
    // right-top
    if (y <= HANDLE_SIZE) return true;
    // right-middle
    if (y >= height_d2 - HANDLE_SIZE_D2 && y <= height_d2 + HANDLE_SIZE_D2)
      return true;
    // right-bottom
    if (y > height - HANDLE_SIZE) return true;
  }
  //if get here, then not over a handle
  return false;
}

bool check_in_line_handles(Am_Object sel_handles, int x, int y) {
  int width = sel_handles.Get(Am_WIDTH);
  int height = sel_handles.Get(Am_HEIGHT);
  bool use_lt_rb = sel_handles.Get(Am_SELECTIONS_HANDLES_USE_LT_RB);
 // int width_d2 = width / 2;
 // int height_d2 = height / 2;		   

  if (use_lt_rb) {
    // left-top
    if (x <=  HANDLE_SIZE && y <= HANDLE_SIZE) return true;
    // right-bottom
    if (x > width - HANDLE_SIZE && y > height - HANDLE_SIZE) return true;
  }
  else {
    //left-bottom
    if (x <= HANDLE_SIZE && y > height - HANDLE_SIZE) return true;
    // right-top
    if (x > width - HANDLE_SIZE && y <= HANDLE_SIZE) return true;
  }
  //if get here, then not over a handle
  return false;
}


// object will be the widget.  
//this is used for growing objects.  Don't need to identify which
//handle it is over, since use Am_WHERE_HIT for grow function of
//move_grow_interactor!
// If over a handle, returns its for_obj
Am_Define_Method(Am_Where_Method, Am_Object,
		 selection_handles_where_function,
		 (Am_Object /* inter */, Am_Object object,
		  Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  //object is the whole sections_handles_widget
  // cout  << "Testing widget " << object << endl << flush;
  if (!Am_Point_In_All_Owners(object, x, y, event_window))
    return Am_No_Object;
  //For each of the selections handles I created:
  Am_Value_List parts;
  //handles map is in the Am_FEEDBACK_OBJECT slot of the widget
  parts = object.Get_Part(Am_FEEDBACK_OBJECT).Get(Am_GRAPHICAL_PARTS);
  Am_Object one_handle, for_obj;
  Am_Value value;
  int x1, y1;
  for(parts.Start(); !parts.Last(); parts.Next()) {
    one_handle = parts.Get();
    // cout  << "   Testing handles " << one_handle << endl << flush;
    //translate to coordinate inside of one_handle, so can test against 0
    //instead of left
    if (Am_Translate_Coordinates(event_window, x, y, one_handle, x1, y1)) {
      //first see if inside me at all
      if (Am_Point_In_Obj (one_handle, x1, y1, one_handle).Valid()) {
	// now see if over a handle
	for_obj = one_handle.Get(Am_ITEM);
	if (for_obj.Valid()) { //otherwise no handles
	  for_obj.Get(Am_AS_LINE, value);
	  bool as_line = value.Valid();
	  if (as_line) {
	    if (check_in_line_handles(one_handle, x1, y1))
	      return for_obj;
	  }
	  else {
	    if (check_in_8_handles(one_handle, x1, y1))
	      return for_obj;
	  }
	}
      }
    }
  }
  //if get here, not over a handle
  return Am_No_Object;
}


Am_Define_Method(Am_Selective_Repeat_New_Method, void, sel_grow_repeat_new,
		 (Am_Object /*command_obj*/, Am_Value /*new_sel*/)){
  Am_Beep();
  cout << "** Sorry Do Again Grow not supported \n" << flush;
}

Am_Define_String_Formula(move_or_grow_label) {
  bool growing = self.GV(Am_GROWING);
  if (growing) return Am_String("Grow");
  else return Am_String("Move");
}
  
//////////////////////////////////////////////////////////////////////////
// Select All Command
//////////////////////////////////////////////////////////////////////////

Am_Define_Method(Am_Object_Method, void, select_all_do, (Am_Object cmd)) {
  Am_Object sel_handles, main_group;
  sel_handles = cmd.Get(Am_SELECTION_WIDGET);
  if (sel_handles.Valid()) {
    main_group = sel_handles.Get(Am_OPERATES_ON);
    if (main_group.Valid()) {
      Am_Value old_value;
      Am_Value_List new_list;
      new_list = main_group.Get(Am_GRAPHICAL_PARTS); // ** may not be all **
      sel_handles.Get(Am_VALUE, old_value);
      cmd.Set(Am_OLD_VALUE, old_value);
      cmd.Set(Am_VALUE, new_list);
      sel_handles.Set(Am_VALUE, new_list);
      cmd.Set(Am_OBJECT_MODIFIED, sel_handles);
    }
  }
}

Am_Define_Value_Formula(get_sel_widgets_impl_parent) {
  Am_Object sel_handles, sel_handles_command;
  //value is the return value;
  value = Am_No_Value;
  sel_handles = self.GV(Am_SELECTION_WIDGET);
  if (sel_handles.Valid()) {
    sel_handles_command = sel_handles.GV_Part(Am_COMMAND);
    if (sel_handles_command.Valid()) {
      //value is the return value;
      sel_handles_command.GVM(Am_IMPLEMENTATION_PARENT, value);
    }
  }
  //cout << "impl parent " << value << " for cmd " << self << " widget "
  //     << sel_handles << " sel command " << sel_handles_command
  //     << endl << flush;
}
    

//////////////////////////////////////////////////////////////////////////
// Initialization
//////////////////////////////////////////////////////////////////////////


//exported objects
Am_Object Am_Selection_Widget = 0;
Am_Object Am_Selection_Widget_Select_All_Command = 0;

//internal objects
Am_Object Am_One_Selection_Handle = 0;

void Am_Selection_Widget_Initialize () {
  Am_Object inter, inter2, inter3;
  
  Am_Object_Advanced obj_adv; // to get at advanced features

  Am_One_Selection_Handle =
    Am_Graphical_Object.Create ("Am_One_Selection_Handle")
     .Set (Am_ITEM, NULL) //will be set by Am_Selection_Widget map
     .Set (Am_SELECTIONS_HANDLES_USE_LT_RB, selections_handles_use_lt_rb)
     .Set (Am_FILL_STYLE, NULL) //if empty, then uses owner's
     //Am_Selections_Handles_LTWH sets all the widget's sizes.
    // .Set (Am_LEFT, selections_handles_ltwh)
     .Set (Am_LEFT, selections_handles_lt)
     .Set (Am_WIDTH, selections_handles_wh)
     .Set (Am_DRAW_METHOD, selection_handles_draw)
     ;

  Am_Selection_Widget = Am_Group.Create("Am_Selection_Widget")
    //parameters
    .Set (Am_START_WHEN, Am_Input_Char("ANY_LEFT_DOWN"))
    .Set (Am_FILL_STYLE, Am_Black)
    .Set (Am_VALUE, Am_Value_List()) // set by the interactors or externally
    .Set (Am_ACTIVE, true) // 
    .Set (Am_OPERATES_ON, NULL) //fill in with group holding parts to select
    .Set (Am_START_WHERE_TEST, Am_Inter_In_Part)

    //internal slots
    .Set (Am_LEFT, 0) //must be zero!
    .Set (Am_TOP, 0)
    .Set (Am_WIDTH, width_from_owner_or_zero)
    .Set (Am_HEIGHT, height_from_owner_or_zero)
    .Set (Am_SET_COMMAND_OLD_OWNER, set_command_and_move_old_owner)
    .Add_Part (Am_LINE_FEEDBACK_OBJECT, Am_Line.Create("Sel_Line_Feedback")
	       .Set(Am_VISIBLE, false)
	       .Set(Am_LINE_STYLE, Am_Dashed_Line)
	       )
    .Add_Part (Am_RECT_FEEDBACK_OBJECT,
	       Am_Rectangle.Create("Sel_Rect_Feedback")
	       .Set(Am_VISIBLE, false)
	       .Set(Am_LINE_STYLE, Am_Dashed_Line)
	       .Set(Am_FILL_STYLE, Am_No_Style)
	       )
    .Add_Part (Am_FAKE_GROUP, Am_Rectangle.Create("Fake_group")
	       .Set (Am_VISIBLE, false)
	       )
    .Add_Part (Am_FEEDBACK_OBJECT, Am_Map.Create("Selection_Handles")
	       .Add_Part (Am_ITEM_PROTOTYPE,
			  Am_One_Selection_Handle.Create ("Sel_Handle_Proto")
			  )
	       //controls the item_prototypes
	       .Set (Am_ITEMS, Am_From_Owner(Am_VALUE))
	       // components lay out themselves.
	       .Set (Am_LAYOUT, NULL) 
	       )
    // components lay out themselves.  This just makes
    // checks for objects to go invalid or invisible
    .Set (Am_LAYOUT, remove_from_value_if_invalid)
    .Add_Part (Am_INTERACTOR,
	       inter = Am_One_Shot_Interactor.Create("inter_in_sel_widget")
	       .Set (Am_PRIORITY, 2.0) //so higher than move
	       .Set (Am_HOW_SET, (int)Am_CHOICE_SET)
	       .Set (Am_START_WHERE_TEST, owner_start_where_or_none)
	       .Set (Am_START_WHEN, Am_From_Owner (Am_START_WHEN)) 
	       .Set (Am_RUN_ALSO, true) //so move will also run afterwards
	       .Set (Am_ACTIVE, Am_From_Owner (Am_ACTIVE))

	       //use the standard choice_start_do
	       .Set(Am_INTERIM_DO_METHOD, NULL)
	       .Set(Am_ABORT_DO_METHOD, NULL)
	       .Set(Am_DO_METHOD, sel_object)
	       )
    .Add_Part (Am_MOVE_INTERACTOR,
	       inter2 = Am_Move_Grow_Interactor.Create("move_inter_in_handle")
	       .Set (Am_START_WHERE_TEST, owner_start_where_or_fake)
	       .Set (Am_FEEDBACK_OBJECT, Am_Compute_MG_Feedback_Object)
	       .Set (Am_START_WHEN, Am_From_Owner (Am_START_WHEN))
	       .Set (Am_ACTIVE, Am_From_Owner (Am_ACTIVE))

	       //use standard start and abort methods
	       .Set(Am_INTERIM_DO_METHOD, sel_move_inter_interim_do)
	       .Set(Am_DO_METHOD, sel_move_do)
	       )
    .Add_Part (Am_GROW_INTERACTOR,
	       inter3 = Am_Move_Grow_Interactor.Create("grow_inter_in_handle")
	       .Set (Am_START_WHERE_TEST, selection_handles_where_function)
	       .Set (Am_GROWING, true)
	       .Set (Am_START_WHEN, compute_mg_start_when)
	       .Set (Am_FEEDBACK_OBJECT, Am_Compute_MG_Feedback_Object)
	       .Set (Am_PRIORITY, 40.0) //so higher than the others
	       )
    .Add_Part (Am_COMMAND, Am_Command.Create("Selection_Command")
	       .Set(Am_IMPLEMENTATION_PARENT, Am_NOT_USUALLY_UNDONE)
	       .Set(Am_LABEL, "Select")
	       )
    .Add_Part (Am_MOVE_GROW_COMMAND, Am_Command.Create("Move_Grow_Command")
	       .Set(Am_GROWING, false)
	       .Set(Am_LABEL, move_or_grow_label)
	       )
    ;

  obj_adv = (Am_Object_Advanced&)Am_Selection_Widget;
  obj_adv.Get_Slot (Am_FILL_STYLE).Set_Demon_Bits (Am_STATIONARY_REDRAW |
						   Am_EAGER_DEMON);
  //selection inter; all the work done by the command
  inter.Get_Part(Am_IMPLEMENTATION_COMMAND)
    .Set_Name("internal_command_in_sel_widget")
    .Set (Am_UNDO_METHOD, NULL)
    .Set (Am_REDO_METHOD, NULL)
    .Set (Am_SELECTIVE_UNDO_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_SAME_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL);

  inter.Get_Part(Am_COMMAND)
    .Set_Name("selection_command_in_sel_widget")
    .Set(Am_IMPLEMENTATION_PARENT, Am_Get_Owners_Command)
    .Set(Am_UNDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_REDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, Am_Widget_Inter_Command_Selective_Undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD,
	 Am_Widget_Inter_Command_Selective_Repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    ;
    
  inter2.Get_Part(Am_IMPLEMENTATION_COMMAND)
    .Set (Am_UNDO_METHOD, NULL)
    .Set (Am_REDO_METHOD, NULL)
    .Set (Am_SELECTIVE_UNDO_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_SAME_METHOD, NULL)
    .Set (Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL);
    ;

  inter2.Get_Part(Am_COMMAND)
    .Set(Am_IMPLEMENTATION_PARENT, get_owners_move_grow_command)
    .Set(Am_START_DO_METHOD, sel_move_start_do)
    .Set(Am_LABEL, "Move")
    .Set(Am_GROWING, false)
    .Set(Am_UNDO_METHOD, sel_move_undo)
    .Set(Am_REDO_METHOD, sel_move_undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, sel_move_selective_undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD, sel_move_selective_repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, sel_move_selective_repeat_new)
    .Set_Name("move_command_in_sel_widget")
    ;
  
  inter3.Get_Part(Am_IMPLEMENTATION_COMMAND)
    .Set (Am_SELECTIVE_REPEAT_NEW_ALLOWED,
	  Am_Selective_New_Allowed_Return_False) //**NIY**
    .Set (Am_SELECTIVE_REPEAT_ON_NEW_METHOD, sel_grow_repeat_new);
  
  inter3.Get_Part(Am_COMMAND)
    .Set(Am_LABEL, "Grow")
    .Set(Am_IMPLEMENTATION_PARENT, get_owners_move_grow_command)
    .Set(Am_DO_METHOD, copy_values_for_grow)
    .Set_Name("grow_command_in_sel_widget")
    ;

  Am_Selection_Widget_Select_All_Command =
    Am_Command.Create("Select_All_Command")
    .Set (Am_LABEL, "Select All")
    .Set (Am_SELECTION_WIDGET, NULL) //set this to associated sel..han..widget
    .Set (Am_DO_METHOD, select_all_do)
    .Set(Am_UNDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_REDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, Am_Widget_Inter_Command_Selective_Undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD,
	 Am_Widget_Inter_Command_Selective_Repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    .Set(Am_IMPLEMENTATION_PARENT, get_sel_widgets_impl_parent)
    .Set(Am_ACCELERATOR, (long)Am_Input_Char("CONTROL_a"))
    ;
}

