/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains the standard widgets
   
   Designed and implemented by Brad Myers
*/

#include <am_inc.h>

#include AM_IO__H

#include WIDGETS_ADVANCED__H
#include STANDARD_SLOTS__H
#include VALUE_LIST__H
#include OBJECT_ADVANCED__H  // for Am_Slot_Advanced
#include OPAL_ADVANCED__H  // for Am_DRAWONABLE, Am_Window_Coordinate
#include INTER_ADVANCED__H  // for inter_tracing

#include GEM__H
#include INTER__H
#include OPAL__H
#include WIDGETS__H

////////////////////////////////////////////////////////////////////////
//  Color Utility Functions
////////////////////////////////////////////////////////////////////////

//  Values for changing the brightness of the foreground-color for MOTIF
#define MOTIF_BACKGROUND_VALUE 0.85f
#define MOTIF_SHADOW_VALUE 0.24f
#define MOTIF_HIGHLIGHT_VALUE 1.45f

Am_Define_Style_Formula (Am_Default_Motif_Fill_Style)
{
  Am_Object win;
  win = self.GV(Am_WINDOW);
  if (win.Valid()) {
    bool color = win.GV(Am_IS_COLOR);
    if (color) return Am_Amulet_Purple;
    else return Am_White;
  }
  else return Am_Amulet_Purple;
}
  

inline float fmax (float f1, float f2) {if (f1>f2) return f1; else return f2;}
inline float fmin (float f1, float f2) {if (f1>f2) return f2; else return f1;}

// This function is used to compute the shades of the Motif colors in the
// Motif gadgets.  Given a color and a brightness adjustment, the function
// returns a color which is "brighter" or "dimmer" based on the adjustment
// factor.
//        | Y |   | .3   .59  .11 | | R |
//        | I | = | .6  -.28 -.32 | | G |       ( Y = Brightness )
//        | G |   | .21 -.52  .31 | | B |
// Given the RGB values, multiply by the above matrix to get YIG values.
// Multiply Y by the adjustment to get the new brightness, then multiply
// the new YIG matrix by the inverse of the original 3x3 matrix.
void convert_colors (float red, float green, float blue, float adjustment,
			float &new_red, float &new_green, float &new_blue) {
  float y = (red * 0.3f) + (green * 0.59f) + (blue * 0.11f);
  float i = (red * 0.6f) + (green * -0.28f) + (blue * -0.32f);
  float q = (red * 0.21f) + (green * -0.52f) + (blue * 0.31f);
  float new_brightness = y * adjustment;
  new_red  = fmax (0.0f, fmin(1.0f, new_brightness + (i *  0.95f) + (q *  0.62f)));
  new_green= fmax (0.0f, fmin(1.0f, new_brightness + (i * -0.28f) + (q * -0.64f)));
  new_blue = fmax (0.0f, fmin(1.0f, new_brightness + (i * -1.10f) + (q *  1.70f)));
}


Am_Style make_converted_style (const Am_Style& old, float adjustment)
{
  float old_red, old_green, old_blue;
  float new_red, new_green, new_blue;
  old.Get_Values(old_red, old_green, old_blue);
  convert_colors(old_red, old_green, old_blue, adjustment,
		 new_red, new_green, new_blue);
  // new color, and line_thickness of 2
  return Am_Style (new_red, new_green, new_blue, 2);
}

Am_WRAPPER_DATA_IMPL (Computed_Colors_Record, (this))

//constructor
Computed_Colors_Record_Data::
    Computed_Colors_Record_Data (const Am_Style& foreground)
{
  refs = 1;

  if (foreground == Am_No_Style) {
    foreground_style = Am_No_Style;
    background_style = Am_No_Style;
    shadow_style     = Am_No_Style;
    highlight_style  = Am_No_Style;
    menu_top_line_style = Am_No_Style;
    menu_bottom_line_style = Am_No_Style;
  }
  else {
    // need to convert foreground to make sure it has a line-thickness of 2
    foreground_style = make_converted_style (foreground, 1.0f);
    background_style = make_converted_style (foreground,
					     MOTIF_BACKGROUND_VALUE);
    shadow_style     = make_converted_style (foreground, MOTIF_SHADOW_VALUE);
    highlight_style  = make_converted_style (foreground,
					     MOTIF_HIGHLIGHT_VALUE);

// a menu line is two thickness-1 lines next to each other; one highlighted,
// one shadowed.
    float red, green, blue;
    shadow_style.Get_Values(red, green, blue);
    menu_top_line_style = Am_Style (red, green, blue);
    highlight_style.Get_Values(red, green, blue);
    menu_bottom_line_style = Am_Style (red, green, blue);
  }
}

Computed_Colors_Record_Data::
    Computed_Colors_Record_Data (Computed_Colors_Record_Data* prev)
{
  refs = 1;

  foreground_style = prev->foreground_style;
  background_style = prev->background_style;
  shadow_style     = prev->shadow_style;
  highlight_style  = prev->highlight_style;
  menu_top_line_style = prev->menu_top_line_style;
  menu_bottom_line_style = prev->menu_bottom_line_style;
}

//////// define the hash table from a style to a Computed_Colors_Record //////

Am_IMPL_MAP (Style2MotifRec, Am_Wrapper*, NULL,
	     Computed_Colors_Record_Data*, NULL)

////////////////////////////////////////////////////////////////////////
// Global color map
////////////////////////////////////////////////////////////////////////

Am_Map_Style2MotifRec computed_colors_map;
Computed_Colors_Record black_and_white_rec;
Computed_Colors_Record_Data black_and_white_rec_data;
Am_Style Am_Motif_Inactive_Stipple;
Am_Style Am_Key_Border_Line; //black, thickness=2

Computed_Colors_Record::
    Computed_Colors_Record (const Am_Style& foreground_color)
{
  Am_Wrapper* foreground_wrapper = foreground_color;
  Computed_Colors_Record_Data* rec =
      computed_colors_map.GetAt (foreground_wrapper);
  if (!rec) {
    rec = new Computed_Colors_Record_Data (foreground_color);
    if (foreground_wrapper)
      foreground_wrapper->Note_Reference ();
    computed_colors_map[foreground_wrapper] = rec;
  }
  else
    rec->Note_Reference ();
  if (foreground_wrapper)
    foreground_wrapper->Release ();
  data = rec;
}

Computed_Colors_Record_Data::~Computed_Colors_Record_Data ()
{
  Am_Wrapper* key = computed_colors_map.GetAt (this);
  if (key)
    key->Release ();
  computed_colors_map.DeleteKey (this);
}

Am_Define_Formula (Am_Wrapper*, Am_Get_Computed_Colors_Record_Form)
{
  Am_Style foreground_color;
  Am_Value value;
  bool is_color = true;
  self.GVM (Am_FILL_STYLE, value);
  if (Am_Type_Class (value.type) == Am_WRAPPER)
    foreground_color = value;
  Am_Object window;
  window = self.GV(Am_WINDOW);
  if (window.Valid ()) {
    Am_Drawonable* draw = Am_Drawonable::Narrow (window.GV (Am_DRAWONABLE));
    is_color = !draw || draw->Is_Color();
  }
  if (is_color)
    return Computed_Colors_Record (foreground_color);
  else
    return black_and_white_rec;
}

///////////////////////////////////////////////////////////////////////////
// Accelerators
///////////////////////////////////////////////////////////////////////////

//matches on the accelerator character
static Am_Object accel_match(Am_Value_List accel_list, Am_Input_Char target_ic,
			     Am_Object window) {
  Am_Input_Char accel_char;
  long accel;
  Am_Object cmd, ret;
  bool changed = false;
  for (accel_list.Start (); !accel_list.Last (); accel_list.Next ()) {
    cmd = accel_list.Get();
    if (cmd.Valid()) {
      accel = cmd.Get(Am_ACCELERATOR);
      if (accel > 1) {
	accel_char = Am_Input_Char::Narrow(accel);
	if (accel_char == target_ic) {
	  ret = cmd;
	  break; //leave loop
	}
      }
    }
    else {
      accel_list.Delete(); //if cmd is no longer valid, remove it from list
      changed = true;
    }
  }
  if (changed && window.Valid())
    window.Set(Am_ACCELERATOR_LIST, accel_list);
  return ret;
}

static void check_for_conflicts(Am_Object window, Am_Value_List accel_list,
				Am_Object command) {
  Am_Input_Char target_ic;
  long accel;
  accel = command.Get(Am_ACCELERATOR);
  if (accel > 1) {
    target_ic = Am_Input_Char::Narrow(accel);
    Am_Object old_cmd = accel_match(accel_list, target_ic, Am_No_Object);
    if (old_cmd.Valid()) {
      cerr << "** Amulet Error, In window " << window
	   << " accelerator character " << target_ic
	   << " for new command " << command;
      Am_Object widget;
      widget = command.Get(Am_SAVED_OLD_OWNER);
      if (widget.Valid()) cerr << " in widget " << widget;
      cerr << " conflicts with accelerator on existing command " << old_cmd;
      widget = old_cmd.Get(Am_SAVED_OLD_OWNER);
      if (widget.Valid()) cerr << " in widget " << widget;
      cerr << endl << flush;
      Am_Error();
    }
  }
}

// widget has a command in its Am_COMMAND slot with an event in the
// Am_ACCELERATOR slot of the command.  Add the widget to the widget's
// window's global list of accelerators.
void Am_Add_Accelerator_Command_To_Window(Am_Object command,
					  Am_Object window) {
  Am_Object inter;
  Am_Value_List accel_list;
  Am_Value value;
  if (window.Valid()) {
    inter = window.Get_Part(Am_ACCELERATOR_INTER);
    if (!inter.Valid()) {
      Am_INTER_TRACE_PRINT(window, "Creating accel inter for "
			   << window);
      inter = Am_Accelerator_Inter.Create();
      window.Add_Part(Am_ACCELERATOR_INTER, inter);
    }
    window.Get(Am_ACCELERATOR_LIST, value);
    if (value.Valid() && Am_Value_List::Test(value))
      accel_list = value;
    accel_list.Start();
    if (!accel_list.Member(command)) {
      check_for_conflicts(window, accel_list, command);
      accel_list.Add(command);
      Am_INTER_TRACE_PRINT(inter, "Adding " << command <<
			   " to accel list for " << window);
      window.Set(Am_ACCELERATOR_LIST, accel_list);
    }
  }
}

//OK if not there
void Am_Remove_Accelerator_Command_From_Window(Am_Object command,
					       Am_Object window) {
  Am_Value_List accel_list;
  Am_Value value;
  if (window.Valid()) {
    window.Get(Am_ACCELERATOR_LIST, value);
    if (value.Valid() && Am_Value_List::Test(value)) {
      accel_list = value;
      accel_list.Start();
      if (accel_list.Member(command)) {
	accel_list.Delete();
	window.Set(Am_ACCELERATOR_LIST, accel_list);
	Am_INTER_TRACE_PRINT(window, "Removing " << command <<
			     " from accel list of " << window);
      }
    }
  }
}

Am_Define_Method(Am_Where_Method, Am_Object, accelerator_start_where,
		 (Am_Object inter, 
		  Am_Object /* object */, Am_Object event_window,
		  Am_Input_Char ic, int /* x */, int /* y */)) {
  Am_INTER_TRACE_PRINT(inter, "testing " << ic << " as accelerator for "
		       << event_window);
  Am_Value_List accel_list;
  Am_Value value;
  event_window.Get(Am_ACCELERATOR_LIST, value);
  if (value.Valid() && Am_Value_List::Test(value)) {
    accel_list = value;
    Am_Object cmd = accel_match(accel_list, ic, event_window);
    if (cmd.Valid()) {
      Am_INTER_TRACE_PRINT(inter, "Found accelerator " << cmd);
      if ((bool) cmd.Get(Am_ACTIVE)) {
	inter.Set(Am_ACCELERATOR, cmd);
	return event_window;
      }
      else {
	Am_INTER_TRACE_PRINT(inter, "-> but not active");
      }
    }
  }
  //if get here, then fail
  return Am_No_Object;
}

Am_Define_Method(Am_Object_Method, void, do_accelerator_action,
		 (Am_Object cmd)) {
  Am_Object inter, accel_cmd;
  inter = cmd.Get_Owner();
  if (inter.Valid())
    accel_cmd = inter.Get(Am_ACCELERATOR);
  cmd.Set(Am_IMPLEMENTATION_PARENT, accel_cmd);
  Am_INTER_TRACE_PRINT(cmd, "Executing accelerator for " << accel_cmd);
}
      
///////////////////////////////////////////////////////////////////////////
// Simple bordered rectangle
///////////////////////////////////////////////////////////////////////////

// Draws the "raised" or "depressed" box that is ubiquitous in Motif.
void Am_Draw_Motif_Box (int left, int top, int width, int height,
		        bool depressed,
			const Computed_Colors_Record& rec, Am_Drawonable* draw)
{
  Am_Style top_fill;
  Am_Style bot_fill;
  Am_Style inside_fill;
  if (depressed) {
    top_fill = rec.data->shadow_style;
    bot_fill = rec.data->highlight_style;
    inside_fill = rec.data->background_style;
  }
  else {
    top_fill = rec.data->highlight_style;
    bot_fill = rec.data->shadow_style;
    inside_fill = rec.data->foreground_style;
  }
  //top edges
  draw->Draw_Rectangle(Am_No_Style, top_fill, left, top, width, height);
  //bottom edges
  draw->Draw_Rectangle(Am_No_Style, bot_fill, left+2, top+2, width-2,
		       height-2);
  //inside of box
  draw->Draw_Rectangle(Am_No_Style, inside_fill, left+2, top+2, width-4,
		       height-4);
}

// draw routine for a plain border'ed rectangle
Am_Define_Method(Am_Draw_Method, void, border_rectangle_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool selected = self.Get (Am_SELECTED);
  Computed_Colors_Record rec = self.Get(Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Box(left, top, width, height, selected, rec, drawonable);
  else
    Am_Error("Sorry, only the Motif Style implemented for now");
}


////////////////////////////////////////////////////////////////////////
//  Starting, Stopping and Aborting Widgets
////////////////////////////////////////////////////////////////////////

Am_Define_Method_Type_Impl(Am_Explicit_Widget_Run_Method);

void Am_Start_Widget(Am_Object widget, Am_Value initial_value) {
  Am_Value v;
  widget.Get(Am_WIDGET_START_METHOD, v);
  if (v.Valid()) {
    Am_Explicit_Widget_Run_Method method = v;
    method.Call(widget, initial_value);
  }
  else Am_ERROR("Widget " << widget << " has no start method");
}

//Explicitly abort an interactor.  If not running, does nothing.  This
//makes sure the Widget's commands are not entered into the command
//history, even if this is called from the widget command's DO method.

void Am_Abort_Widget(Am_Object widget_or_inter_or_command) {
  Am_Object widget, inter, command;
  if (widget_or_inter_or_command.Is_Instance_Of(Am_Command)) {
    command = widget_or_inter_or_command;
    widget_or_inter_or_command = command.Get(Am_SAVED_OLD_OWNER);
  }
  if (widget_or_inter_or_command.Valid()) {
    if (widget_or_inter_or_command.Is_Instance_Of(Am_Interactor)) {
      inter = widget_or_inter_or_command;
      Am_Abort_Interactor(inter);
    }
    else { //hope it is a widget with a method
      widget = widget_or_inter_or_command;
      Am_Value v;
      widget.Get(Am_WIDGET_ABORT_METHOD, v);
      if (v.Valid()) {
	Am_Object_Method method = v;
	method.Call(widget);
      }
    }
  }
  //make sure the command isn't entered into the history
  if (command.Valid())
    command.Set(Am_COMMAND_IS_ABORTING, true);
}

//Explicitly stop a widget.  If final_value is supplied, then this is
//the value used as the value of 
//the widget.  If final_value is not supplied, the widget uses its current
//value.  If the widget is running, commands associated with the
//widget will be invoked just as if the widget had terminated normally.
void Am_Stop_Widget(Am_Object widget, Am_Value final_value) {
  Am_Value v;
  widget.Get(Am_WIDGET_STOP_METHOD, v);
  if (v.Valid()) {
    Am_Explicit_Widget_Run_Method method = v;
    method.Call(widget, final_value);
  }
}

Am_Define_Method(Am_Explicit_Widget_Run_Method, void,
		 Am_Standard_Widget_Start_Method,
		 (Am_Object widget, Am_Value initial_value)) {
  if (initial_value.Valid()) widget.Set(Am_VALUE, initial_value);
  Am_Object inter = widget.Get_Part(Am_INTERACTOR);
  Am_Start_Interactor(inter);
}

Am_Define_Method(Am_Explicit_Widget_Run_Method, void,
		 Am_Standard_Widget_Stop_Method,
		 (Am_Object widget, Am_Value final_value)) {
  if (final_value.Valid()) widget.Set(Am_VALUE, final_value);
  Am_Object inter = widget.Get_Part(Am_INTERACTOR);
  Am_Stop_Interactor(inter);
}


Am_Define_Method(Am_Object_Method, void,
		 Am_Standard_Widget_Abort_Method, (Am_Object widget)) {
  Am_Object inter = widget.Get_Part(Am_INTERACTOR);
  Am_Abort_Interactor(inter);
  //usually, interactor already stopped, so copy the old value from the widget
  Am_Value old_value;
  inter.Get_Part(Am_COMMAND).Get(Am_OLD_VALUE, old_value);
  widget.Set(Am_VALUE, old_value);
}



///////////////////////////////////////////////////////////////////////////
// Create all the widgets
///////////////////////////////////////////////////////////////////////////

// Standard slots of interactors and command objects
void widgets_set_slot_names () {
#ifdef DEBUG
  Am_Register_Slot_Key (Am_FINAL_FEEDBACK_WANTED, "FINAL_FEEDBACK_WANTED");
  Am_Register_Slot_Key (Am_KEY_SELECTED, "KEY_SELECTED");
  Am_Register_Slot_Key (Am_ACTIVE_2, "ACTIVE_2");
  Am_Register_Slot_Key (Am_ITEM_OFFSET, "ITEM_OFFSET");
  Am_Register_Slot_Key (Am_TEXT_OFFSET, "TEXT_OFFSET");
  Am_Register_Slot_Key (Am_WIDGET_LOOK, "WIDGET_LOOK");
  Am_Register_Slot_Key (Am_INTERACTOR, "~INTERACTOR~");
  Am_Register_Slot_Key (Am_STYLE_RECORD, "~STYLE_RECORD~");
  Am_Register_Slot_Key (Am_REAL_STRING_OR_OBJ, "~REAL_STRING_OR_OBJ~");
  Am_Register_Slot_Key (Am_ATTACHED_OBJECT, "~ATTACHED OBJECT~");
  Am_Register_Slot_Key (Am_ATTACHED_COMMAND, "~ATTACHED COMMAND~");
  Am_Register_Slot_Key (Am_LABEL_OR_ID, "~LABEL_OR_ID~");
  Am_Register_Slot_Key (Am_ITEM_TO_COMMAND, "~ITEM_TO_COMMAND~");
  Am_Register_Slot_Key (Am_BOX_ON_LEFT, "BOX_ON_LEFT");
  Am_Register_Slot_Key (Am_BOX_HEIGHT, "BOX_HEIGHT");
  Am_Register_Slot_Key (Am_BOX_WIDTH, "BOX_WIDTH");

  Am_Register_Slot_Key (Am_VALUE_1, "VALUE_1");
  Am_Register_Slot_Key (Am_VALUE_2, "VALUE_2");
  Am_Register_Slot_Key (Am_SMALL_INCREMENT, "SMALL_INCREMENT");
  Am_Register_Slot_Key (Am_LARGE_INCREMENT, "LARGE_INCREMENT");
  Am_Register_Slot_Key (Am_PERCENT_VISIBLE, "PERCENT_VISIBLE");
  Am_Register_Slot_Key (Am_SCROLL_ARROW_DIRECTION, "ARROW_DIRECTION");
  Am_Register_Slot_Key (Am_SCROLL_AREA_MIN, "SCROLL_AREA_MIN");
  Am_Register_Slot_Key (Am_SCROLL_AREA_MAX, "SCROLL_AREA_MAX");
  Am_Register_Slot_Key (Am_SCROLL_AREA_SIZE, "SCROLL_AREA_SIZE");
  Am_Register_Slot_Key (Am_SCROLL_ARROW1, "~SCROLL_ARROW1~");
  Am_Register_Slot_Key (Am_SCROLL_ARROW2, "~SCROLL_ARROW2~");
  Am_Register_Slot_Key (Am_SCROLL_INDICATOR, "~SCROLL_INDICATOR~");
  Am_Register_Slot_Key (Am_ARROW_INTERACTOR, "~ARROW_INTERACTOR~");
  Am_Register_Slot_Key (Am_BACKGROUND_INTERACTOR, "~BACKGROUND_INTERACTOR~");

  Am_Register_Slot_Key (Am_REAL_WIDTH, "~REAL_WIDTH~");
  Am_Register_Slot_Key (Am_REAL_HEIGHT, "~REAL_HEIGHT~");
  Am_Register_Slot_Key (Am_SUB_MENU, "~SUB_MENU~");
  Am_Register_Slot_Key (Am_FOR_ITEM, "~FOR_ITEM~");
  Am_Register_Slot_Key (Am_COMPUTE_INTER_VALUE, "~COMPUTE_INTER_VALUE~");
  Am_Register_Slot_Key (Am_WEB_CONSTRAINT, "~WEB_CONSTRAINT~");
  Am_Register_Slot_Key (Am_TEXT_WIDGET_BOX_LEFT, "~TEXT_WIDGET_BOX_LEFT~");
  Am_Register_Slot_Key (Am_TEXT_WIDGET_TEXT_OBJ, "~TEXT_WIDGET_TEXT_OBJ~");

  Am_Register_Slot_Key (Am_SELECTIONS_HANDLES_USE_LT_RB,
			  "~SELECTIONS_HANDLES_USE_LT_RB~");
  Am_Register_Slot_Key (Am_MOVE_INTERACTOR, "~MOVE_INTERACTOR~");
  Am_Register_Slot_Key (Am_GROW_INTERACTOR, "~GROW_INTERACTOR~");
  Am_Register_Slot_Key (Am_LINE_FEEDBACK_OBJECT, "~LINE_FEEDBACK_OBJECT~");
  Am_Register_Slot_Key (Am_RECT_FEEDBACK_OBJECT, "~RECT_FEEDBACK_OBJECT~");
  Am_Register_Slot_Key (Am_MOVED_ENOUGH, "~MOVED_ENOUGH~");
  Am_Register_Slot_Key (Am_FAKE_GROUP, "~FAKE_GROUP~");
  Am_Register_Slot_Key (Am_ACCELERATOR_STRING, "~ACCELERATOR_STRING~");
  Am_Register_Slot_Key (Am_ACCELERATOR_INTER, "~ACCELERATOR_INTER~");
  Am_Register_Slot_Key (Am_ACCELERATOR_LIST, "~ACCELERATOR_LIST~");

  Am_Register_Slot_Key (Am_H_SCROLL_BAR, "H_SCROLL_BAR");
  Am_Register_Slot_Key (Am_V_SCROLL_BAR, "V_SCROLL_BAR");
  Am_Register_Slot_Key (Am_H_SCROLL_BAR_ON_TOP, "H_SCROLL_BAR_ON_TOP");
  Am_Register_Slot_Key (Am_V_SCROLL_BAR_ON_LEFT, "V_SCROLL_BAR_ON_LEFT");
  Am_Register_Slot_Key (Am_H_SMALL_INCREMENT, "H_SMALL_INCREMENT");
  Am_Register_Slot_Key (Am_H_LARGE_INCREMENT, "H_LARGE_INCREMENT");
  Am_Register_Slot_Key (Am_V_SMALL_INCREMENT, "V_SMALL_INCREMENT");
  Am_Register_Slot_Key (Am_V_LARGE_INCREMENT, "V_LARGE_INCREMENT");
  Am_Register_Slot_Key (Am_INNER_WIDTH, "INNER_WIDTH");
  Am_Register_Slot_Key (Am_INNER_HEIGHT, "INNER_HEIGHT");
  Am_Register_Slot_Key (Am_CLIP_LEFT, "~CLIP_LEFT~");
  Am_Register_Slot_Key (Am_CLIP_TOP, "~CLIP_TOP~");
  Am_Register_Slot_Key (Am_CLIP_WIDTH, "~CLIP_WIDTH~");
  Am_Register_Slot_Key (Am_CLIP_HEIGHT, "~CLIP_HEIGHT~");
  Am_Register_Slot_Key (Am_H_SCROLLER, "~H_SCROLLER~");
  Am_Register_Slot_Key (Am_V_SCROLLER, "~V_SCROLLER~");
  Am_Register_Slot_Key (Am_INNER_FILL_STYLE, "INNER_FILL_STYLE");
  Am_Register_Slot_Key (Am_LABEL_FONT, "LABEL_FONT");
  Am_Register_Slot_Key (Am_OPERATES_ON, "OPERATES_ON");
  Am_Register_Slot_Key (Am_MOVE_GROW_COMMAND, "~MOVE_GROW_COMMAND~");
  Am_Register_Slot_Key (Am_SELECTION_WIDGET, "SELECTION_WIDGET");
  Am_Register_Slot_Key (Am_OBJECT_MODIFIED_PLACES, "~OBJECT_MODIFIED_PLACES~");
  Am_Register_Slot_Key (Am_GET_WIDGET_VALUE_METHOD,
			"GET_WIDGET_VALUE_METHOD");
  Am_Register_Slot_Key (Am_GET_OBJECT_VALUE_METHOD,
			"GET_OBJECT_VALUE_METHOD");
  Am_Register_Slot_Key (Am_SET_OBJECT_VALUE_METHOD,
			"SET_OBJECT_VALUE_METHOD");
  Am_Register_Slot_Key (Am_SLOT_FOR_VALUE, "SLOT_FOR_VALUE");
  Am_Register_Slot_Key (Am_CLIPBOARD, "CLIPBOARD");
  Am_Register_Slot_Key (Am_OLD_CLIPBOARD_OBJECTS, "~OLD_CLIPBOARD_OBJECTS~");
  Am_Register_Slot_Key (Am_CREATED_GROUP, "~CREATED_GROUP~");
  Am_Register_Slot_Key (Am_WIDGET_START_METHOD, "~WIDGET_START_METHOD~");
  Am_Register_Slot_Key (Am_WIDGET_ABORT_METHOD, "~WIDGET_ABORT_METHOD~");
  Am_Register_Slot_Key (Am_WIDGET_STOP_METHOD, "~WIDGET_STOP_METHOD~");

  Am_Register_Slot_Key (Am_DIALOG_BUTTONS, "~DIALOG_BUTTONS~");
  Am_Register_Slot_Key (Am_TEXT_WIDGET, "~TEXT_WIDGET~");
  Am_Register_Slot_Key (Am_DIALOG_GROUP, "~DIALOG_GROUP~");
  Am_Register_Slot_Key (Am_ICON_IN_ABOUT_DB, "~ICON_IN_ABOUT_DB~");

  Am_Register_Slot_Key (Am_VALID_INPUT, "VALID_INPUT");

#endif
}

// exported objects

Am_Object Am_Border_Rectangle;
Am_Object Am_Accelerator_Inter;
Am_Input_Char Am_Default_Widget_Start_Char;

// internal objects


void Am_Widgets_Initialize () {
  Am_Object inter; // interactor in the widget
  Am_Object command_obj; 
  Am_Object_Advanced obj_adv; // to get at advanced features like
			       // local-only and demons.
  // Am_Slot_Advanced *slot;
  // Am_Demon_Set* demons;
  // Am_Demon_Set* proto_demons;

  widgets_set_slot_names();
  
  Am_Motif_Inactive_Stipple = Am_Gray_Stipple;
  Am_Key_Border_Line = Am_Style (0.0f, 0.0f, 0.0f, 2); //black, thickness=2


  // for black and white drawonables:
  black_and_white_rec.data = &black_and_white_rec_data;
  black_and_white_rec_data.foreground_style = Am_White;
  black_and_white_rec_data.background_style = Am_White;
  black_and_white_rec_data.shadow_style = Am_Black;
  black_and_white_rec_data.highlight_style = Am_Opaque_Gray_Stipple;
  black_and_white_rec_data.menu_top_line_style = Am_Black;
  black_and_white_rec_data.menu_bottom_line_style = Am_Opaque_Gray_Stipple;

  //any click, any modifier
  Am_Default_Widget_Start_Char = Am_Input_Char("ANY_LEFT_DOWN");

  //////////// rectangle /////////////
  Am_Border_Rectangle = Am_Graphical_Object.Create("Border_Rectangle")
    .Set (Am_SELECTED, false)
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
    .Set (Am_WIDTH, 50)
    .Set (Am_HEIGHT, 50)
    .Set (Am_FILL_STYLE, Am_Amulet_Purple)
    .Set (Am_DRAW_METHOD, border_rectangle_draw)
    ;
  
  obj_adv = (Am_Object_Advanced&)Am_Border_Rectangle;
  obj_adv.Get_Slot (Am_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_WIDGET_LOOK)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);

  Am_Button_Widgets_Initialize();
  Am_Scroll_Widgets_Initialize();
  Am_Text_Widgets_Initialize ();
  Am_Selection_Widget_Initialize();
  Am_Editing_Commands_Initialize();
  Am_Dialog_Widgets_Initialize();

  Am_Accelerator_Inter = Am_One_Shot_Interactor.Create("Accelerator")
    .Set (Am_START_WHERE_TEST, accelerator_start_where)
    .Set (Am_START_WHEN, Am_Input_Char("ANY_KEYBOARD"))
	  //start_where test does the complex event check, so any event is fine
    ;
  Am_Accelerator_Inter.Get_Part(Am_COMMAND)
    .Set (Am_DO_METHOD, do_accelerator_action)
    ;

} // end widgets initialize
