// $Id: ObjectList.cpp,v 1.5 1997/11/02 23:00:33 jharris Exp $ 

#include <qpainter.h>
#include <qfontmet.h>
#include <qcursor.h>
#include <qbitmap.h>

#include "ProjectWnd.h"
#include "Project.h"
#include "DialogWnd.h"

#include "ObjectList.h"

#include "bmp/bitmaps.h"

QPixmap *ObjectListItem::toBeSaved = NULL;

QPixmap *ObjectListItem::projectFolded = NULL;
QPixmap *ObjectListItem::projectUnfolded = NULL;

ObjectListItem::ObjectListItem( DlgObject *o )
  : QListBoxItem()
{
  obj = o;

  Initialize();
}

void ObjectListItem::Initialize( )
{
  saveItem = FALSE;
  objAreShown = TRUE;
  objClosed = FALSE;

  if( ! toBeSaved )
  {
    toBeSaved = new QPixmap();
    toBeSaved->loadFromData( ToBeSaved_bmp_data, ToBeSaved_bmp_len );
  
    projectFolded = new QPixmap();
    projectFolded->loadFromData( ProjectFolded_bmp_data, 
                                 ProjectFolded_bmp_len );
  
    projectUnfolded = new QPixmap();
    projectUnfolded->loadFromData( ProjectUnfolded_bmp_data, 
                                   ProjectUnfolded_bmp_len );
  }
  
  hItem = toBeSaved->height() + 2;
}


void ObjectListItem::paint( QPainter *painter )
{
  QString str;

  str = obj->GetName();

  if( str.isEmpty() )
    str = "NoName";

  int i = str.findRev( "/" );
  if( i >= 0 )
     str = str.right( str.length() - i - 1 );
  
  if( saveItem && toBeSaved )
    painter->drawPixmap( 0, 1, *toBeSaved );

  int fh = painter->fontMetrics().lineSpacing();

  int w1 = (toBeSaved) ? toBeSaved->width() : 0;
  int w2 = (projectFolded) ? projectFolded->width() : 0;
  int w = w1 + w2;

  if( obj->InProject() )
    w += 10;

  switch( obj->GetObjType() )
  {
  case DlgObject_Project:
    if( objAreShown )
      painter->drawPixmap( w1 + 4, 1, *projectUnfolded );
    else
      painter->drawPixmap( w1 + 4, 1, *projectFolded );

    painter->drawText( w + 8 , hItem / 2 + fh / 2, str );
    break;
  case DlgObject_Dialog:
  case DlgObject_Pixmap:
    if( obj->InProject() )
      painter->drawText( w + 8 , hItem / 2 + fh / 2, str );
    else 
      painter->drawText( w1 + 4 , hItem / 2 + fh / 2, str );
  }
}  

bool ObjectListItem::IsProject()
{
  return obj->GetObjType() == DlgObject_Project;
}

bool ObjectListItem::IsInProject()
{
  return obj->InProject();
}


ObjectList::ObjectList( ProjectWnd *p, const char *name )
  : QListBox( p, name )
{
  theProjectWnd = p;
  showChange = TRUE;
  objectDragged = NULL;
  
  QBitmap dragCursorBmp;
  dragCursorBmp.loadFromData( DragObj_bmp_data, DragObj_bmp_len );
  QBitmap dragCursorMaskBmp;
  dragCursorMaskBmp.loadFromData( DragObjM_bmp_data, DragObjM_bmp_len );
  dragCursor = new QCursor( dragCursorBmp, dragCursorMaskBmp, 7, 7 );
}

ObjectList::~ObjectList()
{
  delete dragCursor;
}

void ObjectList::Append( DlgObject *newObj )
{
  
  ObjectListItem *item = new ObjectListItem( newObj );

  // If the new item is in a project, search the projects entry first, then
  // append to the project entries
  if( newObj->InProject() )
  {
    QList<DlgObject> prjList = newObj->GetProjects();

    DlgObject *prj = prjList.first();
    while( prj )
    {
      AddObjectToProject( (Project*)prj, newObj );
      prj = prjList.next();
    }
  } else
    insertItem( item );
  
  if( newObj->GetObjType() == DlgObject_Project )
  {
    DlgObjList &objects = ((Project *)newObj)->GetObjects();

    for( DlgObject *obj = objects.first();
         obj;
         obj = objects.next() )
    {
      ObjectListItem *item = new ObjectListItem( obj );

      insertItem( item );
    }    
  }
}

void ObjectList::Remove( DlgObject *remObj )
{
  int index = FindItem( remObj );
  
  if( index >= 0 )
  {
    removeItem( index );
  }

  if( remObj->GetObjType() == DlgObject_Project )
  {
    unsigned int j;
    for( j = index + 1;
         j < count();
         j++ )
    {
      ObjectListItem *obj = (ObjectListItem*)item( j );
    
      if( obj->GetObject()->InProject( (Project *) remObj ) )
        removeItem( j );
      else
        break;
    }
  }  
}

DlgObject *ObjectList::CurrentItem()
{
  int index = currentItem();

  if( index < 0 )
    return NULL;

  return ((ObjectListItem *)item( index ))->GetObject();
}
  
void ObjectList::Update( DlgObject *updObj )
{
  int index = FindItem( updObj );

  if( index >= 0 )
  {
    ObjectListItem *objItem = (ObjectListItem *) item( index );
    objItem->SetObject( updObj );
    objItem->ToBeSaved( showChange );
    updateItem( index );
  }
}

int ObjectList::FindItem( DlgObject *fObj )
{
  for( unsigned int i = 0;
       i < count();
       ++i )
  {
    ObjectListItem *objItem = (ObjectListItem *) item( i );
    if( objItem->GetObject() == fObj )
    {
      return i;
    }
  }

  return -1;
}

void ObjectList::mousePressEvent( QMouseEvent *event )
{
  QListBox::mousePressEvent( event );

  if( event->button() != RightButton )
    return;

  DlgObject *obj = UnderMouse( event->pos().y() );
  
  if( obj )
    obj->GetListPopup().popup( QCursor::pos() );
}

void ObjectList::mouseDoubleClickEvent( QMouseEvent *event )
{
  QListBox::mouseDoubleClickEvent( event );

  DlgObject *obj = UnderMouse( event->pos().y() ); 

  if( obj && obj->GetObjType() == DlgObject_Project )
    ToggleView( (Project *)obj );
}

void ObjectList::mouseReleaseEvent( QMouseEvent *event )
{
  if( objectDragged )
  {
    // Get object under mouse
    DlgObject *obj = UnderMouse( event->pos().y() ); 

    // if object is a project, add dragged object to it
    if( obj )
    {
      if( obj->GetObjType() == DlgObject_Project )
      {
        Project *prj = (Project *) obj;

        // Only if the target project isn't the same as the start
        if( ! objectDragged->InProject( prj ) )
        {
          // Dragging is moving
          Remove( objectDragged );

          // Remove object from all its projects
          QList<DlgObject> prjList = objectDragged->GetProjects();
      
          DlgObject *ptr = prjList.first();
          while( ptr )
          {
            Project *prj = (Project *) ptr;

            prj->RemoveFromProject( objectDragged );
            objectDragged->LeaveProject( prj );
        
            RemoveObjectFromProject( prj, objectDragged );
            ptr = prjList.next();
          }

          // Add object
          objectDragged->EnterProject( prj );
          prj->AddToProject( objectDragged );
          AddObjectToProject( prj, objectDragged );
        }
      } 
    } else
    {
      // Dragging is moving
      Remove( objectDragged );

      // Remove object from all its projects
      QList<DlgObject> prjList = objectDragged->GetProjects();
      
      DlgObject *ptr = prjList.first();
      while( ptr )
      {
        Project *prj = (Project *) ptr;

        prj->RemoveFromProject( objectDragged );
        objectDragged->LeaveProject( prj );
        
        RemoveObjectFromProject( prj, objectDragged );
        ptr = prjList.next();
      }

      // Reenter list
      Append( objectDragged );
      
    }

    setCursor( arrowCursor );
    objectDragged = NULL;
  }
}

void ObjectList::mouseMoveEvent( QMouseEvent *event )
{
  QListBox::mouseMoveEvent( event );

  // Return if already dragging
  if( objectDragged )
    return;

  // Get object under mouse
  DlgObject *obj = UnderMouse( event->pos().y() ); 

  // Currently restricted to move only objects (no projects)
  if( obj->GetObjType() != DlgObject_Project )
  {
    objectDragged = obj;
    setCursor( *dragCursor );
  }
  

}

void ObjectList::Saved( DlgObject* obj )
{
    ShowChange( FALSE );
    Update( obj );
    ShowChange( TRUE );
}

int ObjectList::Contains( const QString& name )
{
  int found = FALSE;
  QString str;
  unsigned int i;
  int j;
    
  for( i = 0; (i < count()) && !found; i++ )
  {
    ObjectListItem *objItem = (ObjectListItem*)item(i);

    str = objItem->GetObject()->GetName();

    j = str.findRev( "/" );
    if( j >= 0 )
	    str = str.right( str.length() - j - 1 );

    if( str == name )
    {
	    found = TRUE;
    }
  }
    
  return found;
}

void ObjectList::AddObjectToProject( Project *prj, DlgObject *obj )
{
  int i = FindItem( prj );

  if( i < 0 )
    return;
  
  // Search the last obj in this project
  unsigned int j;
  for( j = i+1;
       j < count();
       j++ )
  {
    ObjectListItem *objItem = (ObjectListItem*)item( j );    
    
    if( ! objItem->GetObject()->InProject( prj ) )
      break;
  }

  ObjectListItem *item = new ObjectListItem( obj );

  insertItem( item, j );

}

void ObjectList::RemoveObjectFromProject( Project *prj, DlgObject *obj )
{
  int i = FindItem( prj );

  if( i < 0 )
    return;
  
  // Search the obj in this project
  unsigned int j;
  bool found = FALSE;
  for( j = i;
       j < count() && ! found;
       j++ )
  {
    ObjectListItem *objItem = (ObjectListItem*)item( j );    
    if( ! objItem->GetObject()->InProject( prj ) )
      break;
    if( objItem->GetObject() == obj )
      found = TRUE;
  }

  if( found )
    removeItem( --j );

}

void ObjectList::ToggleView( Project *prj )
{
  static bool showDlgs = TRUE;

  int i = FindItem( prj );

  if( i < 0 )
    return;

  // Toggle
  showDlgs = showDlgs ? FALSE : TRUE;
  
  if( showDlgs )
  {
     DlgObjList &objects = prj->GetObjects();

     for( DlgObject *obj = objects.first();
          obj;
          obj = objects.next() )
       AddObjectToProject( prj, obj );
  } else
  {
    for( unsigned int j = count() - 1;
         j > (unsigned int)i;
         --j )
    {
      ObjectListItem *objItem = (ObjectListItem*)item( j );    
      if( ! objItem->GetObject()->InProject( prj ))
        continue;
      removeItem( j );
    }
  }

  ObjectListItem *objItem = (ObjectListItem*)item( i );      
  objItem->ProjectView( showDlgs );
  updateItem( i );
}

ObjectListItem *ObjectList::GetCurrentListItem()
{
  return (ObjectListItem*)item( currentItem() );
}

ObjectListItem *ObjectList::GetObjectListItem( DlgObject *obj, Project *prj )
{
  if( ! prj )
  {
    int index = FindItem( obj );
    if( index > -1 )
      return (ObjectListItem*)item( index );
    return NULL;
  } else
  {
    int i = FindItem( prj );

    if( i < 0 )
      return NULL;

    bool found = FALSE;
    unsigned int j;
    for( j = i;
         j < count() && ! found;
         j++ )
    {
      ObjectListItem *objItem = (ObjectListItem*)item( j );    
      if( ! objItem->GetObject()->InProject( prj ))
        break;
      if( objItem->GetObject() == obj )
        found = TRUE;
    }

    if( found )
      return (ObjectListItem*)item( --j );
    return NULL;
  }
}

DlgObject *ObjectList::UnderMouse( int y )
{
  // Get object under mouse
  int i = findItem( y );

  ObjectListItem *objItem; 
  DlgObject *obj = NULL;

  if( i >= 0 )
  {
    objItem = (ObjectListItem *) item( i );
    if( objItem )
      obj = objItem->GetObject();
  }

  return obj;

}  

