// $Id: Project.cpp,v 1.4 1997/08/06 04:02:52 jharris Exp $
#include <qfile.h>

#include "Project.h"

#include "ProjectWnd.h"
#include "Dialog.h"
#include "pixmap/Pixmap.h"
#include "ProjectDialog.h"
#include "MakeGen.h"
#include "MakeEdit.h"

#include "DlgParser.h"

Project::Project( ProjectWnd &projectWnd, const QString &name )
  : DlgObject( projectWnd, name )
{
  /* Create three popupmenus for the List:
   * 1. Not in a project
   * 2. In a project, not hidden
   * 3. In a project, hidden
   */
  unboundPopup = new QPopupMenu;
  unboundPopup->insertItem( "Rename Project", this, SLOT(Rename()) );
  unboundPopup->insertSeparator();
  unboundPopup->insertItem( "Save Project", this, SLOT(Save()) );
  unboundPopup->insertItem( "Save Project As...", this, SLOT(SaveAs()));
  unboundPopup->insertItem( "Close Project", this, SLOT(Hide()) );
  unboundPopup->insertSeparator();
  unboundPopup->insertItem( "Add Object...", this, SLOT(AddObject()) );
  unboundPopup->insertItem( "Generate Objects", this, SLOT(Generate()) );

  // Not yet implemented
  unboundHiddenPopup = NULL;
  boundPopup = NULL;
  boundHiddenPopup = NULL;

  // Will hold the generated files
  genSources.setAutoDelete( TRUE );
  genHeaders.setAutoDelete( TRUE );
  genMeta.setAutoDelete( TRUE );

  makefile.setAutoDelete( TRUE );

};

bool Project::Load()
{
  if( ! itsFilename.isEmpty() )
  {
    itsParser = new DlgParser( itsFilename );
    if( ! itsParser->IsInitialized() )
    {
      delete itsParser;
      return FALSE;
    }

    // Connect the signals which are interesting for us
    connect( itsParser, SIGNAL(ProjectKeyValue(QString &,QString &)), 
             SLOT(RestoreProject(QString &,QString &)) );
    
    // And start the action
    itsParser->StartParsing();

    // That's it for the Parser object
    delete itsParser;    

    itsParser = NULL;

    DlgObject* obj;
    for( obj = itsObjects.first();
         obj;
         obj = itsObjects.next() )
    {
      if( !itsProjectWnd.IsAutoGenerating() )
	{
        obj->Show();
        //     itsProjectWnd.itsObjectList.Append( obj );
	}
	
      obj->Load();
    }
  }

  return TRUE;
}

bool Project::Save()
{
  QFile file( itsFilename );
  if( !file.open( IO_WriteOnly ) )
  {
    // error msg
    return FALSE;
  }
    
  QTextStream stream( &file );
    
  stream << "DlgEdit:v1.2:Project:\n";
  stream << "Project {\n";

  stream << "Name {" << itsName << "}\n";

  DlgObject* obj;
  for( obj = itsObjects.first();
       obj;
       obj = itsObjects.next() )
  {
    stream << obj->GetType() << " {" << obj->GetName() << "}\n";
  }

  stream << "}\n";

  file.close();

   for( obj = itsObjects.first();
        obj;
        obj = itsObjects.next() )
  {
     obj->Save();
  }

  emit Saved( this );
  status = DlgObject_Saved;

  return TRUE;
}

void Project::Show()
{
}

void Project::Hide()
{
    DlgObject* obj;
    for( obj = itsObjects.first();
	 obj;
	 obj = itsObjects.next() )
    {
	obj->Hide();
    }
}

void Project::Raise()
{
}

bool Project::EnterProject( const Project *prj )
{
  return DlgObject::EnterProject( prj );
}

bool Project::LeaveProject( const Project *prj )
{
  return DlgObject::LeaveProject( prj );
}

bool Project::InProject( const Project *prj )
{
  return DlgObject::InProject( prj );
}

void Project::CPAvailable( bool av )
{
  // shut up
  av = av;
}

void Project::AddToProject( DlgObject *newObject )
{
   itsObjects.append( newObject );
   status = DlgObject_Changed;
   emit Changed( this );
}

void Project::RemoveFromProject( DlgObject *obj )
{
   itsObjects.remove( obj );
   obj->Hide();
   status = DlgObject_Changed;
   emit Changed( this );
}

bool Project::Generate( bool aG )
{
  autoGenerate = aG;

  MakeGen makeGen( this );

  if( makeGen.Exec() )
  {
    MakeEdit makeEdit( itsMakeFileName, makefile );
    makeEdit.exec();
  }

  return TRUE;
}

void Project::RestoreProject(QString &key, QString &value )
{
  if( key == "Name" )
  {
    itsName = value;
  } else if( key == "Dialog" )
  {
     Dialog *dialog = new Dialog( itsProjectWnd, "" );

     dialog->SetName( value );
     dialog->EnterProject( this );

     itsObjects.append( dialog );

     connect( dialog, SIGNAL(Changed(DlgObject*)), 
              &(itsProjectWnd.itsObjectList), SLOT(Update(DlgObject*)) );
     connect( dialog, SIGNAL(Saved(DlgObject*)),
              &(itsProjectWnd.itsObjectList), SLOT(Saved(DlgObject*)) );
     connect( dialog, SIGNAL(Closed(DlgObject*)), 
              SLOT(ObjClosed(DlgObject*)) );

     connect( dialog, SIGNAL(CPDataSet(DlgObject*)), 
              &itsProjectWnd, SLOT(NewCPBuffer(DlgObject*)) );
     connect( &itsProjectWnd, SIGNAL(CPBufferAvailable(bool)), dialog,
              SLOT(CPAvailable(bool)) );
  } else if( key == "Pixmap" )
  {
     Pixmap *pix = new Pixmap( itsProjectWnd, "" );

     pix->SetName( value );
     pix->EnterProject( this );

     itsObjects.append( pix );

     connect( pix, SIGNAL(Changed(DlgObject*)), 
              &(itsProjectWnd.itsObjectList), SLOT(Update(DlgObject*)) );
     connect( pix, SIGNAL(Saved(DlgObject*)),
              &(itsProjectWnd.itsObjectList), SLOT(Saved(DlgObject*)) );
     connect( pix, SIGNAL(Closed(DlgObject*)), 
              SLOT(ObjClosed(DlgObject*)) );

     connect( pix, SIGNAL(CPDataSet(DlgObject*)), 
              &itsProjectWnd, SLOT(NewCPBuffer(DlgObject*)) );
     connect( &itsProjectWnd, SIGNAL(CPBufferAvailable(bool)), pix,
              SLOT(CPAvailable(bool)) );
  }
}

void Project::GenerateAllObjects()
{
  QString gSrc, gHdr, gMHdr;

  genSources.clear();
  genHeaders.clear();
  genMeta.clear();

  DlgObject* obj;
  for( obj = itsObjects.first();
       obj;
       obj = itsObjects.next() )
  {
    obj->Generate( autoGenerate );
    obj->GetGeneratedFiles( gSrc, gHdr, gMHdr );
    genSources.append( new QString( gSrc ) );
    genHeaders.append( new QString( gHdr ) );
    genMeta.append( new QString( gMHdr ) );
  }
}

void Project::GenerateMakeFile()
{
  makefile.clear();

  QFile templateMakefile( itsMakeFileTemplateName );

  if( ! templateMakefile.open( IO_ReadOnly ) )
  {
    QMessageBox::message( "Error...", "Unable to read template " + 
                          itsMakeFileTemplateName );
    return;
  }

  QTextStream stream( &templateMakefile );
  while( ! stream.eof() )
  {
    QString line = stream.readLine();

    makefile.append( new QString( line ) );
    if( line == "# DlgEdit-Project: Do not remove this line" )
    {
      StringIter iSrc( genSources );
      if( iSrc.current() )
        makefile.append( new QString( "DLGSOURCES = \\" ) );
      while( iSrc.current() )
      {
        QString *str = iSrc;
        makefile.append( new QString( *str + (iSrc.atLast() ? "" : "\\") ));
        ++iSrc;
      }
      StringIter iHdr( genHeaders );
      if( iHdr.current() )
        makefile.append( new QString( "DLGHEADERS = \\" ));
      while( iHdr.current() )
      {
        QString *str = iHdr;
        makefile.append( new QString( *str + (iHdr.atLast() ? "" : "\\") ));
        ++iHdr;
      }
      StringIter iMHdr( genMeta );
      if( iMHdr.current() )
        makefile.append( new QString( "DLGMETA = \\" ));
      while( iMHdr.current() )
      {
        QString *str = iMHdr;
        makefile.append( new QString( *str + (iMHdr.atLast() ? "" : "\\") ));
        ++iMHdr;
      }
      makefile.append( new QString( "# DlgEdit-Project: End of generated section" ));
    }
  }
  templateMakefile.close();

}
void Project::ObjClosed( DlgObject *obj )
{
  ObjectListItem *item = 
    itsProjectWnd.itsObjectList.GetObjectListItem( obj, this );

  item->ObjectClosed( TRUE );
}


void Project::Rename()
{
  ProjectDialog dialog;

  dialog.SetName( itsName );

  if( dialog.exec() == QDialog::Accepted )
    itsName = dialog.GetName();
}

void Project::AddObject()
{
  QString str;

  str = QFileDialog::getOpenFileName();
    
  if( !str.isEmpty() )
  {
    // Most of the dirty work is done by ProjectWnd, it has to remember 
    // its loaded objects.
    DlgObject *obj = itsProjectWnd.ObjectLoad( str, this );

    AddToProject( obj );
  }
}

