#include <iostream.h>

#include <qobject.h>

#include "layout/BoxLayout.h"
#include "layout/BoxLayoutPropPage.h"
#include "layout/BoxSpacing.h"
#include "layout/BoxStretch.h"
#include "layout/BoxWidget.h"

#include "widget/DlgWidget.h"
#include "widget/DlgMenuBar.h"

#include "WidgetLayout.h"
#include "GuiTools.h"

#define Inherited LayoutItem

BoxLayout::BoxLayout
( 
    QWidget*			parent,
    WidgetLayout*		widgetLayout,
    Frame*			frame
)
    :
    Inherited( widgetLayout, TRUE ),
    itsFrame( frame ),
    itsLayout( NULL ),
    itsDirection( QBoxLayout::Down ),
    itsBorder( 5 ),
    itsAutoBorder( 5 ),
    itsStrut( 0 ),
    itsStretch( 1 ),
    itsMenuBar( NULL ),
    isNewFrame( frame == NULL )
{
    if( parent )
    {
	if( !itsFrame )
	{
	    itsFrame = new Frame( parent, NULL );
	    itsFrame->setFrameStyle( QFrame::Box | QFrame::Raised );
	    SetColor( itsFrame );
	    itsFrame->show();
	}
    
	widgetLayout->connect( itsFrame, 	
			       SIGNAL(MousePress(QWidget*,QMouseEvent*)),
			       SLOT(LayoutMousePress(QWidget*,QMouseEvent*)) );

	if( isNewFrame )
	{
	    widgetLayout->connect( itsFrame, SIGNAL(Paint(QWidget*)),
				   SLOT(LayoutPaint(QWidget*)) );
	    widgetLayout->connect( 
		itsFrame,
		SIGNAL(MouseDoubleClick(QWidget*,QMouseEvent*)),
		SLOT(ProcessMouseDoubleClick(QWidget*,QMouseEvent*)) );
	}

	itsChildren.setAutoDelete( TRUE );
    }
    else
    {
	isNewFrame = FALSE;
    }
}


BoxLayout::~BoxLayout()
{
    delete itsLayout;
    itsLayout = NULL;

    itsChildren.clear();

    if( isNewFrame )
    {
	//itsFrame->parent()->removeChild( itsFrame );
	delete itsFrame;
	itsFrame = NULL;
    }

    if( itsMenuBar )
    {
	itsMenuBar->RemoveFromLayout();
    }
}


void BoxLayout::Recreate( void )
{
    QSize frameSize = itsFrame->size();

    if( itsLayout )
    {
	delete itsLayout;
	itsLayout = NULL;
    }

    if( itsChildren.count() > 0 )
    {
	itsLayout = new QBoxLayout( itsFrame, itsDirection, 	
				    itsBorder, itsAutoBorder );
	itsLayout->addStrut( itsStrut );
	if( itsMenuBar )
	{
	    itsLayout->setMenuBar( (QMenuBar*)itsMenuBar->GetWidget() );
	}

	QListIterator<LayoutItem> iter( itsChildren );
	LayoutItem* item;
    
	while( (item = iter()) )
	{
	    item->AddToLayout( itsLayout );
	}

	itsLayout->activate();
    }

    itsFrame->setFixedSize( frameSize );
}


void BoxLayout::AddToLayout
(
    QBoxLayout*	layout
)
{
    if( itsLayout )
    {
	delete itsLayout;
	
	itsLayout = NULL;
    }

    layout->addWidget( itsFrame, itsStretch );

    if( itsChildren.count() > 0 )
    {
	itsLayout = new QBoxLayout( itsFrame, itsDirection, 
				    itsAutoBorder, itsAutoBorder );

	itsLayout->addStrut( itsStrut );
	
	if( itsMenuBar )
	{
	    itsLayout->setMenuBar( (QMenuBar*)itsMenuBar->GetWidget() );
	}

	QListIterator<LayoutItem> iter( itsChildren );
	LayoutItem* item;
    
	while( (item = iter()) )
	{
	    item->AddToLayout( itsLayout );
	}

	itsLayout->activate();
    }
}


BoxLayout* BoxLayout::AddBoxLayout( void )
{
    BoxLayout* box = new BoxLayout( itsFrame, itsWidgetLayout );

    itsChildren.append( box );

    return box;
}


BoxSpacing* BoxLayout::AddBoxSpacing( void )
{
    BoxSpacing* space = new BoxSpacing( itsFrame, itsWidgetLayout );
    
    itsChildren.append( space );
    
    return space;
}


BoxStretch* BoxLayout::AddBoxStretch( void )
{
    BoxStretch* stretch = new BoxStretch( itsFrame, itsWidgetLayout );
    
    itsChildren.append( stretch );
    
    return stretch;
}


BoxWidget* BoxLayout::AddBoxWidget
(
    DlgWidget*	widget,
    int		stretch,
    int		align
)
{
    // get the geometry before recreating the widget into its new layout parent
    QRect boxRect = GuiTools::GetGlobalGeometry( widget->GetWidget() );

    BoxWidget* boxWidget = new BoxWidget( itsFrame, widget, itsWidgetLayout,
					  stretch, align );
    
    // find the boxWidget's place in the list of children
    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    QRect rect;
    bool found = FALSE;
    int pos = 0;
    
    while( (child = iter()) && !found )
    {
	rect = child->GetGlobalGeometry();
	
	switch( itsDirection )
	{
	    case QBoxLayout::LeftToRight:
	    {
		if( rect.left() > boxRect.left() )
		{
		    found = TRUE;
		}
		break;
	    }
	    case QBoxLayout::RightToLeft:
	    {
		if( rect.right() < boxRect.right() )
		{
		    found = TRUE;
		}
		break;
	    }
	    case QBoxLayout::TopToBottom:
	    {
		if( rect.top() > boxRect.top() )
		{
		    found = TRUE;
		}
		break;
	    }
	    case QBoxLayout::BottomToTop:
	    {
		if( rect.bottom() < boxRect.bottom() )
		{
		    found = TRUE;
		}
		break;
	    }
	}

	pos++;
    }

    if( found )
    {
	itsChildren.insert( pos - 1, boxWidget );
    }
    else
    {
	itsChildren.append( boxWidget );
    }

    return boxWidget;
}


void BoxLayout::SetMenuBar( const QString&	menuBar )
{
    if( itsMenuBar )
    {
	itsMenuBar->RemoveFromLayout();
    }
    
    itsMenuBar = NULL;

    if( !menuBar.isEmpty() )
    {	
	const DlgWidget* widget = itsWidgetLayout->GetWidget( menuBar );
	if( widget && (widget->GetType() == DlgWidget::DW_MenuBar) )
	{
	    itsMenuBar = (DlgMenuBar*) widget;
	    itsMenuBar->SetInLayout( itsFrame, TRUE );
	}
    }
}


void BoxLayout::RemoveWidget
(
    DlgWidget*	widget
)
{
    if( widget == itsMenuBar )
    {
	itsMenuBar = NULL;
    }
    else
    {
	LayoutItem* item = FindLayoutItem( widget->GetWidget() );
	
	if( item )
	{
	    BoxLayout* itemLayout = FindLayout( item );
	    
	    if( itemLayout )
	    {
		itemLayout->RemoveItem( item );
	    }
	}
    }
}


LayoutItem* BoxLayout::NewLayoutItem( const QString&	type,
				      const QString&	id )
{
    LayoutItem* newItem = NULL;
    int pos = id.find( '.' );

    // the new item needs to be added to this layout
    if( pos == -1 )
    {
	if( type == BoxLayout::ItemName() )
	{
	    newItem = new BoxLayout( itsFrame, itsWidgetLayout );
	}
	else if( type == BoxSpacing::ItemName() )
	{
	    newItem = new BoxSpacing( itsFrame, itsWidgetLayout );
	}
	else if( type == BoxStretch::ItemName() )
	{
	    newItem = new BoxStretch( itsFrame, itsWidgetLayout );
	}
	else if( type == BoxWidget::ItemName() )
	{
	    newItem = new BoxWidget( itsFrame, NULL, itsWidgetLayout );
	}

	if( newItem )
	{
	    itsChildren.insert( id.toInt() - 1, newItem );
	}
	else
	{
	    cerr << "Unknown layout item type: " << type << endl;
	}
    }
    // item needs to be added to a sub-layout
    else
    {
	QString index = id.left( pos );
	QString newId = id.mid( pos + 1, id.length() - pos - 1 );
	
	LayoutItem* layoutItem = itsChildren.at( index.toInt() - 1 );
	if( layoutItem->IsLayout() )
	{
	    // Ugly cast, but needed for now
	    newItem = ((BoxLayout*)layoutItem)->NewLayoutItem( type, newId );
	}
	else
	{
	    cerr << "Invalid layout item for type: " << type << 
		" at index: " << id << endl;
	}
    }
    
    return newItem;
}


BoxLayout* BoxLayout::FindLayout
(
    QWidget*	widget
) const
{
    BoxLayout* layout = NULL;

    if( widget == itsFrame )
    {
	layout = (BoxLayout*)this;
    }
    else
    {
	QListIterator<LayoutItem> iter( itsChildren );
	LayoutItem* child;
	
	while( (child = iter()) && !layout )
	{
	    layout = child->FindLayout( widget );
	}
    }
    
    return layout;
}


BoxLayout* BoxLayout::FindLayout
(
    LayoutItem*	item
) const
{
    BoxLayout* layout = NULL;
    
    if( item == (BoxLayout*)this )
    {
	// Its too late, we have already passed its parent layout, so return
	// NULL
    }
    else
    {
	QListIterator<LayoutItem> iter( itsChildren );
	LayoutItem* child;
	
	while( (child = iter()) && !layout )
	{
	    if( child == item )
	    {
		layout = (BoxLayout*)this;
	    }
	    else
	    {
		layout = child->FindLayout( item );
	    }
	}
    }
    
    return layout;
}


BoxLayout* BoxLayout::FindLayout
(
    const QRect&	rect
)
{
    BoxLayout* layout = NULL;
    
    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    
    while( (child = iter()) && !layout )
    {
	layout = child->FindLayout( rect );
    }
    
    if( !layout && GuiTools::GetGlobalGeometry( itsFrame ).contains( rect ) )
    {
	layout = this;
    }
    
    return layout;
}



LayoutItem* BoxLayout::FindLayoutItem
(
    QWidget*	widget
) const
{
    LayoutItem* item = NULL;
    
    if( widget == itsFrame )
    {
	item = (BoxLayout*)this;
    }
    else
    {
	QListIterator<LayoutItem> iter( itsChildren );
	LayoutItem* child;
	
	while( (child = iter()) && !item )
	{
	    item = child->FindLayoutItem( widget );
	}
    }

    return item;
}


BoxWidget* BoxLayout::FindBoxWidget
(
    DlgWidget*	widget
) const
{
    BoxWidget* box = NULL;
    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    
    while( (child = iter()) && !box )
    {
	box = (BoxWidget*)child->FindLayoutItem( widget->GetWidget() );
    }

    return box;
}


void BoxLayout::LowerWidget( void )
{
    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    
    while( (child = iter()) )
    {
	child->LowerWidget();
    }
    
    itsFrame->lower();
}


void BoxLayout::SaveContents( QTextStream&	stream,
			      QString		id ) const
{
    stream << GetItemName() << "-" << id << "{\n";
    
    stream << "\t\tDirection {";
    switch( itsDirection )
    {
	case QBoxLayout::LeftToRight:
	{
	    stream << "LeftToRight";
	    break;
	}
	case QBoxLayout::RightToLeft:
	{
	    stream << "RightToLeft";
	    break;
	}
	case QBoxLayout::TopToBottom:
	{
	    stream << "TopToBottom";
	    break;
	}
	case QBoxLayout::BottomToTop:
	{
	    stream << "BottomToTop";
	    break;
	}
    }
    stream << "}\n";

    if( IsTopLevel() )
    {
	stream << "\t\tBorder {" << itsBorder << "}\n";
    }
    
    stream << "\t\tAutoBorder {" << itsAutoBorder << "}\n";
    stream << "\t\tName {" << itsName << "}\n";
    stream << "\t\tStrut {" << itsStrut << "}\n";

    if( !IsTopLevel() )
    {
	stream << "\t\tStretch {" << itsStretch << "}\n";
    }

    if( itsMenuBar )
    {
	stream << "\t\tMenu {" << itsMenuBar->GetWidgetName() << "}\n";
    }
    
    stream << "}\n";

    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    QString childId;
    int n = 1;
    
    while( (child = iter()) )
    {
	child->SaveContents( stream, id + "." + childId.setNum( n ) );
	n++;
    }
}


bool BoxLayout::RestoreKeyValue( const QString&	key,
				 const QString&	value )
{
    bool rc = TRUE;

    if( key == "Direction" )
    {
	QBoxLayout::Direction dir;
	
	if( value == "TopToBottom" )
	{
	    dir = QBoxLayout::TopToBottom;
	}
	else if( value == "BottomToTop" )
	{
	    dir = QBoxLayout::BottomToTop;
	}
	else if( value == "LeftToRight" )
	{
	    dir = QBoxLayout::LeftToRight;
	}
	else if( value == "RightToLeft" )
	{
	    dir = QBoxLayout::RightToLeft;
	}
	else 
	{
	    rc = FALSE;
	}
	
	if( rc )
	{
	    itsDirection = dir;
	}
    }
    else if( key == "Border" )
    {
	itsBorder = value.toInt();
    }
    else if( key == "AutoBorder" )
    {
	itsAutoBorder = value.toInt();
    }
    else if( key == "Name" )
    {
	itsName = value;
    }
    else if( key == "Strut" )
    {
	itsStrut = value.toInt();
    }
    else if( key == "Stretch" )
    {
	itsStretch = value.toInt();
    }
    else if( key == "Menu" )
    {
	const DlgWidget* widget = itsWidgetLayout->GetWidget( value );
	if( widget && (widget->GetType() == DlgWidget::DW_MenuBar) )
	{
	    itsMenuBar = (DlgMenuBar*)widget;
	}
	else
	{
	    rc = FALSE;
	}
    }
    else 
    {
	rc = FALSE;
    }

    return rc;
}


void BoxLayout::GenerateCtor( QTextStream&	stream,
			      const QString&	layoutName ) const
{
    stream << "\tQBoxLayout* " << layoutName << " = new QBoxLayout( ";
    
    if( IsTopLevel() )
    {
	stream << "this, ";
    }
    
    stream << "QBoxLayout::";
    switch( itsDirection )
    {
	case QBoxLayout::TopToBottom:
	{
	    stream << "TopToBottom";
	    break;
	}
	case QBoxLayout::BottomToTop:
	{
	    stream << "BottomToTop";
	    break;
	}
	case QBoxLayout::LeftToRight:
	{
	    stream << "LeftToRight";
	    break;
	}
	case QBoxLayout::RightToLeft:
	{
	    stream << "RightToLeft";
	    break;
	}
    }
    stream << ", ";
    
    if( IsTopLevel() )
    {
	stream << itsBorder << ", ";
    }
    
    stream << itsAutoBorder << ", ";
    
    QString name = "NULL";
    if( !itsName.isEmpty() )
    {
	name = "\"" + itsName + "\"";
    }
    stream << name << " );\n";
    
}


void BoxLayout::GenerateSource( QTextStream&	stream,
				const QString&	layoutVarName,
				const QString&	layoutId ) const
{
    QString varName = "dlgedit_layout_" + layoutId;
    
    GenerateCtor( stream, varName );
    
    // not top level layout
    if( !IsTopLevel() )
    {
	stream << '\t'<< layoutVarName << "->addLayout( " << varName << ", " <<
	    itsStretch << " );\n";
    }

    stream << '\t' << varName << "->addStrut( " << itsStrut << " );\n";

    if( itsMenuBar )
    {
	stream << '\t' << varName << "->setMenuBar( " << 
	    itsMenuBar->GetGeneratedVarName() << " );\n";
    }
    
    
    QListIterator<LayoutItem> iter( itsChildren );
    LayoutItem* child;
    QString numStr;
    int n = 1;
    
    while( (child = iter()) )
    {
	child->GenerateSource( stream, varName, 	
			       layoutId + "_" + numStr.setNum( n ) );
	n++;
    }
    stream << endl;
}


QString BoxLayout::GetItemName( void ) const
{
    return ItemName();
}


QWidget* BoxLayout::GetPropertyPage
(
    QWidget*	parent
)
{
    return new BoxLayoutPropPage( this, parent, GetItemName() );
}


QRect BoxLayout::GetGlobalGeometry( void ) const
{
    return GuiTools::GetGlobalGeometry( itsFrame );
}


void BoxLayout::GetMenuBarNames( QStrList&	nameList ) const
{
    if( itsWidgetLayout )
    {
	itsWidgetLayout->GetWidgetTypeNames( DlgWidget::DW_MenuBar, nameList );
    }
    else
    {
	nameList.clear();
    }
}


QString BoxLayout::GetMenuBarName( void ) const
{
    QString name;
    if( itsMenuBar )
    {
	name = itsMenuBar->GetWidgetName();
    }
    
    return name;
}

