#ifndef DlgWidget_included
#define DlgWidget_included

#include <qtstream.h>
#include <qwidget.h>
#include <qlist.h>
#include <qlistbox.h>
#include <qstring.h>
#include <qstrlist.h>

#include "widget/SignalConnection.h"

class BoxLayout;

class DlgWidget
{
    
public:

    enum Type
    {
	DW_Button,
	DW_ButtonGroup,
	DW_CheckBox,
	DW_ComboBox,
	DW_Frame,
	DW_GroupBox,
	DW_LCDNumber,
	DW_Label,
	DW_LineEdit,
	DW_ListBox,
	DW_MenuBar,
	DW_MultiLineEdit,
	DW_ProgressBar,
	DW_PushButton,
	DW_RadioButton,
	DW_ScrollBar,
	DW_Slider,
	DW_User,
	DW_None = -1
    };
    
public:

    DlgWidget
    (
	Type		type,
	QWidget*	origParent
    );

    virtual ~DlgWidget();

    static DlgWidget *NewDlgWidget( QWidget *, 
				    DlgWidget::Type, 
				    const QString * = NULL );
    // Construct a new subclass using DlgWidget::Type

    static DlgWidget *NewDlgWidget( QWidget *,
				    const QString&,
				    const QString* = NULL );
    // Construct a new subclass using a Widgetname
    
    virtual QWidget* GetWidget( void ) const = 0;
    // return the widget's widget

    QString GetWidgetName( void ) const;
    // return the widget's name

    QPoint GetPos( void ) const;
    // return the widget's position relative to its original parent

    virtual void SaveContents
    (
	QTextStream&	stream
    );
    // save the contents of the widget

    int DoPropertyDialog
    (
	BoxLayout*	layout
    );
    // Do the property dialog

    virtual void GetPropertyPages
    (
	QWidget*		parent,
	QList<QWidget>&		pages,
	QStrList&		labels
    ) = 0;
    // Get the property pages and their labels for the widget

    virtual bool RestoreKeyValue( QString &key, QString &value );
    // Restore a single key/value pair

    void SaveQt
    (
        QTextStream &,
        const QWidget *
    );
    // Used by original Qt subclasses
 
    bool RestoreQt
    (
        const QString &,
        const QString &,
        QWidget *
    );
    // Used by original Qt subclasses
 
    void SetVariableMapping
    (
	const QString&	variableName
    );
    // if variableName != "", then the widget is mapped to a variable

    bool IsVariableMapped() const;
    // is the widget mapped to a variable

    const QString& GetVariableName() const;
    // return the name of the variable, or "" if not mapped

    QString GetGeneratedVarName( void ) const;
    // return the name of the widget variable during generation, either the
    // name of the variable or a temporary name if not mapped

    Type GetType() const;
    // return the type of the widget

    virtual QString GetHeaderFilename() const = 0;
    // return the header filename for the widget
    
    virtual QString GetWidgetClass() const = 0;
    // return the class name for the widget

    const char* Dot() const;
    // return a '->' since all variables are allocated dynamically

    enum LayoutStatus
    {
	NoLayout	= 0x0000,
	InLayout	= 0x0001,
	LayoutMenuBar	= 0x0003
    };

    LayoutStatus GetLayoutStatus( void ) const;
    // Return the layout status of the widget

    bool IsInLayout( void ) const;
    // Return true if the widget is in a layout or a layout menu bar

    void SetInLayout
    (
	QWidget*	newParent,
	bool		isMenuBar = FALSE
    );
    // make the widget be in a layout, and make it become a child of 
    // the given parent

    void RemoveFromLayout( void );
    // remove the widget from a layout, and make it a child of its
    // original parent
    
    virtual void GenerateSource
    (
	QTextStream&		stream,
	const QString&		varName,
	const QList<DlgWidget>&	zOrdering
    );
    // call GenSrc for the appropriate widget base classes. varName already
    // has a leading tab and the correct dereferencer

    virtual void GenerateExtraSource( QTextStream&		stream,
				      const QString&		varName,
				      const QList<DlgWidget>&	zOrdering );
    // Generate extra source for a widget, after all the widgets have been
    // constructed and initialized

    virtual void GenerateCtor
    (
	QTextStream&		stream,
	const QString&		varName
    );
    // output a constructor for the widget.  The
    // ctor for leaves the output at the start of the next line

    static void GenSrc
    (
	DlgWidget&		dlgWidget,
	QTextStream&		stream,
	const QString&		varName,
	const QList<DlgWidget>&	zOrdering
    );
    // static method to actually generate the source

    QList<SignalConnection>& GetConnectedSignalList( void );
    // return a reference to the list of signals

    void GetSignalList( QStrList& signalList ) const;
    // return a list of the names of the signals that can be connected
    // to this widget


protected: //Methods
  QString CreateName( QString, const char *, int & );
  // Generates a unique WidgetName based on the Class name, a user-
  // provided name (no name generation) and a class specific number

    virtual void AddSignalsToList( QStrList&	signalList ) const;
    // add signals specific to a particular widget type to the list
  
    void UpdateNextWidgetNumber( const QString&	widgetName,
				 int&		nextWidgetNum ) const;
  
private:

    QString	itsVariableName;
    
    Type	itsType;

    QList<SignalConnection>	itsConnectedSignalList;

    LayoutStatus	itsLayoutStatus;
    // the status of the widget with respect to layouts

    QWidget*	itsOriginalParent;
    // the original parent for this widget, the widget frame
};


inline DlgWidget::Type DlgWidget::GetType() const
{
    return itsType;
}


inline QString DlgWidget::GetWidgetName( void ) const
{
    return GetWidget()->name();
}


inline DlgWidget::LayoutStatus DlgWidget::GetLayoutStatus( void ) const
{
    return itsLayoutStatus;
}


inline bool DlgWidget::IsInLayout( void ) const
{
    return itsLayoutStatus & InLayout;
}


inline bool DlgWidget::IsVariableMapped() const
{
    return !itsVariableName.isEmpty();
}


inline const QString& DlgWidget::GetVariableName() const
{
    return itsVariableName;
}


inline void DlgWidget::SetVariableMapping
(
    const QString&	variableName
)
{
    itsVariableName = variableName;
}


inline void DlgWidget::GenerateSource
(
    QTextStream&		stream,
    const QString&		varName,
    const QList<DlgWidget>&	zOrdering
)
{
    GenSrc( *this, stream, varName, zOrdering );
}


inline QList<SignalConnection>& DlgWidget::GetConnectedSignalList( void )
{
    return itsConnectedSignalList;
}


QTextStream& operator <<( QTextStream&	ostream,
			  const QSize&	size );

QTextStream& operator >>( QTextStream&	istream,
			  QSize&	size );


QTextStream& operator <<( QTextStream&	ostream,
			  const QPoint&	point );

QTextStream& operator >>( QTextStream&	istream,
			  QPoint&	point );


QTextStream& operator <<
(
    QTextStream&	ostream,
    const QRect&	rect
);


QTextStream& operator >>
(
    QTextStream&	istream,
    QRect&		rect
);


// write palette to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QPalette&	palette
);


// read palette from stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QPalette&		palette
);


// write color group to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QColorGroup&	group
);


// read color group from a text stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QColorGroup&	group
);


// write font to text stream
QTextStream& operator <<
(
    QTextStream&	ostream,
    const QFont&	font
);


// read a font from the text stream
QTextStream& operator >>
(
    QTextStream&	istream,
    QFont&		font
);


void SaveBool
(
    QTextStream&	ostream,
    const bool& 
);

void RestoreBool
(
    QTextStream&	istream,
    bool &
);


#endif // DlgWidget_included





