// NOTE:
// TFigure is currently based on `int' and `TPencil'; coming versions will use 
// `double' and `TPencil' 

#include <toad/toad.hh>
#include <toad/io/serializable.hh>

#include <vector>

class TPencil;

#define TFigure TFigure


class TFigure;

class TFigureEditor
{
	public:
		enum EMode {
			MODE_SELECT,
			MODE_ROTATE,
			MODE_CREATE,
			MODE_CREATING,
			MODE_EDIT_IN_PLACE
		};
		
		EMode mode;
		TWindow *window;
		
		unsigned result;	// defined in TFigure
		
		virtual void InvalidateFigure(TFigure* g) = 0; 
};

class TFigure:
	public TSerializable
{
	public:
		double x,y,w,h;
	
		virtual ~TFigure();
		virtual void paint(TPencil& pen, bool selected) = 0;
		
		bool filled;
		TRGB line_color;
		TRGB fill_color;
	
		// editor related stuff per gadget for manipulation
		//-------------------------------------------------
		static const unsigned NOTHING  = 0;
		static const unsigned CONTINUE = 1;
		static const unsigned STOP     = 2;
		static const unsigned REPEAT   = 4;
		static const unsigned DELETE   = 8;

		// stage 1: select:
		virtual double distance(double x, double y) = 0;
		
		// stage 2: move
		virtual void paintHandles(TPencil &) {};
		virtual void translate(double dx, double dy) {};
		
		// stage 3: manipulate
		virtual int getHandle(double x, double y) { return -1; }
		virtual int translateHandle(int handle, double x, double y) { return handle; }

		// stage 4: in place editing
		//. return `true' when in-place editing is desired
		virtual bool startInPlace() { return false; }
		virtual unsigned stop() { return NOTHING; };
		virtual unsigned keyDown(TFigureEditor*, TKey, char*, unsigned) { return CONTINUE; }

		// editor related stuff for manipulation & creation
		//--------------------------------------------------
		virtual void startCreate() {};
		virtual unsigned mouseLDown(TFigureEditor*, double, double, unsigned) { return STOP; }
		virtual unsigned mouseMove(TFigureEditor*, double, double, unsigned) { return CONTINUE; }
		virtual unsigned mouseLUp(TFigureEditor*, double, double, unsigned) { return CONTINUE; }

		// editor related stuff for all figures
		//--------------------------------------
		static double Distance2Line(double x, double y, double x1, double y1, double x2, double y2);
		static void DrawHandle(TPencil& pen, double x, double y);
		static bool IsInHandle(double hx, double hy, double x, double y);

		static const double OUT_OF_RANGE = 100.0;
		static const double RANGE = 5.0;
		static const double INSIDE = 2.5;

		// strange stuff
		//--------------------------------------
		void Adjust() {};
		bool IsInside(double px, double py) {
			return (x<=px && px<=x+w && y<=py && py<=y+h);
		}

		// storage stuff
		//--------------------------------------
		void store(TOutBinStream&, ulong);
		void restore(TInBinStream&, ulong);

		// storage stuff for all figures
		//-------------------------------------- 
		static void InitStorage();
		static void Store(TOutBinStream &stream, TFigure *gadget)
      { serialize.Store(stream, gadget); }
		static TFigure* Restore(TInBinStream &stream)
			{ return (TFigure*)serialize.Restore(stream); }
		static TSerializeBase serialize;
};

class TFigRectangle:
	public TFigure
{
		typedef TFigure super;
	public:
		TFigRectangle() {
			filled = false;
			line_color.Set(0,0,0);
			fill_color.Set(0,0,0);
		}
		void paint(TPencil &,bool);

		double distance(double x, double y);
		void paintHandles(TPencil &);
		void translate(double dx, double dy);
		int getHandle(double x, double y);
		int translateHandle(int handle, double mx, double my);

		void startCreate();
		unsigned stop();

		unsigned mouseLDown(TFigureEditor*, double, double, unsigned);
		unsigned mouseMove(TFigureEditor*, double, double, unsigned);
		unsigned mouseLUp(TFigureEditor*, double, double, unsigned);
		
		TCloneable* clone() { return new TFigRectangle(*this); }

		void store(TOutBinStream&, ulong);
		void restore(TInBinStream&, ulong);
};

class TFigCircle:
	public TFigRectangle
{
	public:
		void paint(TPencil &,bool);
		
		double distance(double x, double y);
		
		TCloneable* clone() { return new TFigCircle(*this); }
};

class TFigText:
	public TFigRectangle
{
		typedef TFigRectangle super;
	public:
		TFigText() {w=h=0;}
		void paint(TPencil &,bool);

		double distance(double x, double y);
		int getHandle(double x, double y);

		bool startInPlace();
		void startCreate();
		unsigned stop();

		unsigned keyDown(TFigureEditor*, TKey, char*, unsigned);
		unsigned mouseLDown(TFigureEditor*, double, double, unsigned);
		unsigned mouseMove(TFigureEditor*, double, double, unsigned);
		unsigned mouseLUp(TFigureEditor*, double, double, unsigned);

		TCloneable* clone() { return new TFigText(*this); }

		void store(TOutBinStream&, ulong);
		void restore(TInBinStream&, ulong);
		
	protected:
		string text;
		void CalcSize();
};

class TFigGroup:
	public TFigRectangle
{
	public:
		TFigGroup() {
			dx=dy=0;
		}
		TFigGroup(const TFigGroup &g) {
			dx=g.dx;
			dy=g.dy;
			figures.insert(figures.begin(), 
										 g.figures.begin(), 
										 g.figures.end());
		}
		~TFigGroup();
		void paint(TPencil& pen, bool selected);
		double distance(double x, double y);
		void translate(double dx, double dy);
		int getHandle(double x, double y) { return -1; }
		int translateHandle(int handle, double x, double y) { return handle; }
		
		void CalcSize();
		double dx, dy;
		typedef vector<TFigure*> TFigureVector;
		TFigureVector figures;

		TCloneable* clone() { return new TFigGroup(*this); }

		void store(TOutBinStream&, ulong);
		void restore(TInBinStream&, ulong);
};
