/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/scrollbar.hh>
#include <toad/listbox.hh>

const int xo = 8;

// this class is also defined in `toadbase.cc'

class TActionDeleteWindow:
	public TAction
{
		TWindow *_window;
	public:
		TActionDeleteWindow(TWindow *w) { _window = w; }
		void execute() { delete _window; }
};


#define DBM(A)

// constructor
//---------------------------------------------------------------------------
TListBox::TListBox(TWindow *parent, const string &title, TLBAdapter *adapter)
	:super(parent,title)
{
	ENTRYEXIT("TListBox::TListBox(TWindow*, const char*, TLBAdapter*)");
	_Init();
	SetAdapter(adapter);
}

void TListBox::_Init()
{
	hscroll = NULL;
	vscroll	= NULL;
	_cy = 0;
	_faked_focus = false;
	
	_type = SINGLE;
	_adapter = NULL;
	DeselectAll();
}

TListBox::~TListBox()
{
	ENTRYEXIT("TListBox::~TListBox()");
}

// Select
//---------------------------------------------------------------------------
//. Set selection to item number 'n' starting with 0.
void TListBox::Select(unsigned n)
{
	if (n>=GetItemCount())
		return;

	switch(_type) {
		case SINGLE:
			if (_single==n)
				return;
			_single = n;
			break;
		case MULTIPLE:
			if (_multiple[n])
				return;
		  _multiple[n] = true;
			break;
	}
	_cy = n;
	Center();
	sigSelect(n);
}

// Deselect
//---------------------------------------------------------------------------
void TListBox::Deselect(unsigned n)
{
	if (n>=GetItemCount())
		return;

	switch(_type) {
		case SINGLE:
			_single = npos;
			break;
		case MULTIPLE:
			_multiple[n]=0;
			break;
	}
	Invalidate();
}

// DeselectAll
//---------------------------------------------------------------------------
void TListBox::DeselectAll()
{
	if (!_adapter)
		return;
	_single = npos;
	_multiple.reserve(_adapter->ItemCount());

	for(unsigned i=0; i<GetItemCount(); i++) {
		if (i>=_multiple.size())
			_multiple.push_back(false);
		else
			_multiple[i]=false;
	}
	Invalidate();
}

// IsSelected
//---------------------------------------------------------------------------
bool TListBox::IsSelected(unsigned n) const
{
	if (n>=GetItemCount())
		return false;

	switch(_type) {
		case SINGLE:
			if (n==npos)
				return false;
			return _single == n;
		case MULTIPLE:
			return _multiple[n];
	}
	
	return false;
}

//.	Returns the number of items in the listbox's TLBAdapter.
//---------------------------------------------------------------------------
unsigned TListBox::GetItemCount() const
{
	return _adapter ? _adapter->ItemCount() : 0;
}

//.	Returns the number of selected items.
//---------------------------------------------------------------------------
unsigned TListBox::GetSelectedItemCount() const
{
	cerr << "toad: unsigned TListBox::GetSelectedItemCount() is not implemented\n";
	return 0;
}

//.	Returns the number of the first visible item in the listbox.
//---------------------------------------------------------------------------
unsigned TListBox::GetTopItemPosition() const
{
	return pos;
}

//.	Make item number 'p' the first visible item.
//---------------------------------------------------------------------------
unsigned TListBox::SetTopItemPosition(unsigned p)
{
	DBM(printf("toad: unsigned TListBox::SetTopItemPosition(unsigned p) is not validated yet\n");)

	pos = p;
	
	unsigned ih=_adapter->ItemHeight()+1;
	visible = Height()/ih;

	if (_adapter->ItemCount()<visible)	{
		pos = 0;
	}	else {
		if ( pos > _adapter->ItemCount()-visible )
				pos = _adapter->ItemCount()-visible;
	}

	if (vscroll)
		vscroll->SetValue(pos);
	
	UpdateWindow(true);
	return pos;
}

//.	Returns the TLBAdapter associated with the listbox.
//---------------------------------------------------------------------------
TLBAdapter* TListBox::GetAdapter()
{
	return _adapter;
}

//. Set a new adapter an destroy the old one
//---------------------------------------------------------------------------
void TListBox::SetAdapter(TLBAdapter *adapter)
{
	_adapter = adapter;
	pos = 0;
	_cy = 0;
	if (_adapter)
		_multiple.reserve(_adapter->ItemCount());
	DeselectAll();
	adjust_sb();
}

//.	Get the number of the first selected item. Returns ((unsigned)-1) when
//.	there's no item selected.
//---------------------------------------------------------------------------
unsigned TListBox::GetFirstSelectedItem()
{
	switch(_type)
	{
		case SINGLE:
			return _single;
		case MULTIPLE:
			cerr << "toad: unsigned TListBox::GetFirstSelectedItem() isn't implemented for TLB_MULTIPLE_SELECT\n";
			break;
	}
	return npos;
}

//.	Get the number of the next selected item. Returns ((unsigned)-1) when there
//.	are no more items.
//---------------------------------------------------------------------------
unsigned TListBox::GetNextSelectedItem()
{
	switch(_type) {
		case SINGLE:
			return npos;
		case MULTIPLE:
			cerr << "toad: unsigned TListBox::GetNextSelectedItem() isn't implemented yet\n";
	}
	return npos;
}

//.	The type can be:
//.	<UL>
//.   <LI>
//.			TListBox::SINGLE
//.		<LI>
//.			TListBox::MULTIPLE
//.	</UL>
//---------------------------------------------------------------------------
void TListBox::SetType(TListBox::EType type)
{
	_type = type;
	if (type==MULTIPLE && _adapter) {
		_multiple.reserve(_adapter->ItemCount());
	}
}

void TListBox::AdapterChanged()
{
	_cy = 0;
	DeselectAll();
	adjust_sb();
	UpdateWindow(true);
}

// destroy
//---------------------------------------------------------------------------
void TListBox::destroy()
{
	ENTRYEXIT("TListBox::~TListBox()");
	hscroll = NULL;
	vscroll = NULL;
}

// paint
//---------------------------------------------------------------------------
void TListBox::paint()
{
	if (!_adapter)
		return;

	TPen pen(this);
	unsigned p = pos;				// first visible item
	unsigned ey = 0;				// visible item counter
	unsigned y=yo;					// y screen position
	bool bSelected=false;	

	int width = Width();
	if (vscroll && vscroll->IsRealized()) {
		TRectangle r;
		vscroll->GetShape(&r);
		width=r.x;
	}

	while(ey<visible)	{
		if (p>=_adapter->ItemCount())
			return;
		switch(_type) {
			case SINGLE:
				bSelected = (_single == p)?true:false;
				break;
			case MULTIPLE:
				bSelected = _multiple[p];
				break;
		}
		if (bSelected) {
			pen.SetColor(TColor::SELECTED);
			pen.FillRectangle(0,y,Width(),pen.Height());
			pen.SetColor(TColor::SELECTED_TEXT);
			_adapter->PrintItem(xo,y,p,pen);
			pen.SetColor(TColor::BLACK);
		}	else {
			_adapter->PrintItem(xo,y,p,pen);
		}
		if ((IsFocus() || _faked_focus) && p==_cy) {
			pen.ClrClipBox();
			pen.DrawRectangle(0,y-1,width,pen.Height()+2);
			pen.DrawRectangle(1,y,width-2,pen.Height());
		}
		p++;															// next item
		y+=_adapter->ItemHeight()+1;			// move to next screen position
		ey++;															// visible item counter
	}
}

void TListBox::Center()
{
	unsigned oldpos = pos;
	if (_cy<pos) {
		pos=_cy;
	}
	if (_cy>=pos+visible) {
		pos=_cy-visible+1;
	}
	if (pos!=oldpos) {
		if (vscroll)
			vscroll->SetValue(pos);
#if 0
		int dy = _adapter->ItemHeight()+1;
		TRectangle r(0,yo,Width(),visible*dy);
		ScrollRectangle(r, 0,-dy);
		Invalidate(0,yo-1,Width(),1);
		PaintNow();
#endif
	}
	Invalidate();
}

void TListBox::keyDown(TKey key,char*,unsigned)
{
	unsigned oy = _cy;
	switch(key) {
		case TK_DOWN:
			if (_cy<GetItemCount()-1)
				_cy++;
			Center();
			break;
		case TK_UP:
			if (_cy>0)
				_cy--;
			Center();
			break;
		case ' ':
			if (IsSelected(_cy))
				Deselect(_cy);
			else
				Select(_cy);
			break;
	}
	if (oy==_cy)
		return;
	Invalidate();
}

// mouseLDown
//---------------------------------------------------------------------------
void TListBox::mouseLDown(int x,int y,unsigned m)
{
	ENTRYEXIT("void TListBox::mouseLDown(int,int,unsigned)");
	SetFocus();

	if (!_adapter)
		return;
	
	unsigned n=(y-yo)/(_adapter->ItemHeight()+1)+pos;
	if (n>=_adapter->ItemCount())
		return;

	// calculate screen position of selected item
	int ix,iy;
	ix = xo;
	iy = yo + _adapter->ItemHeight()+1 * ( n - pos );
	_adapter->mouseLDown(x-ix, y-iy, m, n, ix, iy);
	switch(_type) {
		case SINGLE:
			_single = n;
			break;
		case MULTIPLE:
			_multiple[n]=!_multiple[n];
			break;
	}
	_cy=n;
	Invalidate();
	if (m & MK_DOUBLE)
		sigDoubleClick();
	else
		sigSelect(_cy);
}

// resize
//---------------------------------------------------------------------------
void TListBox::resize()
{
	ENTRYEXIT("void TListBox::adjustC2W()");
	adjust_sb();
}

// adjust_sb
//---------------------------------------------------------------------------
void TListBox::adjust_sb()
{
	if (!_adapter)
		return;

	ENTRYEXIT("void TListBox::adjust_sb()");
	unsigned ih=_adapter->ItemHeight()+1;
	visible = Height()/ih;
	yo			= (Height()%ih)>>1;
	if (visible<_adapter->ItemCount())	{
		if (!vscroll)	{
			vscroll = new TVScrollBar(this, "listbox.vscrollbar");
			vscroll->bNoFocus = true;
			
			// this one doesn't do anymore... :(
			OLD_CONNECT(this ,cmdVValueChanged, vscroll,vscroll->sigValueChanged);
//			vscroll->sigValueChanged.Add(this, &cmdVValueChanged, vscroll);

//_toad_connect(this,&cmdVValueChanged, vscroll,&vscroll->sigValueChanged);
			
//			vscroll->sigValueChanged.Add(this, &cmdVValueChanged);
			
			OLD_CONNECT( this , scrollDown	 , vscroll,vscroll->sigIncrement);
			OLD_CONNECT(this,scrollUp	 , vscroll,vscroll->sigDecrement);
		}
		TRectangle r;
		vscroll->GetShape(&r);
		vscroll->SetRange(0,_adapter->ItemCount()-1);
		vscroll->SetVisible(visible);
		vscroll->SetShape(Width()-r.w+1,-1,TSIZE_PREVIOUS,Height()+2);
		vscroll->SetValue(pos);
		if (IsRealized() && !vscroll->IsRealized())
			vscroll->Create();
	}	else {
		if (vscroll) {
			SendMessage(new TActionDeleteWindow(vscroll));
			vscroll = NULL;
		}
	}
}

// cmdVValueChanged
//---------------------------------------------------------------------------
void TListBox::cmdVValueChanged(TVScrollBar *sb)
{
//	printf("void TListBox::msgVValueChanged(int)\n");
	pos = sb->Value();
	UpdateWindow(true);
}

// scrollDown
//---------------------------------------------------------------------------
void TListBox::scrollDown()
{
	pos++;
	int dy = _adapter->ItemHeight()+1;
	TRectangle r(0,yo,Width(),visible*dy);
	ScrollRectangle(r, 0,-dy);
	Invalidate(0,yo-1,Width(),1);
	PaintNow();
}

// scroll up
//---------------------------------------------------------------------------
void TListBox::scrollUp()
{
	pos--;
	int dy = _adapter->ItemHeight()+1;
	TRectangle r(0,yo,Width(),visible*dy);
	ScrollRectangle(r, 0,+dy);
	Invalidate(0,yo+visible*dy-1,Width(),1);
	PaintNow();
}

// cmdHScroll
//---------------------------------------------------------------------------
void TListBox::cmdHScroll()
{
}
