/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/pushbutton.hh>

#define _TOAD_PRIVATE

#include <toad/mdishell.hh>

// this class is also defined in `toadbase.cc'

class TActionDeleteWindow:
	public TAction
{
		TWindow *_window;
	public:
		TActionDeleteWindow(TWindow *w) { _window = w; }
		void execute() { delete _window; }
};

class TTitleBar: public TWindow
{
	public:
		TTitleBar(TMDIShell *p,const string &t="");
	protected:
		void create();
		void mouseLDown(int,int,unsigned);
		void mouseLUp(int,int,unsigned);
		void mouseMove(int,int,unsigned);
		void paint();
	private:
		int _x,_y;
		bool bMoving;
		int ox,oy;
		TMDIShell *shell;
		void cmdButton();
};

class TXButton: public TPushButton
{
	public:
		TXButton(TWindow* p,const string &t):TPushButton(p,t){bNoFocus = true;};
		void paint();
};

/*****************************************************************************
 *                                                                           *
 * TMDIShell																                                 *
 *                                                                           *
 *****************************************************************************/
TMDIShell::TMDIShell(TWindow *p,const string &t):TWindow(p,t)
{
	bFocusManager = true;
	tb = NULL;		// IT'S A MUST OR YOU'LL CRASH BECAUSE OF 'childNotify'
	tb = new TTitleBar(this);
	tb->SetShape(-1,-1, _w+2,18);
	wnd = NULL;
}

void TMDIShell::childNotify(TWindow *child, EChildNotify type)
{
	switch(type)
	{
		case TCHILD_ADD:
			if (tb==NULL || child==tb)
				return;
			wnd = child;
			// break;
		case TCHILD_TITLE:
			if (tb)	{ // this..
				SetTitle(child->Title());
				tb->Invalidate();
			}
			break;
		case TCHILD_FOCUS:
			if (tb)	{
				tb->Invalidate();
			}
			break;
		case TCHILD_BEFORE_CREATE:
			// This is the last and best moment to unset the bShell flag.
			// Otherwise 'wnd' would get managed by the real window manager
			// and not by this TMDIShell object. (If! bShell was true.)
			if (wnd && wnd==child) {
				wnd->bShell = false;
				SetPosition(wnd->XPos(), wnd->YPos());
			}
			break;
		case TCHILD_POSITION:
			if (wnd && wnd==child) {
				SetPosition(wnd->XPos(), wnd->YPos());
			}
			break;
		case TCHILD_RESIZE:
			resize();
			break;
		case TCHILD_REMOVE:
			SendMessage(new TActionDeleteWindow(this));
			break;
		default:;
	}
}

void TMDIShell::created()
{
	resize();
}

void TMDIShell::resize()
{
	TWindow *w;
	TRectangle shape;
	w=FirstChild();
	while(w) {	// this loop assumes only 2 children!!!
		if (w!=tb) {
			tb->SetShape(-1,-1,_w+2,18);
			w->GetShape(&shape);
			w->SetSuppressMessages(true);
			w->SetPosition(-1,16);
			w->SetSuppressMessages(false);
			SetShape(TPOS_PREVIOUS,TPOS_PREVIOUS,shape.w, 17+shape.h );
			break;
		}
		w = NextSibling(w);
	}
}

void TMDIShell::focus()
{
	tb->Invalidate();
}

/*****************************************************************************
 *                                                                           *
 * TTitleBar 																																 *
 *                                                                           *
 *****************************************************************************/

TTitleBar::TTitleBar(TMDIShell *p, const string &t="")
	:TWindow(p,t)
{
	shell = p;
	SetBackground(TColor::INACTIVECAPTION);
	SetMouseMoveMessages(TMMM_LBUTTON);
}

void TTitleBar::create()
{
	TXButton *w;
	w = new TXButton(this, "titlebar.xbutton");
		OLD_CONNECT(this,cmdButton, w,w->sigActivate);
		w->SetBorder(false);
		w->SetPosition(1,1);
		w->SetSize(_h-2,_h-2);
	bMoving=false;
}

void TTitleBar::cmdButton()
{
	Parent()->closeRequest();
}

void TTitleBar::paint()
{
	TPen pen(this);
	pen.SetFont(TOADBase::bold_font);	// hack !!!
	if (Parent()->IsFocus()) {
		SetBackground(TColor::CAPTION);
		pen.SetColor(TColor::CAPTIONTEXT);
	}	else {
		SetBackground(TColor::INACTIVECAPTION);
		pen.SetColor(TColor::INACTIVECAPTIONTEXT);
	}
	ClearWindow();
	pen.ClrClipBox();
	int x = (_w-pen.TextWidth(Parent()->Title())) >> 1;
	int y = (_h-pen.Height()) >> 1;
	pen.DrawString(x, y, Parent()->Title());
}

void TTitleBar::mouseLDown(int x,int y,unsigned)
{
  _x=x; _y=y;
  if (shell->wnd)
	  shell->wnd->SetFocus();
  shell->RaiseWindow();
}

void TTitleBar::mouseMove(int mx,int my,unsigned)
{
	TRectangle shape;
	TPen pen(Parent()->Parent());
	pen.SetClipChildren(false);
	pen.SetMode(TPen::INVERT);
	pen.SetLineWidth(3);

	Parent()->GetShape(&shape);
	int x=mx-_x+shape.x;
	int y=my-_y+shape.y;

	if (!bMoving)	{
		// XGrabServer
		GrabMouse(TMMM_PREVIOUS,Parent()->Parent());
		bMoving = true;
		LockPaintQueue();
	}	else {
		pen.DrawRectangle(ox,oy,shape.w,shape.h);
	}
	pen.DrawRectangle(x,y,shape.w,shape.h);
	ox=x; oy=y;
}

void TTitleBar::mouseLUp(int,int,unsigned)
{
	if (bMoving) {
		UnlockPaintQueue();
		UngrabMouse();
		// XGrabServer
		TPen pen(Parent()->Parent());
		pen.SetClipChildren(false);
		pen.SetMode(TPen::INVERT);
		pen.SetLineWidth(3);
		TRectangle shape;
		Parent()->GetShape(&shape);
		pen.DrawRectangle(ox,oy,shape.w,shape.h);
		bMoving = false;

		Parent()->SetPosition(ox,oy);
	}
}

/*****************************************************************************
 *                                                                           *
 * TXButton  																																 *
 *                                                                           *
 *****************************************************************************/
void TXButton::paint()
{
	TPen pen(this);
	DrawShadow(pen, bDown && bInside);
	int n=bDown && bInside?1:0;

	pen.SetColor(TColor::BTNTEXT);
	pen.DrawLine(3+n,3+n, _w-4+n,_h-4+n);
	pen.DrawLine(3+n,4+n, _w-5+n,_h-4+n);
	pen.DrawLine(4+n,3+n, _w-4+n,_h-5+n);
	pen.DrawLine(_w-4+n,3+n, 3+n,_h-4+n);
	pen.DrawLine(_w-5+n,3+n, 3+n,_h-5+n);
	pen.DrawLine(_w-4+n,4+n, 4+n,_h-4+n);
}
