/******************************************************************************
 JXFontNameMenu.cc

	Since we have to update our index before anybody can request its new
	value, the correct way to use this object is to listen for the
	NameChanged message, not the ItemSelected message.  In addition, since
	we are responsible for marking the menu item, clients should listen
	for NameNeedsUpdate instead of NeedsUpdate.

	BASE CLASS = JXTextMenu

	Copyright  1996 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXFontNameMenu.h>
#include <jXGlobals.h>
#include <JString.h>
#include <JFontManager.h>
#include <jAssert.h>

// JBroadcaster message types

const JCharacter* JXFontNameMenu::kNameNeedsUpdate = "NameNeedsUpdate::JXFontNameMenu";
const JCharacter* JXFontNameMenu::kNameChanged     = "NameChanged::JXFontNameMenu";

/******************************************************************************
 Create (static)

	Work-around for egcs thunks bug.

 ******************************************************************************/

JXFontNameMenu*
JXFontNameMenu::Create
	(
	const JCharacter*	title,
	JXContainer*		enclosure,
	const HSizingOption	hSizing,
	const VSizingOption	vSizing,
	const JCoordinate	x,
	const JCoordinate	y,
	const JCoordinate	w,
	const JCoordinate	h
	)
{
	JXFontNameMenu* menu =
		new JXFontNameMenu(title, enclosure, hSizing, vSizing, x,y, w,h);
	assert( menu != NULL );
	menu->BuildMenu();
	return menu;
}

JXFontNameMenu*
JXFontNameMenu::Create
	(
	JXMenu*			owner,
	const JIndex	itemIndex,
	JXContainer*	enclosure
	)
{
	JXFontNameMenu* menu = new JXFontNameMenu(owner, itemIndex, enclosure);
	assert( menu != NULL );
	menu->BuildMenu();
	return menu;
}

/******************************************************************************
 Constructor (protected)

 ******************************************************************************/

JXFontNameMenu::JXFontNameMenu
	(
	const JCharacter*	title,
	JXContainer*		enclosure,
	const HSizingOption	hSizing,
	const VSizingOption	vSizing,
	const JCoordinate	x,
	const JCoordinate	y,
	const JCoordinate	w,
	const JCoordinate	h
	)
	:
	JXTextMenu(title, enclosure, hSizing, vSizing, x,y, w,h)
{
	itsBroadcastNameChangeFlag = kTrue;
//	BuildMenu();	// egcs thunks
}

JXFontNameMenu::JXFontNameMenu
	(
	JXMenu*			owner,
	const JIndex	itemIndex,
	JXContainer*	enclosure
	)
	:
	JXTextMenu(owner, itemIndex, enclosure)
{
	itsBroadcastNameChangeFlag = kTrue;
//	BuildMenu();	// egcs thunks
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXFontNameMenu::~JXFontNameMenu()
{
}

/******************************************************************************
 SetFontName

 ******************************************************************************/

void
JXFontNameMenu::SetFontName
	(
	const JCharacter* name
	)
{
	const JIndex count = GetItemCount();
	for (JIndex i=1; i<=count; i++)
		{
		if (GetItemText(i) == name)
			{
			itsFontIndex = i;
			SetPopupChoice(itsFontIndex);
			if (itsBroadcastNameChangeFlag)
				{
				Broadcast(NameChanged());
				}
			break;
			}
		}
}

/******************************************************************************
 BuildMenu (private)

 ******************************************************************************/

void
JXFontNameMenu::BuildMenu()
{
	RemoveAllItems();

	// Until JPSPrinter can embed fonts in a Postscript file, we are limited
	// to only the standard Postscript fonts.

	AppendItem(JXGetCourierFontName(), kTrue, kTrue);
	SetItemFontName(1, JXGetCourierFontName());

	AppendItem(JXGetHelveticaFontName(), kTrue, kTrue);
	SetItemFontName(2, JXGetHelveticaFontName());

	AppendItem(JXGetSymbolFontName(), kTrue, kTrue);
	SetItemFontName(3, JXGetSymbolFontName());

	AppendItem(JXGetTimesFontName(), kTrue, kTrue);
	SetItemFontName(4, JXGetTimesFontName());
/*
	JPtrArray<JString> fontNames;
	(GetFontManager())->GetFontNames(&fontNames);

	const JSize count = fontNames.GetElementCount();
	assert( count > 0 );
	for (JIndex i=1; i<=count; i++)
		{
		const JString* fontName = fontNames.NthElement(i);
		AppendItem(*fontName, kTrue, kTrue);
		SetItemFontName(i, *fontName);
		}
	fontNames.DeleteAll();
*/
	SetUpdateAction(kDisableNone);

	SetFontName(JGetDefaultFontName());
	ListenTo(this);
}

/******************************************************************************
 Receive (protected)

 ******************************************************************************/

void
JXFontNameMenu::Receive
	(
	JBroadcaster*	sender,
	const Message&	message
	)
{
	if (sender == this && message.Is(JXMenu::kNeedsUpdate))
		{
		itsBroadcastNameChangeFlag = kFalse;
		Broadcast(NameNeedsUpdate());
		CheckItem(itsFontIndex);
		itsBroadcastNameChangeFlag = kTrue;
		}
	else if (sender == this && message.Is(JXMenu::kItemSelected))
		{
		const JXMenu::ItemSelected* selection =
			dynamic_cast(const JXMenu::ItemSelected*, &message);
		assert( selection != NULL );
		itsFontIndex = selection->GetIndex();
		Broadcast(NameChanged());
		}

	else
		{
		JXTextMenu::Receive(sender, message);
		}
}

/******************************************************************************
 SetToPopupChoice

 ******************************************************************************/

void
JXFontNameMenu::SetToPopupChoice
	(
	const JBoolean isPopup
	)
{
	JXTextMenu::SetToPopupChoice(isPopup, itsFontIndex);
}
