/******************************************************************************
 JXPSPrintSetupDialog.cc

	Clients must call BuildWindow() after constructing the object.
	Derived classes must override BuildWindow() and call SetObjects()
	at the end of their implementation.

	BASE CLASS = JXDialogDirector

	Copyright  1996-99 by John Lindal. All rights reserved.

 ******************************************************************************/

#include <JXPSPrintSetupDialog.h>
#include <JXWindow.h>
#include <JXTextButton.h>
#include <JXStaticText.h>
#include <JXIntegerInput.h>
#include <JXTextCheckbox.h>
#include <JXRadioGroup.h>
#include <JXTextRadioButton.h>
#include <JXFileNameDisplay.h>
#include <jXGlobals.h>
#include <JString.h>
#include <jFileUtil.h>
#include <jDirUtil.h>
#include <jAssert.h>

// mappings from dialog window to JPSPrinter options

enum
{
	kPrintToPrinterID = 1,
	kPrintToFileID
};
static const JXPSPrinter::Destination kIndexToDest[] =
{
	JXPSPrinter::kPrintToPrinter,
	JXPSPrinter::kPrintToFile
};
static const JSize kDestCount = sizeof(kIndexToDest)/sizeof(JXPSPrinter::Destination);

/******************************************************************************
 Constructor

 ******************************************************************************/

JXPSPrintSetupDialog::JXPSPrintSetupDialog()
	:
	JXDialogDirector(JXGetApplication(), kTrue)
{
}

/******************************************************************************
 Destructor

 ******************************************************************************/

JXPSPrintSetupDialog::~JXPSPrintSetupDialog()
{
}

/******************************************************************************
 BuildWindow

 ******************************************************************************/

void
JXPSPrintSetupDialog::BuildWindow
	(
	const JXPSPrinter::Destination	dest,
	const JCharacter*				printCmd,
	const JCharacter*				fileName,
	const JBoolean					collate,
	const JBoolean					bw
	)
{
// begin JXLayout

    JXWindow* window = new JXWindow(this, 370,250, "");
    assert( window != NULL );
    SetWindow(window);

    itsPrintCmdLabel =
        new JXStaticText("Print command:", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 10,70, 100,19);
    assert( itsPrintCmdLabel != NULL );

    JXTextButton* okButton =
        new JXTextButton("Print", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 219,219, 72,22);
    assert( okButton != NULL );
    okButton->SetShortcuts("^M");

    JXTextButton* cancelButton =
        new JXTextButton("Cancel", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 70,220, 70,20);
    assert( cancelButton != NULL );

    JXStaticText* obj1 =
        new JXStaticText("Destination:", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 50,30, 80,20);
    assert( obj1 != NULL );

    itsDestination =
        new JXRadioGroup(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 135,20, 139,39);
    assert( itsDestination != NULL );

    JXTextRadioButton* obj2 =
        new JXTextRadioButton(1, "Printer", itsDestination,
                    JXWidget::kHElastic, JXWidget::kVElastic, 5,8, 70,20);
    assert( obj2 != NULL );
    obj2->SetShortcuts("#P");

    JXTextRadioButton* obj3 =
        new JXTextRadioButton(2, "File", itsDestination,
                    JXWidget::kHElastic, JXWidget::kVElastic, 75,8, 50,20);
    assert( obj3 != NULL );
    obj3->SetShortcuts("#F");

    itsPrintCmd =
        new JXInputField(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 110,70, 240,20);
    assert( itsPrintCmd != NULL );

    itsChooseFileButton =
        new JXTextButton("Choose file", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 30,70, 80,20);
    assert( itsChooseFileButton != NULL );
    itsChooseFileButton->SetShortcuts("#O");

    itsCopyCount =
        new JXIntegerInput(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 170,110, 40,20);
    assert( itsCopyCount != NULL );

    itsBWCheckbox =
        new JXTextCheckbox("Black & White output", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 110,190, 150,20);
    assert( itsBWCheckbox != NULL );
    itsBWCheckbox->SetShortcuts("#B");

    itsFirstPageIndex =
        new JXIntegerInput(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 230,150, 40,20);
    assert( itsFirstPageIndex != NULL );

    itsLastPageIndex =
        new JXIntegerInput(window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 290,150, 40,20);
    assert( itsLastPageIndex != NULL );

    itsPrintAllCB =
        new JXTextCheckbox("Print all pages", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 40,150, 120,20);
    assert( itsPrintAllCB != NULL );
    itsPrintAllCB->SetShortcuts("#A");

    itsFirstPageIndexLabel =
        new JXStaticText("Print page", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 163,153, 66,20);
    assert( itsFirstPageIndexLabel != NULL );

    itsLastPageIndexLabel =
        new JXStaticText("to", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 273,153, 16,20);
    assert( itsLastPageIndexLabel != NULL );

    JXStaticText* obj4 =
        new JXStaticText("Number of copies:", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 55,110, 115,20);
    assert( obj4 != NULL );

    itsCollateCB =
        new JXTextCheckbox("Collate", window,
                    JXWidget::kHElastic, JXWidget::kVElastic, 240,110, 70,20);
    assert( itsCollateCB != NULL );
    itsCollateCB->SetShortcuts("#E");

// end JXLayout

	SetObjects(okButton, cancelButton, itsDestination, dest,
			   itsPrintCmdLabel, itsPrintCmd, printCmd,
			   itsChooseFileButton, fileName, itsCopyCount,
			   itsCollateCB, collate, itsBWCheckbox, bw,
			   itsPrintAllCB, itsFirstPageIndexLabel,
			   itsFirstPageIndex, itsLastPageIndexLabel, itsLastPageIndex);
}

/******************************************************************************
 SetObjects (protected)

 ******************************************************************************/

void
JXPSPrintSetupDialog::SetObjects
	(
	JXTextButton*					okButton,
	JXTextButton*					cancelButton,
	JXRadioGroup*					destinationRG,
	const JXPSPrinter::Destination	dest,
	JXStaticText*					printCmdLabel,
	JXInputField*					printCmdInput,
	const JCharacter*				printCmd,
	JXTextButton*					chooseFileButton,
	const JCharacter*				fileName,
	JXIntegerInput*					copyCount,
	JXTextCheckbox*					collateCheckbox,
	const JBoolean					collate,
	JXTextCheckbox*					bwCheckbox,
	const JBoolean					bw,
	JXTextCheckbox*					printAllCheckbox,
	JXStaticText*					firstPageIndexLabel,
	JXIntegerInput*					firstPageIndex,
	JXStaticText*					lastPageIndexLabel,
	JXIntegerInput*					lastPageIndex
	)
{
	itsPrintButton         = okButton;
	itsDestination         = destinationRG;
	itsPrintCmdLabel       = printCmdLabel;
	itsPrintCmd            = printCmdInput;
	itsChooseFileButton    = chooseFileButton;
	itsCopyCount           = copyCount;
	itsCollateCB           = collateCheckbox;
	itsBWCheckbox          = bwCheckbox;
	itsPrintAllCB          = printAllCheckbox;
	itsFirstPageIndexLabel = firstPageIndexLabel;
	itsFirstPageIndex      = firstPageIndex;
	itsLastPageIndexLabel  = lastPageIndexLabel;
	itsLastPageIndex       = lastPageIndex;

	JXWindow* window = itsDestination->GetWindow();
	window->SetTitle("Print Setup");
	SetButtons(okButton, cancelButton);

	const JRect r = itsPrintCmd->GetFrame();
	itsFileNameDisplay =
		new JXFileNameDisplay(fileName, window,
							  JXWidget::kHElastic, JXWidget::kVElastic,
							  r.left, r.top, r.width(), r.height());
	assert( itsFileNameDisplay != NULL );

	itsPrintCmd->SetText(printCmd);

	ListenTo(itsDestination);
	ListenTo(itsChooseFileButton);
	ListenTo(itsPrintAllCB);

	JBoolean foundDest = kFalse;
	for (JIndex i=1; i<=kDestCount; i++)
		{
		if (kIndexToDest[i-1] == dest)
			{
			SetDestination(i);
			foundDest = kTrue;
			break;
			}
		}
	assert( foundDest );

	itsCopyCount->SetValue(1);
	itsCopyCount->SetLowerLimit(1);

	PrintAllPages(kTrue);

	itsFirstPageIndex->SetLowerLimit(1);
	itsFirstPageIndex->SetIsRequired(kFalse);

	itsLastPageIndex->SetLowerLimit(1);
	itsLastPageIndex->SetIsRequired(kFalse);

	itsCollateCB->SetState(collate);
	itsBWCheckbox->SetState(bw);
}

/******************************************************************************
 OKToDeactivate (virtual protected)

 ******************************************************************************/

JBoolean
JXPSPrintSetupDialog::OKToDeactivate()
{
	if (!JXDialogDirector::OKToDeactivate())
		{
		return kFalse;
		}
	else if (Cancelled())
		{
		return kTrue;
		}

	if (itsDestination->GetSelectedItem() == kPrintToFileID)
		{
		const JString& fullName = itsFileNameDisplay->GetText();
		if (fullName.IsEmpty())
			{
			(JGetUserNotification())->ReportError(
				"Please specify a destination file.");
			return kFalse;
			}

		JString path, fileName;
		JSplitPathAndName(fullName, &path, &fileName);
		const JBoolean fileExists = JFileExists(fullName);
		if (!JDirectoryExists(path))
			{
			(JGetUserNotification())->ReportError(
				"The specified directory does not exist.");
			return kFalse;
			}
		else if (!fileExists && !JDirectoryWritable(path))
			{
			(JGetUserNotification())->ReportError(
				"The file cannot be created because the you do not have write access "
				"to the specified directory.");
			return kFalse;
			}
		else if (fileExists && !JFileWritable(fullName))
			{
			(JGetUserNotification())->ReportError(
				"You do not have write access to the specified file.");
			return kFalse;
			}
		}

	return kTrue;
}

/******************************************************************************
 Receive (protected)

 ******************************************************************************/

void
JXPSPrintSetupDialog::Receive
	(
	JBroadcaster*	sender,
	const Message&	message
	)
{
	if (sender == itsDestination && message.Is(JXRadioGroup::kSelectionChanged))
		{
		const JXRadioGroup::SelectionChanged* selection =
			dynamic_cast(const JXRadioGroup::SelectionChanged*, &message);
		assert( selection != NULL );
		SetDestination(selection->GetID());
		}

	else if (sender == itsChooseFileButton && message.Is(JXButton::kPushed))
		{
		ChooseDestinationFile();
		}

	else if (sender == itsPrintAllCB && message.Is(JXCheckbox::kPushed))
		{
		PrintAllPages(itsPrintAllCB->IsChecked());
		}

	else
		{
		JXDialogDirector::Receive(sender, message);
		}
}

/******************************************************************************
 SetDestination (private)

 ******************************************************************************/

void
JXPSPrintSetupDialog::SetDestination
	(
	const JIndex id
	)
{
	itsDestination->SelectItem(id);

	if (id == kPrintToPrinterID)
		{
		itsPrintButton->Activate();
		itsPrintCmdLabel->Show();
		itsPrintCmd->Show();
		itsPrintCmd->SetIsRequired(kTrue);
		itsPrintCmd->Focus();
		itsCollateCB->Show();
		itsChooseFileButton->Hide();
		itsFileNameDisplay->Hide();
		}
	else
		{
		assert( id == kPrintToFileID );

		itsPrintCmdLabel->Hide();
		itsPrintCmd->Hide();
		itsPrintCmd->SetIsRequired(kFalse);
		itsCollateCB->Hide();
		itsChooseFileButton->Show();
		itsFileNameDisplay->Show();

		if ((itsFileNameDisplay->GetText()).IsEmpty())
			{
			ChooseDestinationFile();
			}
		}
}

/******************************************************************************
 PrintAllPages (private)

 ******************************************************************************/

void
JXPSPrintSetupDialog::PrintAllPages
	(
	const JBoolean all
	)
{
	itsPrintAllCB->SetState(all);

	if (all)
		{
		if (itsFirstPageIndex->HasFocus() || itsLastPageIndex->HasFocus())
			{
			(GetWindow())->KillFocus();		// values become irrelevant
			(GetWindow())->SwitchFocusToFirstWidget();
			}
		itsFirstPageIndex->Hide();
		itsLastPageIndex->Hide();
		itsFirstPageIndexLabel->Hide();
		itsLastPageIndexLabel->Hide();
		}
	else
		{
		itsFirstPageIndex->Show();
		itsFirstPageIndex->Focus();
		itsLastPageIndex->Show();
		itsFirstPageIndexLabel->Show();
		itsLastPageIndexLabel->Show();
		}
}

/******************************************************************************
 ChooseDestinationFile (private)

 ******************************************************************************/

void
JXPSPrintSetupDialog::ChooseDestinationFile()
{
	JString fileName;
	if ((JGetChooseSaveFile())->SaveFile("Save PostScript file as:", NULL,
										 itsFileNameDisplay->GetText(), &fileName))
		{
		itsFileNameDisplay->SetText(fileName);
		itsPrintButton->Activate();
		}
	else if ((itsFileNameDisplay->GetText()).IsEmpty())
		{
		itsPrintButton->Deactivate();
		}
}

/******************************************************************************
 SetParameters

 ******************************************************************************/

void
JXPSPrintSetupDialog::SetParameters
	(
	JXPSPrinter* p
	)
	const
{
	p->SetDestination(kIndexToDest[ itsDestination->GetSelectedItem()-1 ],
					  itsPrintCmd->GetText(), itsFileNameDisplay->GetText());

	JInteger copyCount;
	const JBoolean ok = itsCopyCount->GetValue(&copyCount);
	assert( ok );
	p->SetCopyCount(copyCount);

	const JBoolean printAll = itsPrintAllCB->IsChecked();
	p->PrintAllPages();

	if (!printAll)
		{
		JInteger p1, p2;
		const JBoolean ok1 = itsFirstPageIndex->GetValue(&p1);
		const JBoolean ok2 = itsLastPageIndex->GetValue(&p2);
		if (ok1 && ok2)
			{
			p->SetFirstPageToPrint(JMin(p1, p2));
			p->SetLastPageToPrint(JMax(p1, p2));
			}
		else
			{
			if (ok1)
				{
				p->SetFirstPageToPrint(p1);
				}
			if (ok2)
				{
				p->SetLastPageToPrint(p2);
				}
			}
		}

	p->CollatePages(itsCollateCB->IsChecked());
	p->PSPrintBlackWhite(itsBWCheckbox->IsChecked());
}
