/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/


/* This file contains the definitions for IMPLEMENTING Interactors.
   Normally, this won't be used by users, unless they are implementing
   a new kind of interactor
   
   Designed and implemented by Brad Myers
*/

#ifndef INTER_ADVANCED_H
#define INTER_ADVANCED_H

#include <am_inc.h>

#include GEM__H
#include INTER__H
#include FORMULA__H
#include VALUE_LIST__H  // for Am_Value_List type


// Amount priority of interactors added when running
#define Am_INTER_PRIORITY_DIFF 100.0f

// call this before doing any interactor stuff.  This should be
// called automatically from Am_Initialize();
extern void Am_Initialize_Interactors ();

// for the Am_CURRENT_STATE slot of interactors
enum Am_Inter_State { Am_INTER_WAITING, Am_INTER_RUNNING, Am_INTER_OUTSIDE, 
		       Am_INTER_ABORTING, Am_INTER_ANIMATING };

// type of the function in the various internal _METHOD slots of
// interactors (not commands) 
Am_Define_Method_Type(Am_Inter_Internal_Method, void, 
		      (Am_Object& inter, Am_Object& object,
		       Am_Object& event_window, Am_Input_Event *ev))



/////////////////////////////////////////////////////////////////////////////
// Main top-level input handler.  Figures out which interactors to call for
// the event.  This is called from the Input_Event_Notify method for the
// default event handler class defined in Opal.
/////////////////////////////////////////////////////////////////////////////

extern void Interactor_Input_Event_Notify(Am_Object event_window,
					  Am_Input_Event *ev);


/////////////////////////////////////////////////////////////////////////////
//Debugging: These are used internally to see if should print something
/////////////////////////////////////////////////////////////////////////////

extern bool Am_Inter_Tracing (Am_Inter_Trace_Options trace_code);
extern bool Am_Inter_Tracing (Am_Object inter_to_trace);

#ifdef DEBUG

#define Am_INTER_TRACE_PRINT(condition, printout) \
 if (Am_Inter_Tracing(condition))        \
   cout << printout << endl << flush

#define Am_INTER_TRACE_PRINT_NOENDL(condition, printout) \
 if (Am_Inter_Tracing(condition))        \
   cout << printout

extern void Am_Report_Set_Vis(Am_Object inter, Am_Object obj, bool value);

#define Am_REPORT_VIS_CONDITION(condition, inter, obj, value) \
 if (Am_Inter_Tracing(condition))       		   \
     Am_Report_Set_Vis(inter, obj, value);

#else

#define Am_INTER_TRACE_PRINT(condition, printout)
    /* if not debugging, define it to be nothing */

#define Am_INTER_TRACE_PRINT_NOENDL(condition, printout) 
    /* if not debugging, define it to be nothing */

#define Am_REPORT_VIS_CONDITION(condition, inter, obj, value)
    /* if not debugging, define it to be nothing */

#endif

/////////////////////////////////////////////////////////////////////////////
// Generally useful functions
/////////////////////////////////////////////////////////////////////////////

//checks that object & all owner's are visible (all the way to the Am_Screen).
extern bool Am_Object_And_Owners_Valid_And_Visible(Am_Object obj);

// calls Am_Object_And_Owners_Valid_And_Visible on each object in
// value, which can be a single object or a value list of objects
bool Am_Valid_and_Visible_List_Or_Object(Am_Value value,
					 bool want_visible = true);



// Applies the gridding, if any, defined by the inter's GRID slots
extern void Am_Get_Filtered_Input(Am_Object inter, int x, int y,
				  int& out_x, int & out_y);

// sets the left,top,width,height or x1,y1,x2,y2 of obj from data
extern void Am_Modify_Object_Pos(Am_Object obj, Am_Inter_Location data,
				 bool growing, Am_Object command_obj);

extern void Am_Choice_Set_Value (Am_Object inter);

extern bool Am_Inter_Call_Both_Method(Am_Object& inter, Am_Object& command_obj,
				      Am_Slot_Key method_slot, int x, int y,
				      Am_Object& ref_obj, Am_Input_Char& ic, 
				      Am_Object& object_modified,
				      Am_Inter_Location& points);
extern void Am_Inter_Call_Method(Am_Object& inter_or_cmd,
				 Am_Slot_Key method_slot,
				 int x, int y, Am_Object& ref_obj,
				 Am_Input_Char& ic, Am_Object& object_modified,
				 Am_Inter_Location& points);

typedef Am_Object Am_Impl_Command_Setter(Am_Object inter,
					 Am_Object object_modified,
					 Am_Inter_Location data);
extern void Am_Call_Final_Do_And_Register(Am_Object inter,
				  Am_Object command_obj,
				  int x, int y, Am_Object ref_obj,
				  Am_Input_Char ic,
		 		  Am_Object object_modified,
				  Am_Inter_Location data,
				  Am_Impl_Command_Setter* impl_command_setter);
extern void Am_Register_For_Undo(Am_Object inter, Am_Object command_obj,
				 Am_Object object_modified,
				 Am_Inter_Location data,
				 Am_Impl_Command_Setter* impl_command_setter);

//copies Am_VALUE, Am_OBJECT_MODIFIED and Am_SAVED_OLD_OWNER from
//from_object to its Am_COMMAND part
extern void Am_Copy_Values_To_Command(Am_Object from_object);

//exported for when want to pretend a move-grow interim method is a
//final method, like for scroll bar indicators (scroll_widgets.cc)  
extern void Am_Move_Grow_Register_For_Undo(Am_Object inter);

// formula used in the Am_SET_COMMAND_OLD_OWNER to set the old_owner
// slot of command objects
extern Am_Formula Am_Set_Old_Owner_To_Me;

/////// used for multi-window moves (move-grow and new-points)

// Checks if the window that feedback is in matches the
// window in the interactor, and if not, moves the feedback object to the
// corresponding owner in Am_MULTI_FEEDBACK_OWNERS or Am_MULTI_OWNERS
extern void Am_Check_And_Fix_Feedback_Window(Am_Object &feedback,
					     Am_Object &inter);


// Checks if the window that object is in matches the
// window in interactor (or new_window if supplied), and if not, moves
// the object to the corresponding owner in Am_MULTI_OWNERS.
// new_window is optional, and if missing, use inter's current Am_WINDOW
extern bool Am_Check_And_Fix_Object_Window(Am_Object &obj, Am_Object &inter,
				    Am_Object new_window = Am_No_Object);

//Used when interactor needs to be added or removed from an extra
//window's list, for example when the feedback object is a window
//itself, call this on the window  If add and want_move, then turns
//on mouse-moved events for the window.
extern void Am_Add_Remove_Inter_To_Extra_Window(Am_Object inter,
						Am_Object window, bool add,
						bool want_move);

//Use when Feedback is a window.  Make sure inter_window isn't feedback,
//and if so, find a new window and return it.  Call this AFTER feedback
//has disappeared.  Otherwise, just returns inter_window.
extern Am_Object
  Am_Check_And_Fix_Point_In_Feedback_Window(Am_Object &inter_window,
					    Am_Object &feedback);

//////////////////////////////////////////////////////////////////
// Initialization Routines
//////////////////////////////////////////////////////////////////

extern void Am_Initialize_Top_Command ();
extern void Am_Initialize_Move_Grow_Interactor ();
extern void Am_Initialize_Choice_Interactor ();
extern void Am_Initialize_New_Points_Interactor ();
extern void Am_Initialize_Text_Interactor ();
extern void Am_Initialize_Gesture_Interactor ();

////////////////////////////////////////////////////////////////////////
// For the UNDO mechanisms
////////////////////////////////////////////////////////////////////////



//returns the undo handler attached to the window of inter_or_widget
extern Am_Object Am_Inter_Find_Undo_Handler(Am_Object inter_or_widget);

// command_obj's DO has already been "done", now do the DO methods of
// all the Am_IMPLEMENTATION_PARENT's of command_obj, recursively on up
extern void Am_Process_All_Parent_Commands(Am_Object command_obj,
			   Am_Object undo_handler,
			   const Am_Register_Command_Method &reg_method);


extern Am_Selective_Allowed_Method Am_Standard_Selective_Return_True;
extern Am_Selective_Allowed_Method Am_Selective_Allowed_Return_False;
extern Am_Selective_New_Allowed_Method 
			Am_Selective_New_Allowed_Return_False;
extern Am_Selective_New_Allowed_Method 
			Am_Selective_New_Allowed_Return_True;

extern void Am_Modify_Command_Label_Name(Am_Object cmd, Am_Slot_Key slot);



////////////////////////////////////////////////////////////////////////
// Miscellaneous
////////////////////////////////////////////////////////////////////////

//Initialize the state if not there yet
extern Am_Inter_State Am_Get_Inter_State(Am_Object inter);

//global list of all modal windows currently in force
extern Am_Value_List Am_Modal_Windows;

//like choice interactor, but when repeat on new, actually does repeat
//on same.  This is used by button widgets.
extern Am_Object Am_Choice_Interactor_Repeat_Same;


#endif
