/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

/* This file contains the button-like widgets
   
   Designed and implemented by Brad Myers
*/

#include <am_inc.h>

#include AM_IO__H

#include WIDGETS_ADVANCED__H
#include OBJECT_ADVANCED__H // For cc definition.
#include STANDARD_SLOTS__H
#include VALUE_LIST__H
#include INTER_ADVANCED__H // for Am_Choice_Command_Set_Value and
			           // Am_Choice_Interactor_Repeat_Same
#include OPAL_ADVANCED__H  // for Am_DRAWONABLE, Am_Window_Coordinate

#include WIDGETS__H
#include GEM__H
#include INTER__H
#include OPAL__H

///////////////////////////////////////////////////////////////////////////
// Buttons
///////////////////////////////////////////////////////////////////////////

void Am_Draw_Motif_Radio_Box (int left, int top, int width, int height,
			      bool depressed,
			      const Computed_Colors_Record& rec,
			      Am_Drawonable* draw) {

  const Am_Style top_fill (depressed ? rec.data->shadow_style :
			   rec.data->highlight_style);
  const Am_Style bot_fill (depressed ? rec.data->highlight_style :
			   rec.data->shadow_style);
  const Am_Style inside_fill (depressed ? rec.data->background_style :
			      rec.data->foreground_style);
  
  int da[10];
  // Don't want to have to calculate these things by hand, but
  // right now there's no legal opal-level way to get the points back out of
  // a point list.  Hack: +/- 1 to account for line width of 2.
  da[0] = left + (width + 1)/2 - 1;
  da[1] = top + 1;
  da[2] = left + width - 1;
  da[3] = top + (height + 1)/2;
  da[4] = da[0];
  da[5] = top + height;
  da[6] = left;
  da[7] = da[3];
  da[8] = da[0];
  da[9] = da[1];
  Am_Point_List diamond (da, 10);

  //inside of box
  draw->Draw_Lines(Am_No_Style, inside_fill, diamond, Am_DRAW_COPY);

  //top edges
  draw->Draw_Line(top_fill, da[0], da[1], da[2], da[3]);
  draw->Draw_Line(top_fill, da[6], da[7], da[0], da[1]);
  
  //bottom edges
  draw->Draw_Line(bot_fill, da[2], da[3], da[4], da[5]);
  draw->Draw_Line(bot_fill, da[4], da[5], da[6], da[7]);
}

//////
// Am_Draw_Motif_Button
// Draws a single Motif Button of any style: normal, radio, check, menu
// defaults:

// Draws a single Motif Button
static void Am_Draw_Motif_Button(Am_Object /* self */,
			  int left, int top, int width, int height,
			  const char* string, Am_Object obj,
			  bool interim_selected, bool selected,
			  bool active, bool key_selected, Am_Font font,
			  const Computed_Colors_Record& rec,
			  Am_Drawonable* draw,
			  Am_Button_Type but_type,
			  int box_height, int box_width,
			  bool box_on_left, int align, int offset_left)
{
  int box_left, box_top, real_left, real_top, real_width, real_height;

  if (key_selected) // draw box showing that the keyboard is over this item
    draw->Draw_Rectangle (Am_Key_Border_Line, Am_No_Style, left, top,
			  width, height);
  switch (but_type) {
  case Am_PUSH_BUTTON: 
    // draw the external box, leave 2 pixel border for the key-selected box
    // selected and interim selected look the same in Motif
    Am_Draw_Motif_Box(left+2, top+2, width-4, height-4,
		      interim_selected || selected, rec, draw);
    real_left = left + 4;
    real_width = width - 8;
    real_top = top + 4;
    real_height = height - 8;
    break;
  case Am_CHECK_BUTTON:
    // draw the check box.
    // on left if box_on_left; on right otherwise.  Center vertically.
    // leave 2 pixel border for the key-selected box
    // selected and interim selected look the same in Motif  ??
    box_left = box_on_left ? left + 2 : left + width - 2 - box_width;
    box_top = top + (height - box_height)/2;
    Am_Draw_Motif_Box(box_left, box_top, box_width, box_height,
		      interim_selected || selected, rec, draw);
    real_left = box_on_left ? left + box_width + 3 : left + 2;
    real_width = width - box_width - 4;
    real_top = top;
    real_height = height;
    break;
  case Am_RADIO_BUTTON:
    // draw the radio box
    // on left if box_on_left; on right otherwise.  Center vertically.
    // leave 2 pixel border for the key-selected box
    // selected and interim selected look the same in Motif  ??
    box_left = box_on_left ? left + 2 : left + width - 2 - box_width;
    box_top = top + (height - box_height)/2;
    Am_Draw_Motif_Radio_Box(box_left, box_top, box_width, box_height,
			    interim_selected || selected, rec, draw);
    real_left = box_on_left ? left + box_width + 3 : left + 2;
    real_width = width - box_width - 4;
    real_top = top;
    real_height = height;
    break;
  default:
    Am_Error ("Am_Draw_Motif_Button: incorrect button type.\n");
  }
  // now draw the string if any
  if(string) {
    int str_width, ascent, descent, a, b, str_left, str_top;
    const Am_Style text_style(active ? Am_Black : Am_Motif_Inactive_Stipple);
    
    draw->Get_String_Extents (font, string, strlen (string), str_width,
			      ascent, descent, a, b);
    // center the text
    switch (align) {
    case Am_LEFT_ALIGN:
      str_left = real_left + offset_left;
      break;
    case Am_RIGHT_ALIGN:
      str_left = real_left + (real_width - str_width) - 2;
      break;
    case Am_CENTER_ALIGN:
    default:
      str_left = real_left + (real_width - str_width)/2;
      break;
    }
    str_top = real_top + (real_height - ascent - descent) / 2;
    // set a clip region in case string bigger than the button
    draw->Push_Clip (real_left, real_top, real_width, real_height);
    draw->Draw_Text (text_style, string, strlen (string), font,
		     str_left, str_top);
    draw->Pop_Clip ();
  }
  else if (obj.Valid ()) {
    // center the object; since a part of the button, will be offset from
    // buttons' left and top automatically.
    //int obj_left = obj.Get(Am_LEFT);
    //int obj_top = obj.Get(Am_TOP);
    int obj_width = obj.Get(Am_WIDTH);
    int obj_height = obj.Get(Am_HEIGHT);
    // int x_offset = real_left - obj_left + (real_width - obj_width) / 2;
    // int y_offset = real_top - obj_top + (real_height - obj_height) / 2;
    int obj_left;
    switch (align) {
    case Am_LEFT_ALIGN:
      obj_left = real_left - left + offset_left;
      break;
    case Am_RIGHT_ALIGN:
      obj_left = real_left - left + (real_width - obj_width) - 2;
      break;
    case Am_CENTER_ALIGN:
    default:
      obj_left = real_left - left + (real_width - obj_width) / 2;
      break;
    }
    int obj_top = real_top - top + (real_height - obj_height) / 2;
    obj.Set(Am_LEFT, obj_left);
    obj.Set(Am_TOP, obj_top);
    draw->Push_Clip (real_left, real_top, real_width, real_height);
    // call the object's draw method to draw the component
    // Am_Draw (obj, draw, x_offset, y_offset);
    Am_Draw (obj, draw, left, top);
    draw->Pop_Clip ();
  }
}


//////
// Am_Draw_Motif_Menu_Item
// Draws a single Motif menu item
void Am_Draw_Motif_Menu_Item(Am_Object /* self */,
			     int left, int top, int width, int height,
			     const char* string, Am_Object obj, bool line,
			     bool interim_selected, bool selected,
			     bool active, bool key_selected, Am_Font& font,
			     const Computed_Colors_Record& rec,
			     Am_Drawonable* draw,
			     int offset, char* accel_string)
     // offset is horizontal offset; object is always centered vertically
{
  // if it's just a line, draw it.  Otherwise do key selection, selection, etc
  if (line) {
    // motif menus in garnet had lines that didn't extand all the way to the
    // edges of the menu panel
    int line_left = left + 2;
    int line_right = left + width -3;
    draw->Draw_Line (rec.data->menu_top_line_style, line_left, top,
		     line_right, top);
    draw->Draw_Line (rec.data->menu_bottom_line_style, line_left, top+1,
		     line_right, top+1);
    return;
  }
  // draw key selection box, if it's key selected
  if (key_selected) 
    draw->Draw_Rectangle (Am_Key_Border_Line, Am_No_Style, left, top,
			  width, height);

  // draw the selection box, if selected
  if (selected || interim_selected)
    Am_Draw_Motif_Box(left+2, top+2, width-4, height-4,
		      false, rec, draw);
  // now draw the string if any
  int str_width, ascent, descent, a, b, str_left, str_top;
  Am_Style text_style;
  if(string || accel_string) {
    if (active) text_style = Am_Black;
    else text_style = Am_Motif_Inactive_Stipple;
  }
  if (string) {
    draw->Get_String_Extents (font, string, strlen (string), str_width,
			      ascent, descent, a, b);
    // always left justify the text for now
    str_left = left + 4 + offset;
    str_top = top + (height - ascent - descent) / 2;
    // set a clip region in case string bigger than the button
    draw->Push_Clip (left + 4, top + 4, width - 8, height-8);
    draw->Draw_Text (text_style, string, strlen (string), font,
		     str_left, str_top);
    draw->Pop_Clip ();
  }
  else if (obj.Valid ()) {
    // left justify the object; since a part of the button, will be offset from
    // buttons' left and top automatically.
    // int obj_left = obj.Get(Am_LEFT);
    // int obj_top = obj.Get(Am_TOP);
    // int obj_width = obj.Get(Am_WIDTH);
    int obj_height = obj.Get(Am_HEIGHT);
    // int x_offset = left + 4 + offset;
    // int y_offset = top - obj_top + (height - obj_height) / 2;
    int obj_left = 4 + offset;
    int obj_top = (height - obj_height) / 2;
    obj.Set(Am_LEFT, obj_left);
    obj.Set(Am_TOP, obj_top);
    // call the object's draw method to draw the component
    draw->Push_Clip (left + 4, top + 4, width - 8, height-8);
    Am_Draw (obj, draw, left, top);
    // Am_Draw (obj, draw, x_offset, y_offset);
    draw->Pop_Clip ();
  }
  if(accel_string) {
    // always right justify the accel text
    draw->Get_String_Extents (font, accel_string, strlen(accel_string),
			      str_width, ascent, descent, a, b);
    str_left = left + width - str_width - 5; 
    str_top = top + (height - ascent - descent) / 2;
    // set a clip region in case string bigger than the button
    draw->Push_Clip (left + 4, top + 4, width - 8, height-8);
    draw->Draw_Text (text_style, accel_string, strlen (accel_string), font,
		     str_left, str_top);
    draw->Pop_Clip();
  }
}

Am_Define_Method(Am_Draw_Method, void, menu_draw,
		 (Am_Object menu, Am_Drawonable *drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)menu.Get(Am_LEFT) + x_offset;
  int top = (int)menu.Get(Am_TOP) + y_offset;
  int width = menu.Get(Am_WIDTH);
  int height = menu.Get(Am_HEIGHT);
  Computed_Colors_Record rec = menu.Get (Am_STYLE_RECORD);

  // first draw a motif box behind the menu parts
  Am_Draw_Motif_Box(left, top, width, height,
		    false, rec, drawonable);

  // now draw the graphical parts of the aggregate, using Am_Aggregate's
  // draw method.
  Am_Draw_Method method;
  method = Am_Aggregate.Get(Am_DRAW_METHOD);
  method.Call (menu, drawonable, x_offset, y_offset);
}

//Formula to get the real object or string to use in the widget.
// It can be directly in the Am_COMMAND slot or there can be a command
// object there, and then the real value is in the command's Am_LABEL slot.
// Assumes that the only GRAPHICAL_PART of the aggregate is the object from
// the label object unless it is the Am_TEXT part of the aggregate (this is
// used by Text_Input widgets)
Am_Define_Value_Formula(Am_Get_Real_String_Or_Obj)
{
  // "value" is a ref parameter.  Set it with the right type and value
  Am_Object obj;
  Am_Object command = self.GV_Part(Am_COMMAND);
//cout << "Get_Real_String for " << self << " command = " << command
//     << endl << flush;
  if (!command.Valid ()) { // try as a value instead of as a part
    self.GVM(Am_COMMAND, value);
    if (!value.Valid ()) {
      // The Am_COMMAND slot is empty, just return NULL
      value = Am_No_Object;
      return;
    }
    else if (value.type == Am_OBJECT) obj = value;
    else if (value.type == Am_STRING) ; // value is already set correctly
    else {// bad type value in command or item
      cerr << "** Amulet Error: Value in COMMAND slot of widget " << self
	   << " should be object or string, but is ";
      Am_Print_Type (cerr, value.type);
      cerr << endl;
      Am_Error();
    }
  }
  // else there is a command
  else if (command.Is_Instance_Of(Am_Menu_Line_Command))
    value = true; // this takes care of the menu line case
  else if (command.Is_Instance_Of(Am_Command)) {
    // then get the value out of the command object
    command.GVM(Am_LABEL, value);
    if (value.type == Am_STRING) ; // have a string in value
    else if (value.type == Am_OBJECT) obj = value;  // have a new object
    else {
      cerr << "** Amulet Error: Widget " << self
	   << " COMMAND object " << obj
	   << " Am_LABEL slot contains object of type ";
      Am_Print_Type (cerr, value.type);
      cerr << " but should be string or Am_Object\n";
      Am_Error();
    }
  }
  else {
    obj = command; // have object in command slot;
    value = obj;
  }
  
  // now deal with the components list


  
  //first, get the "other part" if any. This is used by text-input widgets
  Am_Object other_part = self.GV_Part(Am_TEXT);

//if (!self.Is_Instance_Of(Am_Button))
//cout << "***** Other part for " << self << " is "
//     << other_part << endl << flush;

  Am_Value_Type type = self.Get_Slot_Type(Am_GRAPHICAL_PARTS);
  if ((type == Am_NONE) || (type == Am_UNINIT)) {
    // doesn't have graphical parts	(usually when being destroyed)
	value = Am_No_Object;
//cout << "***** Aborting because no graphical parts for " << self
//     << endl << flush;
	return;
  }

  Am_Value_List components;
  components = self.Get (Am_GRAPHICAL_PARTS);
  components.Start();
  Am_Object old_object;
  if (!components.Empty ()) {
    if (components.Length() == 2) { //first is other_part, then old_object
      Am_Object old_other_part;
      old_other_part = components.Get();
      if (old_other_part != other_part) {
	if (other_part.Valid())
	  components.Set(other_part); //replace with new other_part
	else components.Delete(); //no other_part now, remove old one
      }
      else components.Next(); //other_part is OK, go on to old-object
      old_object = components.Get ();
    }
    else { //length == 1, see if is other_part or old_object
      old_object = components.Get();
      if (other_part.Valid() && other_part == old_object) {
	old_object = Am_No_Object;
      }
    }
  }
  else if (other_part.Valid()) {
    // here, no old components list so old_object ==0, but need to add
    // other_part 
    components.Add(other_part);
  }
  
  //if old_object is valid, then components list internal pointer is pointing
  //at it so can delete it if necessary .
  if (obj.Valid ()) {
    Am_Object owner = obj.Get_Owner();
    if (obj == old_object) {
      //make sure obj not in another group (or button) already
      if (owner.Valid () && (owner != self)) {
	// then obj is already in another object, use an instance of it
	obj = obj.Create();
	value = obj; // the return value from this formula is the new object
	owner = Am_No_Object;
      }
      //else old_object == obj and already part of self, so fine
    }
    else { //old_object is different
      if (old_object.Valid()) { //then remove old_object from components
	components.Delete();
	//if old_object was part of me, then remove it
	if (old_object.Get_Owner() == self)
	  self.Remove_Part(old_object);
      }
      // Make obj be a component
      components.Add (obj);
      if (!owner.Valid ()) // if owner is valid then obj is already part of me
	self.Add_Part(Am_ATTACHED_OBJECT, obj);
    }
  }
  else { // no new obj, make sure no old components
    if (old_object.Valid ()) {
      components.Delete();
      self.Remove_Part(old_object);
    }
  }
  self.Set (Am_GRAPHICAL_PARTS, components);
  // value is already set with the return value
}

Am_Define_Method(Am_Draw_Method, void, button_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool selected = self.Get (Am_SELECTED);
  bool interim_selected = self.Get (Am_INTERIM_SELECTED);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  bool want_final_selected = self.Get (Am_FINAL_FEEDBACK_WANTED);
  
  Am_Font font(self.Get (Am_FONT));
  int align = self.Get(Am_H_ALIGN);

  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else Am_Error("String slot of widget should have string or object value");

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Button(self, left, top, width, height, string, obj,
 			 interim_selected, selected && want_final_selected,
			 active, key_selected, font, rec, drawonable,
			 Am_PUSH_BUTTON, 0, 0, false, align, 0);
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

Am_Define_Method(Am_Draw_Method, void, radio_button_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  int box_width = self.Get (Am_BOX_WIDTH);
  int box_height = self.Get (Am_BOX_HEIGHT);
  bool selected = self.Get (Am_SELECTED);
  bool interim_selected = self.Get (Am_INTERIM_SELECTED);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  bool want_final_selected = self.Get (Am_FINAL_FEEDBACK_WANTED);
  bool box_on_left = self.Get (Am_BOX_ON_LEFT);
  Am_Font font(self.Get (Am_FONT));
  int align = self.Get (Am_H_ALIGN);

  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else Am_Error("String slot of widget should have string or object value");

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Button(self, left, top, width, height,
			 string, obj, interim_selected,
			 selected && want_final_selected,
			 active, key_selected, font, rec, drawonable,
			 Am_RADIO_BUTTON, box_width, box_height, box_on_left,
			 align, 0);
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

Am_Define_Method(Am_Draw_Method, void, menu_item_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool selected = self.Get (Am_SELECTED);
  bool interim_selected = self.Get (Am_INTERIM_SELECTED);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  bool want_final_selected = self.Get (Am_FINAL_FEEDBACK_WANTED);
  Am_Font font(self.Get (Am_FONT));
  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  bool line = false;
  int offset = 0;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING) {
    string = value;
    offset = self.Get(Am_TEXT_OFFSET);
  }
  else if (value.type == Am_OBJECT) {
    obj = value;
    offset = self.Get(Am_ITEM_OFFSET);
  }
  else if (value.type == Am_INT || value.type == Am_BOOL)
    line = true;
  else Am_Error("String slot of widget should have string or object type");

  Am_String accel_string;
  Am_Value accel_value;
  self.Get(Am_ACCELERATOR_STRING, accel_value);
  if (accel_value.type == Am_STRING) 
    accel_string = accel_value;

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Menu_Item(self, left, top, width, height,
			    string, obj, line, interim_selected,
			    selected && want_final_selected,
			    active, key_selected, font, rec, drawonable,
			    offset, accel_string);
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

Am_Define_Method(Am_Draw_Method, void, checkbox_draw,
		 (Am_Object self, Am_Drawonable* drawonable,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  int box_width = self.Get (Am_BOX_WIDTH);
  int box_height = self.Get (Am_BOX_HEIGHT);
  bool selected = self.Get (Am_SELECTED);
  bool interim_selected = self.Get (Am_INTERIM_SELECTED);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  bool want_final_selected = self.Get (Am_FINAL_FEEDBACK_WANTED);
  bool box_on_left = self.Get (Am_BOX_ON_LEFT);

  Am_Font font(self.Get (Am_FONT));
  int align = self.Get (Am_H_ALIGN);

  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else Am_Error("String slot of widget should have string or object value");

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK)
    Am_Draw_Motif_Button(self, left, top, width, height,
			 string, obj, interim_selected,
			 selected && want_final_selected,
			 active, key_selected, font, rec, drawonable,
			 Am_CHECK_BUTTON, box_width, box_height, box_on_left,
			 align, 0);
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

// in the width slot of the button
Am_Define_Formula(int, get_button_width) {
  // adds 8 to the width: 2 on each side for the selection box, plus 2 on each
  // side for the key_selection box.
  Am_String string;
  Am_Object obj;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else return 20;
  int offset = self.GV(Am_ITEM_OFFSET);
  if ((const char*)string) {
    Am_Object window(self.GV(Am_WINDOW));
    Am_Font font(self.GV (Am_FONT));
    
    if (window.Valid ()) {
      Am_Drawonable* draw = GV_a_drawonable (window, cc);
      if (draw) {
	int str_width, ascent, descent, a, b;
	draw->Get_String_Extents(font, string, strlen (string), str_width,
				 ascent, descent, a, b);
	return str_width + 10 + offset + offset;
      }
    }
  }
  else if (obj.Valid ()) {
    int obj_width = obj.GV(Am_WIDTH);
    return obj_width + 8 + offset + offset;
  }
  return 20;
}

//in the height slot of the button
Am_Define_Formula (int, get_button_height) {
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING) string = value;
  else if (value.type == Am_OBJECT) obj = value;
  else return 2; // height of Am_Menu_Line_Command
  int offset = self.GV (Am_ITEM_OFFSET);
  if ((const char*)string) {
    Am_Drawonable* draw;
    Am_Object window(self.GV (Am_WINDOW));
    Am_Font font(self.GV(Am_FONT));
    if (window.Valid () && (draw = GV_a_drawonable (window, cc))) {
      int str_width, ascent, descent, a, b;
      draw->Get_String_Extents (font, string, strlen (string), str_width,
				ascent, descent, a, b);
      return ascent + descent + 8 + offset + offset;
    }
  }
  else if (obj.Valid ())
    return (int)obj.GV(Am_HEIGHT) + 8 + offset + offset;
  return 10;
}
//in the width slot of the checkbox
Am_Define_Formula(int, get_checkbox_width) {
  Am_String string;
  Am_Object  obj = 0;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot
  int box_width = self.GV(Am_BOX_WIDTH);
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else return 20 + box_width;
  int offset = self.GV(Am_ITEM_OFFSET);
  if ((const char*)string) {
    Am_Object window;
    Am_Font font;
    window = self.GV (Am_WINDOW);
    font = self.GV (Am_FONT);
    if (window) {
      Am_Drawonable* draw = GV_a_drawonable (window, cc);
      if (draw) {
	int str_width, ascent, descent, a, b;
	draw->Get_String_Extents(font, string, strlen (string), str_width,
				 ascent, descent, a, b);
	return offset + box_width + str_width + 8 + offset;
      }
    }
  }
  else if (obj.Valid()) {
    int obj_width = obj.GV(Am_WIDTH);
    return offset + box_width + obj_width + 8 + offset;
  }
  return 20 + box_width;
}

//in the height slot of the checkbox
Am_Define_Formula (int, get_checkbox_height) {
  Am_String string;
  Am_Object obj = 0;
  Am_Value value;
  int box_height = self.GV(Am_BOX_HEIGHT);
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.GVM(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING) string = value;
  else if (value.type == Am_OBJECT) obj = value;
  else return box_height + 4;
  int offset = self.GV (Am_ITEM_OFFSET);
  if ((const char*)string) {
    Am_Drawonable* draw;
    Am_Object window;
    Am_Font font;
    window = self.GV (Am_WINDOW);
    font = self.GV(Am_FONT);
    if (window && (draw = GV_a_drawonable (window, cc))) {
      int str_width, ascent, descent, a, b;
      draw->Get_String_Extents (font, string, strlen (string), str_width,
				ascent, descent, a, b);
      int str_height = ascent + descent;
      return (str_height > box_height ? str_height  : box_height)
	+ 4 + offset + offset;
    }
  }
  else if (obj.Valid()) {
    int obj_height = (int)obj.GV(Am_HEIGHT);
    return (obj_height > box_height ? obj_height : box_height)
      + 4 + offset + offset;
  }
  return box_height + 4;
}

Am_Define_String_Formula(check_accel_string) {
  Am_Value cmd_value;
  Am_Object cmd_obj, old_cmd_obj, old_cmd_window;
  self.Get(Am_ACCELERATOR_LIST, cmd_value);
  if (cmd_value.Valid()) old_cmd_obj = cmd_value;
  self.Get(Am_ACCELERATOR_INTER, cmd_value);
  if (cmd_value.Valid()) old_cmd_window = cmd_value;
  self.GVM(Am_COMMAND, cmd_value);
  if (cmd_value.Valid() && cmd_value.type == Am_OBJECT) {
    cmd_obj = cmd_value;
    if (cmd_obj.Is_Instance_Of(Am_Command)) {
      long accel = 0;
      Am_Value accel_value;
      Am_Input_Char accel_char;
      cmd_obj.GVM(Am_ACCELERATOR, accel_value);
      if ((accel_value.type == Am_INT) || (accel_value.type == Am_LONG)) {
	accel = accel_value;
	if (accel > 1) //then is a legal accelerator character
	  accel_char = Am_Input_Char::Narrow(accel);
      }
      else if (accel_value.type == Am_STRING) {
	// convert string into right type
	Am_String sval;
	sval = accel_value;
	accel_char = Am_Input_Char(sval);
	// store it back into the slot so more efficient next time
	// (because no parsing will be needed next time)
	accel = accel_char;
	cmd_obj.Set(Am_ACCELERATOR, accel);
      }
      // now do comparison
      if (accel > 1) { //then is a legal accelerator character
	char s[Am_LONGEST_CHAR_STRING];
	accel_char.As_Short_String(s);
	Am_Object owner_widget, window;
	owner_widget = cmd_obj.GV(Am_SAVED_OLD_OWNER);
	if (owner_widget.Valid()) 
	  window = owner_widget.GV(Am_WINDOW);

	if ((old_cmd_obj != cmd_obj ||
	     old_cmd_window != window)) {
	  if (old_cmd_obj.Valid() && old_cmd_window.Valid())
	    Am_Remove_Accelerator_Command_From_Window(old_cmd_obj,
						      old_cmd_window);
	  if (window.Valid())
	    Am_Add_Accelerator_Command_To_Window(cmd_obj, window);
	  self.Set(Am_ACCELERATOR_LIST, cmd_obj);
	  self.Set(Am_ACCELERATOR_INTER, window);
	}
	return Am_String(s);
      }
    }
  }
  if (old_cmd_obj.Valid() && old_cmd_window.Valid())
    Am_Remove_Accelerator_Command_From_Window(old_cmd_obj, old_cmd_window);

  //no accel string
  return NULL;
}

//in the width slot of the menu item
Am_Define_Formula(int, get_menu_item_width) {
  // adds 8 to the width: 2 on each side for the selection box, plus 2 on each
  // side for the key_selection box.
  Am_String string;
  Am_Object obj, window;
  Am_Value value, cmd;
  Am_Font font;
  window = self.GV (Am_WINDOW);
  font = self.GV (Am_FONT);
  int str_width, ascent, descent, a, b;
  int accel_width = 0;
  if (window.Valid ()) {
    Am_Drawonable* draw = GV_a_drawonable (window, cc);
    if (draw) {
      self.GVM(Am_COMMAND, cmd);
      if (cmd.Valid() && cmd.type == Am_OBJECT) {
	Am_Object cmd_obj = (Am_Object)cmd;
	if (cmd_obj.Is_Instance_Of(Am_Menu_Line_Command))
	  return 20; // don't worry about the width of menu-lines
	else {  //can only have an accelerator if have a command
	  self.GVM(Am_ACCELERATOR_STRING, value);
	  if (value.type == Am_STRING) {
	    Am_String accel_string;
	    accel_string = value;
	    draw->Get_String_Extents(font, accel_string, strlen(accel_string),
				     accel_width, ascent, descent, a, b);
	    accel_width += 6; //6 pixels between accel string and item
	  }
	}
      }
      
      // slot contains a formula which gets the real object based on the
      // value of the COMMAND slot 
      self.GVM(Am_REAL_STRING_OR_OBJ, value);
      if (value.type == Am_STRING)
	string = value;
      else if (value.type == Am_OBJECT)
	obj = value;
      else return 20;
      int offset = self.GV(Am_ITEM_OFFSET);
      int text_offset = self.GV(Am_TEXT_OFFSET);
      if ((const char*)string) {
	draw->Get_String_Extents(font, string, strlen (string), str_width,
				 ascent, descent, a, b);
	return offset + text_offset + str_width + 8 + text_offset + offset
	  + accel_width;
      }
      else if (obj.Valid ()) {
	int obj_width = obj.GV(Am_WIDTH);
	return obj_width + 8 + offset + offset + accel_width;
      }
    }
  }
  //if get here, something wrong, don't have a real size
  return 20;
}

// in the active slot of a widget
Am_Define_Formula(bool, Am_Active_From_Command) {
  bool ret = true;
  Am_Value v;
  self.GVM(Am_COMMAND, v);
  if (v.Valid() && v.type == Am_OBJECT) {
    Am_Object cmd = v;
    if (cmd.Is_Instance_Of(Am_Command)) ret = cmd.GV(Am_ACTIVE);
  }
  return ret;
}

// goes in the active slot of the interactor
Am_Define_Formula(bool, Am_Active_And_Active2) {
  bool ret = true;
  Am_Object button = self.Get_Owner();
  if (button.Valid ())
    ret = (bool)button.GV(Am_ACTIVE) && (bool)button.GV(Am_ACTIVE_2);
  return ret;
}

// goes in the Am_IMPLEMENTATION_PARENT slot of the command in the
// interactor to get the Command of the widget
Am_Define_Object_Formula (Am_Get_Owners_Command) {
  //owner is inter, owner of inter is widget, get command from widget 
  Am_Value v;
  Am_Object command;
  self.GV_Owner().GV_Owner().GVM(Am_COMMAND, v);
  if (v.Valid() && v.type == Am_OBJECT) {
    command = v;
    if(!command.Is_Instance_Of(Am_Command))
      command = Am_No_Object;
  }
  return command;
}

// in selected slot of a button widget, for circular constraints so if VALUE
// set, the button gets highlighted correctly
Am_Define_Formula(bool, button_sel_from_value) {
  Am_Value value;
  self.GVM(Am_VALUE, value);
  return value.Valid();
}

void set_command_from_button(Am_Object parent_command, Am_Object button) {
  Am_Value value = 0;
  if (button.Valid ()) 
    button.Get(Am_LABEL_OR_ID, value);
  parent_command.Set(Am_VALUE, value);
}

//////////////////////////////////////////////////////////////////////
// Single button Undo stuff
//////////////////////////////////////////////////////////////////////

//no repeat on new for button widgets
void Am_Widget_General_Undo_Redo(Am_Object command_obj,
				 bool undo, bool selective) {
  Am_Value new_value, old_value;
  Am_Object inter, widget;
  // this command was in the inter in the widget.  The SAVED_OLD_OWNER
  // will be the inter.  Want to actually set the widget.  The
  // old_value in the command object is for the widget, not for the
  // interactor.  The interactor's value is reset by impl_command.
  inter = command_obj.Get(Am_SAVED_OLD_OWNER);
  if (inter.Valid()) 
    widget = inter.Get_Owner();
  
  if (selective) { // then get current value from the interactor
    if (widget.Valid()) widget.Get(Am_VALUE, new_value);
  }
  else // get current value from the command_obj
    command_obj.Get(Am_VALUE, new_value);

  if (undo) command_obj.Get(Am_OLD_VALUE, old_value);
  else  // repeat
    command_obj.Get(Am_VALUE, old_value);

  command_obj.Set(Am_OLD_VALUE, new_value);
  command_obj.Set(Am_VALUE, old_value);
  //also set widget
  if (widget.Valid()) {
    #ifdef DEBUG
    if (Am_Inter_Tracing(Am_INTER_TRACE_SETTING)) {
      cout << "++ ";
      if (selective) cout << "selective ";
      if (undo) cout << "undo ";
      else cout << "repeat ";
      cout << "setting the Am_VALUE of " << widget << " to "
	   << old_value << endl << flush;
    }
    #endif
    widget.Set(Am_OLD_VALUE, new_value);
    widget.Set(Am_VALUE, old_value);
  }
}

Am_Define_Method(Am_Object_Method, void, Am_Widget_Inter_Command_Undo,
		 (Am_Object command_obj)) {
  Am_Widget_General_Undo_Redo(command_obj, true, false);
}
Am_Define_Method(Am_Object_Method, void,
		 Am_Widget_Inter_Command_Selective_Undo,
		 (Am_Object command_obj)) {
  Am_Widget_General_Undo_Redo(command_obj, true, true);
}
Am_Define_Method(Am_Object_Method, void,
		 Am_Widget_Inter_Command_Selective_Repeat,
		 (Am_Object command_obj)) {
  Am_Widget_General_Undo_Redo(command_obj, false, true);
}

// Do method for the command object in the interator for the single button.
// Set the value of the widget, me and my parent command
Am_Define_Method(Am_Object_Method, void, button_inter_command_do,
		 (Am_Object command_obj)) {
  Am_Value old_value, new_value, inter_value;

  // set the widget's and parent's value.
  Am_Object parent, inter, widget;
  parent = command_obj.Get(Am_IMPLEMENTATION_PARENT);
  inter = command_obj.Get_Owner();
  widget = inter.Get_Owner();
  widget.Get(Am_VALUE, old_value);
  inter.Get (Am_VALUE, inter_value);
  if (inter_value.Valid())
    widget.Get(Am_LABEL_OR_ID, new_value);
  else new_value = NULL;

  Am_INTER_TRACE_PRINT(Am_INTER_TRACE_SETTING,
		    "++ DO method setting the Am_VALUE of " << widget << " to "
		    << new_value);
  widget.Set(Am_VALUE, new_value);
  command_obj.Set(Am_OLD_VALUE, old_value);
  command_obj.Set(Am_VALUE, new_value);
  
  if(parent.Valid ()) {
    //set old value to current value
    parent.Get(Am_VALUE, old_value);
    parent.Set(Am_OLD_VALUE, old_value);
    parent.Set(Am_VALUE, new_value);
  }
}

//demon procedure
//see if have an allocated object attached to me, and if so, destroy
//it also.
// Also removes any accelerators
void Am_Destroy_Button (Am_Object object)
{
  Am_Object attached, command;
  attached = object.Get_Part (Am_ATTACHED_OBJECT);
  if (attached.Valid ()) {
    attached.Remove_From_Owner ();
  }
  if (object.Get_Slot_Type (Am_ATTACHED_COMMAND) != Am_NONE) {
    command = object.Get_Part (Am_COMMAND);
    attached = object.Get (Am_ATTACHED_COMMAND);
    if (command.Valid () && (attached == command)) {
      command.Remove_From_Owner ();
    }
  }
  if (object.Get_Slot_Type (Am_SUB_MENU) != Am_NONE) {
    attached = object.Get (Am_SUB_MENU);
    attached.Destroy ();
  }
  Am_Object_Demon* proto_demon =
      ((Am_Object_Advanced&)Am_Aggregate).Get_Demons ()
      .Get_Object_Demon (Am_DESTROY_OBJ);
  if (proto_demon)
    proto_demon (object);

  //now handle accelerators, if any
  Am_Value cmd_value;
  Am_Object old_cmd_obj, old_cmd_window;
  object.Get(Am_ACCELERATOR_LIST, cmd_value);
  if (cmd_value.Valid()) {
    old_cmd_obj = cmd_value;
    object.Get(Am_ACCELERATOR_INTER, cmd_value);
    if (cmd_value.Valid()) {
      old_cmd_window = cmd_value;
      Am_Remove_Accelerator_Command_From_Window(old_cmd_obj, old_cmd_window);
    }
  }
}

/* **** NOT NEEDED, constraints do the work ****
// See if I am part of a panel, and if so, update the panel.  If a standalone
// button, then just set to my old value.   
Am_Define_Method(Am_Object_Method, void, Am_Button_Command_Undo,
		 (Am_Object command_obj)) {
  Am_Value value, old_value, cur_panel_value;
  Am_Object owner, panel, panel_command;
  command_obj.Get (Am_VALUE, value);
  command_obj.Get (Am_OLD_VALUE, old_value);
  owner = command_obj.Get (Am_SAVED_OLD_OWNER);
  panel = command_obj.Get (Am_SAVED_OLD_OWNER_PANEL);
  if (panel.Valid()) {
    panel_command = panel.Get_Part(Am_COMMAND);
    if (panel_command.Valid())
      panel_command.Get(Am_VALUE, cur_panel_value);
  }

  //If this was a command in an item of a panel, then owner will the the item
  //and panel will be the panel.
  //If this was a command in a panel itself, then panel will be the panel, and
  //owner will be NULL.
  //If this was a command in a stand-alone button, then owner will be the
  //button, and panel will be NULL.
  
  //cout << "Undoing " << command_obj << " old_value = " << old_value
  //     << " new value " << value << " owner " << owner << " panel " << panel
  //     << endl << flush;
  //first, swap old and current values in the command object
  command_obj.Set(Am_VALUE, old_value);
  command_obj.Set(Am_OLD_VALUE, value);

  if (owner.Valid() && !panel.Valid()) {
    //then this command is for an item NOT in a panel, so set slots directly
    owner.Set(Am_SELECTED, old_value.Valid());
    Am_Object cur_owner_command, inter_command;
    cur_owner_command = owner.Get_Part(Am_COMMAND);
    if (cur_owner_command.Valid())
      cur_owner_command.Set(Am_VALUE, old_value);
    inter_command = owner.Get_Part(Am_INTERACTOR).Get_Part(Am_COMMAND);
    if (inter_command.Valid()) {
      if (old_value.Valid())
	inter_command.Set(Am_VALUE, owner);
      else inter_command.Set(Am_VALUE, NULL);
    }
  }
  else if (panel.Valid()) {
    //set everything based on the panel value using the constraints
    if (owner.Valid()) { //then is an item, set item's value
      Am_Object item_command;
      Am_Value old_panel_value;
      item_command = owner.Get_Part(Am_COMMAND);
      if (panel_command.Valid()) {
	//get the old value from the current command object
	command_obj.Get(Am_OWNER_PANEL_OLD_VALUE, old_panel_value);
	//cout << "Panel command " << panel_command << " old value "
	//     << old_panel_value << " cur value "
	//     << cur_panel_value << endl << flush;
      }
      if (item_command.Valid())
	item_command.Set(Am_VALUE, old_value);
      // Also need to set the panel's command's value.
      // should be after setting item's value because the web will
      // cause the panel to get the wrong value from setting just the item 
      if (panel_command.Valid()) {
	//swap the old value in the current command object
	command_obj.Set(Am_OWNER_PANEL_OLD_VALUE, cur_panel_value);
	// panel set must be second since web on item_command value will
	// set the panel's value wrong.
	//Do a get first to cause the constraints to run so Web constraint will
	// finish before setting the panel command.
	command_obj.Get(Am_OWNER_PANEL_OLD_VALUE, cur_panel_value);
	panel_command.Set(Am_VALUE, old_panel_value);
	//cout << "Setting VALUE of " << panel_command << " to "
	//     << old_panel_value << endl << flush;
      }
    }
    else { // command is from the panel itself
      if (panel_command.Valid())
	panel_command.Set(Am_VALUE, old_value);
    }
  }
  
  // now, call the prototype's method, to do the maintenance
  Am_Object_Method method;
  method = Am_Command.Get(Am_UNDO_METHOD);
  method.Call(command_obj);
}
****** */

//goes in the VALUE slot of the interactor of a single
//button, to get the value of the widget.  This is
//necessary to set up a circular constraint in case someone sets the value of
//the command from the outside.
Am_Define_Object_Formula(get_button_widget_value) {
  Am_Object widget = self.GV_Owner(); // widget the interactor is in
  if (widget.Valid ()) {
    Am_Value value;
    widget.GVM(Am_VALUE, value);
    if (value.Valid()) return widget;
  }
  return Am_No_Object;
}

// put into the Am_LABEL_OR_ID slot of the widget.  If there is a command and
// it has an ID field, use that value, otherwise use Am_REAL_STRING_OR_OBJ
// value
Am_Define_Value_Formula(Am_Get_Label_Or_ID) {
  //return will be set into value
  value = Am_No_Value;
  Am_Value v;
  self.GVM(Am_COMMAND, v);
  if (v.Valid() && v.type == Am_OBJECT) {
    Am_Object cmd = v;
    if (cmd.Is_Instance_Of(Am_Command))
      cmd.GVM(Am_ID, value);
  }
  if (!value.Valid())
    self.GVM(Am_REAL_STRING_OR_OBJ, value);
}

//////////////////////////////////////////////////////////////////////////////
//  Button Panels
//////////////////////////////////////////////////////////////////////////////

int set_parts_list_commands_old_owner(Am_Constraint_Context& cc,
				      Am_Value_List parts, Am_Object widget) {
  Am_Value item_value;
  Am_Object part, item;
  int ret = 0; //not used, just for debugging
  for (parts.Start (); !parts.Last (); parts.Next ()) {
    part = parts.Get();
    part.GVM(Am_COMMAND, item_value);
    if (item_value.type == Am_OBJECT) {
      item = (Am_Object)item_value;
      if (item.Is_Instance_Of(Am_Command)) {
	item.Set(Am_SAVED_OLD_OWNER, widget);
	++ret;
      }
    }
  }
  return ret;
}

Am_Define_Formula (int, Am_Panel_Set_Old_Owner_To_Me) {
  int ret = 0;
  Am_Object cmd;
  Am_Value_List parts;
  cmd = self.GV_Part(Am_COMMAND);
  if (cmd.Valid()) {
    cmd.Set(Am_SAVED_OLD_OWNER, self);
    ret = 1;
  }
  parts = self.GV(Am_GRAPHICAL_PARTS);
  ret = set_parts_list_commands_old_owner(cc, parts, self);
    
  //ret not used, different values just for debugging
  return ret;
}

//Button in a panel is active if owner is and if specific command object is 
Am_Define_Formula(bool, active_from_command_panel) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid () && !(bool)owner.GV(Am_ACTIVE)) return false;
  //now check my command object, if any
  Am_Value v;
  self.GVM(Am_COMMAND, v);
  if (v.Valid() && v.type == Am_OBJECT) {
    Am_Object cmd = v;
    if (cmd.Is_Instance_Of(Am_Command)) return cmd.GV(Am_ACTIVE);
  }
  return true;
}

//copy look from owner
Am_Define_Formula(int, look_from_owner) {
  return self.GV_Owner().GV(Am_WIDGET_LOOK);
}

Am_Define_Font_Formula (Am_Font_From_Owner) {
  return self.GV_Owner().GV (Am_FONT);
}
Am_Define_Formula(bool, final_feedback_from_owner) {
  return self.GV_Owner().GV(Am_FINAL_FEEDBACK_WANTED);
}
// goes in the interactor for a button panel
Am_Define_Formula(int, Am_How_Set_From_Owner) {
  return self.GV_Owner().GV(Am_HOW_SET);
}
Am_Define_Formula(bool, box_on_left_from_owner) {
  return self.GV_Owner().GV(Am_BOX_ON_LEFT);
}
Am_Define_Formula(int, box_width_from_owner) {
  return self.GV_Owner().GV(Am_BOX_WIDTH);
}
Am_Define_Formula(int, box_height_from_owner) {
  return self.GV_Owner().GV(Am_BOX_HEIGHT);
}
Am_Define_Formula(int, Am_Align_From_Box_On_Left) {
  bool bol = self.GV(Am_BOX_ON_LEFT);
  return bol ? Am_LEFT_ALIGN : Am_RIGHT_ALIGN;
}
Am_Define_Formula(int, Am_Left_From_Owner) {
  return self.GV_Owner().GV(Am_LEFT);
}
Am_Define_Formula(int, Am_Top_From_Owner) {
  return self.GV_Owner().GV(Am_TOP);
}
Am_Define_Formula(int, Am_Width_From_Owner) {
  return self.GV_Owner().GV(Am_WIDTH);
}
Am_Define_Formula(int, Am_Height_From_Owner) {
  return self.GV_Owner().GV(Am_HEIGHT);
}

// Panel item (width, height) calculate the (width, height) of an
// item in a button panel or menu by checking its owner's fixed_width slot.
// 0 or false means, use real width.
// 1 or true means use max. width (from owner).
// Otherwise, set it to that int value
Am_Define_Formula(int, panel_item_width) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    Am_Value fw;
    owner.GVM(Am_FIXED_WIDTH, fw);
    if (fw.type == Am_BOOL)
      if ((bool)fw)
	return owner.GV(Am_MAX_WIDTH);
      else return self.GV(Am_REAL_WIDTH);
    else // not Am_BOOL
      if (fw.type == Am_INT) {
	int n = fw;
	if (n == 0) return self.GV(Am_REAL_WIDTH);
	if (n == 1) return owner.GV(Am_MAX_WIDTH);
	return self.GV(Am_FIXED_WIDTH);
      }
      else { // neither int nor bool: error.
	Am_Error
	  ("Am_Panel_Item_Width: wrong type for Am_FIXED_WIDTH slot.\n");
	return 0;
      }
  }
  else return self.GV(Am_REAL_WIDTH);
}

Am_Define_Formula(int, panel_item_height) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid()) {
    Am_Value fh;
    owner.GVM(Am_FIXED_HEIGHT, fh);
    if (fh.type == Am_BOOL)
      if ((bool)fh)
	return owner.GV(Am_MAX_HEIGHT);
      else return self.GV(Am_REAL_HEIGHT);
    else // not Am_BOOL
      if (fh.type == Am_INT) {
	int n = fh;
	if (n == 0) return self.GV(Am_REAL_HEIGHT);
	if (n == 1) return owner.GV(Am_MAX_HEIGHT);
	return self.GV(Am_FIXED_HEIGHT);
      }
      else { // neither int nor bool: error.
	Am_Error
	  ("Am_Panel_Item_Height: wrong type for Am_FIXED_HEIGHT slot.\n");
	return 0;
      }
  }
  else return self.GV(Am_REAL_HEIGHT);
}

//  The height is just the vertical extent of the menu's parts, plus border
//  width.

Am_Define_Formula (int, menu_height) {
  // based on Am_map_height in opal.cc
  // simply add 2 to the calculated map height, to account for border.
  
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int height = 0;
  Am_Object item;
  for (components.Start (); !components.Last (); components.Next ()) {
    item = components.Get ();
    if ((bool)item.GV (Am_VISIBLE)) {
      int item_top = item.GV (Am_TOP);
      int item_height = item.GV (Am_HEIGHT);
      int item_bottom = item_top + item_height;
      if (item_bottom > height)
        height = item_bottom;
    }
  }
  return height + 2; // +2 for bottom border of menu panel
}

Am_Define_Formula (int, menu_width) {
  // based on Am_map_width in opal.cc
  // simply add 2 to the calculated map width, to account for border.
  
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int width = 0;
  Am_Object item;
  for (components.Start (); !components.Last (); components.Next ()) {
    item = components.Get ();
    if ((bool)item.GV (Am_VISIBLE)) {
      int item_left = item.GV (Am_LEFT);
      int item_width = item.GV (Am_WIDTH);
      int item_right = item_left + item_width;
      if (item_right > width)
        width = item_right;
    }
  }
  return width + 2; // +2 for right border of menu panel
}


// Max Item Height simply calculates maximum width of all parts, not maximum
// extents of the objects.
// potential circular constraint: dependancy on Am_WIDTH instead of
// Am_REAL_WIDTH in some cases.

Am_Define_Formula (int, max_item_width) {
  // based on Am_map_width in opal.cc
  // finds width of widest visible item
  
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int width = 0;
  Am_Object item;
  for (components.Start (); !components.Last (); components.Next ()) {
    item = components.Get ();
    if ((bool)item.GV (Am_VISIBLE)) {
      int item_width;
      if (item.Get_Slot_Type(Am_REAL_WIDTH) == Am_INT)
	item_width = item.GV (Am_REAL_WIDTH);
      else item_width = item.GV (Am_WIDTH);
      if (item_width > width)
        width = item_width;
    }
  }
  return width;
}

Am_Define_Formula (int, max_item_height) {
  // based on Am_map_width in opal.cc
  // finds width of widest visible item
  
  Am_Value_List components;
  components = self.GV (Am_GRAPHICAL_PARTS);
  int height = 0;
  for (components.Start (); !components.Last (); components.Next ()) {
    Am_Object item;
    item = components.Get ();
    if ((bool)item.GV (Am_VISIBLE)) {
      int item_height;
      if (item.Get_Slot_Type(Am_REAL_HEIGHT) == Am_INT)
	item_height = item.GV (Am_REAL_HEIGHT);
      else item_height = item.GV (Am_HEIGHT);
      if (item_height > height)
        height = item_height;
    }
  }
  return height;
}

//used by Am_Copy_Item_To_Command and Am_Menu_Bar_Copy_Item_To_Command
//  Converts the Am_ITEM slot set by the map into a command part or value
static void copy_item_to_command_proc (Am_Object& panel_item,
				       const Am_Value& value) {
  if (value.type == Am_OBJECT && Am_Object (value).Valid () &&
      Am_Object (value).Is_Instance_Of(Am_Command)) {
    // then is a command
    Am_Object command = value;
    Am_Object owner = command.Get_Owner();
    if (owner == panel_item) {
      if (panel_item.Get_Part(Am_COMMAND) == command)
        return; // already fine
      else {
	panel_item.Remove_Part(command); // part of me in the wrong slot??
      }
    }
    else if (owner.Valid ()) { // make new instance, leave command where it is
      command = command.Create();
    }
    panel_item.Remove_Part(Am_COMMAND); // prevent deleting the prior command
    panel_item.Add_Part(Am_COMMAND, command);
    panel_item.Set(Am_ATTACHED_COMMAND, command);
  }
  else { // not a command object, add to COMMAND slot as a value
    panel_item.Remove_Part(Am_COMMAND); // prevent deleting the prior command
    panel_item.Set(Am_COMMAND, value);
  }
}

// Get the ITEM value set by the Map. 
// If value is a command object, add it as a part otherwise just set
// the Command slot with it
Am_Define_Value_Formula(Am_Copy_Item_To_Command) {
  self.GVM(Am_ITEM, value);
  //cout << "regular copy_item_to_command for " << self << " value = " << value
  //     << endl << flush;
  copy_item_to_command_proc(self, value);
}


// For the where test of interactor: Find a component of owner which is active.
Am_Define_Method(Am_Where_Method, Am_Object, Am_In_Active_Widget_Part,
		 (Am_Object /* inter */, Am_Object object,
		  Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  Am_Object result = Am_No_Object;
  if (Am_Point_In_All_Owners(object, x, y, event_window))
    result = Am_Point_In_Part (object, x, y, event_window);
  if (result.Valid ()) {
    Am_Value value;
    result.Get (Am_ACTIVE, value);
    // if slot exists and is zero then return 0 (if slot doesn't
    // exist, return result)
    if (value.type != Am_NONE && !value.Valid())
      return Am_No_Object;
    else
      return result;
  }
  else return Am_No_Object;
}

bool inter_value_is_or_contains(Am_Object inter, Am_Object new_object) {
  Am_Value inter_value;
  inter.Get(Am_VALUE, inter_value);
  if (Am_Value_List::Test(inter_value)) { // is a list
    Am_Value_List inter_list;
    inter_list = inter_value;
    inter_list.Start();
    if (inter_list.Member(new_object)) return true;
    else return false;
  }
  else { //not list
    Am_Object val;
    val = inter_value;
    if (val == new_object) return true;
    else return false;
  }
}

//assign to the panel value based on the interactor's value.
void set_panel_value_from_inter_value(Am_Object inter) {
  Am_Object panel = inter.Get_Owner();
  if (!(inter.Valid())) return;
  Am_Value inter_value, label_or_id, value;
  inter.Get(Am_VALUE, inter_value);
  // cout << "** setting from inter " << inter  << " value " << inter_value
  //      << endl << flush;
  if (Am_Value_List::Test(inter_value)) { // is a list
    Am_Value_List inter_list, panel_list;
    bool allocated_list = true;
    inter_list = inter_value;
    Am_Object item;
    for (inter_list.Start(); !inter_list.Last(); inter_list.Next()) {
      item = inter_list.Get();
      item.Get(Am_LABEL_OR_ID, label_or_id);
      panel_list.Add(label_or_id);
    } 
    value = panel_list;
  } //end if value is a list
  else { //value is a single value, should be an object or null
    if (inter_value.Valid()) {
      Am_Object value_obj;
      value_obj = inter_value;
      value_obj.Get(Am_LABEL_OR_ID, value);
    }
    else value = 0;
  }
  Am_INTER_TRACE_PRINT(Am_INTER_TRACE_SETTING,
		    "++ Panel DO method setting the Am_VALUE of " << panel
		    << " to " << value);
  panel.Set(Am_VALUE, value);
}


// do method for the command in the interactor in the button panel.
// Sets the panel widget's value,
// set the IMPLEMENTATION_PARENT slot of me
// set the Am_VALUE slot of me and of the parent command
Am_Define_Method(Am_Object_Method, void, Am_Inter_For_Panel_Do,
		 (Am_Object command_obj)) {
  Am_Object new_object, item, item_command, panel, panel_command,
    parent_command; 
  Am_Object inter = command_obj.Get_Owner(); // owner will be interactor
  Am_Value old_value, new_value;
  if (inter.Valid ()) {
    //get old_value
    panel = inter.Get_Owner(); // panel the interactor is in
    panel.Get(Am_VALUE, old_value);
    set_panel_value_from_inter_value(inter);
    //set command_obj's slots
    panel.Get(Am_VALUE, new_value);
    command_obj.Set(Am_OLD_VALUE, old_value);
    command_obj.Set(Am_VALUE, new_value);

    new_object = inter.Get(Am_INTERIM_VALUE);
    // new_object is a sub-widget (an individual button)
    if (panel.Valid()) {
      //don't use get-part here because option_sub_menus compute command with a
      //formula instead of having it as a part
      panel_command = panel.Get(Am_COMMAND);
      if (!(panel_command.Is_Instance_Of(Am_Command)))
	panel_command = 0;
    }

    if (new_object.Valid ()) {
      item_command = new_object.Get_Part(Am_COMMAND);
      if (item_command.Valid () && item_command.Is_Instance_Of(Am_Command)) {
	item = new_object;
	parent_command = item_command;
	//set the value of the item_command to the item's label_or_id or 0
	if (inter_value_is_or_contains(inter, new_object))
	  new_object.Get(Am_LABEL_OR_ID, new_value);
	else new_value = 0;
      }
      else item_command = 0; // wrong type object
    }
    if (!item_command.Valid ()) { //not in the item, try to find a global cmd
      if (panel_command.Valid()) {
	parent_command = panel_command;
	//set the value of the panel_command to be the same as the panel's
	panel.Get(Am_VALUE, new_value);
      }
    }
    parent_command.Get(Am_VALUE, old_value);
    parent_command.Set(Am_OLD_VALUE, old_value);
    parent_command.Set(Am_VALUE, new_value);

    // set my parent to either the item's or panel's command object
    command_obj.Set(Am_IMPLEMENTATION_PARENT, parent_command);
  }
}

//arbitrarily use first part as the start object
Am_Define_Method(Am_Explicit_Widget_Run_Method, void,
		 widget_first_member_start_method,
		 (Am_Object widget, Am_Value initial_value)) {
  Am_Object inter = widget.Get_Part(Am_INTERACTOR);
  if (initial_value.Valid()) widget.Set(Am_VALUE, initial_value);
  Am_Value_List parts = widget.Get(Am_GRAPHICAL_PARTS);
  parts.Start();
  Am_Object first_obj = parts.Get();
  Am_Start_Interactor(inter, first_obj);
}



///////////////////////////////////////////////////////////////////////////
// Constraints for button panels
///////////////////////////////////////////////////////////////////////////

void get_inter_value_from_panel_value(const Am_Value& panel_value,
				      const Am_Object& panel,
				      Am_Constraint_Context& cc,
				      Am_Value &value) {
  Am_Value_List parts, panel_value_list, inter_value_list;
  Am_Value v, label_or_id;
  Am_Object inter, item;
  bool is_list = false;
  panel.Get(Am_GRAPHICAL_PARTS, v);
  if (!v.Valid()) return;
  parts = v;
  value = NULL; //initialize return value
  if (panel_value.Valid() && Am_Value_List::Test(panel_value)) { // is a list
    panel_value_list = panel_value;
    is_list = true;
  }
  for (parts.Start(); !parts.Last(); parts.Next()) {
    item = parts.Get();
    item.GVM(Am_LABEL_OR_ID, label_or_id);
    if (is_list) {
      panel_value_list.Start();
      if (panel_value_list.Member(label_or_id)) { // then this one should be on
	item.Set(Am_SELECTED, true);
	inter_value_list.Add(item);	
      }
      else  // this one should be off
	item.Set(Am_SELECTED, false);
    }
    else { // panel value not a list, should be Am_LABEL_OR_ID
      if (panel_value == label_or_id) { // then this one should be on
	item.Set(Am_SELECTED, true);
	value = item;
      }
      else  // this one should be off
	item.Set(Am_SELECTED, false);
    }
  } // end for parts
  if (is_list) value = inter_value_list;
}

//Calculate inter value from panel's value.  Used when
//the programmer sets the panel's value.  Also sets the SELECTED slot
Am_Define_Value_Formula(inter_value_from_panel_value) {
  Am_Object panel = self.GV_Owner();
  if (!panel.Valid()) {
    return;
  }
  Am_Value panel_value;
  panel.GVM(Am_VALUE, panel_value);
  //cout << "** computing inter " << self << " value from panel value " <<
  //  panel_value << endl << flush;
  get_inter_value_from_panel_value(panel_value, panel, cc, value);
}

///////////////////////////////////////////////////////////////////////////
// Menu_bars
///////////////////////////////////////////////////////////////////////////

// DESIGN:
// The top-level Am_ITEMS slot should contain a list of command
// objects.  Unlike other menus and panels, the first level members of
// the value_list in the Am_ITEMS slot MUST be command objects. The
// Am_LABEL of each of these command objects will be the top-level
// names of the sub-menus.  Each of these command objects should
// contain a Am_ITEMS slot containing the sub-menu items, which can be
// strings, objects or commands like regular menus and panels.
//
// The top level menu_bar is a horizontal menu.  Each item is set with a
// Am_SUB_MENU slot containing the window containing a Am_SUB_MENU part
// containing a vertical menu.  The interactor in the top level menu_bar works
// over all the windows, and its interim_do deals with turning the visibility
// on and off.  The individual menu items know how to take the right part of
// the Am_ITEM list for their own use.


Am_Define_Formula(int, sub_menu_set_old_owner) {
  Am_Object window, for_item, menu_bar;
  window = self.GV_Owner();
  int ret = 0;
  if (window.Valid()) {
    for_item = window.GV(Am_FOR_ITEM);
    if (for_item.Valid()) {
      menu_bar = for_item.GV_Owner();
      if (menu_bar.Valid()) {
	Am_Value_List parts;
	parts = self.GV(Am_GRAPHICAL_PARTS);
	ret = set_parts_list_commands_old_owner(cc, parts, menu_bar);
      }
    }
  }
  return ret;
}

inline int imax(int i1, int i2) {if (i1>i2) return i1; else return i2;}

Am_Define_Formula (int, menu_bar_width) {
  return self.GV_Owner().GV(Am_WIDTH);
}
Am_Define_Formula (int, menu_bar_height) {
  // based on height_of_parts_procedure in opal.cc.  Just add 2 for border
  int max_y = 0, comp_bottom;
  Am_Value_List components;
  Am_Object comp;
  components = self.GV(Am_GRAPHICAL_PARTS);
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    // compute how much of the component extends below the origin
    comp_bottom = ((int)(comp.GV(Am_TOP)) + (int)(comp.GV(Am_HEIGHT)));
    max_y = imax (max_y, comp_bottom);
  }
  return max_y + 2;
}

Am_Define_Formula(int, popup_sub_win_left) {
  Am_Object for_item;
  for_item = self.GV(Am_FOR_ITEM);
  //cout << "<><>Calc left for " << self << " for_item= " << for_item
  //     << endl << flush;
  if (for_item.Valid()) {
    int x = 0; //initializations so no compiler warnings for translate_coords
    int y = 0;
    // get the coordinates of the left of the menu item w.r.t. the screen
    Am_Translate_Coordinates(for_item, 0, 0, Am_Screen, x, y, cc);
    //cout << " Menu bar sub win left finds " << endl << "  Sub win = " << self
    //     << endl << "  x = " << x << endl;
    return x + 2;
  }
  else return 0;
}
Am_Define_Formula(int, menu_bar_sub_win_top) {
  Am_Object for_item;
  for_item = self.GV(Am_FOR_ITEM);
  if (for_item.Valid()) {
    int height = for_item.GV(Am_HEIGHT);
    int x = 0; //initializations so no compiler warnings for translate_coords
    int y = 0;
    // get the coordinates of the bottom of the menu item w.r.t. the screen
    Am_Translate_Coordinates(for_item, 0, height, Am_Screen, x, y, cc);
//cout << "sub menu " << self << " appearing at y " << y << endl << flush;
    return y - 2;
  }
  else return 0;
}
  
Am_Define_Formula(int, popup_sub_win_width) {
  return self.GV_Part(Am_SUB_MENU).GV(Am_WIDTH);
}

Am_Define_Formula(int, popup_sub_win_height) {
  return self.GV_Part(Am_SUB_MENU).GV(Am_HEIGHT);
}

Am_Define_Object_Formula(popup_sub_win_undo_handler) {
  Am_Object for_item, main_window, undo_handler;
  for_item = self.GV(Am_FOR_ITEM);
  if (for_item.Valid()) {
    Am_Value v;
    for_item.GVM(Am_WINDOW, v);
    if (v.Valid()) {
      main_window = v;
      if (main_window.Valid()) {
	main_window.GVM(Am_UNDO_HANDLER, v);
	if (v.Valid()) {
	  undo_handler = v;
	}
      }
    }
  }
  return undo_handler;
}

Am_Object Am_Menu_Bar_Sub_Window_Proto; //defined below

void create_sub_menu_window(Am_Object menu_item) {
  Am_Object new_window = Am_Menu_Bar_Sub_Window_Proto.Create();
  new_window.Set(Am_FOR_ITEM, menu_item);
  Am_Screen.Add_Part(new_window);
  menu_item.Set(Am_SUB_MENU, new_window);
//cout << "~~Setting Am_SUB_MENU slot of " << menu_item << " to be "
//     << new_window << endl << flush;
}

//Put into the Am_ITEM_TO_COMMAND slot of each top-level item of menu_bar.
// Gets the ITEM value set by the Map. If value is a command object, add it as
// a part.  Also creates the sub-menus
Am_Define_Value_Formula(menu_bar_copy_item_to_command) {
  self.GVM(Am_ITEM, value);
  if (value.type == Am_OBJECT && Am_Object (value).Valid () &&
      Am_Object (value).Is_Instance_Of(Am_Command)) {
    //then fine
  }
  else {
    Am_ERROR("In a menu_bar, the top-level items must be\n"
	     << "command objects, but for " << self << " item # "
	     << (int)self.Get(Am_RANK) + 1 << " is " << value);
  }
  copy_item_to_command_proc(self, value);

  //now create the sub-menu if necessary
  if (self.Get_Slot_Type (Am_SUB_MENU) != Am_OBJECT)
    create_sub_menu_window(self);
}


// formula for the items list of a sub-menu: get the list from the Am_FOR_ITEM
// of my window.
Am_Define_Value_Formula(sub_menu_items) {
  value = 0;
  Am_Object window, for_item;
  window = self.GV(Am_WINDOW);
  if (window.Valid()) {
    for_item = window.GV(Am_FOR_ITEM);
    if (for_item.Valid()) {
      Am_Object cmd;
      cmd = for_item.GV(Am_ITEM);
      // all top-level items must be commands
      if (cmd.Valid()) {
	// then set the return value of this formula with the contents of the
	// Am_ITEMS slot of the command.
	cmd.GVM(Am_ITEMS, value);
	if (!value.Valid ())
          value = 0;
      }
    }
  }
}

//set the sub-window visibility, and also deal with interim selected of the
//associated main item
void set_sub_window_vis(Am_Object sub_window, bool vis) {
  sub_window.Set(Am_VISIBLE, vis);
  Am_Object for_item;
  for_item = sub_window.Get(Am_FOR_ITEM);
  if (for_item.Valid()) for_item.Set(Am_INTERIM_SELECTED, vis);
}
  
// For the where test of the interactor in the menu bar: Work over all windows
// and items, and pop-up sub-windows when necessary. 
Am_Define_Method(Am_Where_Method, Am_Object, in_menu_bar_item,
		 (Am_Object inter, Am_Object menu_bar, Am_Object event_window,
		  Am_Input_Char /* ic */, int x, int y)) {
  Am_Object result = Am_No_Object;
  Am_Object menu_bar_window, old_window, new_window;
  Am_Value v;
  // get the old sub_window which used to be visible
  inter.Get(Am_SUB_MENU, v);
  if (v.Valid()) old_window = v;
  menu_bar_window = menu_bar.Get(Am_WINDOW);
  if (menu_bar_window == event_window) {
    // in the top level window, 
    result = Am_Point_In_Part (menu_bar, x, y, event_window);
    // now deal with visibility of sub-windows
    if (result.Valid()) {
      result.Get(Am_SUB_MENU, v);
      if (v.Valid())
        new_window = v;
    }
    if (new_window != old_window && new_window.Valid()) {
      if (old_window.Valid()) set_sub_window_vis(old_window, false);
      if (new_window.Valid()) {
	set_sub_window_vis(new_window, true);
	Am_To_Top(new_window);
      }
      inter.Set(Am_SUB_MENU, new_window);
    }
    return result;  //don't test if top-level menu items are active!!
  }
  else { //must be in a sub-menu
    Am_Object sub_menu = event_window.Get_Part(Am_SUB_MENU);
    if (sub_menu.Valid())
      result = Am_Point_In_Part (sub_menu, x, y, event_window);
    // test if active
    if (result.Valid ()) {
      Am_Value value;
      result.Get (Am_ACTIVE, value);
      // if slot exists and is zero then return 0.
      // If slot does NOT exist, return result.
      if (value.type != Am_NONE && !value.Valid())
        return Am_No_Object;
      else
        return result;
    }
  }
  return Am_No_Object;
}

//menu_bar has a list of parts which are Am_Item_In_Menu and each one should
//have a Am_SUB_MENU slot which contains the appropriate sub-menu
Am_Define_Value_List_Formula(menu_bar_window_list) {
  Am_Object menu_bar = self.GV_Owner();
  Am_Object menu_bar_main_win = menu_bar.GV(Am_WINDOW);
  Am_Value_List window_list;
  Am_Value_List components;
  Am_Object comp;
  Am_Value v;
  components = menu_bar.GV(Am_GRAPHICAL_PARTS);
  if (menu_bar_main_win.Valid())
    window_list.Add(menu_bar_main_win); // main window must be there also
  for (components.Start(); !components.Last(); components.Next()) {
    comp = components.Get ();
    comp.GVM(Am_SUB_MENU, v);
    if (v.Valid()) {
      window_list.Add(v);
    }
  }
  return window_list;
}

//A custom destroy demon for menu bar to destroy the menu popup windows.
void Am_Destroy_Menu_Bar (Am_Object object)
{
  Am_Value_List parts;
  Am_Object part, sub_menu;
  parts = object.Get (Am_GRAPHICAL_PARTS);
  for (parts.Start (); !parts.Last (); parts.Next ()) {
    part = parts.Get ();
    sub_menu = part.Get (Am_SUB_MENU);
    sub_menu.Destroy ();
  }
  Am_Object_Demon* proto_demon = ((Am_Object_Advanced&)Am_Menu).Get_Demons ()
      .Get_Object_Demon (Am_DESTROY_OBJ);
  if (proto_demon)
    proto_demon (object);
}

//current object should be already set into Am_INTERIM_VALUE and old
//value in Am_OLD_INTERIM_VALUE.  This will replace the choice_inter's interim
//do method.
Am_Define_Method(Am_Object_Method, void, menu_bar_inter_interim_do,
		 (Am_Object inter)) {
  Am_Object old_object, new_object, menu_bar_win, old_object_win,
    new_object_win, main_item;
  menu_bar_win = inter.Get_Owner().Get(Am_WINDOW);
  old_object = inter.Get(Am_OLD_INTERIM_VALUE);
  new_object = inter.Get(Am_INTERIM_VALUE);
  if (new_object.Valid ())
    new_object.Set(Am_INTERIM_SELECTED, true);
  if (old_object.Valid ()) {
    old_object_win = old_object.Get(Am_WINDOW);
    //never clear the Am_INTERIM_SELECTED of top-level items
    if (old_object_win != menu_bar_win)
      old_object.Set(Am_INTERIM_SELECTED, false);
  }
}

// returns main-item, which might be obj if this is a main-item
Am_Object clear_interim_sel(Am_Object obj, Am_Object menu_bar_win) {
  Am_Object obj_win, main_item;
  if (obj.Valid ()) {
    obj.Set(Am_INTERIM_SELECTED, false);
    obj_win = obj.Get(Am_WINDOW);
    if (obj_win.Valid () &&
	obj_win != menu_bar_win) { // then clear old main menu selection
      main_item = obj_win.Get(Am_FOR_ITEM);
      if (main_item.Valid())
	main_item.Set(Am_INTERIM_SELECTED, false);
    }
    else main_item = obj;
  }
  return main_item;
}

Am_Define_Method(Am_Object_Method, void, menu_bar_inter_abort,
		 (Am_Object inter)) {
  Am_Object obj, menu_bar_win, sub_window;
  menu_bar_win = inter.Get_Owner().Get(Am_WINDOW);
  obj = inter.Get(Am_OLD_INTERIM_VALUE);
  clear_interim_sel(obj, menu_bar_win);
  obj = inter.Get(Am_INTERIM_VALUE);
  clear_interim_sel(obj, menu_bar_win);
  // now make sub_window go away
  sub_window = inter.Get(Am_SUB_MENU);
  if (sub_window.Valid()) set_sub_window_vis(sub_window, false);
  inter.Set(Am_SUB_MENU, Am_No_Object);
}

Am_Define_Method(Am_Object_Method, void, menu_bar_inter_do,
		 (Am_Object inter)) {
  Am_Object new_object, parent_command, item_command, main_item_command,
    main_item, menu_bar, sub_window, menu_bar_win;
  Am_Value value;
  new_object = inter.Get(Am_INTERIM_VALUE);
  menu_bar = inter.Get_Owner();
  menu_bar_win = menu_bar.Get(Am_WINDOW);

  // make sub_window go away
  sub_window = inter.Get(Am_SUB_MENU);
  if (sub_window.Valid()) set_sub_window_vis(sub_window, false);
  inter.Set(Am_SUB_MENU, Am_No_Object);

  // if click on top-level item, it might not be valid, so check first
  if (new_object.Valid()) {
    new_object.Get (Am_ACTIVE, value);
    // if slot exists and is zero then return 0.
    // If slot does NOT exist, return result.
    if (value.type != Am_NONE && !value.Valid()) {
      Am_Abort_Interactor(inter);
      return;
    }
    // else continue
  }
  
  // Now the standard choice interactor stuff.
  //   clear interim selection of sub-item and main item.  Returns
  //   main-item, which might be new_object if it is a main-item
  main_item = clear_interim_sel(new_object, menu_bar_win);

  // sets the interactor's command value and SELECTED of the individual widgets
  Am_Choice_Set_Value(inter);
  inter.Set(Am_OBJECT_MODIFIED, new_object);

  // now find and set the right parent command object
  // This code is like Am_Inter_For_Panel_Do
  if (main_item.Valid()) {
    main_item_command = main_item.Get_Part(Am_COMMAND);
    if (!main_item_command.Is_Instance_Of(Am_Command))
      main_item_command = 0;  //wrong type object
  }
  
  if (new_object.Valid ()) {
    item_command = new_object.Get_Part(Am_COMMAND);
    if (item_command.Valid () && item_command.Is_Instance_Of(Am_Command))
      parent_command = item_command;
    else item_command = 0; // wrong type object
  }
  if (!item_command.Valid ()) { //not in the item, use main item
    parent_command = main_item_command;
  }

  // set the value field of the parent_command
  set_command_from_button(parent_command, new_object);
 
  // install the found command into my command object's parent
  Am_Object command_obj = inter.Get_Part(Am_COMMAND);
  if (command_obj.Valid()) {
    command_obj.Set(Am_IMPLEMENTATION_PARENT, parent_command);
  }
}

//this goes in the window for the sub-menu
Am_Define_Style_Formula (popup_fill_from_for_item) {
  Am_Value v;
  self.GV_Object(Am_FOR_ITEM).GVM(Am_FILL_STYLE, v);
  if (v.Valid()) return v;
  return Am_Motif_Light_Blue;
}

// the next three go into the sub-menu itself.  FOR_ITEM is in my window
Am_Define_Formula (int, sub_menu_look_from_win_for_item) {
  Am_Value v;
  self.GV_Object(Am_WINDOW).GV_Object(Am_FOR_ITEM).GVM(Am_WIDGET_LOOK, v);
  if (v.Valid()) return v;
  else return (int)Am_MOTIF_LOOK;
}
  
Am_Define_Font_Formula(sub_menu_font_from_win_for_item) {
  Am_Value v;
  self.GV_Object(Am_WINDOW).GV_Object(Am_FOR_ITEM).GVM(Am_FONT, v);
  if (v.Valid()) return v;
  else return Am_Default_Font;
}

//Button in a menu_bar is active if owner and main-menu is, and if specific
//command object is 
Am_Define_Formula(bool, menu_bar_sub_item_active) {
  Am_Object owner = self.GV_Owner();
  if (owner.Valid () && !(bool)owner.GV(Am_ACTIVE)) return false;
  // now check the main-item I am attached to, if any
  Am_Object win;
  win = self.GV(Am_WINDOW);
  if (win.Valid()) {
    Am_Object main_item;
    main_item = win.GV(Am_FOR_ITEM);
    if (main_item.Valid() && !(bool)main_item.GV(Am_ACTIVE)) return false;
  }
  //now check my command object, if any
  Am_Object cmd = self.GV_Part(Am_COMMAND);
  if (cmd.Valid () && cmd.Is_Instance_Of(Am_Command)) return cmd.GV(Am_ACTIVE);
  else return true;
}

///////////////////////////////////////////////////////////////////////////
// Option Button
//    single button, but pops up a menu of choices
///////////////////////////////////////////////////////////////////////////
	  
//self is menu, window's for_item is main option button, get its ITEMS
Am_Define_Value_Formula(option_button_items) {
  value = 0;
  Am_Value v;
  Am_Object o;
  self.GVM(Am_WINDOW, v);
  //cout << "Calculating items for " << self << " win=" << v;
  if (v.Valid()) {
    o = v;
    o.GVM(Am_FOR_ITEM, v);
    if (v.Valid()) {
      o = v;
      o.GVM(Am_ITEMS, value);
    }
  }
  //cout << " returning=" << value << endl << flush;
}

Am_Object Am_Option_Button_Sub_Window_Proto; //defined below

//doesn't need the cc parameter since only evaluated once per object
int create_sub_menu_proc (Am_Constraint_Context& /* cc */,
				 Am_Object& self) {
  Am_Object new_window = self.Get(Am_SUB_MENU);
  if (new_window.Valid())
    Am_Error("create_sub_menu called but already has a menu");
  new_window = Am_Option_Button_Sub_Window_Proto.Create()
    .Set(Am_FOR_ITEM, self);
  Am_Screen.Add_Part(new_window);
  self.Set(Am_SUB_MENU, new_window);
  return -1;
}
Am_Formula create_sub_menu (create_sub_menu_proc, "create_sub_menu");

Am_Define_Formula(int, option_sub_win_top) {
  Am_Object for_item = self.GV(Am_FOR_ITEM);
  if (for_item.Valid()) {
    //find top of item in my menu that is currently selected
    int x = 0;
    int y = 0;
    Am_Value inter_value;
    for_item.GVM(Am_COMPUTE_INTER_VALUE, inter_value);
    if (inter_value.Valid()) {
      Am_Object sel_menu_item = inter_value;
      y = (int)sel_menu_item.Get(Am_TOP);
    }
    // get the coordinates of the current item w.r.t. the screen
    Am_Translate_Coordinates(for_item, 0, -y, Am_Screen, x, y, cc);
    //cout << "sub menu " << self << " appearing at y " << y << endl << flush;
    return y + 4;
  }
  else return 0;
}

//get the value out of my sub-menu, if any, otherwise use first of items
Am_Define_Value_Formula(option_button_value) {
  Am_Object sub_menu = self.GV_Object(Am_SUB_MENU).GV_Object(Am_SUB_MENU);
  sub_menu.GVM(Am_VALUE, value);
  if (!value.Valid()) {
    Am_Value v;
    self.GVM(Am_ITEMS, v);
    value = Am_No_Value;
    if (v.Valid() && Am_Value_List::Test(v)) {
      Am_Value_List items = v;
      Am_Value firstv;
      items.Start();
      firstv = items.Get();
      if (firstv.Valid()) {
	value = firstv;
	if (firstv.type == Am_OBJECT) {
	  Am_Object cmd = firstv;
	  if (cmd.Is_Instance_Of(Am_Menu_Line_Command))
	    value = true; // this takes care of the menu line case
	  else if (cmd.Is_Instance_Of(Am_Command)) 
	    // then get the value out of the command object
	    cmd.GVM(Am_LABEL, value);
	}
      }
    }
  }
}

const int NOTCH_OFFSET = 8; //distance from edge of button to notch
const int NOTCH_WIDTH = 12; 
const int NOTCH_HEIGHT = 8; 

//width of menu, plus notch offset plus notch size 
Am_Define_Formula(int, get_option_button_width) {
  Am_Object sub_menu_win = self.GV(Am_SUB_MENU);
  int menu_width = sub_menu_win.GV(Am_WIDTH);
  return menu_width + NOTCH_OFFSET + NOTCH_WIDTH + NOTCH_OFFSET;
}

Am_Define_Method(Am_Draw_Method, void, option_button_draw,
		 (Am_Object self, Am_Drawonable* draw,
		  int x_offset, int y_offset)) {
  int left = (int)self.Get (Am_LEFT) + x_offset;
  int top = (int)self.Get (Am_TOP) + y_offset;
  int width = self.Get (Am_WIDTH);
  int height = self.Get (Am_HEIGHT);
  bool active = self.Get (Am_ACTIVE);
  bool key_selected = self.Get (Am_KEY_SELECTED);
  Am_Font font;
  font = self.Get (Am_FONT);
  Computed_Colors_Record rec = self.Get (Am_STYLE_RECORD);
  Am_Widget_Look look = (Am_Widget_Look)(int)self.Get (Am_WIDGET_LOOK);

  // now find the contents to draw in the button
  Am_String string;
  Am_Object obj;
  Am_Value value;
  // string slot contains a formula which gets the real object based on the
  // value of the COMMAND slot 
  self.Get(Am_REAL_STRING_OR_OBJ, value);
  if (value.type == Am_STRING)
    string = value;
  else if (value.type == Am_OBJECT)
    obj = value;
  else Am_Error("String slot of widget should have string or object value");
  int offset = 4 + 2; //default text offset is 2

  // finally ready to draw it
  if (look == Am_MOTIF_LOOK) {
    Am_Draw_Motif_Button(self, left, top, width, height, string, obj,
 			 false, false,
			 active, key_selected, font, rec, draw,
			 Am_PUSH_BUTTON, 0, 0, false, false, offset);
    //now draw notch
    int x = left+width - NOTCH_OFFSET - NOTCH_WIDTH - 1;
    //center in Y
    int y = top + (height - NOTCH_HEIGHT)/2;
    draw->Push_Clip (left, top, width, height);
    Am_Draw_Motif_Box(x, y, NOTCH_WIDTH, NOTCH_HEIGHT, false, rec, draw);
    draw->Pop_Clip ();
  }
  else Am_Error("Sorry, only the Motif Style implemented for now");
}

//displays sub-menu and starts its interactor
Am_Define_Method(Am_Object_Method, void, option_button_start_do,
		 (Am_Object inter)) {
  Am_Object option_button = inter.Get_Owner();
  Am_Object sub_menu_window = option_button.Get(Am_SUB_MENU);
  if (sub_menu_window.Valid()) {
    sub_menu_window.Set(Am_VISIBLE, true);
    Am_To_Top(sub_menu_window);
    Am_Object inter = sub_menu_window.Get_Part(Am_SUB_MENU)
      .Get_Part(Am_INTERACTOR);
    Am_Value inter_value;
    option_button.Get(Am_COMPUTE_INTER_VALUE, inter_value);
    Am_Start_Interactor(inter, inter_value);
  }
}

void hide_sub_menu(Am_Object command_obj) {
  Am_Object inter = command_obj.Get_Owner(); // owner will be interactor
  //cout << "Calling hide sub_menu for " << command_obj << " inter "
  //     << inter << endl << flush;
  if (inter.Valid ()) {
    Am_Object window = inter.Get_Owner().Get(Am_WINDOW);
    if (window.Valid())
      window.Set(Am_VISIBLE, false);
  }
}
  
Am_Define_Method(Am_Object_Method, void, option_sub_menu_do,
		 (Am_Object command_obj)) {
  //standard stuff
  Am_Inter_For_Panel_Do_proc(command_obj);
  //make my menu disappear
  hide_sub_menu(command_obj);
}

Am_Define_Method(Am_Object_Method, void, option_hide_sub_menu,
		 (Am_Object command_obj)) {
  hide_sub_menu(command_obj);
}

Am_Define_Object_Formula(command_from_for_item) {
  return self.GV_Owner().GV_Object(Am_FOR_ITEM).GV(Am_COMMAND);
}

//A custom destroy demon for option button to destroy the menu popup window.
void destroy_option_button (Am_Object object)
{
  Am_Object sub_menu = object.Get(Am_SUB_MENU);
  sub_menu.Destroy ();

  //now run the regular button destroy demon, if any
  Am_Object_Demon* proto_demon = ((Am_Object_Advanced&)Am_Button).Get_Demons ()
      .Get_Object_Demon (Am_DESTROY_OBJ);
  if (proto_demon)
    proto_demon (object);
}


Am_Define_Value_Formula(compute_inter_value) {
  Am_Value panel_value, v;
  Am_Object panel = self.GV_Object(Am_SUB_MENU).GV_Part(Am_SUB_MENU);
  self.GVM(Am_VALUE, panel_value);
  if (!panel.Valid()) {
    value = NULL;
    return;
  }
  get_inter_value_from_panel_value(panel_value, panel, cc, value);
}

Am_Define_Value_Formula(get_real_string_from_inter_val) {
  Am_Value v;
  self.GVM(Am_COMPUTE_INTER_VALUE, v);
  if (v.Valid()) {
    Am_Object button = v;
    button.GVM(Am_REAL_STRING_OR_OBJ, value);
  }
}

Am_Define_Value_List_Formula(option_button_window_list) {
  Am_Value_List window_list;
  Am_Object menu = self.GV_Owner();
  if (menu.Valid()) {
    Am_Object menu_win = menu.GV(Am_WINDOW);
    if (menu_win.Valid()) {
      Am_Object option_button = menu_win.GV_Object(Am_FOR_ITEM);
      if (option_button.Valid()) {
	Am_Object option_button_window = option_button.GV(Am_WINDOW);
	if (option_button_window.Valid())
	  window_list.Add(menu_win)
	    .Add(option_button_window);
      }
    }
  }
  return window_list;
}
  

///////////////////////////////////////////////////////////////////////////
// Initializing
///////////////////////////////////////////////////////////////////////////

// exported objects

Am_Object Am_Button = 0;
Am_Object Am_Button_Panel = 0;
Am_Object Am_Checkbox_Panel = 0;
Am_Object Am_Radio_Button_Panel = 0;
Am_Object Am_Menu = 0;
Am_Object Am_Menu_Line_Command = 0;

// internal objects

Am_Object Am_Checkbox = 0;
Am_Object Am_Menu_Item = 0;
Am_Object Am_Radio_Button = 0;

Am_Object Am_Button_In_Panel;
Am_Object Am_Radio_Button_In_Panel = 0;
Am_Object Am_Checkbox_In_Panel = 0;
Am_Object Am_Item_In_Menu = 0;
Am_Object Am_Menu_Bar;
Am_Object Am_Option_Button;
Am_Object Am_Pop_Up_Window_Proto;

void Am_Button_Widgets_Initialize () {
  Am_Object inter; // interactor in the widget
  Am_Object command_obj; 
  Am_Object_Advanced obj_adv; // to get at advanced features like
			       // local-only and demons.

  //////////// Command Objects ////////////////////////////

  Am_Menu_Line_Command = Am_Command.Create("Menu_Line_Command")
    .Set (Am_LABEL, "Menu_Line_Command")
    .Set (Am_ACTIVE, false) // menu lines aren't active menu members.
    .Set (Am_VALUE, NULL) 
    ;
  
  //////////// button /////////////
  // instance of a group so can have a part (the contents)
  Am_Button = Am_Aggregate.Create("Button")
    .Set (Am_VALUE, NULL)
    .Set (Am_SELECTED, false) //set by interactor OR computed from the value
			       //of the command object
    .Set (Am_SELECTED, button_sel_from_value)
    .Set (Am_ITEM_OFFSET, 5) // how far to indent the string or obj
    .Set (Am_INTERIM_SELECTED, false)
    .Set (Am_ACTIVE, Am_Active_From_Command)
    .Set (Am_ACTIVE_2, true) // used by interactive tools
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_KEY_SELECTED, false)
    .Set (Am_FONT, Am_Default_Font)
    .Set (Am_FILL_STYLE, Am_Amulet_Purple)
    .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
    .Set (Am_FINAL_FEEDBACK_WANTED, false)
    .Set (Am_DRAW_METHOD, button_draw)
    .Set (Am_REAL_STRING_OR_OBJ, Am_Get_Real_String_Or_Obj)
    .Set (Am_WIDTH, get_button_width)
    .Set (Am_HEIGHT, get_button_height)
    .Set (Am_H_ALIGN, Am_CENTER_ALIGN)
    .Set (Am_LABEL_OR_ID, Am_Get_Label_Or_ID)
    .Set (Am_SET_COMMAND_OLD_OWNER, Am_Set_Old_Owner_To_Me)
    .Set (Am_ACCELERATOR_STRING, check_accel_string)
    .Set (Am_WIDGET_START_METHOD, Am_Standard_Widget_Start_Method)
    .Set (Am_WIDGET_ABORT_METHOD, Am_Standard_Widget_Abort_Method)
    .Set (Am_WIDGET_STOP_METHOD, Am_Standard_Widget_Stop_Method)
    .Add_Part (Am_COMMAND, Am_Command.Create("Button_Command")
	       .Set (Am_LABEL, "Button")
	       .Set (Am_VALUE, 0))
    .Add_Part (Am_INTERACTOR,
	       inter = Am_Choice_Interactor_Repeat_Same.Create("inter_in_button")
	       .Set (Am_HOW_SET, (int)Am_CHOICE_TOGGLE)
	       .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
	       .Set (Am_START_WHERE_TEST, Am_Inter_In)
	       .Set (Am_ACTIVE, Am_Active_And_Active2)
  // set up a circular constraint between the value in the command of
  // the interactor and the value in the top-level button widget so setting
  // one will cause the other to change also
	        .Set (Am_VALUE, get_button_widget_value)
		)
    ;
  inter.Get_Part(Am_COMMAND)
    .Set(Am_IMPLEMENTATION_PARENT, Am_Get_Owners_Command)
    .Set(Am_DO_METHOD, button_inter_command_do)
    .Set(Am_UNDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_REDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, Am_Widget_Inter_Command_Selective_Undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD,
	 Am_Widget_Inter_Command_Selective_Repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    ;

  obj_adv = (Am_Object_Advanced&)inter;
  obj_adv.Get_Slot (Am_VALUE).Set_Single_Constraint_Mode (false);

  obj_adv = (Am_Object_Advanced&)Am_Button;
  //formula that contains a copy of the real string or object to display in
  //the button

  obj_adv.Get_Slot (Am_VALUE).Set_Single_Constraint_Mode (false);
  obj_adv.Get_Slot (Am_SELECTED).Set_Single_Constraint_Mode (false);

  obj_adv.Get_Slot (Am_REAL_STRING_OR_OBJ) 
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_INTERIM_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_KEY_SELECTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_ACTIVE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_WIDGET_LOOK)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FINAL_FEEDBACK_WANTED)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FONT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_H_ALIGN)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  // all the next slots should not be inherited
  obj_adv.Get_Slot(Am_SELECTED).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_INTERIM_SELECTED).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_KEY_SELECTED).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_ACTIVE).Set_Inherit_Rule(Am_COPY);
  obj_adv.Get_Slot(Am_ACTIVE_2).Set_Inherit_Rule(Am_COPY);

//  obj_adv.Get_Slot(Am_WIDTH)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_HEIGHT)->Set_Single_Constraint_Mode (true);
  Am_Demon_Set demons = obj_adv.Get_Demons ().Copy ();
  demons.Set_Object_Demon (Am_DESTROY_OBJ, Am_Destroy_Button);
  obj_adv.Set_Demons (demons);

  
  //////////// Check box ////////////
  // Just a button with a different draw method.
  Am_Checkbox = Am_Button.Create("Checkbox")
    .Set (Am_BOX_ON_LEFT, true)
    .Set (Am_BOX_WIDTH, 15)
    .Set (Am_BOX_HEIGHT, 15)
    .Set (Am_FINAL_FEEDBACK_WANTED, true)
    .Set (Am_WIDTH, get_checkbox_width)
    .Set (Am_HEIGHT, get_checkbox_height)
    .Set (Am_H_ALIGN, Am_Align_From_Box_On_Left)
    .Set (Am_DRAW_METHOD, checkbox_draw)
    ;

  inter = Am_Checkbox.Get(Am_INTERACTOR);
  inter.Set(Am_FIRST_ONE_ONLY, true);
   
  obj_adv = (Am_Object_Advanced&)Am_Checkbox;
  
  obj_adv.Get_Slot (Am_BOX_ON_LEFT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_BOX_WIDTH)
    .Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_BOX_HEIGHT)
    .Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);

  
  //////////// Radio Button ////////////
  // Just another button with a different draw method.
  Am_Radio_Button = Am_Button.Create("Radio_Button")
    .Set (Am_BOX_ON_LEFT, true)
    .Set (Am_BOX_WIDTH, 15)
    .Set (Am_BOX_HEIGHT, 15)
    .Set (Am_ITEM_OFFSET, 3)
    .Set (Am_WIDTH, get_checkbox_width)
    .Set (Am_HEIGHT, get_checkbox_height)
    .Set (Am_H_ALIGN, Am_Align_From_Box_On_Left)
    .Set (Am_DRAW_METHOD, radio_button_draw)
    ;

  inter = Am_Radio_Button.Get(Am_INTERACTOR);
  inter.Set(Am_FIRST_ONE_ONLY, true);

  
  obj_adv = (Am_Object_Advanced&)Am_Radio_Button;
  
  obj_adv.Get_Slot (Am_BOX_ON_LEFT)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_BOX_WIDTH)
    .Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);
  obj_adv.Get_Slot (Am_BOX_HEIGHT)
    .Set_Demon_Bits (Am_MOVING_REDRAW | Am_EAGER_DEMON);


  //////////// Menu Item ////////////
  // Just another button with a different draw method.
  Am_Menu_Item = Am_Button.Create("Menu_Item")
    .Set (Am_WIDTH, get_menu_item_width)
    .Set (Am_DRAW_METHOD, menu_item_draw)
    .Set (Am_ITEM_OFFSET, 0)
    .Set (Am_TEXT_OFFSET, 2)
    ;

//  obj_adv = (Am_Object_Advanced&)Am_Menu_Item;
//  obj_adv.Get_Slot (Am_WIDTH).Set_Single_Constraint_Mode (true);

  
  //////////// Button Panel /////////////
  // Design: Basically, use the regular button as an Item_Prototype and the
  // standard mapping stuff to copy it as needed.
  // Setting the ITEM slot of a Am_Button_In_Panel will cause the
  // value to be copied into the COMMAND slot either as a value or a part so
  // the regular button functions will work with it.  Other values are also
  // copied down.

  Am_Button_Panel = Am_Map.Create("Button Panel")
    .Set (Am_VALUE, NULL)
    .Set (Am_ITEM_OFFSET, 3) // how far to indent the string or obj
    // active here is whether whole widget is active.  Use command part if any
    .Set (Am_FIXED_WIDTH, true)
    .Set (Am_FIXED_HEIGHT, false)
    .Set (Am_KEY_SELECTED, false)
    .Set (Am_MAX_WIDTH, max_item_width)
    .Set (Am_MAX_HEIGHT, max_item_height)
    .Set (Am_WIDTH, Am_Width_Of_Parts)
    .Set (Am_HEIGHT,Am_Height_Of_Parts)
    .Set (Am_ACTIVE, Am_Active_From_Command)
    .Set (Am_ACTIVE_2, true) // used by interactive tools
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_FONT, Am_Default_Font)
    .Set (Am_FILL_STYLE, Am_Amulet_Purple)
    .Set (Am_FINAL_FEEDBACK_WANTED, false)
    .Set (Am_HOW_SET, (int)Am_CHOICE_SET) //toggle is also a good choice
    .Set (Am_LAYOUT, Am_Vertical_Layout) // or horiz
    .Set (Am_H_ALIGN, Am_LEFT_ALIGN)
    .Set (Am_ITEMS, 0)
    .Set (Am_SET_COMMAND_OLD_OWNER, Am_Panel_Set_Old_Owner_To_Me)
    .Set (Am_WIDGET_START_METHOD, widget_first_member_start_method)
    .Set (Am_WIDGET_ABORT_METHOD, Am_Standard_Widget_Abort_Method)
    .Set (Am_WIDGET_STOP_METHOD, Am_Standard_Widget_Stop_Method)

    // Plus all the slots of a Map: Am_H_ALIGN, Am_V_ALIGN, Am_FIXED_WIDTH,
    //     Am_FIXED_HEIGHT 
    .Add_Part (Am_INTERACTOR,
	       inter = Am_Choice_Interactor_Repeat_Same.Create("inter_in_button_panel")
	       .Set (Am_HOW_SET, Am_How_Set_From_Owner)
	       .Set (Am_START_WHEN, Am_Default_Widget_Start_Char)
	       .Set (Am_START_WHERE_TEST, Am_In_Active_Widget_Part)
	       .Set (Am_ACTIVE, Am_Active_And_Active2)
	       .Set (Am_VALUE, inter_value_from_panel_value )
	       )
    // top-level command do method is called after the do method of commands
    // in the items list.  But there is no top-level command by default
    .Add_Part (Am_ITEM_PROTOTYPE, Am_Button_In_Panel =
		Am_Button.Create ("Button_In_Panel_Proto")
	       .Set (Am_REAL_WIDTH, get_button_width)
	       .Set (Am_REAL_HEIGHT, get_button_height)
	       .Set (Am_WIDTH, panel_item_width)
	       .Set (Am_HEIGHT, panel_item_height)
	       .Set (Am_ACTIVE, active_from_command_panel)
	       .Set (Am_SELECTED, false)
	       .Set (Am_WIDGET_LOOK, look_from_owner)
	       .Set (Am_FONT, Am_Font_From_Owner)
	       .Set (Am_ITEM_OFFSET, Am_From_Owner (Am_ITEM_OFFSET))
	       .Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
	       .Set (Am_FINAL_FEEDBACK_WANTED, final_feedback_from_owner)
	       .Set (Am_SET_COMMAND_OLD_OWNER, NULL)
	       )
    .Add_Part (Am_COMMAND, Am_Command.Create("Command_In_Button_Panel")
	       .Set (Am_LABEL, "Panel Button"))
    ;

  //this do method is in addition to the impl_command's of the interactor
  inter.Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, Am_Inter_For_Panel_Do)
    .Set(Am_UNDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_REDO_METHOD, Am_Widget_Inter_Command_Undo)
    .Set(Am_SELECTIVE_UNDO_METHOD, Am_Widget_Inter_Command_Selective_Undo)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD,
	 Am_Widget_Inter_Command_Selective_Repeat)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    .Set_Name("Command_In_Button_Inter")
    ;

//  obj_adv = (Am_Object_Advanced&)Am_Button_Panel;
//  obj_adv.Get_Slot(Am_WIDTH)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_HEIGHT)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_WEB_CONSTRAINT)->Set_Single_Constraint_Mode (false);

  obj_adv = (Am_Object_Advanced&)Am_Button_In_Panel;
  obj_adv.Get_Slot(Am_SELECTED).Set_Single_Constraint_Mode (false);
  obj_adv.Get_Slot(Am_VALUE).Set_Single_Constraint_Mode (false);

  obj_adv = (Am_Object_Advanced&)Am_Button_Panel;
  obj_adv.Get_Slot(Am_VALUE).Set_Single_Constraint_Mode (false);

  // don't want the individual interactor from the button
  Am_Button_In_Panel.Remove_Part(Am_INTERACTOR);
  // when in a panel, the button's command object is gotten from the item slot
  // which is set automatically by the Map
  command_obj = Am_Button_In_Panel.Get_Part(Am_COMMAND);
  Am_Button_In_Panel.Set(Am_ITEM, command_obj); // default value
  Am_Button_In_Panel.Set(Am_ITEM_TO_COMMAND, Am_Copy_Item_To_Command);

  // set value of interactor so is multi constraint mode.
  obj_adv = (Am_Object_Advanced&)inter;
  obj_adv.Get_Slot(Am_VALUE).Set_Single_Constraint_Mode(false);
  
  ///////////////////////////////////////////////////////////////////////////
  // Radio button panel
  ///////////////////////////////////////////////////////////////////////////

  Am_Radio_Button_Panel = Am_Button_Panel.Create("Radio button Panel");
  Am_Radio_Button_Panel.Remove_Part (Am_ITEM_PROTOTYPE);
  Am_Radio_Button_Panel
    .Set (Am_BOX_ON_LEFT, true)
    .Set (Am_BOX_HEIGHT, 15)
    .Set (Am_BOX_WIDTH, 15)
    .Set (Am_ITEM_OFFSET, 3)
    .Set (Am_H_ALIGN, Am_Align_From_Box_On_Left)
    .Set (Am_FIXED_WIDTH, false) // fixed width makes the labels centered
    .Set (Am_FINAL_FEEDBACK_WANTED, true)
    .Add_Part (Am_ITEM_PROTOTYPE, Am_Radio_Button_In_Panel =
       Am_Radio_Button.Create ("Radio_Button_In_Panel_Proto")
	       .Set (Am_REAL_WIDTH, get_checkbox_width)
	       .Set (Am_REAL_HEIGHT, get_checkbox_height)
	       .Set (Am_WIDTH, panel_item_width)
	       .Set (Am_HEIGHT, panel_item_height)
	       .Set (Am_BOX_ON_LEFT, box_on_left_from_owner)
	       .Set (Am_BOX_WIDTH, box_width_from_owner)
	       .Set (Am_BOX_HEIGHT, box_height_from_owner)
	       .Set (Am_ACTIVE, active_from_command_panel)
	       .Set (Am_SELECTED, false)
	       .Set (Am_WIDGET_LOOK, look_from_owner)
	       .Set (Am_FONT, Am_Font_From_Owner)
	       .Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
	       .Set (Am_ITEM_OFFSET, Am_From_Owner (Am_ITEM_OFFSET))
	       .Set (Am_SET_COMMAND_OLD_OWNER, NULL)
	       .Set (Am_FINAL_FEEDBACK_WANTED, final_feedback_from_owner)
	       )
    ;
  
  inter = Am_Radio_Button_Panel.Get (Am_INTERACTOR);
  inter.Set(Am_FIRST_ONE_ONLY, true);
  
  // don't want the individual interactor from the button
  Am_Radio_Button_In_Panel.Remove_Part(Am_INTERACTOR);
  // when in a panel, the box's command object is gotten from the item slot
  // which is set automatically by the Map
  command_obj = Am_Radio_Button_In_Panel.Get_Part(Am_COMMAND);
  Am_Radio_Button_In_Panel.Set(Am_ITEM, command_obj); // default value
  Am_Radio_Button_In_Panel.Set(Am_ITEM_TO_COMMAND, Am_Copy_Item_To_Command);
  
  ///////////////////////////////////////////////////////////////////////////
  // Check Boxes
  ///////////////////////////////////////////////////////////////////////////

  Am_Checkbox_Panel = Am_Button_Panel.Create("Checkbox Panel");
  Am_Checkbox_Panel.Remove_Part (Am_ITEM_PROTOTYPE);
  Am_Checkbox_Panel
    .Set (Am_HOW_SET, (int)Am_CHOICE_LIST_TOGGLE)
    .Set (Am_FINAL_FEEDBACK_WANTED, true)
    .Set (Am_FIXED_WIDTH, false) // fixed width makes the labels centered
    .Set (Am_BOX_ON_LEFT, true)
    .Set (Am_BOX_HEIGHT, 15)
    .Set (Am_ITEM_OFFSET, 3)
    .Set (Am_BOX_WIDTH, 15)
    .Set (Am_H_ALIGN, Am_Align_From_Box_On_Left)
    .Add_Part (Am_ITEM_PROTOTYPE, Am_Checkbox_In_Panel =
	       Am_Checkbox.Create ("Checkbox_In_Panel_Proto")
	       .Set (Am_REAL_WIDTH, get_checkbox_width)
	       .Set (Am_REAL_HEIGHT, get_checkbox_height)
	       .Set (Am_WIDTH, panel_item_width)
	       .Set (Am_HEIGHT, panel_item_height)
	       .Set (Am_BOX_ON_LEFT, box_on_left_from_owner)
	       .Set (Am_BOX_WIDTH, box_width_from_owner)
	       .Set (Am_BOX_HEIGHT, box_height_from_owner)
	       .Set (Am_ACTIVE, active_from_command_panel)
	       .Set (Am_SELECTED, false)
	       .Set (Am_WIDGET_LOOK, look_from_owner)
	       .Set (Am_FONT, Am_Font_From_Owner)
	       .Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
	       .Set (Am_ITEM_OFFSET, Am_From_Owner (Am_ITEM_OFFSET))
	       .Set (Am_FINAL_FEEDBACK_WANTED, final_feedback_from_owner)
	       .Set (Am_SET_COMMAND_OLD_OWNER, NULL)
	       )
    ;
  
  inter = Am_Checkbox_Panel.Get (Am_INTERACTOR);
  inter.Set(Am_FIRST_ONE_ONLY, true);

  // don't want the individual interactor from the button
  Am_Checkbox_In_Panel.Remove_Part(Am_INTERACTOR);
  // when in a panel, the box's command object is gotten from the item slot
  // which is set automatically by the Map
  command_obj = Am_Checkbox_In_Panel.Get_Part(Am_COMMAND);
  Am_Checkbox_In_Panel.Set(Am_ITEM, command_obj); // default value
  Am_Checkbox_In_Panel.Set(Am_ITEM_TO_COMMAND, Am_Copy_Item_To_Command);

  ///////////////////////////////////////////////////////////////////////////
  // Menus 
  ///////////////////////////////////////////////////////////////////////////

  // Based on button panel.

  Am_Menu = Am_Button_Panel.Create("Menu")
    .Set (Am_WIDTH, menu_width)
    .Set (Am_HEIGHT, menu_height)
    .Set (Am_HOW_SET, (int)Am_CHOICE_SET)
    .Set (Am_ITEM_OFFSET, 1)
    .Set (Am_X_OFFSET, 2)
    .Set (Am_Y_OFFSET, 2)
    .Set (Am_V_SPACING, -2)
    .Set (Am_KEY_SELECTED, false)
    .Set (Am_FINAL_FEEDBACK_WANTED, false)
    .Set (Am_STYLE_RECORD, Am_Get_Computed_Colors_Record_Form)
    .Set (Am_DRAW_METHOD, menu_draw)
    ;
  
//  obj_adv = (Am_Object_Advanced&)Am_Menu;
//  obj_adv.Get_Slot(Am_WIDTH)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_HEIGHT)->Set_Single_Constraint_Mode (true);

  Am_Menu.Remove_Part (Am_ITEM_PROTOTYPE);
  Am_Menu.Add_Part (Am_ITEM_PROTOTYPE, Am_Item_In_Menu =
		    Am_Menu_Item.Create ("Item_In_Menu_Proto")
		    .Set (Am_REAL_WIDTH, get_menu_item_width)
		    .Set (Am_REAL_HEIGHT, get_button_height)
		    .Set (Am_WIDTH, panel_item_width)
		    .Set (Am_HEIGHT, panel_item_height)
		    .Set (Am_ACTIVE, active_from_command_panel)
		    .Set (Am_SELECTED, false)
		    .Set (Am_WIDGET_LOOK, look_from_owner)
		    .Set (Am_FONT, Am_Font_From_Owner)
		    .Set (Am_ITEM_OFFSET, Am_From_Owner (Am_ITEM_OFFSET))
		    .Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		    .Set (Am_FINAL_FEEDBACK_WANTED, final_feedback_from_owner)
		    .Set (Am_SET_COMMAND_OLD_OWNER, NULL)
		    )
    ;
  
  // don't want the individual interactor from the button
  Am_Item_In_Menu.Remove_Part(Am_INTERACTOR);
  // when in a panel, the box's command object is gotten from the item slot
  // which is set automatically by the Map
  command_obj = Am_Item_In_Menu.Get_Part(Am_COMMAND);
  Am_Item_In_Menu.Set(Am_ITEM, command_obj); // default value
  Am_Item_In_Menu.Set(Am_ITEM_TO_COMMAND, Am_Copy_Item_To_Command);

  ///////////////////////////////////////////////////////////////////////////
  // Menu_Bars 
  ///////////////////////////////////////////////////////////////////////////

  // Based on menus.

  // internal: prototype for the sub-menu windows
  Am_Object proto, sub_menu;
  
  Am_Pop_Up_Window_Proto = Am_Window.Create("Pop_Up_Window")
       .Set(Am_OMIT_TITLE_BAR, true)
       .Set(Am_SAVE_UNDER, true)
       .Set(Am_FOR_ITEM, Am_No_Object) //for_item is menu button widget
       .Set(Am_LEFT, 0) //usually overridden with formula
       .Set(Am_TOP, 0)  //usually overridden with formula
       .Set(Am_WIDTH,  popup_sub_win_width)
       .Set(Am_HEIGHT, popup_sub_win_height)
       .Set(Am_FILL_STYLE, popup_fill_from_for_item)
       .Set(Am_VISIBLE, false)
       // copy the main-window's undo handler here so that commands in
       // the menu bar can be undone if the main window is undo-able
       .Set(Am_UNDO_HANDLER, popup_sub_win_undo_handler)
       .Add_Part(Am_SUB_MENU, sub_menu = Am_Menu.Create("Sub_Menu")
		 .Set(Am_ITEMS, sub_menu_items)
		 .Set(Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
		 .Set(Am_WIDGET_LOOK, sub_menu_look_from_win_for_item)
		 .Set(Am_FONT, sub_menu_font_from_win_for_item)
		 .Set(Am_SET_COMMAND_OLD_OWNER, sub_menu_set_old_owner)
		 )
       ;
  //remove the interactor from the sub-menu
  sub_menu.Remove_Part(Am_INTERACTOR);

  obj_adv = (Am_Object_Advanced&)Am_Pop_Up_Window_Proto;
  obj_adv.Get_Slot(Am_WIDTH).Set_Single_Constraint_Mode (false);
  obj_adv.Get_Slot(Am_HEIGHT).Set_Single_Constraint_Mode (false);

  proto = sub_menu.Get_Part(Am_ITEM_PROTOTYPE);
  proto.Set_Name("popup_sub_item")
    .Set (Am_ACTIVE, menu_bar_sub_item_active);

  Am_Menu_Bar_Sub_Window_Proto =
    Am_Pop_Up_Window_Proto.Create("Sub_Menu_Window")
       .Set(Am_LEFT, popup_sub_win_left)
       .Set(Am_TOP, menu_bar_sub_win_top)
       ;

  Am_Menu_Bar = Am_Menu.Create("Menu_Bar")
 	//default width = width of container (usually a window)
    .Set (Am_WIDTH, menu_bar_width)
    .Set (Am_HEIGHT, menu_bar_height)
    .Set (Am_HOW_SET, (int)Am_CHOICE_SET)
    .Set (Am_FIXED_WIDTH, false)
    .Set (Am_FIXED_HEIGHT, false)
    .Set (Am_LAYOUT, Am_Horizontal_Layout)
    .Set (Am_ITEMS, 0)
    ;
  obj_adv = (Am_Object_Advanced&)Am_Menu_Bar;
//  obj_adv.Get_Slot(Am_WIDTH)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_HEIGHT)->Set_Single_Constraint_Mode (true);
  obj_adv.Get_Slot (Am_FILL_STYLE)
    .Set_Demon_Bits (Am_STATIONARY_REDRAW | Am_EAGER_DEMON);
  demons = obj_adv.Get_Demons ().Copy ();
  demons.Set_Object_Demon (Am_DESTROY_OBJ, Am_Destroy_Menu_Bar);
  obj_adv.Set_Demons (demons);

  proto = Am_Menu_Bar.Get_Part(Am_ITEM_PROTOTYPE);
  proto.Remove_Slot(Am_ITEM_TO_COMMAND); // make sure the inherited
			                 // constraint is destroyed
  proto.Set(Am_ITEM_TO_COMMAND, menu_bar_copy_item_to_command);
  proto.Set(Am_SUB_MENU, Am_No_Object);  //fix a bug that if create slot
				 //later, doesn't re-evaluate formulas 
  
  //make Am_SUB_MENU slot be local
  obj_adv = (Am_Object_Advanced&)proto;
  obj_adv.Get_Slot(Am_SUB_MENU).Set_Inherit_Rule(Am_LOCAL);

  inter = Am_Menu_Bar.Get_Part(Am_INTERACTOR)
    .Set(Am_START_WHERE_TEST, in_menu_bar_item)
    .Set(Am_MULTI_OWNERS, menu_bar_window_list)
    .Set (Am_INTERIM_DO_METHOD, menu_bar_inter_interim_do)
    .Set (Am_ABORT_DO_METHOD, menu_bar_inter_abort)
    .Set (Am_DO_METHOD, menu_bar_inter_do)
    .Set_Name("inter_in_menu_bar")
    ;

  inter.Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, NULL) //get rid of Am_Inter_For_Panel_Do
    .Set(Am_UNDO_METHOD, NULL)
    .Set(Am_REDO_METHOD, NULL)
    .Set(Am_SELECTIVE_UNDO_METHOD, NULL)
    .Set(Am_SELECTIVE_REPEAT_SAME_METHOD, NULL)
    .Set(Am_SELECTIVE_REPEAT_ON_NEW_METHOD, NULL)
    ;

  ///////////////////////////////////////////////////////////////////////////
  // Option button: single button that pops up a menu
  ///////////////////////////////////////////////////////////////////////////

  //can't be a part of option button since needs to be stand-alone
  //window at the top level

  Am_Option_Button_Sub_Window_Proto =
      Am_Pop_Up_Window_Proto.Create("Option_Sub_Window")
      .Set(Am_FOR_ITEM, NULL)
      .Set(Am_LEFT, popup_sub_win_left)
      .Set(Am_TOP, option_sub_win_top)
      .Set(Am_VISIBLE, false)
      // copy the main-window's undo handler here so that commands in
      // the option button can be undone if the main window is undo-able
     ;
  Am_Object submenu = Am_Option_Button_Sub_Window_Proto.Get_Part(Am_SUB_MENU);
  submenu.Set(Am_ITEMS, option_button_items)
    .Set_Name("sub_menu_in_option_button")
    .Add_Part(Am_INTERACTOR, inter = Am_Button_Panel.Get_Part(Am_INTERACTOR)
	       .Create("Inter_In_Option_PopUp"))
     ;
  inter.Set(Am_MULTI_OWNERS, option_button_window_list);
  inter.Get_Part(Am_COMMAND)
    .Set(Am_DO_METHOD, option_sub_menu_do)
    .Set(Am_ABORT_DO_METHOD, option_hide_sub_menu)
    ;
  submenu.Remove_Part(Am_COMMAND);
  submenu.Set(Am_COMMAND, command_from_for_item);

  // Based on menus (like menubars)
  Am_Option_Button = Am_Button.Create("Option_Button")
    .Set (Am_ITEMS, 0)
    .Set (Am_WIDTH, get_option_button_width)
    .Set (Am_DRAW_METHOD, option_button_draw)
    .Set (Am_FOR_ITEM, create_sub_menu)
    .Set (Am_SUB_MENU, 0)
    .Set_Inherit_Rule(Am_SUB_MENU, Am_LOCAL)
    .Set_Single_Constraint_Mode (Am_VALUE, false)
    .Set (Am_VALUE, option_button_value)
    .Set (Am_COMPUTE_INTER_VALUE, compute_inter_value)
    .Set (Am_REAL_STRING_OR_OBJ, get_real_string_from_inter_val)
    ;

  obj_adv = (Am_Object_Advanced&)Am_Option_Button;
//  obj_adv.Get_Slot(Am_WIDTH)->Set_Single_Constraint_Mode (true);
//  obj_adv.Get_Slot(Am_HEIGHT)->Set_Single_Constraint_Mode (true);
  demons = obj_adv.Get_Demons ().Copy ();
  demons.Set_Object_Demon (Am_DESTROY_OBJ, destroy_option_button);
  obj_adv.Set_Demons (demons);

  inter = Am_Option_Button.Get_Part(Am_INTERACTOR)
    .Set (Am_CONTINUOUS, false)
    .Set (Am_DO_METHOD, option_button_start_do)
    .Set_Name("inter_in_option_button")
    ;
  //no command in interactor, instead use command in interactor of the menu
  inter.Remove_Part(Am_COMMAND);
  inter.Get_Part(Am_IMPLEMENTATION_COMMAND)
    .Set(Am_IMPLEMENTATION_PARENT, Am_NOT_USUALLY_UNDONE);
  
}
