/*
 * TPaint -- a simple bitmap editor
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <toad/toad.hh>
#include <toad/dnd/color.hh>

#include "BitmapEditor.hh"
#include "Tool.hh"

extern TTool *tool;
extern TBitmapEditor *bme;
extern void RegisterBitmapEditor(TBitmapEditor *ed);
extern void RemoveBitmapEditor(TBitmapEditor *ed);

TBitmapEditor::TBitmapEditor(TWindow *p,const string &t)
	:TWindow(p,t)
{
	w=32; h=32; zoom=1;
	filename = NULL;
	bitmap = NULL;
	bNoBackground = true;
}

void TBitmapEditor::create()
{
	::RegisterBitmapEditor(this);

	if (!bitmap)
	{	
		bitmap = new TAlterBitmap(w,h,TBITMAP_TRUECOLOR);
		for(int y=0; y<h; y++)
			for(int x=0; x<w; x++)
				bitmap->SetPixel(x,y,255,255,255);
	}
	SetSize(w*zoom,h*zoom);
	SetMouseMoveMessages(TMMM_LBUTTON);
}

void TBitmapEditor::destroy()
{
	::RemoveBitmapEditor(this);
	if (bitmap)
	{
		delete bitmap;
		bitmap=NULL;
	}
}

void TBitmapEditor::focus()
{
//	printf("TBitmapEditor::focus(%s)\n",b?"true":"false");
	if (IsFocus()) bme = this;	// set current TBitmapEditor
}

void TBitmapEditor::keyDown(TKey, char *string,unsigned)
{
	switch(string[0])
	{
		case '+': case '>':
			if (zoom<16)
			{
				zoom++;
				SetSize(w*zoom,h*zoom);
			}
			
			break;
		case '-': case '<':
			if(zoom>1)
			{
				zoom--;
				SetSize(w*zoom,h*zoom);
			}
			break;
	}
}

void TBitmapEditor::SetEditSize(int w,int h)
{
	this->w = w;
	this->h = h;
}

void TBitmapEditor::SetEditZoom(int z)
{
	zoom = z;
	if (IsRealized())
	{
		SetSize(w*zoom,h*zoom);
	}
}

bool TBitmapEditor::Save(const char* filename, void *xtra)
{
	if (!bitmap) return false;
	return bitmap->Save(filename,xtra);
}

bool TBitmapEditor::Load(const char* fn)
{
	if (filename)
		delete filename;
	filename=new char[strlen(fn)+1];
	strcpy(filename,fn);
	if (!bitmap)
	{
		bitmap = new TAlterBitmap;
	}

	try {
		bitmap->Load(filename);
	}
	catch(runtime_error &e) {
		MessageBox(NULL, "Load Bitmap Failed", e.what(), MB_ICONSTOP|MB_OK);
		return false;
	}
//		SetTitle(filename);
	w = bitmap->width;
	h = bitmap->height;
	SetSize(w*zoom, h*zoom);
	if (!IsRealized()) {
		Create();
		//TPen pen(this);
		//pen.SetColorMode(TColor::DITHER);
		// bitmap->SetZoom(zoom);
		//pen.DrawBitmap(0,0,*bitmap);
	}	else {
		UpdateWindow();
	}
	return true;
}

void TBitmapEditor::thread()
{
}

inline bool TBitmapEditor::IsInside(int x,int y)
{
	return (x>=0 && x<w && y>=0 && y<h);
}

void TBitmapEditor::paint()
{
	TPen pen(this);
	TRect rect;
	pen.GetClipBox(&rect);
	rect.x/=zoom;
	rect.y/=zoom;
	rect.w=(rect.w/zoom)+((rect.w%zoom)?2:1);
	rect.h=(rect.h/zoom)+((rect.h%zoom)?2:1);

	pen.SetColorMode(TColor::DITHER);

	bitmap->SetZoom(zoom);
	pen.DrawBitmap(0,0,*bitmap);
}

void TBitmapEditor::mouseLDown(int x,int y,unsigned m)
{
	x/=zoom;
	y/=zoom;
	tool->mouseLDown(this,x,y,m);
}

void TBitmapEditor::mouseMDown(int x,int y,unsigned modifier)
{
	TColor c;
	x/=zoom; y/=zoom;
	GetPixel(x,y, &c);
	StartDrag(
		new TDnDColor(c), 
		modifier);
}

void TBitmapEditor::mouseLUp(int x,int y,unsigned m)
{
	x/=zoom;
	y/=zoom;
	tool->mouseLUp(this,x,y,m);
}

void TBitmapEditor::mouseMove(int x,int y,unsigned m)
{
	x/=zoom;
	y/=zoom;
	tool->mouseMove(this,x,y,m);
}

void TBitmapEditor::SetPixel(int x,int y,TColor &color,TPen &pen)
{
	pen.SetColorMode(TColor::DITHER);
	pen.SetColor(color.r,color.g,color.b);
	pen.FillRectangle(x*zoom,y*zoom,zoom,zoom);
	
	bitmap->SetPixel(x,y,color.r,color.g,color.b);
}

bool TBitmapEditor::GetPixel(int x,int y,TColor *color)
{
	return bitmap->GetPixel(x,y,color);
}

void TBitmapEditor::InvertPixel(int x,int y,TPen &pen)
{
	pen.SetMode(TPen::INVERT);
	pen.DrawRectangle(x*zoom,y*zoom,zoom,zoom);
}
