/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-2000 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toad.hh>
#include <toad/gadget.hh>
#include <toad/gadgeteditor.hh>
#include <toad/scrollbar.hh>
#include <toad/gadgeteditor/undoablemove.hh>
#include <toad/gadgeteditor/undoablehandlemove.hh>
#include <toad/gadgeteditor/undoabledelete.hh>
#include <toad/gadgeteditor/undoablecreate.hh>

#include <algorithm>

#define DBM(CMD)
#define VERBOSE 0

//! TGadgetEditor
//. TGadgetEditor is a common graphical editor for TGadget objects.
//. <P>
//. It's still experimental so expect major changes in the future before
//. using it. One major goal is to make it possible to edit scaleable and
//. rotateable 2D gadgets and to create 3D objects.
//. <P>
//. Originally written for the dialog editor, it can be used for lots of
//. other jobs, e.g. a MacOS(tm) alike file folder.
//. <P>
//. TGadgetEditor can be used as a window of its own or just as an object 
//. to delegate events from other windows to. The later was needed for the
//. dialog editor.

TGadgetEditor::TGadgetEditor():
	super(NULL, "(no window)")
{
	Init();
	window = NULL;
}

void TGadgetEditor::SetWindow(TWindow *w)
{
	// [store gadgets to `window']
	if (window)
		window->Invalidate();
	window = w;
	// [get gadgets from `window']
	if (window)
		window->Invalidate();
}

TGadgetEditor::TGadgetEditor(TWindow *p, const string &t):
	super(p, t)
{
	Init();
	SetMouseMoveMessages(TMMM_LBUTTON);
	bNoBackground = true;
	window = this;
}

void TGadgetEditor::Init()
{
	line_color.Set(0,0,0);
	fill_color.Set(255,255,255);
	filled = false;
	background_color.Set(192,192,192);

	gridx = gridy = 4;
	draw_grid = true;
	handle = -1;
	gadget = gtemplate = NULL;
	operation = OP_SELECT;
	state = STATE_NONE;
	use_scrollbars = true;
	vscroll = NULL;
	hscroll = NULL;
	x1=y1=x2=y2=0;
	update_scrollbars = false;
}

TGadgetEditor::~TGadgetEditor()
{
//	SetMode(MODE_SELECT);
//	cout << "gadgets total: " << gadgets.size() << endl;
}

void TGadgetEditor::EnableScroll(bool b)
{
	use_scrollbars = b;
}

void TGadgetEditor::EnableGrid(bool b)
{
	draw_grid = b;
}

void TGadgetEditor::SetBackground(int r,int g,int b)
{
	background_color.Set(r,g,b);
}

void TGadgetEditor::resize()
{
	UpdateScrollbars();
}

void TGadgetEditor::paint()
{
	if (!window) {
		cout << __PRETTY_FUNCTION__ << ": no window" << endl;
		return;
	}
	if (update_scrollbars) {
		// cout << "paint: update_scrollbars" << endl;
		UpdateScrollbars();
		update_scrollbars = false;
	}
	TBitmap bmp(window->Width(), window->Height(), TBITMAP_SERVER);
	TPen pen(&bmp);

	if (draw_grid && gridx && gridy) {
		TBitmap bitmap(gridx,gridy, TBITMAP_SERVER);
		TPen bpen(&bitmap);
		bpen.SetColor(background_color);
		bpen.FillRectangle(0,0,gridx,gridy);
		bpen.SetColor(
			background_color.r > 128 ? background_color.r-128 : background_color.r+128,
			background_color.g > 128 ? background_color.g-128 : background_color.g+128,
			background_color.b > 128 ? background_color.b-128 : background_color.b+128
		);
		int ox = window->OriginX() % gridx;
		if (ox<0)
			ox+=gridx;
		int oy = window->OriginY() % gridy;
		if (oy<0)
			oy+=gridy;
		
		bpen.DrawPoint(ox, oy);
		pen.SetBitmap(&bitmap);
	} else {
		pen.SetColor(background_color);
	}
	pen.FillRectangle(0,0,window->Width(),window->Height());

	pen.SetOrigin(window->OriginX(), window->OriginY());
	pen.SetColor(TColor::BLACK);
	
	TGadgetStorage::iterator p,e;
	
	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		TGadget::EPaintType pt = TGadget::NORMAL;
		if (gadget==*p) {
			pt = TGadget::EDIT;
		} else if (selection.find(*p)!=selection.end()) {
			pt = TGadget::SELECT;
		}
		(*p)->paint(pen, pt);
		p++;
	}

	pen.SetColor(0,0,0);
	TGadgetSet::iterator sp,se;
	sp = selection.begin();
	se = selection.end();
	while(sp!=se) {
		(*sp)->paintSelection(pen);
		sp++;
	}

	if (vscroll && vscroll->Mapped() &&
			hscroll && hscroll->Mapped() )
	{
		TRectangle r(window->Width() - TScrollBar::FixedSize(),
								 window->Height()- TScrollBar::FixedSize(),
								 TScrollBar::FixedSize(), TScrollBar::FixedSize());
		pen.SetColor(TColor::LIGHTGRAY);
		pen.SetOrigin(0,0);
		pen|=r;
		pen.FillRectangle(r);
	}

	TPen scr(window);
	scr.DrawBitmap(-window->OriginX(),-window->OriginY(), &bmp);
}

void TGadgetEditor::SetLineColor(const TRGB &rgb)
{
	line_color = rgb;
	TGadgetSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->line_color = line_color;
		p++;
	}
	if (gtemplate)
		gtemplate->line_color = line_color;
	Invalidate();
}

void TGadgetEditor::SetFillColor(const TRGB &rgb)
{
	fill_color = rgb;
	TGadgetSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->fill_color = fill_color;
		p++;
	}
	if (gtemplate)
		gtemplate->fill_color = fill_color;
	Invalidate();
}

void TGadgetEditor::SetFilled(bool b)
{
	filled = b;
	TGadgetSet::iterator p,e;
	p = selection.begin();
	e = selection.end();
	while(p!=e) {
		(*p)->filled = b;
		p++;
	}
	if (gtemplate)
		gtemplate->filled = b;
	Invalidate();
}

void TGadgetEditor::Add(TGadget *g)
{
	gadgets.push_back(g);
	update_scrollbars = true;
	InvalidateFigure(g);
}

void TGadgetEditor::DeleteGadget(TGadget *g)
{
	if (g==gadget)
		gadget=NULL;
	if (g==gtemplate)
		gtemplate=NULL;
	
	TGadgetStorage::iterator p,e;
	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		if (g==*p) {
			gadgets.erase(p);
			break;
		}
		p++;
	}

	TGadgetSet::iterator s;
	s = selection.find(g);
	if (s!=selection.end())
		selection.erase(s);	

	delete g;
}

void TGadgetEditor::ClearSelection()
{
	selection.erase(selection.begin(), selection.end());
	Invalidate();
}

//. Delete all selected objects with `removeable' being true.
void TGadgetEditor::DeleteSelection()
{
//cout << "delete selection" << endl;
	history.Add(new TUndoableDelete(gadgets, selection));
//cout << "selection size: " << selection.size() << endl;

	TGadgetStorage::iterator p,e,del;
	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end() &&
				(*p)->removeable )
		{
			del = p;
			p++;
			if (gadget==*del)
				gadget=NULL;
//			cout << "removing gadget " << *del << endl;
			gadgets.erase(del);
		} else {
			p++;
		}
	}

	ClearSelection();
	update_scrollbars = true;
	Invalidate();
}

void TGadgetEditor::SelectAll()
{
	TGadgetStorage::iterator p,e;
	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		selection.insert(*p);
		p++;
	}	
}

void TGadgetEditor::DeleteAll()
{
	SelectAll();
	DeleteSelection();
	SetOrigin(0,0);
	if (vscroll)
		vscroll->SetValue(0);
	if (hscroll)
		hscroll->SetValue(0);
	UpdateScrollbars();
}

void TGadgetEditor::Selection2Top()
{
	TGadgetStorage::iterator p,b,np;
	p = gadgets.end();
	b = gadgets.begin();

	if (p==b)
		return;

	p--; np=p;											// p, np @ last element

	if (p==b)
		return;

	while(true) {
		if (selection.find(*p)!=selection.end()) {
			if (p!=np) {
				TGadget *akku = *p;
				TGadgetStorage::iterator mp = p;
				while(mp!=np) {
					TGadgetStorage::iterator op = mp;
					mp++;
					*op = *mp;
				}
				*np = akku;
			}
			np--;
		}
		if (p==b)
			break;
		p--;
	}
	
	Invalidate();
}

void TGadgetEditor::Selection2Bottom()
{
	TGadgetStorage::iterator p,e,np;
	p = np = gadgets.begin();
	e = gadgets.end();
	if (p==e)
		return;
	if (selection.find(*p)!=selection.end())
		np++;
	p++;
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			TGadget *akku = *p;
			TGadgetStorage::iterator mp = p;
			while(mp!=np) {
				TGadgetStorage::iterator op = mp;
				mp--;
				*op = *mp;
			}
			*np = akku;
			np++;
		}
		p++;
	}
	Invalidate();
}

void TGadgetEditor::SelectionUp()
{
	TGadgetStorage::iterator p,e,b,prev;
	p = e = prev = gadgets.end();
	b = gadgets.begin();
	if (p==b)
		return;
	while(true) {
		if (selection.find(*p)!=selection.end()) {
			if (prev!=e) {
				TGadget* a = *p;
				*p = *prev;
				*prev = a;
			}
			prev = e;
		} else {
			prev = p;
		}
		if (p==b)
			break;
		p--;
	}
	Invalidate();
}

void TGadgetEditor::SelectionDown()
{
	TGadgetStorage::iterator p,e,prev;
	p = gadgets.begin();
	e = prev = gadgets.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			if (prev!=e) {
				TGadget* a = *p;
				*p = *prev;
				*prev = a;
			}
			prev=e;
		} else {
			prev=p;
		}
		p++;
	}
	Invalidate();
}

void TGadgetEditor::Gadget2Top(TGadget *gadget)
{
	cerr << __PRETTY_FUNCTION__ << ": sorry, not implemented yet" << endl;
}

void TGadgetEditor::Gadget2Bottom(TGadget *gadget)
{
	TGadgetStorage::iterator p,e,np;
	p = np = gadgets.begin();
	e = gadgets.end();
	if (p==e)
		return;
	if (*p==gadget)
		return;
	p++;
	while(p!=e) {
		if (*p==gadget) {
			TGadget *akku = *p;
			TGadgetStorage::iterator mp = p;
			while(mp!=np) {
				TGadgetStorage::iterator op = mp;
				mp--;
				*op = *mp;
			}
			*np = akku;
			return;
		}
		p++;
	}
	Invalidate();
}

void TGadgetEditor::GadgetUp(TGadget *gadget)
{
	cerr << __PRETTY_FUNCTION__ << ": sorry, not implemented yet" << endl;
}

void TGadgetEditor::GadgetDown(TGadget *gadget)
{
	cerr << __PRETTY_FUNCTION__ << ": sorry, not implemented yet" << endl;
}

void TGadgetEditor::Group()
{
	if (selection.size()<2)
		return;
	TGGroup *group = new TGGroup();
	TGadgetStorage::iterator p,e;

	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			group->gadgets.push_back(*p);
			TGadgetStorage::iterator del = p;
			p++;
			gadgets.erase(del);
		} else {
			p++;
		}
	}

	ClearSelection();
	group->CalcSize();
	gadgets.insert(p, group);
	selection.insert(group);
}

void TGadgetEditor::Ungroup()
{
	TGadgetStorage::iterator p,e;
	p = gadgets.begin();
	e = gadgets.end();
	while(p!=e) {
		if (selection.find(*p)!=selection.end()) {
			TGGroup *group = dynamic_cast<TGGroup*>(*p);
			if (group) {
				TGGroup::TGadgetStorage::iterator vp,ve;
				vp = group->gadgets.begin();
				ve = group->gadgets.end();
				gadgets.insert(p, vp,ve);
				group->gadgets.erase(group->gadgets.begin(),group->gadgets.end());
				delete group;
				TGadgetStorage::iterator del = p;
				p++;
				gadgets.erase(del);
				continue;
			}
		}
		p++;
	}
	ClearSelection();
}

//. Abondon the current mode of operation and select a new mode.
//. <P>
void TGadgetEditor::SetOperation(unsigned op)
{
#if VERBOSE
	cout << "Setting Operation " << op << endl;
#endif
	StopOperation();
	if (window)
		window->SetFocus();
	operation = op;
}

void TGadgetEditor::SetCreate(TGadget *t)
{
	if (gtemplate) {
		delete gtemplate;
	}
	TCloneable *clone = t->clone();
	gtemplate = dynamic_cast<TGadget*>(clone);
	if (!gtemplate) {
		cerr << "TGadget::clone() didn't delivered a TGadget" << endl;
		delete clone;
		return;
	}
	gtemplate->line_color = line_color;
	gtemplate->fill_color = fill_color;
	gtemplate->filled     = filled;
	gtemplate->removeable = true;
	ClearSelection();
	SetOperation(OP_CREATE);
}

void TGadgetEditor::StopOperation()
{
	switch(state) {
		case STATE_CREATE:
			ClearSelection();
			if (gadget) {
				selection.insert(gadget);
				history.Add(new TUndoableCreate(gadgets, selection));
				ClearSelection();
			}
			// Invalidate(gadget);
			break;
	}
	gadget = NULL;
	state = STATE_NONE;
}

void TGadgetEditor::keyDown(TKey key, char *s, unsigned m)
{
redo:
	switch(operation) {
		case OP_SELECT: {
			if (state!=STATE_EDIT) {
				switch(key) {
					case TK_DELETE:
						DeleteSelection();
						break;
				}
				break;
			}
		}
		case OP_CREATE: {
			if (state==STATE_NONE)
				break;
			assert(gadget!=NULL);
			unsigned r = gadget->keyDown(this,key,s,m);
			if (r & TGadget::DELETE)
				DeleteGadget(gadget);
			if (r & TGadget::STOP)
				StopOperation();
			if (r & TGadget::REPEAT)
				goto redo;
		} break;
	}
}

void TGadgetEditor::mouseLDown(int mx,int my, unsigned m)
{
	#if VERBOSE
		cout << __PRETTY_FUNCTION__ << endl;
	#endif

	int x = ((mx+gridx/2)/gridx)*gridx;
	int y = ((my+gridy/2)/gridy)*gridy;

	down_x = x;
	down_y = y;

	if (window)
		window->SetFocus();

	// handle special operation modes
	//--------------------------------
redo:

	switch(state) {

		case STATE_NONE: {
			#if VERBOSE
				cout << "  STATE_NONE" << endl;
			#endif
			switch(operation) {
				case OP_SELECT: {
					#if VERBOSE
						cout << "    OP_SELECT" << endl;
					#endif

					// handle the handles
					//--------------------
					if ( !selection.empty() && !(m&MK_DOUBLE) ) {
						TGadgetSet::iterator p,e;
						p = selection.begin();
						e = selection.end();
						#if VERBOSE
							cout << "      mouse @ " << mx << ", " << my << endl;
						#endif
						while(p!=e) {
							unsigned h=0;
							while(true) {
								if (!(*p)->getHandle(h,memo_pt))
									break;
								if (memo_pt.x-2<=mx && mx<=memo_pt.x+2 && 
										memo_pt.y-2<=my && my<=memo_pt.y+2) {
									#if VERBOSE
										cout << "      found handle at cursor => STATE_MOVE_HANDLE" << endl;
									#endif
									handle = h;
									#if VERBOSE
									cout << "      handle " << h << " @ " << memo_pt.x << ", " << memo_pt.y << endl;
									#endif
									state = STATE_MOVE_HANDLE;
									if (selection.size()>1) {
										TGadget *g = *p;
										ClearSelection();
										selection.insert(g);
										sigSelectionChanged();
									}
									return;
								}
								h++;
							}
							p++;
						}
					}

					// selection, start movement, start edit
					//--------------------------------------
					TGadget *g = FindGadgetAt(mx, my);
					if (g) {
						#if VERBOSE
							cout << "      gadget at cursor";
						#endif
						TGadgetSet::iterator gi = selection.find(g);
						if (m & MK_DOUBLE) {
							#if VERBOSE
								cout << ", double click => ";
							#endif
							if (g->startInPlace()) {
								#if VERBOSE
									cout << "STATE_EDIT" << endl;
								#endif
								ClearSelection();
								sigSelectionChanged();
								gadget = g;
								state = STATE_EDIT;
								goto redo;
							}
							#if VERBOSE
								cout << "not editing" << endl;
							#endif
						} else if (m & MK_SHIFT) {
							#if VERBOSE
								cout << ", shift => ";
							#endif
							if (gi==selection.end()) {
								#if VERBOSE
									cout << "  adding object to selection" << endl;
								#endif
								selection.insert(g);
							} else {
								#if VERBOSE
									cout << " removing object from selection" << endl;
								#endif
								selection.erase(gi);
							}
							sigSelectionChanged();
							Invalidate();
						} else {
							#if VERBOSE
								cout << " => ";
							#endif
							if (gi==selection.end()) {
								ClearSelection();
								selection.insert(g);
								sigSelectionChanged();
							}
							state = STATE_MOVE;
							memo_x = memo_y = 0;
							#if VERBOSE
								cout << "STATE_MOVE" << endl;
							#endif
						}
					} else {
						#if VERBOSE
							cout << "      nothing at cursor => STATE_SELECT_RECT" << endl;
						#endif
						if (!(m & MK_SHIFT)) {
							ClearSelection();
							sigSelectionChanged();
						}
						state =  STATE_SELECT_RECT;
					}
				} break;

				case OP_CREATE: {
					#if VERBOSE
						cout << "    OP_CREATE => STATE_CREATE" << endl;
					#endif
					ClearSelection();
					gadget = static_cast<TGadget*>(gtemplate->clone());
					gadgets.push_back(gadget);
					Invalidate();
					state = STATE_START_CREATE;
					gadget->startCreate();
					unsigned r = gadget->mouseLDown(this,x,y,m);
					state = STATE_CREATE;
					if (r & TGadget::DELETE)
						DeleteGadget(gadget);
					if (r & TGadget::STOP)
						StopOperation();
					if (r & TGadget::REPEAT)
						goto redo;
					return;
				} break;
			}
		} break;
		
		case STATE_CREATE: 
		case STATE_EDIT: {
			assert(gadget!=NULL);
			#if VERBOSE
			if (state==STATE_CREATE)
				cout << "  STATE_CREATE" << endl;
			else
				cout << "  STATE_EDIT" << endl;
			#endif
			unsigned r = gadget->mouseLDown(this,x,y,m);
			if (r & TGadget::DELETE) {
				#if VERBOSE
					cout << "    delete gadget" << endl;
				#endif
				DeleteGadget(gadget);
			}
			if (r & TGadget::STOP) {
				#if VERBOSE
					cout << "    stop" << endl;
				#endif
				StopOperation();
			}
			if (r & TGadget::REPEAT) {
				#if VERBOSE
					cout << "    repeat event" << endl;
				#endif
				goto redo;
			}
		} break;
	}
}


void TGadgetEditor::mouseMove(int x, int y, unsigned m)
{
	#if VERBOSE
		cout << __PRETTY_FUNCTION__ << endl;
	#endif
	x = ((x+gridx/2)/gridx)*gridx;
	y = ((y+gridy/2)/gridy)*gridy;

redo:

	switch(state) {
		case STATE_CREATE: 
		case STATE_EDIT: {
			#if VERBOSE
				if (state==STATE_CREATE)
					cout << "  STATE_CREATE => mouseMove to gadget" << endl;
				else
					cout << "  STATE_EDIT => mouseMove to gadget" << endl;
			#endif
			assert(gadget!=NULL);
			unsigned r = gadget->mouseMove(this,x,y,m);
			if (r & TGadget::DELETE) {
				#if VERBOSE
					cout << "    delete gadget" << endl;
				#endif
				DeleteGadget(gadget);
			}
			if (r & TGadget::STOP) {
				#if VERBOSE
					cout << "    stop" << endl;
				#endif
				StopOperation();
			}
			if (r & TGadget::REPEAT) {
				#if VERBOSE
					cout << "    repeat event" << endl;
				#endif
				goto redo;
			}
			return;
		} break;

		case STATE_MOVE: {
#if VERBOSE
			cout << "  STATE_MOVE => moving selection" << endl;
#endif
			int dx = x-down_x; down_x=x;
			int dy = y-down_y; down_y=y;
			TGadgetSet::iterator p,e;
			p = selection.begin();
			e = selection.end();
			memo_x+=dx;
			memo_y+=dy;
			while(p!=e) {
				InvalidateFigure(*p);
				(*p)->translate(dx, dy);
				InvalidateFigure(*p);
				p++;
			}
			UpdateScrollbars();
		} break;

		case STATE_MOVE_HANDLE: {
			#if VERBOSE
				cout << "  STATE_MOVE_HANDLE => moving handle" << endl;
			#endif
			InvalidateFigure(*selection.begin());
			(*selection.begin())->translateHandle(handle, x, y);
			InvalidateFigure(*selection.begin());
		} break;

		case STATE_SELECT_RECT: {
			#if VERBOSE
				cout << "  STATE_SELECT_RECT => redrawing rectangle" << endl;
			#endif
			window->Invalidate();
			window->PaintNow();
			TPen pen(window);
			pen.SetLineStyle(TPen::DOT);
			TRectangle r(TPoint(down_x,down_y), TPoint(x,y));
			pen.DrawRectangle(r);
		} break;
	}
}

void TGadgetEditor::mouseLUp(int x, int y, unsigned m)
{
#if VERBOSE
	cout << __PRETTY_FUNCTION__ << endl;
#endif

	x = ((x+gridx/2)/gridx)*gridx;
	y = ((y+gridy/2)/gridy)*gridy;

redo:

	switch(state) {
		case STATE_CREATE:
		case STATE_EDIT: {
			assert(gadget!=NULL);
			unsigned r = gadget->mouseLUp(this,x,y,m);
			if (r & TGadget::DELETE) {
				#if VERBOSE
					cout << "    delete gadget" << endl;
				#endif
				DeleteGadget(gadget);
			}
			if (r & TGadget::STOP) {
				#if VERBOSE
					cout << "    stop" << endl;
				#endif
				StopOperation();
			}
			if (r & TGadget::REPEAT) {
				#if VERBOSE
					cout << "    repeat event" << endl;
				#endif
				goto redo;
			}
			return;
		}	break;

		case STATE_MOVE: {
			#if VERBOSE
				cout << "  STATE_MOVE => STATE_NONE" << endl;
			#endif
			int dx = x-down_x; down_x=x;
			int dy = y-down_y; down_y=y;
			TGadgetSet::iterator p,e;
			p = selection.begin();
			e = selection.end();
			memo_x += dx;
			memo_y += dy;
			TUndoableMove *undo = new TUndoableMove(memo_x, memo_y, selection);
			history.Add(undo);
			while(p!=e) {
				InvalidateFigure(*p);
				(*p)->translate(dx, dy);
				InvalidateFigure(*p);
				p++;
			}
			UpdateScrollbars();
			state = STATE_NONE;
		}	break;

		case STATE_MOVE_HANDLE: {
			#if VERBOSE
				cout << "  STATE_MOVE_HANDLE => updating scrollbars, STATE_NONE" << endl;
			#endif
			InvalidateFigure(*selection.begin());
			(*selection.begin())->translateHandle(handle, x, y);
			InvalidateFigure(*selection.begin());
			state = STATE_NONE;
			UpdateScrollbars();
			
			TPoint pt(x,y);
			history.Add(new TUndoableHandleMove(*selection.begin(), handle, memo_pt, pt));
		} break;

		case STATE_SELECT_RECT: {
			#if VERBOSE
				cout << "  STATE_SELECT_RECT => ";
			#endif
			TGadgetStorage::iterator p, e;
			p = gadgets.begin();
			e = gadgets.end();
			TRectangle r1(TPoint(down_x,down_y), TPoint(x,y));
			TRectangle r2;
			while(p!=e) {
				(*p)->getShape(r2);
				if (r1.IsInside( r2.x, r2.y ) &&
						r1.IsInside( r2.x+r2.w, r2.y+r2.h ) )
				{
					selection.insert(*p);
				}
				p++;
			}
			#if VERBOSE
				cout << selection.size() << " objects selected, STATE_NONE" << endl;
			#endif
			Invalidate();	// ??
			state = STATE_NONE;
		} break;
	}
}


void TGadgetEditor::InvalidateFigure(TGadget* gadget)
{
	if (window) {
		TRectangle r;
		gadget->getShape(r);
//cout << "invalidating shape " << r.x << "," << r.y << ","	<< r.w << "," << r.h << endl;
		r.x-=3;
		r.y-=3;
		r.w+=6;
		r.h+=6;
		r.x+=window->OriginX();
		r.y+=window->OriginY();
		window->Invalidate(r);
	}
}

//. Find the gadget at position at (mx, my).
//. <P>
//. This method doesn't find gadgets which are currently created or edited.
TGadget* TGadgetEditor::FindGadgetAt(int mx, int my)
{
	double distance = TGadget::OUT_OF_RANGE;
	TGadgetStorage::iterator p,b,found;
	p = found = gadgets.end();
	b = gadgets.begin();
	while(p!=b) {
		p--;
		if (*p!=gadget) {
			double d = (*p)->distance(mx, my);
			if (d<distance) {
				distance = d;
				found = p;
			}
		}
	}
	if (distance > TGadget::RANGE)
		return NULL;
	return *found;
}

void TGadgetEditor::UpdateScrollbars()
{
	if (!window || !use_scrollbars)
		return;
DBM(cout << __PRETTY_FUNCTION__ << ": entry" << endl;)

	// determine area size
	//-----------------------------------------------------------------
	int x1,x2, y1,y2;
	x1=-window->OriginX();
	if (x1>0)
		x1=0;
	y1=-window->OriginY();
	if (y1>0)
		y1=0;

	x2=x1+1;
	y2=y1+1;

DBM(cout << "x-org     : " << window->OriginX() << endl;)
DBM(cout << "x-axis (1): " << x1 << " - " << x2 << endl;)
DBM(cout << "y-axis (1): " << y1 << " - " << y2 << endl;)
	
	TGadgetStorage::iterator p, e;
	p = gadgets.begin();
	e = gadgets.end();
	TRectangle r;
	while(p!=e) {
		int a;
		(*p)->getShape(r);
		if (r.x<x1)
			x1=r.x;
		a = r.x+r.w;
		if (a>x2)
			x2=a;
		if (r.y<y1)
			y1=r.y;
		a = r.y+r.h;
		if (a>y2)
			y2=a;
		p++;
	}

DBM(cout << "x-axis (2): " << x1 << " - " << x2 << endl;)
DBM(cout << "y-axis (2): " << y1 << " - " << y2 << endl;)

	// setup scrollbars
	//-----------------------------------------------------------------
	bool vs, hs;
	vs = hs = false;

	int w, h;
	w = window->Width();
	h = window->Height();

	int x22=x2, y22=y2;

	x22=-window->OriginX() + w-1;
	if (x22<x2)
		x22=x2;
	if (x22<w)
		x22=w;
		
	y22=-window->OriginY() + h-1;
	if (y22<y2)
		y22=y2;
	if (y22<h)
		y22=h;

DBM(cout << "x-axis (3): " << x1 << " - " << x22 << endl;)
DBM(cout << "y-axis (3): " << y1 << " - " << y22 << endl;)

	if (x1<0 || x22>w) {
		hs   = true;
		h   -= TScrollBar::FixedSize();
		y22 -= TScrollBar::FixedSize();
		if (y22<y2)
			y22=y2;
	}
	if (y1<0 || y22>h) {
		vs   = true;
		w   -= TScrollBar::FixedSize();
		x22 -= TScrollBar::FixedSize();
		if (x22<x2)
			x22=x2;
	}
	if (!hs && (x1<0 || x2>w)) {
		hs   = true;
		h   -= TScrollBar::FixedSize();
		y22 -= TScrollBar::FixedSize();
		if (y22<y2)
			y22=y2;
	}
	x2=x22;
	y2=y22;

DBM(cout << "x-axis (4): " << x1 << " - " << x22 << endl;)
DBM(cout << "y-axis (4): " << y1 << " - " << y22 << endl;)


	if (hs) {
		if (!hscroll) {
			hscroll = new THScrollBar(window, "hscroll");
			CONNECT(hscroll->sigValueChanged, this, actHScroll);
		}
		hscroll->SetShape(0,h,w,TScrollBar::FixedSize());
		hscroll->SetRange(x1,x2);
		hscroll->SetVisible(w);
		hscroll->SetValue(-window->OriginX());
DBM(cout << "hscroll: " << x1 << ", " << -window->OriginX() << ", " << x2 << endl;)
		if (!hscroll->IsRealized()) {
			hscroll->Create();
		} else {
			hscroll->SetMapped(true);
		}
	} else {
		if (hscroll) {
			hscroll->SetMapped(false);
		}
	}

	if (vs) {
		if (!vscroll) {
			vscroll = new TVScrollBar(window, "vscroll");
			CONNECT(vscroll->sigValueChanged, this, actVScroll);
		}
		vscroll->SetShape(w,0,TScrollBar::FixedSize(),h);
		vscroll->SetRange(y1,y2);
		vscroll->SetVisible(h);
		vscroll->SetValue(-window->OriginY());
		if (!vscroll->IsRealized()) {
			vscroll->Create();
		} else {
			vscroll->SetMapped(true);
		}
	} else {
		if (vscroll) {
			vscroll->SetMapped(false);
		}
	}
DBM(cout << __PRETTY_FUNCTION__ << ": exit" << endl << endl;)
}

void TGadgetEditor::actVScroll(int v)
{
	window->ScrollTo(window->OriginX(), -v);
	UpdateScrollbars();
}

void TGadgetEditor::actHScroll(int v)
{
DBM(cout << "actHScroll: v=" << v << " hscroll range=" << hscroll->nMin << " - " << hscroll->nMax << endl;)
	window->ScrollTo(-v, window->OriginY());
	UpdateScrollbars();
}

void TGadgetEditor::Undo()
{
	ClearSelection();
	if (history.BackSize()>0) {
		history.Current()->undo();
		history.Back();
		Invalidate();
		UpdateScrollbars();
	}
}

void TGadgetEditor::Redo()
{
	ClearSelection();
	if (history.ForwardSize()>0) {
		history.Forward();
		history.Current()->redo();
		Invalidate();
		UpdateScrollbars();
	}
}
