/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/rowcolumn.hh>
#include <toad/pulldown.hh>
#include <toad/menubar.hh>

#include <assert.h>

struct TMenuBar::TPulldownNode
{
	TPulldown *pull;
	TPulldownNode *next;
};

/*****************************************************************************
 *                                                                           *
 *                                TMenuBar                                   *
 *                                                                           *
 *****************************************************************************/

//! TMenuBar
//. This is a typical example for the usage of TMenuBar:
//. <PRE>
//. class TMainWindow:
//.   public TForm
//. {
//.    ...
//. };
//.
//. TMainWindow::TMainWindow(TWindow *parent, const string &title):
//.   TForm(parent, title)
//. {
//.   TMenuBar *mb = new TMenuBar(this, "menubar");
//.   Attach(mb, SIDE_TOP | SIDE_LEFT | SIDE_RIGHT);
//.
//.   mb->BgnPulldown("File");
//.     mb->AddItem("New")   ->sigActivate.Add(this, menuNew);
//.     mb->AddItem("Open..")->sigActivate.Add(this, menuOpen);
//.     mb->AddItem("Save")  ->sigActivate.Add(this, menuSave);
//.     mb->AddItem("Quit")  ->sigActivate.Add(this, closeRequest);
//.   mb->EndPulldown();
//. }
//. </PRE>

// constructor
//----------------------------------------------------------------------------
TMenuBar::TMenuBar(TWindow *parent, const string &title)
	:TRowColumn(parent,title)
{
	SetBackground(TColor::MENU);
	SetSize(1,TOADBase::bold_font->Height()+2+8); // useful, when TMenuBar has no children
	active_button=NULL;
	pulldown_stack=NULL;
	bNoFocus = true;
	bFocusTraversal = false;
	bHaveFocus=false;		// ???
	bCloseAll=false;
	bLBDown=false;
	// TRowColumn properties:
	nOrientation = TS_HORIZONTAL;	
}

void TMenuBar::BgnPulldown(const string &title)
{
	TPulldownNode *ptr = new TPulldownNode;
	if (!pulldown_stack) {
		ptr->pull = new TPulldown(new TMenuButton(this,title), "pulldown");
	} else {
		ptr->pull = new TPulldown(new TMenuItem(pulldown_stack->pull,title), "pulldown");
	}
	ptr->pull->bExplicitCreate = true;

	ptr->next = pulldown_stack;
	pulldown_stack = ptr;
}

// AddItem
//---------------------------------------------------------------------------
TMenuItem* TMenuBar::AddItem(const string &title)
{
	if (!pulldown_stack) {
		printf("toad: TMenuBar.AddItem; can't add an item without pulldown menu\n");
		exit(1);
	}
	return new TMenuItem(pulldown_stack->pull, title);
}

TMenuCheck* TMenuBar::AddCheck(const string &title, bool)
{
	if (!pulldown_stack) {
		printf("toad: TMenuBar.AddCheck; can't add an item without pulldown menu\n");
		exit(1);
	}
	return new TMenuCheck(pulldown_stack->pull, title);
}

TMenuRadio* TMenuBar::AddRadio(const string &title, int group, int value, bool)
{
	if (!pulldown_stack) {
		printf("toad: TMenuBar.AddRadio; can't add an item without pulldown menu\n");
		exit(1);
	}
	return new TMenuRadio(pulldown_stack->pull, title);
}

//+---------------------------------------------------------------------------+
//| EndSub                                                                    |
//+---------------------------------------------------------------------------+
void TMenuBar::EndPulldown()
{
	if (!pulldown_stack)
	{
		fprintf(stderr, "toad; warning; 'TMenuBar.EndSub()' called without pulldown\n");
		return;
	}
	TPulldownNode *ptr = pulldown_stack;
	pulldown_stack = ptr->next;
	delete ptr;
}

// SetActive
//---------------------------------------------------------------------------
//. Make <VAR>btn</VAR> the active button.<BR>
//. <UL>
//. 	<LI> When no button was active before, TMenuBar grab's the mouse and 
//.        the keyboard.
//.   <LI> When 'btn' is NULL and the grab is active it drops the grab.
//. </UL>
void TMenuBar::SetActive(TMenuButton* btn)
{
	if (!active_button && btn) GetFocus();
	else if (active_button && !btn) DropFocus();

	// deactivate previous button
	if (active_button) active_button->Deactivate();
	active_button=btn;
}

// menuSelect
//---------------------------------------------------------------------------
void TMenuBar::menuSelect()
{
	SetActive(NULL);
	if (Parent())
		sigActivate.DelayedTrigger();
}

/*
// keyDown
//---------------------------------------------------------------------------
void TMenuBar::keyDown(TKey key,char*,unsigned)
{
	switch(key)
	{
		case TK_LEFT:
			break;
		case TK_RIGHT:
			break;
		case TK_DOWN:
			break;
		case TK_UP:
			break;
		case TK_RETURN:
			break;
		default:
			printf("TMenuBar.keyDown\n");
	}
}
*/

// GetFocus
//---------------------------------------------------------------------------
void TMenuBar::GetFocus()
{
//	printf("TMenuBar.GetFocus\n");
	if (!bHaveFocus)
	{
		PushModal(this);
		bHaveFocus=true;
	}
	GrabPopupMouse();
}

// DropFocus
//---------------------------------------------------------------------------
void TMenuBar::DropFocus()
{
//	printf("TMenuBar.DropFocus\n");
	if (bHaveFocus)
	{
		bHaveFocus=false;
		bCloseAll=false;
		UngrabMouse();
		PopModal(this);
	}
}

// mouseLDown
//---------------------------------------------------------------------------
void TMenuBar::mouseLDown(int,int,unsigned)
{
//printf("TMenuBar::mouseLDown\n");
	SetActive(NULL);
}

// mouseLUp
//---------------------------------------------------------------------------
void TMenuBar::mouseLUp(int x,int y,unsigned)
{
//printf("TMenuBar::mouseLUp\n");
 	if (x<0 || x>=_w || y<0 || y>=_h)
		SetActive(NULL);
}

// closeRequest
//---------------------------------------------------------------------------
void TMenuBar::closeRequest()
{
	SetActive(NULL);
}

/*****************************************************************************
 *                                                                           *
 *                              TSimpleButton                                *
 *                                                                           *
 *****************************************************************************/

// constructor
//-------------
TSimpleButton::TSimpleButton(TWindow *parent, const string &title)
	:TControl(parent, title)
{
	bDown = false;
	SetBorder(false);
	SetBackground(TColor::MENU);
}

void TSimpleButton::adjust()
{
	// calculate button size
	SetSize( TOADBase::bold_font->TextWidth(Title())+12+8, 
					 TOADBase::bold_font->Height()+2+8);
	text_y = 1;
}

//+---------------------------------------------------------------------------+
//| paint                                                                     |
//+---------------------------------------------------------------------------+
void TSimpleButton::paint()
{
	TPen pen(this);
	pen.SetFont(TOADBase::bold_font);	// hack!!!
	if (bDown) {
		SetBackground(TColor::MENUTEXT);
		pen.SetColor(TColor::MENU);
	}	else {
		SetBackground(TColor::MENU);
		pen.SetColor(TColor::MENUTEXT);
	}
	ClearWindow();
	pen.ClrClipBox();
	pen.DrawString(6+4, text_y+4, Title());
}

//+---------------------------------------------------------------------------+
//| menubar                                                                   |
//+---------------------------------------------------------------------------+
TMenuBar* TSimpleButton::menubar()
{
	TWindow *ptr;
	TMenuBar *mb;
	
	ptr=Parent();
	while(ptr)
	{
		if ( (mb=dynamic_cast<TMenuBar*>(ptr)) != NULL )
			return mb;
		ptr = ptr->Parent();
	}
	fprintf(stderr, "toad: internal error; couldn't find TMenuBar parent\n");
	exit(1);
	return NULL;	// for the compiler
}

/*****************************************************************************
 *                                                                           *
 *                               TMenuButton                                 *
 *                                                                           *
 *****************************************************************************/
//+---------------------------------------------------------------------------+
//| wmLButtonDown                                                             |
//+---------------------------------------------------------------------------+
void TMenuButton::mouseLDown(int,int,unsigned)
{
//printf("TMenuButton::mouseLDown: \"%s\"\n",Title().c_str());
	if (!bDown)	{
		// mark button
		menubar()->SetActive(this);
		bDown=true;
		Invalidate();
		// open pulldown window
		if (FirstChild())	{
			int x,y;
			GetRootPos(&x,&y);
			static_cast<TPulldown*>(FirstChild())->Open(x,y+_h);
		}
	}	else {
		// next TMenuButton.mouseLUp will close whole menubar
		menubar()->bCloseAll = true;
		// unmark pulldown item
		if (FirstChild())
			((TPulldown*)FirstChild())->SetActive(NULL);
	}
}

// mouseLUp
//---------------------------------------------------------------------------
void TMenuButton::mouseLUp(int,int,unsigned)
{
	if (!menubar()->bCloseAll)
	{
		if (FirstChild())
			((TPulldown*)FirstChild())->SetFirstActive();
	}
	else
	{
		menubar()->SetActive(NULL);
	}
}

// mouseEnter
//---------------------------------------------------------------------------
void TMenuButton::mouseEnter(int,int,unsigned modifier)
{
//printf("TMenuButton::mouseEnter: \"%s\"\n",Title().c_str());
	if ( !bDown && (modifier&MK_LBUTTON) && (menubar()->IsActive()) )
	{
		// unmark and close other TMenuButton
		menubar()->SetActive(this);
		// mark button
		bDown=true;
		UpdateWindow();
		// open pulldown
		if (FirstChild())
		{
			int x,y;
			GetRootPos(&x,&y);
			((TPulldown*)FirstChild())->Open(x,y+_h);
		}
	}
}

//+---------------------------------------------------------------------------+
//| Deactivate                                                                |
//+---------------------------------------------------------------------------+
void TMenuButton::Deactivate()
{
	// close pulldown
	if (FirstChild())
		((TPulldown*)FirstChild())->Close();
	// unmark button
	bDown=false;
	UpdateWindow();
}

/*****************************************************************************
 *                                                                           *
 *                                TMenuItem                                  *
 *                                                                           *
 *****************************************************************************/

// adjust
//---------------------------------------------------------------------------
void TMenuItem::adjust()
{
	// calculate item size
	SetSize( TOADBase::bold_font->TextWidth(Title())+30, 
					 TOADBase::bold_font->Height()+2);
	text_y = 1;
}

// paint
//---------------------------------------------------------------------------
void TMenuItem::paint()
{
	TPen pen(this);
	if (!Enabled() || ( 
			!sigActivate.IsConnected() && 
			!FirstChild()) )
	{
		SetBackground(TColor::MENU);
		ClearWindow();
    pen.SetColor(TColor::BTNLIGHT);
    pen.DrawString(15+1,text_y+1,Title());
    pen.SetColor(TColor::BTNSHADOW);
		pen.DrawString(15,text_y,Title());
		return;
	}

	if (bDown) {
		SetBackground(TColor::MENUTEXT);
		pen.SetColor(TColor::MENU);
	} else {
		SetBackground(TColor::MENU);
		pen.SetColor(TColor::MENUTEXT);
	}
	ClearWindow();
	pen.DrawString(15, text_y, Title());
	if(FirstChild()) {
		int y=_h>>1;
		TPoint tri[3];
		tri[0].x=_w-7; tri[0].y=y-4;
		tri[1].x=_w-3; tri[1].y=y;
		tri[2].x=_w-7; tri[2].y=y+4;
		pen.FillPolygon(tri, 3);
	}
}

// wmLButtonDown
//---------------------------------------------------------------------------
void TMenuItem::mouseLDown(int,int,unsigned)
{
	if (!Enabled())
		return;

	menubar()->GetFocus();
	// calculate pulldown position
	int x = _w + 1;
	int y = 1;

		x+=_x-7;
		y+=_y;

		x+=Parent()->XPos();
		y+=Parent()->YPos();

	TPulldown *pd = (TPulldown*)FirstChild();
	if (bDown) {
		// menubar()->bCloseAll = true;
		if (pd)
			pd->SetActive(NULL);
	} else {
		pulldown()->SetActive(this);
		bDown=true;
		UpdateWindow();
		if (pd) {
			printf("TPulldownButton.mouseLDown: open pulldown at 0,0\n");
			pd->Open(x,y);
		}
	}
}

// mouseLUp
//---------------------------------------------------------------------------
void TMenuItem::mouseLUp(int,int,unsigned)
{
	if (!Enabled())
		return;

	if (!FirstChild()) {
//		printf("TPulldownButton.mouseLUp: begin\n");
			if (!sigActivate.DelayedTrigger())
				menubar()->menuSelect();
			else
				menubar()->SetActive(NULL);
//		printf("TPulldownButton.mouseLUp: end\n");
	}
}

// mouseLeave
//---------------------------------------------------------------------------
void TMenuItem::mouseLeave(int x,int y,unsigned modifier)
{
	if (!Enabled())
		return;

	if ((modifier & MK_LBUTTON) && (x>=_w || x<0 || y<0 || y>=_h)) {
		((TPulldown*)Parent())->SetActive(NULL);
	}
}

// mouseEnter
//---------------------------------------------------------------------------
void TMenuItem::mouseEnter(int,int,unsigned modifier)
{
	if (!Enabled())
		return;

	if (modifier & MK_LBUTTON) {
		((TPulldown*)Parent())->SetActive(this);
	}
}

// Activate
//---------------------------------------------------------------------------
void TMenuItem::Activate()
{
	bDown=true;
	Invalidate();

	// calculate pulldown position
	int x = _w + 1 - 8;
	int y = 1;

		x+=_x;
		y+=_y;

		x+=Parent()->XPos();
		y+=Parent()->YPos();

	if (FirstChild())	{
		((TPulldown*)FirstChild())->Open(x,y);
	}
}

// Deactivate
//---------------------------------------------------------------------------
void TMenuItem::Deactivate()
{
	if (FirstChild())
	{
		((TPulldown*)FirstChild())->Close();
	}
	bDown=false;
	Invalidate();
}

TPulldown* TMenuItem::pulldown()
{
	TWindow *ptr;
	TPulldown *mb;
	
	ptr=Parent();
	while(ptr)
	{
		if ( (mb=dynamic_cast<TPulldown*>(ptr)) != NULL )
			return mb;
		ptr = ptr->Parent();
	}
	fprintf(stderr, "toad: internal error; couldn't find TPulldown parent\n");
	exit(1);
	return 0;
}

/*****************************************************************************
 *                                                                           *
 *                                TMenuCheck                                 *
 *                                                                           *
 *****************************************************************************/
TMenuCheck::TMenuCheck(TWindow* parent, const string &title)
	:super(parent, title)
{
	bChecked = false;
}

void TMenuCheck::paint()
{
	TPen pen(this);

	if (bDown) {
		SetBackground(TColor::MENUTEXT);
		pen.SetColor(TColor::MENU);
	}	else {
		SetBackground(TColor::MENU);
		pen.SetColor(TColor::MENUTEXT);
	}
	ClearWindow();
/*
	SetBackground(TColor::MENU);
	pen.SetColor(TColor::MENUTEXT);
*/	
	pen.DrawString(15, text_y, Title());
	
	if (bChecked)	{
		pen.SetLineWidth(2);
		pen.DrawLine(  2,text_y+2+5,  2+3+1,text_y+2+9+1);
		pen.DrawLine(2+3+1,text_y+2+9-1,  2+9,text_y+2-2);
	}
}

// mouseLUp
//---------------------------------------------------------------------------
void TMenuCheck::mouseLUp(int,int,unsigned)
{
	bChecked = !bChecked;
	Invalidate();
	menubar()->SetActive(NULL);
}
