/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,   
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#define NO_FLICKER

#include <unistd.h>

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/region.hh>
#include <toad/pushbutton.hh>
#include <toad/scrollbar.hh>
#include <toad/simpletimer.hh>

enum EArrowType {ARROW_DOWN=1, ARROW_UP, ARROW_RIGHT, ARROW_LEFT};

class TNArrowButton: public TPushButton, TSimpleTimer
{
	private:
		int delay;
		EArrowType direction;

	public:
		TNArrowButton(TWindow *parent, EArrowType d)
			:TPushButton(parent,"scrollbar.arrowbutton"){direction=d;bNoFocus=true;}
			
	protected:
		void mouseLDown(int,int,unsigned);
		void mouseLUp(int,int,unsigned);
		void tick();
		void paint();
};

/*****************************************************************************
 *																																					 *
 * Scrollbar                                                                 *
 *																																					 *
 *****************************************************************************/

#define DEFAULT_FIXED_SIZE 15
#define DEFAULT_FIXED_BORDER 1

// Constructor
//---------------------------------------------------------------------------
TScrollBar::TScrollBar(TWindow *parent, const string &title,bool v)
	:TControl(parent,title)
{
	_b = DEFAULT_FIXED_BORDER;
	bNoBackground = true;
	bVertical 	= v;
	nMin 				= 0;
	nMax 				= 10;
	nVisible 		= 1;
	_data 			= 0;
	nMouseDown	= -1;
	_w=_h=DEFAULT_FIXED_SIZE;

	SetMouseMoveMessages(TMMM_LBUTTON);
	btn1	= new TNArrowButton(this, bVertical ? ARROW_UP : ARROW_LEFT);
	CONNECT(btn1->sigActivate, this, button1);
	btn2 	= new TNArrowButton(this, bVertical ? ARROW_DOWN : ARROW_RIGHT);
	CONNECT(btn2->sigActivate, this, button2);
}

int TScrollBar::FixedSize()
{
	return DEFAULT_FIXED_SIZE+DEFAULT_FIXED_BORDER*2;
}

void TScrollBar::resize()
{
	_PlaceChildren();
	_PlaceSlider();
}

void TScrollBar::_DrawSlider(TPen &pen, TRectangle &r)
{
	TPoint p[6];

	// draw slider face
	//------------------
	pen.SetColor(TColor::SLIDER_FACE);
	pen.FillRectangle(r.x+3, r.y+3, r.w-6, r.h-6);

	// draw border
	//-------------
	pen.SetColor(0,0,0);
	pen.DrawRectangle(r.x, r.y, r.w, r.h);
	
	// draw shadow(s)
	//----------------
	pen.SetColor(TColor::SLIDER_SHADOW);
	int c;
	if (bVertical)
	{
		c = r.y+(r.h>>1)-2;
		pen.FillRectangle(r.x+5, c+1, r.w-8, 5);
	}	else {
		c = r.x+(r.w>>1)-2;
		pen.FillRectangle(c+1, r.y+5, 5, r.h-8);
	}

	p[0].Set(r.x+1		, r.y+r.h-2);
	p[1].Set(r.x+r.w-2, r.y+r.h-2);
	p[2].Set(r.x+r.w-2, r.y+1		 );
	p[3].Set(r.x+r.w-3, r.y+2    );
	p[4].Set(r.x+r.w-3, r.y+r.h-3);
	p[5].Set(r.x+2		, r.y+r.h-3);
	pen.DrawLines(p,6);
	
	// draw light
	//------------
	pen.SetColor(TColor::SLIDER_LIGHT);
	if (bVertical) {
		pen.DrawLine(r.x+4,c  , r.x+5+r.w-10,c  );
		pen.DrawLine(r.x+4,c+2, r.x+5+r.w-10,c+2);
		pen.DrawLine(r.x+4,c+4, r.x+5+r.w-10,c+4);
		pen.DrawLine(r.x+4,c  , r.x+4       ,c+5);
	}	else {
		pen.DrawLine(c  ,r.y+4, c  ,r.y+5+r.h-10);
		pen.DrawLine(c+2,r.y+4, c+2,r.y+5+r.h-10);
		pen.DrawLine(c+4,r.y+4, c+4,r.y+5+r.h-10);
		pen.DrawLine(c  ,r.y+4, c+5,r.y+4);
	}
	
	p[0].Set(r.x+1    , r.y+r.h-2);
	p[1].Set(r.x+1		, r.y+1		 );
	p[2].Set(r.x+r.w-2, r.y+1		 );
	p[3].Set(r.x+r.w-3, r.y+2    );
	p[4].Set(r.x+2    , r.y+2    );
	p[5].Set(r.x+2    , r.y+r.h-3);
	pen.DrawLines(p,6);
	
	// draw sliders shadow
	//---------------------
	pen.SetColor(TColor::BTNSHADOW);
	if (bVertical) {
		pen.DrawLine(r.x, r.y+r.h,     r.x+r.w-1, r.y+r.h);
		pen.DrawLine(r.x+2, r.y+r.h+1, r.x+r.w-1, r.y+r.h+1);
	}	else {
		pen.DrawLine(r.x+r.w, r.y,     r.x+r.w, r.y+r.h-1);
		pen.DrawLine(r.x+r.w+1, r.y+2, r.x+r.w+1, r.y+r.h-1);
	}
}

// this method draws the area where the slider is moving; to reduce
// flicker and double buffering, it excludes the region of the slider
void TScrollBar::_DrawArea(TPen &pen)
{
	TPoint p[3];
	int v, n, m;

	if (IsFocus()) {
		pen.SetColor(0,0,0);
		pen.DrawRectangle(0,0, _w, _h);
		v=1;
	}	else {
		v=0;
	}
	
	if (bVertical) {
		// background
		//------------
		pen.SetColor(TColor::BTNFACE);
		n = rectSlider.y-_w-2;
		if (n>0)
			pen.FillRectangle(v+1,_w+2, _w-2-(v<<1), n);
		n += rectSlider.h + _w + 3;
		m = _h-_w-n-2;
		if (m>0)
			pen.FillRectangle(v+1, n, _w-2-(v<<1), m);
	
		// shadow
		//------------
		pen.SetColor(TColor::BTNSHADOW);
		if (_w+1<=rectSlider.y-1) {
			p[0].Set(_w-2-v, _w+1);
			p[1].Set(			v, _w+1);
			p[2].Set(			v, rectSlider.y-1);
			pen.DrawLines(p,3);
		}
		pen.DrawLine(v, rectSlider.y + rectSlider.h + 1,
								 v, _h-_w-2 );

		// light
		//------------
		pen.SetColor(TColor::BTNLIGHT);
		pen.DrawLine(_w-1-v,0, _w-1-v,rectSlider.y-1);
		p[0].Set(_w-1-v,rectSlider.y+rectSlider.h+1);		// slider bottom, right
		p[1].Set(_w-1-v,_h-_w-2);												// bottom,right
		p[2].Set(1+v,_h-_w-2);													// bottom,left
		pen.DrawLines(p,3);
	}	else {
		// background
		//------------
		pen.SetColor(TColor::BTNFACE);
		n = rectSlider.x-_h-2;
		if (n>0)
			pen.FillRectangle(_h+2, v+1, n, _h-(v<<1)-2);
		n = rectSlider.x+rectSlider.w+1;
		m = _w - n - _h - 2;
		pen.FillRectangle(n, v+1, m, _h-(v<<1)-2);
	
		// shadow
		//------------
		pen.SetColor(TColor::BTNSHADOW);
		p[0].Set(_h+1,_h-1-v);
		p[1].Set(_h+1,v);
		p[2].Set(rectSlider.x-1,v);
		pen.DrawLines(p,3);
		pen.DrawLine(rectSlider.x+rectSlider.w+1, v,
								 _w-_h-2, v);

		// light
		//------------
		pen.SetColor(TColor::BTNLIGHT);
		pen.DrawLine(_h+2, _h-1-v,
								 rectSlider.x-1, _h-1-v);
		p[0].Set(rectSlider.x+rectSlider.w+2,_h-1-v);
		p[1].Set(_w-_h-2, _h-1-v);
		p[2].Set(_w-_h-2,v+1);
		pen.DrawLines(p,3);
	}
}

void TScrollBar::focus()
{
	_PlaceChildren();
	_PlaceSlider();
	Invalidate();
}

void TScrollBar::button1()
{
	SetFocus();
	if (_data>nMin)	{
		_data--;
		_PlaceSlider();
		sigDecrement();
		UpdateWindow(true);
		ValueChanged();
	}
}

void TScrollBar::button2()
{
	SetFocus();
	if (_data<(nMax-nVisible+1)) {
		_data++;
		_PlaceSlider();
		sigIncrement();
		UpdateWindow(true);
		ValueChanged();
	}
}

void TScrollBar::SetVisible(int n)
{
	nVisible=n;
	if (IsRealized())	{
		Invalidate();
		_PlaceSlider();
	}
}

void TScrollBar::valueChanged()
{
	if (IsRealized())	{
		Invalidate();
		_PlaceSlider();
	}
}

void TScrollBar::SetRange(int min,int max)
{
	nMin = min;
	nMax = max;
  if (IsRealized()) {
  	Invalidate();
		_PlaceSlider();
	}
}

void TScrollBar::keyDown(TKey key,char*,unsigned)
{
	switch(key)	{
		case TK_UP:
			if (bVertical)
				button1();
			break;
		case TK_DOWN:
			if (bVertical)
				button2();
			break;
		case TK_LEFT:
			if (!bVertical)
				button1();
			break;
		case TK_RIGHT:
			if (!bVertical)
				button2();
			break;
		case TK_PAGEUP:
			break;
		case TK_PAGEDOWN:
			break;
	}
}

void TScrollBar::mouseLDown(int x,int y,unsigned)
{
	SetFocus();
	int v = _data;
	if (bVertical ? y<rectSlider.y : x<rectSlider.x ) {
		// page up
		//----------
		v-=nVisible-1;
		if (v<nMin)
			v=nMin;
		if (v!=_data)	{
			_data = v;
			_PlaceSlider();
			UpdateWindow(true);
			sigPageDecrement();
			ValueChanged();
		}
	}
	else if (bVertical ? y>rectSlider.y+rectSlider.h : x>rectSlider.x+rectSlider.w)
	{
		// page down
		//-----------
		v+=nVisible-1;
		if ( v> nMax-nVisible+1 )
			v=nMax-nVisible+1;
    if (v!=_data)
    {
      _data = v;
	    _PlaceSlider();
	    UpdateWindow(true);
	    sigPageIncrement();
	    ValueChanged();
		}
	}	else {
		// move slider
		//-------------
		TPen pen(this);
		_DrawSlider(pen, rectSlider);
		nMouseDown = bVertical ? y-rectSlider.y : x-rectSlider.x;
	}
}

void TScrollBar::mouseMove(int x,int y,unsigned)
{
	if (nMouseDown!=-1)	{
		TRectangle rectOld;
		rectOld = rectSlider;
		bool b = bVertical ? _MoveSliderTo(y - nMouseDown)
											 : _MoveSliderTo(x - nMouseDown);
		
		if (bVertical ? rectOld.y!=rectSlider.y : rectOld.x!=rectSlider.x) {
#if 1
			Invalidate();
			PaintNow();
#else
			TPen pen(this);
			
			TRegion rgn;
			TRectangle w;
			w.Set(0,0,_w,_h);
			rgn|=rectSlider;
			rgn^=w;
			_DrawSlider(pen,rectSlider);
			pen.SetClipRegion(&rgn);
			pen.SetColor(TColor::BTNFACE);
			pen.FillRectangle(rectOld.x-1,rectOld.y-1,rectOld.w+4,rectOld.h+4);
			paint();
			
			Flush();
#endif			
		}
		if (b) {
			ValueChanged();
			sigDrag();
		}
	}
}

void TScrollBar::mouseLUp(int,int,unsigned)
{
	if (nMouseDown!=-1)	{
		nMouseDown = -1;
		ValueChanged();
	}
}

void TScrollBar::paint()
{
	TPen pen(this);
	_DrawArea(pen);
	_DrawSlider(pen, rectSlider);
}

void TScrollBar::_PlaceSlider()
{
	if (nVisible<0)
		return;
			
	// calculate range of possible slider positions 
	//----------------------------------------------
	int nRange  = nMax - nMin - nVisible + 1;

	if (nRange<0)
		nRange = 0;

	// calculate size of the area in which the slider is
	//---------------------------------------------------
	TRectangle rect1,rect2;
	btn1->GetShape(&rect1);
	btn2->GetShape(&rect2);
	int nSize = bVertical ? _h - rect1.h - rect1.y - _h+ rect2.y + 2
												: _w  - rect1.w - rect1.x - _w + rect2.x + 2;
		
	// calculate slider size 
	//-----------------------
	int nSlider = (nVisible * nSize) / (nVisible + nRange);

	// limit slider size
	if (bVertical) {
		if (nSlider<_w+2)
			nSlider=_w+2;
		else if (nSlider>nSize)
			nSlider=nSize;
	}	else {
		if (nSlider<_h+2)
			nSlider=_h+2;
		else if (nSlider>nSize)
			nSlider=nSize;
	}	

	// calculate slider position
	//---------------------------
	int nSetRange = nSize-nSlider+1;
	int pos;
	if (nRange==0)
		pos = 0;
	else
		pos = (_data-nMin) * nSetRange / nRange;

	pos += bVertical ? rect1.h+rect1.y-1 : rect1.w+rect1.x-1;

	// correct slider position 
	//-------------------------
	if (bVertical) {
		if ( pos+nSlider-1 > rect2.y ) 
			pos = rect2.y - nSlider + 1;
		rectSlider.Set(IsFocus()?1:0,pos, _w+(IsFocus()?-2:0), nSlider);
	}	else {
		if ( pos+nSlider-1 > rect2.x ) 
			pos = rect2.x - nSlider + 1;
		rectSlider.Set(pos, IsFocus()?1:0, nSlider, _h+(IsFocus()?-2:0));
	}
}

bool TScrollBar::_MoveSliderTo(int pos)
{
	TRectangle rect1,rect2;
	btn1->GetShape(&rect1);
	btn2->GetShape(&rect2);

	// validate new position and place slider 
	//----------------------------------------
	if (bVertical) {
		if (pos < rect1.y+rect1.h-1 )
			pos = rect1.y+rect1.h-1;
		else if ( pos+rectSlider.h-1 > rect2.y )
			pos = rect2.y - rectSlider.h + 1;
		rectSlider.x = IsFocus()?1:0;
		rectSlider.y = pos;
	}	else {
		if (pos < rect1.x+rect1.w-1 )
			pos = rect1.x+rect1.w-1;
		else if ( pos+rectSlider.w-1 > rect2.x )
			pos = rect2.x - rectSlider.w + 1;
		rectSlider.x = pos;
		rectSlider.y = IsFocus()?1:0;
	}

	// change _data 
	//---------------
	// slider size
	int nSlider = bVertical ? rectSlider.h : rectSlider.w;
	// size of slider area
  int nSize = bVertical ? _h - rect1.h - rect1.y - _h+ rect2.y + 2
                        : _w  - rect1.w - rect1.x - _w + rect2.x + 2;
	// size of area to place the slider
	int nSetRange = nSize-nSlider;
	
	// avoid division by zero
	//------------------------
	if (nSetRange==0 || nSize==0)	{
		_data = nMin;
		return true;
	}
	
	int nValue2;
	int nRange = nMax - nMin - nVisible + 1;
	
	if (nRange<0)
		nRange=0;
	
	pos -= bVertical ? rect1.y+rect1.h-1 : rect1.x+rect1.w-1;
	nValue2 = ( pos * nRange ) / nSetRange + nMin;

	if (nValue2<nMin) nValue2 = nMin;
	if (nValue2>nMax-nVisible+1) nValue2 = nMax - nVisible + 1;

	if (_data!=nValue2) {
		_data=nValue2;
		return true;
	}
	return false;
}

void TScrollBar::_PlaceChildren()
{
	if (bVertical) {
		if (IsFocus()) {
			btn1->SetShape(0,0,_w,_w+1);
			btn2->SetShape(0,_h-_w-1,_w,_w+1);
		}	else {
			btn1->SetShape(-1,-1,_w+2,_w+2);
			btn2->SetShape(-1,_h-_w-1,_w+2,_w+2);
		}
	}	else {	
		if (IsFocus()) {
			btn1->SetShape(0,0,_h+1,_h);
			btn2->SetShape(_w-_h-1,0,_h+1,_h);
		}	else {
			btn1->SetShape(-1,-1,_h+2,_h+2);
			btn2->SetShape(_w-_h-1,-1,_h+2,_h+2);
		}
	}
}

/*****************************************************************************
 *																																					 *
 * TNArrowButton	                                                           *
 *																																					 *
 *****************************************************************************/
void TNArrowButton::paint()
{
	TPen pen(this);

	DrawShadow(pen, bDown && bInside);

	int n=bDown && bInside ? 1:0;

	const int d=4;
	TPoint p[3];
	
	switch(direction)
	{
		case ARROW_DOWN:
			p[0].Set(n+(_w>>1), n+_h-d-2);
			p[1].Set(n+_w-d   , n+d-1+2);
			p[2].Set(n+d-1 				, n+d-1+2);
			break;
		case ARROW_UP:
			p[2].Set(n+_w-d   , n+_h-d-2);
			p[1].Set(n+d-1 				, n+_h-d-2);
			p[0].Set(n+(_w>>1), n+d-1+2);
			break;
		case ARROW_LEFT:
			p[0].Set(n+d-1+2			, n+(_h)>>1);
			p[1].Set(n+_w-d-2	, n+d-1);
			p[2].Set(n+_w-d-2	, n+_h-d);
			break;
		case ARROW_RIGHT:
			p[0].Set(n+d-1+2			, n+d-1);
			p[1].Set(n+_w-d-2	, n+(_h)>>1);
			p[2].Set(n+d-1+2 			, n+_h-d);
			break;
	}
	pen.SetColor(TColor::BTNTEXT);
	pen.FillPolygon(p,3);
}

void TNArrowButton::mouseLDown(int,int,unsigned)
{
	bDown=true;
	UpdateWindow(true);
	sigArm();
	sigActivate();
	delay = 0;
	StartTimer(0, 1000000/12);
}

void TNArrowButton::mouseLUp(int,int,unsigned)
{
	bDown=false;
	UpdateWindow(true);
	sigDisarm();
	StopTimer();
}

void TNArrowButton::tick()
{
	if(delay<3)
		delay++;
	else if (bInside)
		sigActivate();
}
