/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

//! TWindow
//. TWindow is the super class of all windows. A window is a rectangle
//. area on the screen which receives input events, eg. from the 
//. mouse pointer, the keyboard, displays graphics and can contain
//. other windows.

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xmd.h>

#define _TOAD_PRIVATE

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/region.hh>
#include <toad/dragndrop.hh>
#include <toad/dialogeditor.hh>

#include <vector>
#include <stack>
#include <map>

// obsolete:
struct TWndPtrComp
{
	bool operator()(const TWindow *a, const TWindow *b) const
	{
		return a<b;
	}
};

#ifdef __TOAD_THREADS
	#undef __TOAD_THREADS
#endif

#ifdef __TOAD_THREADS
	static TThreadMutex mutexPaintQueue;
	#define THREAD_LOCK(A) A.Lock()
	#define THREAD_UNLOCK(A) A.Unlock();
#else
	#define THREAD_LOCK(A)
	#define THREAD_UNLOCK(A)
#endif



// window icons
//---------------------------------------------------------------------------
// Since bitmaps are barely used, we don't store them in TWindow but in two
// external maps. The intention is to keep TWindow small.
typedef map<TWindow*,TBitmap*,TWndPtrComp> TWndBmpList;
static TWndBmpList iconlist;
static TWndBmpList backgroundlist;

// some extra functions
//---------------------------------------------------------------------------
inline bool IsTopLevel(TWindow *w)
{
	if (w->bPopup || !w->parent)
		return true;
	return false;
}

// LessTif Window Manager Hints
//---------------------------------------------------------------------------
#define PROP_MOTIF_WM_HINTS_ELEMENTS 5

#define MWM_HINTS_FUNCTIONS     (1L << 0)
#define MWM_HINTS_DECORATIONS   (1L << 1)
#define MWM_HINTS_INPUT_MODE    (1L << 2)
#define MWM_HINTS_STATUS        (1L << 3)
                    
#define MWM_FUNC_ALL            (1L << 0)
#define MWM_FUNC_RESIZE         (1L << 1)
#define MWM_FUNC_MOVE           (1L << 2)
#define MWM_FUNC_MINIMIZE       (1L << 3)
#define MWM_FUNC_MAXIMIZE       (1L << 4)
#define MWM_FUNC_CLOSE          (1L << 5)
                    
#define MWM_DECOR_ALL           (1L << 0)
#define MWM_DECOR_BORDER        (1L << 1)
#define MWM_DECOR_RESIZEH       (1L << 2)
#define MWM_DECOR_TITLE         (1L << 3)
#define MWM_DECOR_MENU          (1L << 4)
#define MWM_DECOR_MINIMIZE      (1L << 5)
#define MWM_DECOR_MAXIMIZE      (1L << 6)
                    
#define MWM_INPUT_MODELESS 									0
#define MWM_INPUT_PRIMARY_APPLICATION_MODAL 1
#define MWM_INPUT_SYSTEM_MODAL 							2
#define MWM_INPUT_FULL_APPLICATION_MODAL 		3
#define MWM_INPUT_APPLICATION_MODAL MWM_INPUT_PRIMARY_APPLICATION_MODAL
                    
#define MWM_TEAROFF_WINDOW      (1L<<0)

struct PropMotifWmHints	{
	CARD32 flags;
	CARD32 functions;
	CARD32 decorations;
	INT32 inputMode;
	CARD32 status;
};

// virtual method auto selection
//---------------------------------------------------------------------------
//  X11 requires the selection of events. The VMAS checks if a virtual method
// is still part of TWindow or overridden in a derived class. A table with
// pointers is created once by the TWindow constructor and used by
// `TWindow::build_eventmask()'.
//  The table became necessary with g++ 2.8.x due to changes in the behaviour
// of the `&' operator on virtual methods. [MAH]

enum {
	VMAS_PAINT,
	VMAS_MOUSELDOWN,
	VMAS_MOUSEMDOWN,
	VMAS_MOUSERDOWN,
	VMAS_MOUSELUP,
	VMAS_MOUSEMUP,
	VMAS_MOUSERUP,
	VMAS_MOUSEMOVE,
	VMAS_MOUSEENTER,
	VMAS_MOUSELEAVE,
	VMAS_KEYDOWN,
	VMAS_KEYUP,
	VMAS_MAX
};

#define _vmas_cast(f) ((void*)f)

#define SET_VMAS(I, F) \
	_vmas_table[I] = &TWindow::F
#define CHK_VMAS(I, F) \
	(_vmas_table[I] != _vmas_cast(&TWindow::F))
static const void* _vmas_table[VMAS_MAX] = { NULL,  };

// Constructor                                                               
//---------------------------------------------------------------------------
TWindow::TWindow(TWindow *p, const string &new_title)
{
	// set up vmas table
	if (!*_vmas_table) {
		SET_VMAS(VMAS_PAINT, paint);
		SET_VMAS(VMAS_MOUSELDOWN, mouseLDown);
		SET_VMAS(VMAS_MOUSEMDOWN, mouseMDown);
		SET_VMAS(VMAS_MOUSERDOWN, mouseRDown);
		SET_VMAS(VMAS_MOUSELUP, mouseLUp);
		SET_VMAS(VMAS_MOUSEMUP, mouseMUp);
		SET_VMAS(VMAS_MOUSERUP, mouseRUp);
		SET_VMAS(VMAS_MOUSEMOVE, mouseMove);
		SET_VMAS(VMAS_MOUSEENTER, mouseEnter);
		SET_VMAS(VMAS_MOUSELEAVE, mouseLeave);
		SET_VMAS(VMAS_KEYDOWN, keyDown);
		SET_VMAS(VMAS_KEYUP, keyUp);
	}

//printf("TWindow: constructing %lx\n",(long)this);

	x11window	= 0;
	parent	= NULL;
	taborder = 0;
	
	// public flags
	bShell = bPopup = bExplicitCreate = bSaveUnder = bStaticFrame =
	bBackingStore = bNoBackground = bX11GC = bFocusManager = bNoFocus = 
	bTabKey = bDialogEditRequest = bDoubleBuffer = false;
	
	bFocusTraversal = true;

	// private flags
	bEraseBe4Paint = false;
	_bOwnsFocus = false;
	_bResizedBeforeCreate = false;
	_bToolTipAvailable = false;
	_visible = true;

	_x		=	0;
	_y		=	0;
	_w	=	320;
	_h	=	200;
	_b	=	1;
	_dx = _dy	= 0;	// origin for TPen
	_cursor = TCursor::DEFAULT;
	paint_rgn = NULL;
	background.Set(255,255,255);
	iflags	=	0;
#ifdef DEBUG
	debug_flags = 0;
#endif
	child = prev_sibling = next_sibling = NULL;

	if (&new_title == NULL)
		cerr << "toad: warning, window title reference is NULL\n"; 
	else
		SetTitle(new_title);

	if (p)
		p->AddChild(this);
}

TWindow::~TWindow()
{
	SetToolTip("");

	// remove window from paint queue
	//--------------------------------
	THREAD_LOCK(mutexPaintQueue);
	if (paint_rgn)
		paint_rgn->wnd = NULL;
	THREAD_UNLOCK(mutexPaintQueue);

	if (x11window)
		Destroy();

	// delete all children
	//--------------------------------
	while(child)
		delete child;

	// remove window from parent
	//--------------------------------
	if (parent)	{
		_ChildNotify(TCHILD_REMOVE);
		parent->RemoveChild(this);
	}

	// remove messages for this window
	//--------------------------------
	RemoveMessage(this);
	if (x11window) {
		// WARNING !
		// The XSaveContext(...,0) is a source for errors. When a new window
		// is being created with 'window' pointing to _this_ window, it'll
		// get the messages which XSaveContext(...,0) should suppress.
		XSaveContext(x11display, x11window, nClassContext, (XPointer)0);
		XDestroyWindow(x11display, x11window);
		x11window = 0;
	}

	// free bitmaps (icon & background)
	//--------------------------------
	TWndBmpList::iterator p = iconlist.find(this);
	if (p!=iconlist.end()) {
		if ((*p).second)
			delete (*p).second;
		iconlist.erase(p);
	}
	
	p = backgroundlist.find(this);
	if (p!=backgroundlist.end()) {
		if ((*p).second)
			delete (*p).second;
		backgroundlist.erase(p);
	}
}

void TWindow::CreateModal()
{
	Create();
}

// Create
//---------------------------------------------------------------------------
//. Create the X11 window.
void TWindow::Create()
{
	if (x11window || is_iflag(IFLAG_CREATE))
		return;
	set_iflag(IFLAG_CREATE);

	if (parent && !parent->x11window ) {
		parent->Create();
	} else if (!x11window) {
		set_iflag(IFLAG_EXPLICIT_CREATE);
		_init();			// call create top-down
		_adjustW2C();	// call adjust bottom-up
		_create();		// now create windows top-down
		clr_iflag(IFLAG_EXPLICIT_CREATE);
	}
	clr_iflag(IFLAG_CREATE);
}

void TWindow::_init()
{
	// only init windows with bExplicitCreate style, when Create() has
	// been called for this window directly
	//-----------------------------------------------------------------
	if ( bExplicitCreate && !is_iflag(IFLAG_EXPLICIT_CREATE) )
		return;

	// mark children that have been added _before_ init(); these children
	// won't be deleted when calling 'Destroy()'
	//--------------------------------------------------------------------
	TWindow *ptr=FirstChild();
	while(ptr)
	{
		ptr->set_iflag(IFLAG_BEFORE_CREATE);
		ptr=NextSibling(ptr);
	}

	// call 'create()'; it's a common error to call methods now that need
	// a valid 'window', DFLAG_WMINIT will tell that this error appeared
	// during 'init()'
	//--------------------------------------------------------------------
	#ifdef DEBUG
		debug_flags|=DFLAG_WMINIT;
		create();
		debug_flags&=~DFLAG_WMINIT;
	#else
		create();
	#endif

	// init all children
	//-------------------
	ptr=FirstChild();
	while(ptr)
	{
		ptr->_init();
		ptr=NextSibling(ptr);
	}
}

void TWindow::_adjustW2C()
{
	if (bExplicitCreate && !is_iflag(IFLAG_EXPLICIT_CREATE))
		return;

	TWindow *ptr = FirstChild();
	while(ptr)
	{
		ptr->_adjustW2C();
		ptr=NextSibling(ptr);
	}
	#ifdef DEBUG
		debug_flags|=DFLAG_WMADJUST;
		adjust();
		debug_flags&=~DFLAG_WMADJUST;
	#else
		adjust();
	#endif
}

void TWindow::_create()
{
	#ifdef SECURE
	if (x11window) {
		cerr << "toad: internal error; mustn't create an existing window";
		return;
	}
	#endif

	if (bExplicitCreate && !is_iflag(IFLAG_EXPLICIT_CREATE))
		return;

	_ChildNotify(TCHILD_BEFORE_CREATE);

	// set bShell flag for all top level windows
	//-------------------------------------------
	if (IsTopLevel(this))	// parent == NULL || bPopup
		bShell=true;

	// each window handling keyboard events may get the focus
	//--------------------------------------------------------
	if (!bNoFocus) {
      bNoFocus = !( CHK_VMAS(VMAS_KEYDOWN, keyDown) || 
      	   					CHK_VMAS(VMAS_KEYUP, keyUp) );
	}

	// get all window attributes
	//---------------------------
	unsigned long mask=0;

	XSetWindowAttributes attr;

	TWndBmpList::iterator p = backgroundlist.find(this);
	if (p!=backgroundlist.end()) {
		mask|=CWBackPixmap;
		(*p).second->Update();
		attr.background_pixmap = (*p).second->pixmap;
	} else if (!bNoBackground && !bDoubleBuffer) {
		mask|=CWBackPixel;
		attr.background_pixel = background._GetPixel();
	}

	mask|=CWSaveUnder;
	attr.save_under = bSaveUnder;
	
	mask|=CWBackingStore;
	attr.backing_store = bBackingStore ? WhenMapped : NotUseful;

	mask|=CWEventMask;
	attr.event_mask = build_eventmask();
	
	mask|=CWColormap;
	attr.colormap = x11colormap;

	if (_cursor!=TCursor::DEFAULT) {
		mask|=CWCursor;
		attr.cursor = TCursor::X11Cursor(static_cast<TCursor::EType>(_cursor));
	}
	
	if (bPopup)	{
		mask|=CWOverrideRedirect;
		attr.override_redirect = true;
	}

	// 'createX11Window' messsage is needed for things like OpenGL support
	//---------------------------------------------------------------------
	static TX11CreateWindow x11;
	x11.display 	= x11display;
	x11.parent		= ( parent && !bShell ) ? parent->x11window : DefaultRootWindow(x11display);
	x11.x					= _x;
	x11.y					= _y;
	x11.width			= _w;
	x11.height		= _h;
	x11.border		= _b;
	x11.depth			= x11depth;
	x11.wclass		= InputOutput;
	x11.visual		= x11visual;
	x11.valuemask	= mask;
	x11.attributes= &attr;
	createX11Window(&x11);

// cout << "creating with depth: " << x11depth << endl;

	// create the window
	//-------------------
	x11window = XCreateWindow(
		x11.display,
		x11.parent,
		x11.x, x11.y, x11.width, x11.height, x11.border,
		x11.depth, x11.wclass, x11.visual,
		x11.valuemask, x11.attributes
	);

	// save TWindow in Xlib window context (it's a hash table)
	//---------------------------------------------------------
	if(XSaveContext(x11display, x11window, nClassContext, (XPointer)this))	{
		cerr << "toad: XSaveContext failed\n";
		exit(1);
	}

	
	FocusNewWindow(this);	// inform focus management about the new window

	// set additional WM parameters for top level windows
	//----------------------------------------------------
	if (bShell)	{
		// tell the WM that we want to destroy the window ourself
		XSetWMProtocols(x11display, x11window, &xaWMDeleteWindow, 1);
		// XSetWMProtocols(x11display, x11window, &xaWMSaveYourself, 1);

		// tell the WM which title to use
		XTextProperty tp;
		const char *p1 = title.c_str();
		const char **p2 = &p1;
		if (!XStringListToTextProperty((char**)p2, 1, &tp))	{
			cerr << "toad: internal error; XStringListToTextProperty failed\n";
		}	else {
			XSetWMName(x11display, x11window, &tp);
			XFree(tp.value);
		}

		// all windows which are not the mainwindow are transient, e.g. dialogs
		if (twMainWindow && this!=twMainWindow)	{
			XSetTransientForHint(x11display, x11window, twMainWindow->x11window);
		}

		XWMHints *wh;
		if ( (wh=XAllocWMHints())==NULL) {
			cerr << "toad: internal error; XAllocWMHints failed (out of memory)\n";
		}	else {
			// icon
			TWndBmpList::iterator p = iconlist.find(this);
			if (p!=iconlist.end()) {
				(*p).second->Update();
				wh->flags |= IconPixmapHint;
				wh->icon_pixmap = (*p).second->pixmap;
			}

			// group hint
			if (twMainWindow)	{		
				wh->flags |= WindowGroupHint;
				wh->window_group = twMainWindow->x11window;
			}
			
			XSetWMHints(x11display, x11window, wh);
			XFree(wh);
		}
		
		DnDNewShellWindow(this);
	} // end of `if (bShell)'
	
	if (bStaticFrame && !(bDialogEditRequest && TDialogEditor::running )) {
		// set standard X11 Window Manager Hints
		//---------------------------------------
		XSizeHints sh;
		sh.flags = PMinSize | PMaxSize;
		sh.min_width = sh.max_width = _w;
		sh.min_height= sh.max_height= _h;
		XSetWMSizeHints(x11display, x11window, &sh, XA_WM_NORMAL_HINTS);

		PropMotifWmHints motif_hints;

		motif_hints.flags = MWM_HINTS_FUNCTIONS|MWM_HINTS_DECORATIONS|MWM_HINTS_INPUT_MODE;

		motif_hints.decorations = 
			MWM_DECOR_BORDER
		| MWM_DECOR_TITLE 
		| MWM_DECOR_MENU;

		motif_hints.functions = 
			MWM_FUNC_MOVE
		| MWM_FUNC_CLOSE ;

		motif_hints.inputMode = MWM_INPUT_MODELESS;
		motif_hints.status = 0;

		XChangeProperty (
			x11display,
			x11window,
			xaWMMotifHints,															// Atom for "_MOTIF_WM_HINTS"
			xaWMMotifHints,															// type
			32,																					// format (32 bit quantities)
			PropModeReplace,														// mode
			reinterpret_cast<unsigned char*>(&motif_hints),	// data
			PROP_MOTIF_WM_HINTS_ELEMENTS								// nelements
		);
	}

	// create children
	//-----------------
	TWindow *ptr = FirstChild();
	while(ptr) {
		ptr->_create();
		ptr=NextSibling(ptr);
	}

	if (_bResizedBeforeCreate) {
		resize();
		_bResizedBeforeCreate = false;
	}
	created();
	_ChildNotify(TCHILD_CREATE);

	if (_visible)
		XMapRaised(x11display, x11window);
}

//----------------------------------------------------------------------------
//. Destroy and remove all children that have been added since Create().
void TWindow::Destroy()
{
	// this is a situation where it would be nice to add something like
	// a window event listener as in Java AWT/Swing:
	if (TDialogEditor::EditWindow()==this) {
		TDialogEditor::SetEditWindow(0);
	}

	_destroy();
}

void TWindow::_destroy()
{
	ENTRYEXIT("TWindow::_destroy");
	if(x11window) {
		// take care of pointer in TOADBase
		if (this==TOADBase::wndTopPopup)
			UngrabMouse();
		
		destroy();
		_ChildNotify(TCHILD_DESTROY);
		if (child) {
			TWindow *ptr = child;
			TWindow *last= child->prev_sibling;
			TWindow *next;
			while(ptr!=last) {
				next=ptr->next_sibling;

				ptr->_destroy();
				if (!ptr->is_iflag(IFLAG_BEFORE_CREATE))
					delete ptr;
				ptr=next;
			}
			ptr->_destroy();
			if (!ptr->is_iflag(IFLAG_BEFORE_CREATE))
				delete ptr;
		}
		RemoveMessage(this);
		XSaveContext(x11display, x11window, nClassContext, (XPointer)0);
		XDestroyWindow(x11display, x11window);
		x11window = 0;

		// take care of pointer in TFocusManager
		//---------------------------------------
		FocusDelWindow(this);
	}
}

// SetIcon
//----------------------------------------------------------------------------
void TWindow::SetIcon(TBitmap* bmp)
{
	TWndBmpList::iterator p = iconlist.find(this);
	if (p!=iconlist.end()) {
		delete (*p).second;
		(*p).second = bmp;
	}	else {
		iconlist[this]=bmp;
	}
}

// Raise & Lower Window
//----------------------------------------------------------------------------
void TWindow::RaiseWindow()
{
	CHECK_REALIZED("RaiseWindow");
	XRaiseWindow(x11display, x11window);
}

void TWindow::LowerWindow()
{
	CHECK_REALIZED("LowerWindow");
	XLowerWindow(x11display, x11window);
}

// Paint Queue
//----------------------------------------------------------------------------

#if defined __GNUC__ && (__GNUC__<2 || (__GNUC__==2 && __GNUC_MINOR__<8))

typedef vector<TWindow::TPaintRegion*> TPaintRegionContainer;
static stack<TPaintRegionContainer> paint_region_stack;

#else

static stack<TWindow::TPaintRegion*> paint_region_stack;

#endif

// Static method returning `true' when there are event in the paint queue.
//---------------------------------------------------------------------------
bool TWindow::_HavePaintEvents()
{
	if (lock_paint_queue)
		return false;
	return !paint_region_stack.empty();
}

//. Static method to fetch and handle one paint event in the paint queue.
//---------------------------------------------------------------------------
void TWindow::_DispatchPaintEvent()
{
	THREAD_LOCK(mutexPaintQueue);
	if (paint_region_stack.empty()) {
		THREAD_UNLOCK(mutexPaintQueue);
		return;
	}
		
	TPaintRegion *rgn = paint_region_stack.top();
	paint_region_stack.pop();
//	THREAD_UNLOCK(mutexPaintQueue);
	
	if (rgn->wnd)	{
		if (rgn->wnd->x11window)	{
			TRectangle wrect(0, 0, rgn->wnd->_w, rgn->wnd->_h);
			
			// clip update region to window (needed after scrolling)
			(*rgn) &= wrect;
			
			// clear the background
			if (rgn->wnd->bEraseBe4Paint &&		// do we have to?
					!rgn->wnd->bNoBackground &&
					!rgn->wnd->bDoubleBuffer)
			{
				long n = rgn->GetNumRects();
				TRectangle r;
				for(long i=0; i<n; i++) {
					rgn->GetRect(i, &r);
					XClearArea(x11display, rgn->wnd->x11window, r.x,r.y,r.w,r.h, false);
				}
			}

			if (TDialogEditor::running && 
					TDialogEditor::enabled &&
      		rgn->wnd->bDialogEditRequest &&
      		TDialogEditor::EditWindow()==rgn->wnd ) 
      {
        TDialogEditor::DialogEditor()->paint();
			} else {
				rgn->wnd->paint();
			}
		}
		rgn->wnd->bEraseBe4Paint = false;
		rgn->wnd->paint_rgn = NULL;
	}
	delete rgn;
	THREAD_UNLOCK(mutexPaintQueue);
}

//. Update the whole invalidated window region right now. The normal 
//. behaviour is to wait until no other events than paint events are left in 
//. the message queue.
//---------------------------------------------------------------------------
void TWindow::PaintNow()
{
//	cout << "void TWindow::PaintNow()" << endl;
THREAD_LOCK(mutexPaintQueue);
	if (paint_rgn) {
		if (bEraseBe4Paint &&
				!bNoBackground &&
				!bDoubleBuffer) {
			long n = paint_rgn->GetNumRects();
			TRectangle r;
			for(long i=0; i<n; i++) {
				paint_rgn->GetRect(i, &r);
				XClearArea(x11display, x11window, r.x,r.y,r.w,r.h, false);
			}
		}
		bEraseBe4Paint = false;
		
		if (TDialogEditor::running && 
				TDialogEditor::enabled &&
     		bDialogEditRequest &&
     		TDialogEditor::EditWindow()==this ) 
    {
      TDialogEditor::DialogEditor()->paint();
		} else {
			paint();
		}
		Flush();
		paint_rgn->wnd = NULL;
		paint_rgn = NULL;
	}
THREAD_UNLOCK(mutexPaintQueue);
}

// _ProvidePaintRgn
//---------------------------------------------------------------------------
//. Sees that TRegion *paint_rgn contains a region.
void TWindow::_ProvidePaintRgn()
{
	if (paint_rgn==NULL) {			// make sure window has a paint region
//printf("TWindow: creating paint region for %lx\n",(long)this);
		paint_rgn = new TPaintRegion;
		paint_rgn->wnd = this;
		paint_region_stack.push(paint_rgn);
	}
}

TRegion* TWindow::UpdateRegion()
{
	return paint_rgn;
}

// Invalidate
//----------------------------------------------------------------------------
//. Invalidate an area of the window. This will generate a paint event when
//. all other events in the message queue have been processed. Multiple calls
//. to Invalidate sum up into a single paint event. Before paint is called,
//. the background of the invalidated area is cleared with the current 
//. background color.<P>
void TWindow::Invalidate(bool clear)
{
	THREAD_LOCK(mutexPaintQueue);
	_ProvidePaintRgn();
	paint_rgn->AddRect(0,0,_w,_h);
	bEraseBe4Paint |= clear;
	THREAD_UNLOCK(mutexPaintQueue);
}

void TWindow::Invalidate(int x,int y,int w,int h, bool clear)
{
	THREAD_LOCK(mutexPaintQueue);
	_ProvidePaintRgn();
	paint_rgn->AddRect(x,y,w,h);
	bEraseBe4Paint |= clear;
	THREAD_UNLOCK(mutexPaintQueue);
}

void TWindow::Invalidate(const TRectangle &r, bool clear)
{
	THREAD_LOCK(mutexPaintQueue);
	_ProvidePaintRgn();
	(*paint_rgn)|=r;
	bEraseBe4Paint |= clear;
	THREAD_UNLOCK(mutexPaintQueue);
}

void TWindow::Invalidate(const TRegion &r, bool clear)
{
	THREAD_LOCK(mutexPaintQueue);
	_ProvidePaintRgn();
	(*paint_rgn)|=r;
	bEraseBe4Paint |= clear;
	THREAD_UNLOCK(mutexPaintQueue);
}


// ScrollWindow
//-------------------------------------------------------------------

#define SCROLL_WITH_SERVER_GRAB

static Bool CheckEvent(Display*, XEvent *event, char *window)
{
	if (event->xany.window==reinterpret_cast<Window>(window) && 
			( event->type == Expose || event->type == GraphicsExpose) )
		return True;
	return False;	
}

//. Scroll window contents:
//. <TABLE BORDER=1 NOSHADE>
//.   <TR><TD>dy&gt;0</TD><TD>down</TD><TR>
//.   <TR><TD>dy&lt;0</TD><TD>up</TD><TR>
//.   <TR><TD>dx&gt;0</TD><TD>right</TD><TR>
//.   <TR><TD>dx&lt;0</TD><TD>left</TD><TR>
//. </TABLE>
void TWindow::ScrollWindow(int dx, int dy, bool bClrBG)
{
	if (!x11window || (dx==0 && dy==0)) 
		return;

	if (abs(dx)>=_w || abs(dy)>=_h) {
		Invalidate(bClrBG);
		return;
	}

	#ifdef SCROLL_WITH_SERVER_GRAB
	XGrabServer(x11display);
	XFlush(x11display);
	#endif

	// move paint events from the queue to the update region
	//-------------------------------------------------------
	XSync(x11display, False);
	XEvent event;
	while( XCheckIfEvent(x11display, &event, CheckEvent, (char*)x11window ) )
		Invalidate(event.xexpose.x, event.xexpose.y, event.xexpose.width, event.xexpose.height);

	// move update region
	//--------------------
	THREAD_LOCK(mutexPaintQueue);
	if (paint_rgn)
		paint_rgn->Translate(dx,dy);
	THREAD_UNLOCK(mutexPaintQueue);

	// scroll the windows contents
	//-----------------------------
	XCopyArea(x11display, x11window, x11window, x11gc, 0,0, _w, _h, dx,dy);
	
	// should do something like `*static_cast<TRegion*>(paint_rgn) & *this' here

	// decide which parts of the window must be redrawn; the
	// 'true' in 'XClearArea' will generate 'paint' events, but we
	// could handle them localy
	//-------------------------------------------------------------
	if (bClrBG)	{
		if (dy>0)	// scroll down, clear top
			XClearArea(x11display, x11window, 0,0, _w, dy, True);
		else if (dy<0)	// scroll up, clear bottom
			XClearArea(x11display, x11window, 0,_h+dy, _w, -dy, True);
		if (dx>0)	// scroll right, clear left
			XClearArea(x11display, x11window, 0,0, dx, _h, True);
		else if (dx<0)	// scroll left, clear right
			XClearArea(x11display, x11window, _w+dx, 0, -dx, _h, True);
	}

	#ifdef SCROLL_WITH_SERVER_GRAB
	XUngrabServer(x11display);
	#endif
}

//. Scroll area within the given rectangle.<P>
//. No scrolling occures, when <VAR>dx</VAR> or <VAR>dy</VAR> are &gt;= the size
//. of the rectangle.
void TWindow::ScrollRectangle(const TRectangle &r, int dx,int dy, bool bClrBG)
{
	if (!x11window || (dx==0 && dy==0)) 
		return;

	if (abs(dx)>=r.w || abs(dy)>=r.h)	{
		if (bClrBG)
			XClearArea(x11display, x11window, r.x,r.y, r.w, r.h, True);
		return;
	}

	#ifdef SCROLL_WITH_SERVER_GRAB
	XGrabServer(x11display);
	XFlush(x11display);
	#endif

	// move paint events from the queue to the update region
	//-------------------------------------------------------
	XSync(x11display, False);
	XEvent event;
	while( XCheckIfEvent(x11display, &event, CheckEvent, (char*)x11window ) )
		Invalidate(event.xexpose.x,event.xexpose.y,event.xexpose.width,event.xexpose.height);

	// move rectangle within update region
	//-------------------------------------
	THREAD_LOCK(mutexPaintQueue);
	if (paint_rgn) {
		TRegion r1;
		r1|=*paint_rgn;
		r1.Translate(dx,dy);
		r1&=r;
		*paint_rgn-=r;
		*paint_rgn|=r1;
	}
	THREAD_UNLOCK(mutexPaintQueue);

	// scroll the windows contents
	//-----------------------------
	struct {int x,y;} s,d;
	int xs,ys;

	if (dy>0)	{
		s.y = r.y;
		d.y = r.y+dy;
		ys  = r.h - dy;
	}	else if (dy<0)
	{
		s.y = r.y-dy;
		d.y = r.y;
		ys  = r.h + dy;
	}	else {
		s.y = r.y;
		d.y = r.y;
		ys  = r.h;
	}

	if (dx>0)	{
		s.x = r.x;
		d.x = r.x+dx;
		xs  = r.w - dx;
	}	else if (dx<0) {
		s.x = r.x-dx;
		d.x = r.x;
		xs  = r.w + dx;
	}	else {
		s.x = r.x;
		d.x = r.x;
		xs  = r.w;
	}

	//printf("scroll %3i,%3i - %3i,%3i by %3i,%3i\n", r.x,r.y, r.x+r.w,r.y+r.h, dx,dy);
	//printf("copy   %3i,%3i - %3i,%3i to %3i,%3i\n", r.x,r.y, r.x+xs,r.y+ys, d.x,d.y);

	XCopyArea(x11display, x11window, x11window, x11gc, s.x,s.y, xs, ys, d.x,d.y);
	
	// decide which parts of the window must be painted again; the
	// 'true' in 'XClearArea' will generate 'paint' events
	//-------------------------------------------------------------
	if (bClrBG)	{
		if (dy>0)	// scroll down, clear top
			XClearArea(x11display, x11window, r.x,        r.y, _w, dy, True);
		else if (dy<0)	// scroll up, clear bottom
			XClearArea(x11display, x11window, r.x,r.y+r.h+dy, _w, r.y-dy, True);
		if (dx>0)	// scroll right, clear left
			XClearArea(x11display, x11window, r.x,        r.y, dx, _h, True);
		else if (dx<0)	// scroll left, clear right
			XClearArea(x11display, x11window, r.x+r.w+dx, r.y, -dx, _h, True);
	}

	#ifdef SCROLL_WITH_SERVER_GRAB
	XUngrabServer(x11display);
	#endif
}

//. Requests the keyboard focus for the window.
bool TWindow::SetFocus()
{
	SetFocusWindow(this);
	return true;
}

//. Toggles the <VAR>_bOwnsFocus</VAR> flag.
void TWindow::_SetFocus(bool b)
{
	if (b!=_bOwnsFocus)	{
		_bOwnsFocus = b;
		focus();
		if (parent)
			parent->childNotify(this, TCHILD_FOCUS);
	}
}

//. Delivers `true' when the window owns the keyboard focus or when the
//. window is an active focus manager.
bool TWindow::IsFocus()
{
	return (_bOwnsFocus);
}

/*---------------------------------------------------------------------------*
 | AddChild                                                                  |
 *---------------------------------------------------------------------------*/
void TWindow::AddChild(TWindow* window)
{
	childNotify(window, TCHILD_BEFORE_ADD);
	if (window->parent)
		return;
	if (child==NULL) {
		window->prev_sibling=window;
		window->next_sibling=window;
		child=window;
	}	else {
		window->prev_sibling=child->prev_sibling;
		window->next_sibling=child;
		child->prev_sibling->next_sibling=window;
		child->prev_sibling=window;
	}
	window->parent = this;
	if (!window->is_iflag(IFLAG_SUPPRESSMSG))
		childNotify(window, TCHILD_ADD);
}

/*---------------------------------------------------------------------------*
 | RemoveChild                                                               |
 *---------------------------------------------------------------------------*/
void TWindow::RemoveChild(TWindow* window)
{
	if (window == window->next_sibling)	{
		child=NULL;
	}	else {
		window->prev_sibling->next_sibling = window->next_sibling;
		window->next_sibling->prev_sibling = window->prev_sibling;
		if (child==window)
			child=child->next_sibling;
	}
	window->parent = window->prev_sibling = window->next_sibling = NULL;
}

/*---------------------------------------------------------------------------*
 | FirstChild                                                                |
 *---------------------------------------------------------------------------*/
TWindow* TWindow::FirstChild()
{
	return child;
}

/*---------------------------------------------------------------------------*
 | LastChild                                                                 |
 *---------------------------------------------------------------------------*/
TWindow* TWindow::LastChild()
{
	return (child ? child->prev_sibling : (TWindow*)NULL);
}

/*---------------------------------------------------------------------------*
 | NextSibling                                                               |
 *---------------------------------------------------------------------------*/
TWindow* TWindow::NextSibling(TWindow *ptr)
{
	if (!ptr)
		return NULL;

	// top window has no siblings
	if (!ptr->parent)
		return NULL;

	if ( ptr->next_sibling == ptr->parent->child)
		return NULL;

	return ptr->next_sibling;
}

/*---------------------------------------------------------------------------*
 | PrevSibling                                                               |
 *---------------------------------------------------------------------------*/
TWindow* TWindow::PrevSibling(TWindow *ptr)
{
	if (!ptr)
		return NULL;

	// top window has no siblings
	if (!ptr->parent)
		return NULL;

 	if (ptr ==  ptr->parent->child)
		return NULL;
		
	return ptr->prev_sibling;
}

/*---------------------------------------------------------------------------*
 | IsChildOf                                                                 |
 *---------------------------------------------------------------------------*/
bool TWindow::IsChildOf(TWindow *wnd)
{
	TWindow *ptr=this;
	while(ptr!=NULL) {
		if (ptr==wnd) return true;
		ptr = ptr->parent;
	}
	return false;
}

/*---------------------------------------------------------------------------*
 | closeRequest		                                                           |
 *---------------------------------------------------------------------------*/

//. The default action is to call `EndDialog()'.
void TWindow::closeRequest()
{
	EndDialog(this);
}

/*---------------------------------------------------------------------------*
 | functions for dummy message handling that do (almost) nothing             |
 *---------------------------------------------------------------------------*/
void TWindow::adjust(){}
void TWindow::resize(){}
void TWindow::childNotify(TWindow*, EChildNotify){}
void TWindow::destroy(){}

//. Called when parts of the window must be redrawn.
//. <P>
//. See <A HREF="TPen.html">TPen</A> and <A HREF="TPencil.html"> for more
//. information.
void TWindow::paint(){}

//. Notifies the window that the focus has changed. <I>IsFocus()</I> will
//. return `true' when the window has the focus.
void TWindow::focus(){}

//. Called after the window was created.
void TWindow::created(){}

//. The are two ways to initialize a window:
//. <UL>
//.   <LI> in the constructor, which is usually the best way
//.   <LI> in <I>create()</I>
//. </UL>
//. <I>create()</I> is invoked after a call to <I>Create()</I> or <I>Run()</I>,
//. right before the actual window is mapped on the desktop.
//. <P>
//. When you have a window needing some sophisticated configuration
//. and you've decided not to do it all in the constructor but after you've 
//. created the object, then you might need a method being called after this
//. configuration is done and before the window will be created.<BR>
//. And this method is <I>create()</I>.
//. <P>
//. When you've decided to use <I>create()</I> you will have to keep some things
//. in mind:
//. <UL>
//.   <LI> 
//.     Child windows created after <I>Create()</I> will be removed during
//.     <I>Destroy()</I>.
//.   <LI>
//.     Other objects than child windows you've created during <I>create()</I>
//.     should be removed during <I>destroy()</I>, otherwise multiple
//.     <I>Create()</I>, <I>Destroy()</I> would result in memory leaks.<BR>
//.     Another good thing you can do is to set pointers to child windows to
//.     <CODE>NULL</CODE> during <I>destroy</I>. This won't modify the
//.     behaviour of your program but it's easier to see during debugging
//.     that a window is actually gone.
//. </UL>
//. Otherwise you will get into some fine mess.
void TWindow::create(){}

//. This method is called after <I>create()</I> right before the X11 window
//. is created and gives you a chance to modifiy X11 specific window parameters.
//. <P>
//. It was originally introduced to support OpenGL support on SGI machines.
void TWindow::createX11Window(TX11CreateWindow*){}

//. This method is called before an event is dispatched to one of the
//. windows methods. In case you have to handle X11 events on your own,
//. you can access the current XEvent in <I>TOADBase::x11event</I>.
//. <P>
//. The default implementation does nothing.
void TWindow::handleX11Event(){}

void TWindow::keyDown(TKey,char*,unsigned){}
void TWindow::keyUp(TKey,char*,unsigned){}

void TWindow::mouseMove(int,int,unsigned){}
void TWindow::mouseEnter(int,int,unsigned){}
void TWindow::mouseLeave(int,int,unsigned){}
void TWindow::mouseLDown(int,int,unsigned){}
void TWindow::mouseMDown(int,int,unsigned){}
void TWindow::mouseRDown(int,int,unsigned){}
void TWindow::mouseLUp(int,int,unsigned){}
void TWindow::mouseMUp(int,int,unsigned){}
void TWindow::mouseRUp(int,int,unsigned){}

//. The window manager send a `save yourself'-message to the window.
void TWindow::saveYourself(){printf("saveYourself\n");}

// SetSize
//----------------------------------------------------------------------------
//. Set the size of the window.
void TWindow::SetSize(int w,int h)
{
	if (this==NULL)	{
		cerr << "toad: TWindow.SetSize: this==NULL (set breakpoint @ exit() )\n";
		exit(1);
	}
	
	if (w==TSIZE_PREVIOUS) w=_w;
	if (h==TSIZE_PREVIOUS) h=_h;

	if (w<=0 || h<=0)	{
		// we should unmap the window instead...
		if (w<=0) w=1;
		if (h<=0) h=1;
	}

	if (_w==w && _h==h)
		return;

	_w=w;
	_h=h;
	
	if (x11window) {
		XResizeWindow(x11display, x11window ,_w,_h);
	}	else {
		_bResizedBeforeCreate = true;
		return;
	}

//cout << "resizing window \"" << Title() << "\"" << endl;

	// `is_iflag(IFLAG_WMRESIZE)' is true when this method was called
	// as consquence of `_ChildNotify(TCHILD_RESIZE)' below so before
	// spend the rest of our life with an infinite recursion we return
	// at once
	// but maybe i can remove this check because IFLAG_CHILD_NOTIFY
	// servers a similar purpose
	//-----------------------------------------------------------------
	if ( IsSuppressMessages() || is_iflag(IFLAG_WMRESIZE) )	{
		cout << "don't calling resize() and parents childNotify()" << endl;
		return;
	}
	set_iflag(IFLAG_WMRESIZE);
	resize();
	if (parent)	{
		_ChildNotify(TCHILD_RESIZE);
	}
	clr_iflag(IFLAG_WMRESIZE);
}

//. Notify parent of a modified child by calling it's
//. `childNotify' method.
void TWindow::_ChildNotify(TWindow::EChildNotify type)
{
	if(parent && !is_iflag(IFLAG_SUPPRESSMSG)) {
		if (!parent->is_iflag(IFLAG_CHILD_NOTIFY)) {
			parent->set_iflag(IFLAG_CHILD_NOTIFY);
			parent->childNotify(this,type);
			parent->clr_iflag(IFLAG_CHILD_NOTIFY);
		}
	}
}

//. Show or hide the window.
void TWindow::SetMapped(bool b)
{
	if (_visible == b)
		return;
	_visible = b;
	if (!x11window)
		return;
	if (b) {
		XMapWindow(x11display, x11window);
	}	else {
		XUnmapWindow(x11display, x11window);
	}
}

//. Returns `true' when the window is visible.
bool TWindow::Mapped()
{
	if (!x11window)
		return false;
	return _visible;
}

//. Set the origin for all drawing operations.
void TWindow::SetOrigin(int dx, int dy)
{
	_dx = dx; _dy = dy;
}

//. Set the origin for all drawing operations and scroll the windows content
//. to the new position.<BR>
//. <B>Attention: Child windows will not be moved!</B>
void TWindow::ScrollTo(int nx, int ny)
{
	int dx = nx - _dx;
	int dy = ny - _dy;
	ScrollWindow(dx, dy, true);
	_dx = nx; _dy = ny;
}

void TWindow::SetTitle(const string &title)
{
	this->title=title;
	if (x11window) {
		if (bShell) {
			XTextProperty tp;
			const char *p1 = title.c_str();
			const char **p2 = &p1;
			if (!XStringListToTextProperty((char**)p2, 1, &tp)) {
			 	cerr << "toad: internal error; XStringListToTextProperty failed\n";
			} else {
				XSetWMName(x11display, x11window, &tp);
				XFree(tp.value);
			} 
		}
		Invalidate();
	}
	
	_ChildNotify(TCHILD_TITLE);
}

// SetPosition
//---------------------------------------------------------------------------
void TWindow::SetPosition(int x,int y)
{
	if (this==NULL)	{
		cerr << "toad: TWindow.SetPostion: this==NULL (set breakpoint @ exit() )\n";
		exit(1);
	}

	if (x==_x && y==_y)
		return;
	_x = x;
	_y = y;
	if (x11window)
		XMoveWindow(x11display, x11window,x,y);
	_ChildNotify(TCHILD_POSITION);
}

// GetShape
//---------------------------------------------------------------------------
void TWindow::GetShape(TRectangle *r)
{
	if (this==NULL)	{
		cerr << "toad: TWindow::GetShape(TRectangle*): this==NULL (set breakpoint @ exit() )\n";
		exit(1);
	}

	r->Set(_x, _y, _w + (_b<<1), _h+ (_b<<1));
} 

// SetShape
//---------------------------------------------------------------------------
//. Changes the location and size of the the window including the windows
//. border.
void TWindow::SetShape(int x,int y,int w,int h)
{
	if (this==NULL)	{
		cerr << "toad: TWindow::SetShape(TRectangle*): this==NULL (set breakpoint @ exit() )\n";
		exit(1);
	}

	int x2=x, y2=y, w2=w, h2=h;
	if (x==TPOS_PREVIOUS) x=0;
	if (y==TPOS_PREVIOUS) y=0;
	if (w==TSIZE_PREVIOUS) w=0;
	if (h==TSIZE_PREVIOUS) h=0;
	TRectangle r(x,y,w,h);
	r.w -= (_b<<1);
	r.h -= (_b<<1);
	if (x2==TPOS_PREVIOUS) r.x=_x;
	if (y2==TPOS_PREVIOUS) r.y=_y;
	if (w2==TSIZE_PREVIOUS) r.w=_w;
	if (h2==TSIZE_PREVIOUS) r.h=_h;
	SetPosition(r.x, r.y);
	SetSize(r.w, r.h);
}

// SetBackground
//---------------------------------------------------------------------------
void TWindow::SetBackground(const TColor &nc)
{
	background.Set(nc.r,nc.g,nc.b);
	if(x11window)
		XSetWindowBackground(x11display, x11window, background._GetPixel());
}

//. Select a background bitmap for the window. 
void TWindow::SetBackground(TBitmap *bmp)
{
	TWndBmpList::iterator p = backgroundlist.find(this);
	if (p!=backgroundlist.end()) {
		delete (*p).second;
		(*p).second = bmp;
	} else {
		backgroundlist[this]=bmp;
	}

	if (x11window) {
		if (bmp) {
			bmp->Update();
			XSetWindowBackgroundPixmap(x11display, x11window, bmp->pixmap);
		} else {
			XSetWindowBackgroundPixmap(x11display, x11window, None);
		}
	}
}

//. Enables/disables the windows background.
//. <P>
//. In some situations, e.g. double buffering with TBitmap or when you
//. fill the whole window during <I>paint()</I>, this can be helpfull to
//. reduce flicker.
void TWindow::SetHasBackground(bool b)
{
	bNoBackground = !b;
	
	if (!x11window)
		return;

	unsigned long mask = 0UL;
	XSetWindowAttributes attr;

	if (!b) {
		attr.background_pixmap = None;
		mask|=CWBackPixmap;
	} else {
		TWndBmpList::iterator p = backgroundlist.find(this);
		if (p!=backgroundlist.end()) {
			mask|=CWBackPixmap;
			(*p).second->Update();
			attr.background_pixmap = (*p).second->pixmap;
		} else {
			mask|=CWBackPixel;
			attr.background_pixel = background._GetPixel();
		}
	}
	XChangeWindowAttributes(
		x11display, x11window,
		mask,
		&attr);
	Invalidate();
}

/*---------------------------------------------------------------------------*
 | ClearWindow                                                               |
 *---------------------------------------------------------------------------*/
void TWindow::ClearWindow()
{
  XClearWindow(x11display, x11window);
}

/*---------------------------------------------------------------------------*
 | UpdateWindow                                                              |
 *---------------------------------------------------------------------------*/
//. Obsolete, use 'Invalidate' instead.
void TWindow::UpdateWindow(bool bEraseBackground)
{
	if (!x11window) return;
  TRectangle r;
  r.x=0; r.y=0; r.w=_w; r.h=_h;
  if (bEraseBackground && !bNoBackground )
	  XClearWindow(x11display, x11window);
  paint();
}

/*---------------------------------------------------------------------------*
 | GetRootPos		                                                             |
 *---------------------------------------------------------------------------*/
/*	Get window coordiantes relative to the root window.<BR>
		(Returns <CODE>(0,0)</CODE>, when the window is not realized.) */ 
void TWindow::GetRootPos(int* x, int* y)
{
	if (!x11window) {
		*x=0;
		*y=0;
	}	else {
		Window wnd;
		XTranslateCoordinates(x11display,
													x11window, 
													DefaultRootWindow(x11display),
													0,0,x,y, &wnd);
		*x -= _b;
		*y -= _b;
	}
}

#ifdef DEBUG
void TWindow::debug_check_realized(const char *txt)
{
	if(!x11window)	{
		fprintf(stderr,
			"TOAD: FATAL ERROR\n"
			"      %s needs an existing window.\n",txt);
		if (debug_flags&TWindow::DFLAG_WMINIT)
		{
			fprintf(stderr, "      Occured during create() in window \"%s\".\n",
			title.c_str());
		} else if (debug_flags&TWindow::DFLAG_WMADJUST)
		{
			fprintf(stderr, "      Occured during adjust() in window \"%s\".\n",
			title.c_str());
		}
		fprintf(stderr, "      Occured in window \"%s\".\n"
										"      But NOT during create() or adjust(). Could be the constructor...\n",
										title.c_str());
		exit(1);
	}
}
#endif

// GrabMouse
//---------------------------------------------------------------------------
void TWindow::GrabMouse(ulong ulMouseMessages, TWindow* confine_window, TCursor::EType cursor)
{
//printf("%s.GrabMouse(%lx)\n",Title(),ulMouseMessages);

	bSimulatedAutomaticGrab = false;

	// set flags for MouseMoveMessages
	//---------------------------------
	ulong old_iflags = iflags;
	if (ulMouseMessages!=TMMM_PREVIOUS)	{
		// modify iflag for 'build_mouse_event_mask()'
		if ( CHK_VMAS(VMAS_MOUSEMOVE, mouseMove)  || bDialogEditRequest) {
			clr_iflag(IFLAG_MMM_MASK);
			set_iflag(ulMouseMessages & IFLAG_MMM_MASK);
			set_iflag(IFLAG_MMM_MODIFIED);
		}	else {
			cerr<< "toad: GrabMouse; 'mouseMove' message selected for window '"
					<< title << "',\n"
					   "      which has no 'mouseMove' method or bDialogEditRequest style.\n";
		}
	}

	// grab the mouse pointer
	//------------------------
	if (XGrabPointer(	x11display, x11window,
		False,	// all mouse messages for this window (not it's application)
		build_mouse_eventmask() | ButtonReleaseMask,
		/*
				the extra ButtonReleaseMask is needed to end the simulated
				automatic grab
		*/
		GrabModeAsync,
		GrabModeAsync,
		confine_window ? confine_window->x11window : None,
		TCursor::X11Cursor(cursor),
		CurrentTime ) != GrabSuccess )
	{
		cerr << "toad: warning; GrapPointer failed\n";
	}
	
	// restore iflag
	//---------------
	iflags = old_iflags;
}

// GrabPopupMouse
//---------------------------------------------------------------------------
//. Same as GrabMouse but window will receive a 'closeRequest' when
//. button is pressed outside the window and the other window will get
//. the mouse event.
void TWindow::GrabPopupMouse(ulong ulMouseMessages, TCursor::EType cursor)
{
//printf("%s.GrabPopupMouse(%03lx) (MMM:%03lx)\n",Title(),ulMouseMessages,iflags%IFLAG_MMM_MASK);

	bSimulatedAutomaticGrab = false;

	// set flags for MouseMoveMessages
	//---------------------------------
	ulong old_iflags = iflags;
	if (ulMouseMessages!=TMMM_PREVIOUS)	{
		// modify iflag for 'build_mouse_event_mask()'
		if ( CHK_VMAS(VMAS_MOUSEMOVE, mouseMove) || bDialogEditRequest)	{
			clr_iflag(IFLAG_MMM_MASK);
			set_iflag(ulMouseMessages & IFLAG_MMM_MASK);
			set_iflag(IFLAG_MMM_MODIFIED);
		}	else {
			cerr << "toad: GrabPopupMouse; 'mouseMove' message selected for window "
					"'" << title << "'\n"
				    "      which has no 'mouseMove' method or bDialogEditRequest style.\n";
		}
	}

	// grab the mouse pointer
	//------------------------
	if (XGrabPointer(	x11display, x11window,
		True,	// grab mouse events outside the application
		build_mouse_eventmask() | ButtonReleaseMask,
		GrabModeAsync,
		GrabModeAsync,
		None,
		TCursor::X11Cursor(cursor),
		CurrentTime ) != GrabSuccess )
	{
		cerr << "toad: warning; GrapPointer failed\n";
	}
	
	TOADBase::wndTopPopup = this;
	
	// restore iflag
	//---------------
	iflags = old_iflags;
}

// UngrabMouse
//---------------------------------------------------------------------------
void TWindow::UngrabMouse()
{
	XUngrabPointer(x11display, CurrentTime);
	TOADBase::bSimulatedAutomaticGrab = false;
	TOADBase::wndTopPopup = NULL;
}

// MouseMoveMessages
//---------------------------------------------------------------------------
//. Before a 'mouseMove(int,int,unsigned)' message is called you must select
//. the desired event types. TOAD will deliver a warning when 'mouseMove' is
//. implemented but no messages have been selected. The following values
//. define <VAR>mode</VAR> and can be joined by the '|' operator:
//. <TABLE BORDER=1>
//. <TR><TH>TMMM_NONE</TH><TD>
//.   Don't select any messages now but <I>GrabMouse</I> or 
//.   <I>GrabPopupMouse</I> might select some messages later.
//. </TD></TR><TR><TH>TMMM_ALL</TH><TD>
//.   Get all mouse move event types.
//. </TD></TR><TR><TH>TMMM_LBUTTON</TH><TD>
//.   Enable mouseMove when the left mouse button is held down.
//. </TD></TR><TR><TH>TMMM_MBUTTON</TH><TD>
//.   Enable mouseMove when the middle mouse button is held down.<BR>
//.   (Attention: This might be reserved for drag and drop.)
//. </TD></TR><TR><TH>TMMM_RBUTTON</TH><TD>
//.   Enable mouseMove when the right mouse button is held down.
//. </TD></TR><TR><TH>TMMM_ANYBUTTON</TH><TD>
//. </TD></TR><TR><TH>TMMM_FIRST</TH><TD>
//.   Get only the first event. (I've never tried that [MAH])
//. </TD></TR><TR><TH>TMMM_LAST</TH><TD>
//.   <B>Not available yet.</B><BR>
//.   Only deliver the last mouseMove, skip previous events.
//. </TD></TR><TR><TH>TMMM_PREVIOUS</TH><TD>
//.   Don't make any changes.
//. </TD></TR>
//. </TABLE>
//. The reason why mouseMove is so complicated to handle is performance.
void TWindow::SetMouseMoveMessages(ulong mode)
{
//printf("%s.SetMouseMoveMessages(%lx)\n",Title(),mode);
	if ( !CHK_VMAS(VMAS_MOUSEMOVE, mouseMove) )
	{
		fprintf(stderr,"toad: SetMouseMoveMessages; 'mouseMove' message selected for window '%s',\n"
									 "      which has no 'mouseMove' method.\n",title.c_str());
		return;
	}
	set_iflag(mode & IFLAG_MMM_MASK);
	set_iflag(IFLAG_MMM_MODIFIED);
	if (x11window)
		XSelectInput(x11display, x11window, build_eventmask());
}

void TWindow::ClrMouseMoveMessages(ulong mode)
{
	clr_iflag(mode & IFLAG_MMM_MASK);
	set_iflag(IFLAG_MMM_MODIFIED);
	if (x11window)
		XSelectInput(x11display, x11window, build_eventmask());
}

// build_eventmask
//---------------------------------------------------------------------------
//. determine what events the window wants
long TWindow::build_eventmask()
{
	long mask = build_mouse_eventmask();
	if ( CHK_VMAS(VMAS_PAINT, paint) )
		mask |= ExposureMask;

	// only shell windows will get key events
	if (bShell)	{
		mask |= KeyPressMask;
		mask |= KeyReleaseMask;
		mask |= FocusChangeMask;
		mask |= StructureNotifyMask;
	}
  
	// this is for DnD experiments, removed it by time
	mask |= PropertyChangeMask;

	return mask;              
}

//. private
long TWindow::build_mouse_eventmask()
{
	long mask = 0L;

	if ( CHK_VMAS(VMAS_MOUSELDOWN, mouseLDown)
		|| CHK_VMAS(VMAS_MOUSEMDOWN, mouseMDown)
		|| CHK_VMAS(VMAS_MOUSERDOWN, mouseRDown) 
		|| bDialogEditRequest)
		mask |= ButtonPressMask;

	if ( CHK_VMAS(VMAS_MOUSELUP, mouseLUp)
		|| CHK_VMAS(VMAS_MOUSEMUP, mouseMUp)
		|| CHK_VMAS(VMAS_MOUSERUP, mouseRUp)
		|| bDialogEditRequest)
		mask |= ButtonReleaseMask;

	if ( CHK_VMAS(VMAS_MOUSEENTER, mouseEnter) )
		mask |= EnterWindowMask;

	if ( CHK_VMAS(VMAS_MOUSELEAVE, mouseLeave) )
		mask |= LeaveWindowMask;

	if (_bToolTipAvailable) {
		mask |= EnterWindowMask | LeaveWindowMask;
	}

	if ( CHK_VMAS(VMAS_MOUSEMOVE, mouseMove) || bDialogEditRequest)	{
		if (!is_iflag(IFLAG_MMM_MASK) && !bDialogEditRequest) {
			cerr << "toad: No 'mouseMove' message selected for window "
				 <<	"'" << title << "'.\n"
				 <<	"      Method will be ignored.\n";
			return mask;
		}
		if (is_iflag(TMMM_FIRST))
			mask |= PointerMotionHintMask;
		if (is_iflag(TMMM_ALL))
			mask |= PointerMotionMask;
		if (is_iflag(TMMM_LBUTTON))
			mask |= Button1MotionMask;
		if (is_iflag(TMMM_MBUTTON))
			mask |= Button2MotionMask;
		if (is_iflag(TMMM_RBUTTON))
			mask |= Button3MotionMask;
		if (is_iflag(TMMM_ANYBUTTON))
			mask |= ButtonMotionMask;
	}	else {
		if (is_iflag(IFLAG_MMM_MASK))	{
			cerr << "toad: 'mouseMove' message selected for window "
				 << "'" << title << "'.\n"
				 <<	"      which has no 'mouseMove' method.\n";
		}
	}
	return mask;
}

//. <B>BEWARE: OLD DESCRIPTION</B><BR>
//.	Some action methods like <I>SetPostion</I> or <I>SetSize</I> call reaction 
//.	methods, e.g. when calling <I>SetSize</I> it will call <I>resize()</I> and
//.	the parents <I>childResize</I> method. Calling <I>SetSuppressMessages(true)
//.	</I> will avoid this for some methods. Please don't use this method until 
//.	you know what you're doing. See also <I>IsSuppressMessages</I>.
void TWindow::SetSuppressMessages(bool b)
{
	if (b)
		set_iflag(IFLAG_SUPPRESSMSG);
	else
		clr_iflag(IFLAG_SUPPRESSMSG);
}

bool TWindow::IsSuppressMessages()
{
	return is_iflag(IFLAG_SUPPRESSMSG);
}
