% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fix_costmatrix.R
\name{fix_costmatrix}
\alias{fix_costmatrix}
\title{Fixes a costmatrix that has inconsistent costs}
\usage{
fix_costmatrix(costmatrix, message = TRUE)
}
\arguments{
\item{costmatrix}{A costMatrix object.}

\item{message}{A logical indicating whether messages should be output (defaults to \code{TRUE}).}
}
\value{
A costMatrix object with self-consistent transition costs.
}
\description{
Given a costmatrix where transition costs are not self-consistent finds and returns a costmatrix that does.
}
\details{
A user may wish to consider a complicated set of transition costs between states when modelling discrete character evolution. This can be achieved with a custom costmatrix in Claddis (and elsewhere). However, some caution is urged when using such matrices to ensure that these costs are \emph{self-consistent} (Maddison and Maddison 2003). More specifically, no direct state-to-state transition cost should be greater than is possible with an indirect path via one or more intermediate states.

This function offers a solution through an algorithm that will iteratively alter a costmatrix until all direct transition costs are self-consistent. It does so by finding the shortest state-to-state path for all possible transitions using the \link{find_shortest_costmatrix_path} function. Because the first solution may itself be inconsistent (as it relied on costs that have since updated) the algorithm is repeated until an equilibrium is reached. (This scenario is unlikely in most real world cases, but may be possible with very large matrices representing many states so was implemented here for safety.)

Note: infinite costs are allowed in costmatrices but unless they fill entire rows or columns (excluding the diagonal) they will not be self-consistent as there will always be a cheaper indirect cost.

Note: that both PAUP* (Swofford 2003) TNT (Goloboff et al. 2008; Goloboff and Catalano, 2016) offerthe same correction using the triangle inequality.

Note: other issues with a costmatrix may arise that are better revealed by using the \link{check_costMatrix} function, which returns informative error messages (with fix suggestions) where issues are found.
}
\examples{

# Build a custom costmatrix with non-self consistent path lengths:
costmatrix <- make_costmatrix(
  min_state = 0,
  max_state = 2,
  character_type = "irreversible"
)
costmatrix$costmatrix[1:9] <- c(0, 2, 4, 1, 0, 3, 5, 3, 0)
costmatrix$symmetry <- "Asymmetric"
costmatrix$type <- "custom"

# Fix costmatrix:
fixed_costmatrix <- fix_costmatrix(costmatrix = costmatrix)

# Compare transition costs:
costmatrix$costmatrix
fixed_costmatrix$costmatrix
}
\references{
Goloboff, P. A. and Catalano, S. A., 2016. TNT version 1.5, including a full implementation of phylogenetic morphometrics. \emph{Cladistics}, \bold{32}, 221-238.

Goloboff, P., Farris, J. and Nixon, K., 2008. TNT, a free program for phylogenetic analysis. \emph{Cladistics}, \bold{24}, 774-786.

Maddison, D. R. and Maddison, W. P., 2003. \emph{MacClade 4: Analysis of phylogeny and character evolution}. Version 4.06. Sinauer Associates, Sunderland, Massachusetts.

Swofford, D. L., 2003. \emph{PAUP*. Phylogenetic Analysis Using Parsimony (*and Other Methods). Version 4}. Sinauer Associates, Sunderland, Massachusetts.
}
\author{
Graeme T. Lloyd \email{graemetlloyd@gmail.com}
}
