% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SL.R
\name{SL}
\alias{SL}
\title{Schmid-Leiman Transformation}
\source{
Schmid, J. & Leiman, J. M. (1957). The development of hierarchical
factor solutions. Psychometrika, 22(1), 53–61. doi:10.1007/BF02289209

Wolff, H.-G., & Preising, K. (2005). Exploring item and higher order
factor structure with the Schmid-Leiman solution: Syntax codes for SPSS and
SAS. Behavior Research Methods, 37 , 48–58. doi:10.3758/BF03206397
}
\usage{
SL(
  x,
  Phi = NULL,
  type = c("EFAtools", "psych", "SPSS", "none"),
  method = c("PAF", "ML", "ULS"),
  g_name = "g",
  ...
)
}
\arguments{
\item{x}{object of class \code{\link{EFA}}, class \code{\link[psych:fa]{psych::fa}},
class \code{\link[lavaan]{lavaan}} or matrix. If class \code{\link{EFA}} or
class \code{\link[psych:fa]{psych::fa}}, pattern coefficients and factor
intercorrelations are taken from this object. If class \code{\link[lavaan]{lavaan}},
it must be a second-order CFA solution. In this case first-order and second-order
 factor loadings are taken from this object and the \code{g_name} argument has
 to be specified.
x can also be a pattern matrix from an oblique factor solution (see \code{Phi})
or a matrix of first-order factor loadings from a higher-order confirmatory factor
analysis (see \code{L2}).}

\item{Phi}{matrix. A matrix of factor intercorrelations from an oblique factor
solution. Only needs to be specified if a pattern matrix is entered directly
into \code{x}.}

\item{type}{character. One of "EFAtools" (default), "psych", "SPSS", or "none".
This is used to control the procedure of the second-order factor analysis. See
\code{\link{EFA}} for details.}

\item{method}{character. One of "PAF", "ML", or "ULS" to use
principal axis factoring, maximum likelihood, or unweighted least squares
(also called minres), respectively, used in \code{\link{EFA}} to find the second-order
loadings.}

\item{g_name}{character. The name of the general factor. This needs only be
specified if \code{x} is a \code{lavaan} second-order solution. Default is "g".}

\item{...}{Arguments to be passed to \code{\link{EFA}}.}
}
\value{
A list of class SL containing the following
\item{orig_R}{Original correlation matrix.}
\item{sl}{A matrix with general factor loadings, group factor loadings, communalities,
and uniquenesses.}
\item{L2}{Second-order factor loadings.}
\item{vars_accounted}{A matrix of explained variances and sums of squared loadings.}
\item{iter}{The number of iterations needed for convergence in EFA.}
\item{settings}{list. The settings (arguments) used in EFA to get the
second-order loadings.}
}
\description{
This function implements the Schmid-Leiman (SL) transformation
(Schmid & Leiman, 1957). It takes the pattern coefficients and factor
intercorrelations from an oblique factor solution as
input and can reproduce the results from \code{\link[psych:schmid]{psych::schmid}}
and from the SPSS implementation from Wolff & Preising (2005). Other arguments
from \code{\link{EFA}} can be used to control the procedure to find the
second-order loadings more flexibly. The function can also be used on a
second-order confirmatory factor analysis (CFA) solution from lavaan.
}
\details{
The SL transformation (also called SL orthogonalization) is a procedure with
which an oblique factor solution is transformed into a hierarchical,
orthogonalized solution. As a first step, the factor intercorrelations are
again factor analyzed to find second-order factor loadings. If there is only
one higher-order factor, this step of the procedure stops there, resulting in
a second-order factor structure. The first-order factor and the second-order
factor are then orthogonalized, resulting in an orthogonalized factor solution
with proportionality constraints. The procedure thus makes a suggested
hierarchical data structure based on factor intercorrelations explicit. One
major advantage of SL transformation is that it enables variance
partitioning between higher-order and first-order factors, including the
calculation of McDonald's omegas (see \code{\link{OMEGA}}).
}
\examples{
## Use with an output from the EFAtools::EFA function, both with type EFAtools
EFA_mod <- EFA(test_models$baseline$cormat, N = 500, n_factors = 3,
               type = "EFAtools", method = "PAF", rotation = "promax")
SL_EFAtools <- SL(EFA_mod, type = "EFAtools", method = "PAF")

\donttest{
## Use with an output from the psych::fa function with type psych in SL
fa_mod <- psych::fa(test_models$baseline$cormat, nfactors = 3, n.obs = 500,
                    fm = "pa", rotate = "Promax")
SL_psych <- SL(fa_mod, type = "psych", method = "PAF")
}

## Use more flexibly by entering a pattern matrix and phi directly (useful if
## a factor solution found with another program should be subjected to SL
## transformation)

## For demonstration, take pattern matrix and phi from an EFA output
## This gives the same solution as the first example
EFA_mod <- EFA(test_models$baseline$cormat, N = 500, n_factors = 3,
               type = "EFAtools", method = "PAF", rotation = "promax")
SL_flex <- SL(EFA_mod$rot_loadings, Phi = EFA_mod$Phi, type = "EFAtools",
              method = "PAF")

\donttest{
## Use with a lavaan second-order CFA output

# Create and fit model in lavaan (assume all variables have SDs of 1)
mod <- 'F1 =~ V1 + V2 + V3 + V4 + V5 + V6
        F2 =~ V7 + V8 + V9 + V10 + V11 + V12
        F3 =~ V13 + V14 + V15 + V16 + V17 + V18
        g =~ F1 + F2 + F3'
fit <- lavaan::cfa(mod, sample.cov = test_models$baseline$cormat,
                   sample.nobs = 500, estimator = "ml")

SL_lav <- SL(fit, g_name = "g")

}
}
