% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fun_estimate_mean_treat.R
\name{esti_mean_treat}
\alias{esti_mean_treat}
\title{Estimate treatment effect and the corresponding variance estimation on the treatment arm using different covariate adjustment methods.}
\usage{
esti_mean_treat(X, Y, A, H = NULL)
}
\arguments{
\item{X}{The n by p covariates matrix.}

\item{Y}{Vector of n dimensional observed response.}

\item{A}{Vector of n dimensional treatment assignment.}

\item{H}{The n by n hat projection matrix corresponding to X.}
}
\value{
A list with two named vectors:
\describe{
  \item{point_est}{Point estimates for all estimators:
    \itemize{
      \item{\code{unadj}:} Unadjusted estimator
      \item{\code{db}:} Debiased estimator (Lu et al., 2023)
      \item{\code{adj2c}:} HOIF-inspired debiased estimator (Zhao et al., 2024), the same as \code{db}
      \item{\code{adj2}:} HOIF-motivated adjusted estimator (Zhao et al., 2024)
      \item{\code{adj3}:} Bias-free adjusted estimator based on \code{adj2}
      \item{\code{lin}:} Covariate-adjusted estimator (Lin, 2013)
      \item{\code{lin_db}:} Debiased estimator with population leverage scores (Lei, 2020)
    }}
  \item{var_est}{Variance estimates corresponding to each estimator:
    \itemize{
      \item{\code{unadj}:} Variance estimate for unadjusted estimator
      \item{\code{db}:} Variance estimate for debiased estimator (Lu et al., 2023)
      \item{\code{adj2c}:} Variance for \code{adj2c}, using formulas given in (Lu et al., 2023)
      \item{\code{adj2c_v2}:} Conservative variance for \code{adj2c} (Zhao et al., 2024)
      \item{\code{adj2}:} Variance for \code{adj2}, with formulas  motivated by (Lu et al., 2023)
      \item{\code{adj2_v2}:} Conservative variance for \code{adj2} (Zhao et al., 2024)
      \item{\code{adj3}:} Variance for \code{adj3},  with formulas  motivated by (Lu et al., 2023)
      \item{\code{adj3_v2}:} Conservative variance for \code{adj3} (Zhao et al., 2024)
      \item{\code{lin}:} HC3-type variance for Lin's (2013) estimator
      \item{\code{lin_db}:} HC3-type variance for Lei's (2020) estimator
    }}
}
}
\description{
Implements a unified framework for comparing covariate adjustment method for completely randomized experiments under randomization-based framework.
}
\examples{
set.seed(100)
n <- 500
p <- n * 0.3
beta <- runif(p, -1 / sqrt(p), 1 / sqrt(p))

X <- mvtnorm::rmvt(n, sigma = diag(1, p), df = 3)
Y1 <- as.numeric(X \%*\% beta)
Y0 <- rep(0, n)

pi1 <- 2/3
n1 <- ceiling(n * pi1)
ind <- sample(n, size = n1)
A <- rep(0, n)
A[ind] <- 1
Y <- Y1 * A + Y0 * (1 - A)

Xc_svd <- svd(X)
H <- Xc_svd$u \%*\% t(Xc_svd$u)

result_ls <- esti_mean_treat(X, Y, A, H)
point_est <- result_ls$point_est
var_est <- result_ls$var_est
print(paste0('True mean treat:', round(mean(Y1), digits = 3), '.'))
print('Absolute bias:')
print(abs(point_est - mean(Y1)))
print('Estimate variance:')
print(var_est)

}
\references{
Lin, W. (2013). \emph{Agnostic notes on regression adjustments to experimental data: Reexamining Freedman's critique. The Annals of Statistics, Vol. 7(1), 295–318}, \doi{10.1214/12-AOAS583}. \cr
Lei, L. and Ding, P. (2020) \emph{Regression adjustment in completely randomized experiments with a diverging number of covariates. Biometrika, Vol. 108(4), 815–828}, \doi{10.1093/biomet/asaa103}. \cr
Lu, X., Yang, F. and Wang, Y. (2023) \emph{Debiased regression adjustment in completely randomized experiments with moderately high-dimensional covariates. arXiv preprint, arXiv:2309.02073}, \doi{10.48550/arXiv.2309.02073}. \cr
Zhao, S., Wang, X., Liu, L. and Zhang, X. (2024) \emph{Covariate Adjustment in Randomized Experiments Motivated by Higher-Order Influence Functions. arXiv preprint, arXiv:2411.08491}, \doi{10.48550/arXiv.2411.08491}.
}
