% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Kriging_Ordinary.R
\name{Kriging_Ordinary}
\alias{Kriging_Ordinary}
\title{Ordinary Kriging interpolation}
\usage{
Kriging_Ordinary(
  BD_Obs,
  BD_Coord,
  shapefile,
  grid_resolution,
  variogram_model = c("exponential", "spherical", "gaussian", "linear"),
  max_dist = NULL,
  n_lags = 15,
  min_stations = 2,
  n_round = NULL,
  training = 1,
  stat_validation = NULL,
  Rain_threshold = NULL,
  save_model = FALSE,
  name_save = NULL
)
}
\arguments{
\item{BD_Obs}{A \code{data.table} or \code{data.frame} containing observational data with the following structure:
\itemize{
\item The first column (\code{Date}): A \code{Date} object representing the observation date.
\item The remaining columns: Each column corresponds to a unique ground station, where the column name is the station identifier.
}

The dataset should be structured as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{> BD_Obs
# A data.table or data.frame with n rows (dates) and m+1 columns (stations + Date)
   Date        ST001  ST002  ST003  ST004  ...
   <date>      <dbl>  <dbl>  <dbl>  <dbl>  ...
1  2015-01-01    0      0      0      0    ...
2  2015-01-02    0      0      0     0.2   ...
3  2015-01-03   0.1     0      0     0.1   ...
}\if{html}{\out{</div>}}
\itemize{
\item Each station column contains numeric values representing observed measurements.
\item The column names (station identifiers) must be unique and match those in \code{BD_Coord$Cod} to ensure proper spatial referencing.
}}

\item{BD_Coord}{A \code{data.table} or \code{data.frame} containing the metadata of the ground stations. It must include the following columns:
\itemize{
\item \code{"Cod"}:
Unique identifier for each ground station.
\item \code{"X"}:
Latitude of the station in UTM format.
\item \code{"Y"}:
Longitude of the station in UTM format.
}}

\item{shapefile}{A shapefile defining the study area, used to constrain the interpolation to the region of interest.
The shapefile must be of class \code{SpatVector} (from the \code{terra} package) and should have a UTM coordinate reference system.}

\item{grid_resolution}{A numeric value indicating the resolution of the interpolation grid in kilometers (\code{km}).}

\item{variogram_model}{Character string specifying the variogram model to fit. Options are:
\itemize{
\item \code{"exponential"} (default): Exponential variogram model
\item \code{"spherical"}: Spherical variogram model
\item \code{"gaussian"}: Gaussian variogram model
\item \code{"linear"}: Linear variogram model
}}

\item{max_dist}{Numeric value specifying the maximum distance (in meters) for variogram calculation and kriging prediction.
If \code{NULL} (default), it will be set to half the maximum distance between stations.}

\item{n_lags}{Integer specifying the number of lag bins to use in the empirical variogram calculation. Default is 15.}

\item{min_stations}{Integer specifying the minimum number of stations required for kriging interpolation.
If fewer stations are available (due to NAs), the function will return a constant field or use fallback methods.
Default is 2, which is more permissive for sparse precipitation data.}

\item{n_round}{An integer specifying the number of decimal places to round the interpolated results.
If set to \code{NULL}, all decimal places will be preserved. The default value is \code{1}.}

\item{training}{Numerical value between 0 and 1 indicating the proportion of data used for model training. The remaining data are used for validation. Note that if you enter, for example, 0.8 it means that 80 \% of the data will be used for training and 20 \% for validation.
If you do not want to perform validation, set training = 1. (Default training = 1).}

\item{stat_validation}{A character vector specifying the names of the stations to be used for validation.
This option should only be filled in when it is desired to manually enter the stations used for validation. If this parameter is NULL, and the formation is different from 1, a validation will be performed using random stations.
The vector must contain the names of the stations selected by the user for validation.
For example, stat_validation = c("ST001", "ST002"). (Default stat_validation = NULL).}

\item{Rain_threshold}{List of numerical vectors defining precipitation thresholds to classify precipitation into different categories according to its intensity.
This parameter should be entered only when the validation is to include categorical metrics such as Critical Success Index (CSI), Probability of Detection (POD), False Alarm Rate (FAR), etc.
Each list item should represent a category, with the category name as the list item name and a numeric vector specifying the lower and upper bounds of that category.
\strong{Note:} See the "Notes" section for additional details on how to define categories, use this parameter for validation, and example configurations.}

\item{save_model}{Logical value indicating whether the interpolation file should be saved to disk. The default value is \code{FALSE}. indicating that the interpolated file should not be saved.
If set to \code{TRUE}, be sure to set the working directory beforehand using \code{setwd(path)} to specify where the files should be saved.}

\item{name_save}{Character string indicating the name under which the interpolation raster file will be saved. By default the algorithm sets as output name: 'Model_Kriging'.}
}
\value{
The return value will depend on whether validation has been performed or not.
If validation is not performed, the function will return a \code{SpatRaster} object with the interpolated values.
If validation is performed, the function will return a list with two elements:
\itemize{
\item \code{Ensamble}: A \code{SpatRaster} object with the interpolated values.
\item \code{Validation}: A \code{data.table} with the validation results, including goodness-of-fit metrics and categorical metrics (if \code{Rain_threshold} is provided).
}
}
\description{
This function performs Ordinary Kriging, which is a spatial interpolation method that provides the best linear unbiased estimator
for unknown values based on the spatial correlation structure of the observed data. Unlike IDW, Kriging incorporates the spatial
autocorrelation through variogram modeling to determine optimal weights.
}
\section{Details}{

Ordinary Kriging is a geostatistical interpolation technique that provides the best linear unbiased estimator (BLUE)
for spatial data. Unlike deterministic methods like IDW, Kriging incorporates the spatial correlation structure of the
data through variogram modeling.

The Ordinary Kriging estimator is defined as:
\deqn{\hat{Z}(s_0) = \sum_{i=1}^{n} \lambda_i Z(s_i)}
where:
\describe{
\item{\eqn{\hat{Z}(s_0)}}{ is the estimated value at the unknown point.}
\item{\eqn{Z(s_i)}}{ are the known values.}
\item{\eqn{\lambda_i}}{ are the Kriging weights that sum to 1.}
\item{\eqn{n}}{ is the total number of known points used.}
}

The Kriging weights are obtained by solving the Kriging system:
\deqn{\begin{bmatrix} \Gamma & \mathbf{1} \\ \mathbf{1}^T & 0 \end{bmatrix} \begin{bmatrix} \boldsymbol{\lambda} \\ \mu \end{bmatrix} = \begin{bmatrix} \boldsymbol{\gamma} \\ 1 \end{bmatrix}}
where:
\describe{
\item{\eqn{\Gamma}}{ is the variogram matrix between observation points.}
\item{\eqn{\boldsymbol{\gamma}}}{ is the variogram vector between prediction and observation points.}
\item{\eqn{\mu}}{ is the Lagrange multiplier ensuring unbiasedness.}
}

The variogram models available are:
\describe{
\item{Exponential}{ \eqn{\gamma(h) = \sigma^2(1 - \exp(-3h/a))} for \eqn{h > 0}}
\item{Spherical}{ \eqn{\gamma(h) = \sigma^2(1.5h/a - 0.5(h/a)^3)} for \eqn{h \leq a}, \eqn{\sigma^2} for \eqn{h > a}}
\item{Gaussian}{ \eqn{\gamma(h) = \sigma^2(1 - \exp(-3(h/a)^2))} for \eqn{h > 0}}
\item{Linear}{ \eqn{\gamma(h) = \sigma^2 \cdot h/a} for \eqn{h \leq a}, \eqn{\sigma^2} for \eqn{h > a}}
}

The \code{Rain_threshold} parameter is used to calculate categorical metrics such as the Critical Success Index (CSI),
Probability of Detection (POD), False Alarm Rate (FAR), success ratio (SR), Hit BIAS (HB),Heidke Skill Score (HSS);
Hanssen-Kuipers Discriminant (HK); Equal Threat Score (ETS) or Gilbert Skill Score.
The parameter should be entered as a named list, where each item represents a category and the name of the item is the category name.
The elements of each category must be a numeric vector with two values: the lower and upper limits of the category.
For example:
\code{Rain_threshold = list(
  no_rain = c(0, 1),
  light_rain = c(1, 5),
  moderate_rain = c(5, 20),
  heavy_rain = c(20, 40),
  violent_rain = c(40, Inf)
)}
}

\examples{
\donttest{
# Load data from on-site observations
data("BD_Obs", package = "InterpolateR")
data("BD_Coord", package = "InterpolateR")

# Load the study area where the interpolation is performed.
shp_path = system.file("extdata", "study_area.shp", package = "InterpolateR")
shapefile = terra::vect(shp_path)

# Perform the interpolation
Interpolated_data <- Kriging_Ordinary(BD_Obs, BD_Coord, shapefile,
  grid_resolution = 5, variogram_model = "exponential",
  max_dist = NULL, n_lags = 15, n_round = 1, training = 0.8,
  Rain_threshold = NULL, stat_validation = NULL,
  save_model = FALSE, name_save = NULL)
}

}
\references{
Matheron, G. (1963). Principles of geostatistics. Economic Geology, 58(8), 1246-1266.
Cressie, N. (1993). Statistics for Spatial Data. John Wiley & Sons.
}
\author{
Marco Mogro \href{mailto:marcov.mogro@ucuenca.edu.ec}{marcov.mogro@ucuenca.edu.ec}
}
