% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_vcpsr.R
\name{sim_vcpsr}
\alias{sim_vcpsr}
\title{Varying-coefficient single-index signal regression using tensor P-splines.}
\usage{
sim_vcpsr(
  y,
  X,
  t_var,
  x_index = c(1:ncol(X)),
  nsegs = rep(10, 4),
  bdegs = rep(3, 4),
  lambdas = rep(1, 4),
  pords = rep(2, 4),
  max_iter = 100,
  mins = c(min(x_index), min(t_var)),
  maxs = c(max(x_index), max(t_var))
)
}
\arguments{
\item{y}{a response vector of length \code{m}, usually continuous.}

\item{X}{the signal regressors with dimension \code{m} by \code{p1}.}

\item{t_var}{the varying coeffient indexing variable of length \code{m}.}

\item{x_index}{an index of length \code{p} for columns of signal matrix;
default is simple sequence.}

\item{nsegs}{a vector of length 4 containing
the number of evenly spaced segments between min and max, for each
the coefficient surface (row and col) and
link surface (row and col), resp. (default \code{rep(10, 4)}.}

\item{bdegs}{a vector of length 4 containing
the degree of B-splines, for each
the coefficient surface (row and col) and link surface (row and col), resp.
(default cubic \code{rep(3, 4)}).}

\item{lambdas}{a vector of length 4 containing
the positive tuning parameters, for each
the coefficient surface (row and col) and link surface (row and col), resp.
(default \code{rep(1, 4)}).}

\item{pords}{a vector of length 4 containing
the difference penalty order, for each
the coefficient surface (row and col) and link surface (row and col), resp.
(default \code{rep(2, 4)}).}

\item{max_iter}{a scalar for the maximum number of iterations (default 100)}

\item{mins}{A vector length 2, containing min for signal index and \code{t_var}, default
associated with \code{x_index} and \code{t_var} minimums; default is respective minimums.}

\item{maxs}{A vector length 2, containing max for signal index and \code{t_var}, default
associated with \code{x_index} and \code{t_var} maximums; default is respective maximums.}
}
\value{
\item{y}{the response vector of length \code{m}.}
\item{alpha}{the P-spline coefficient vector (unfolded) of length \code{(nsegs[1]+bdeg[1])*(negs[2]+bdeg[2])}.}
\item{iter}{the number of iterations used for the single-index fit.}
\item{yint}{the estimated y-intercept for the single-index model.}
\item{Bx}{the B-spline matrix built along the signal index, using \code{nsegs[1]}, used for the coefficient surface.}
\item{By}{the B-spline matrix built along the \code{t_var} index,
using \code{nsegs[2]}, used for the coefficient surface.}
\item{Q}{the effective regressors from the \code{psVCSignal} portion of the single-index
fit with dimension \code{m} by \code{length(alpha)}.}
\item{t_var}{the VC indexing variable of length \code{m}.}
\item{nsegs}{a vector of length 4 containing
the number of evenly spaced segments between min and max, for each
the coefficient surface (row and col) and
link surface (row and col).}
\item{bdegs}{a vector of length 4 containing
the degree of B-splines, for each
the coefficient surface (row and col) and link surface (row and col).}
\item{lambdas}{a vector of length 4 containing
the positive tuning parameters, for each
the coefficient surface (row and col) and link surface (row and col).}
\item{pords}{a vector of length 4 containing
the difference penalty order, for each
the coefficient surface (row and col) and link surface (row and col).}
\item{mins}{a vector length 2, containing min for signal index and \code{t_var}.}
\item{maxs}{a vector length 2, containing max for signal index and \code{t_var}.}
\item{eta}{the estimated linear predictor for the single-index fit.}
\item{Pars}{a matrix of 2 rows associated with the signal coefficient surface
design parameters, each row: \code{c(min, max, nseg, bdeg, lambda, pord)} for
linear predictor \code{x_index} and \code{t_var}, resp.}
\item{pPars}{a matrix of 2 rows associated with the link function
design parameters, each row: \code{c(min, max, nseg, bdeg, lambda, pord)} for linear
predictor \code{eta} and \code{t_var}, resp.}
\item{cv}{the leave-one-out cross-validation statistic
or the standard error of prediction for the single-index fit.}
\item{delta_alpha}{change measure in signal-coefficent parameters at
convergence.}
\item{fit2D}{\code{ps2DNormal} object, fitting f(\code{eta}, \code{t_var}).}
}
\description{
\code{sim_vcpsr} is a varying-coefficient single-index
signal regression approach that allows both the signal coefficients
and the unknown link function to vary with
an indexing variable \code{t}, e.g. temperature. Two surfaces
are estimated (coefficent and link) that can be sliced at arbitary \code{t}.
Anisotripic penalization with P-splines is used on both.
}
\examples{
# Load libraries
library(fields) # Needed for plotting

# Get the data
Dat <- Mixture

# Dimensions: observations, temperature index, signal
m <- 34
p1 <- 401
p2 <- 12

# Stacking mixture data, each mixture has 12 signals stacked
# The first differenced spectra are also computed.
mixture_data <- matrix(0, nrow = p2 * m, ncol = p1)
for (ii in 1:m)
{
  mixture_data[((ii - 1) * p2 + 1):(ii * p2), 1:p1] <-
    t(as.matrix(Dat$xspectra[ii, , ]))
  d_mixture_data <- t(diff(t(mixture_data)))
}

# Response (typo fixed) and index for signal
y_mixture <- Dat$fractions
y_mixture[17, 3] <- 0.1501
index_mixture <- Dat$wl

# Select response and replicated for the 12 temps
# Column 1: water; 2: ethanediol; 3: amino-1-propanol
y <- as.vector(y_mixture[, 2])
y <- rep(y, each = p2)

bdegs = c(3, 3, 3, 3)
pords <- c(2, 2, 2, 2)
nsegs <- c(12, 5, 5, 5) # Set to c(27, 7, 7 ,7) for given lambdas
mins <- c(700, 30)
maxs <- c(1100, 70)
lambdas <- c(1e-11, 100, 0.5, 1) # based on svcm search
x_index <- seq(from = 701, to = 1100, by = 1) # for dX
t_var_sub <- c(30, 35, 37.5, 40, 45, 47.5, 50, 55, 60, 62.5, 65, 70)
t_var <- rep(t_var_sub, m)
max_iter <- 2 # Set higher in practice, e.g. 100
int <- TRUE

# Defining x as first differenced spectra, number of channels.
x <- d_mixture_data


# Single-index VC model using optimal tuning
fit <- sim_vcpsr(y, x, t_var, x_index, nsegs, bdegs, lambdas, pords,
             max_iter = max_iter, mins = mins, maxs = maxs)

plot(fit, xlab = "Wavelength (nm)", ylab = "Temp C")
}
\references{
Marx, B. D. (2015). Varying-coefficient single-index signal
regression. \emph{Chemometrics and Intelligent Laboratory Systems}, 143, 111–121.

Eilers, P.H.C. and Marx, B.D. (2021). \emph{Practical Smoothing, The Joys of
P-splines.} Cambridge University Press.
}
\author{
Paul Eilers and Brian Marx
}
