% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metric_nvR.R
\name{nvRC}
\alias{nvRC}
\title{Non-visual circadian response}
\usage{
nvRC(
  MEDI.vector,
  Illuminance.vector,
  Time.vector,
  epoch = "dominant.epoch",
  sleep.onset = NULL
)
}
\arguments{
\item{MEDI.vector}{Numeric vector containing the melanopic EDI data.}

\item{Illuminance.vector}{Numeric vector containing the Illuminance data.}

\item{Time.vector}{Vector containing the time data. Can be \link[base]{POSIXct},
\link[hms]{hms}, \link[lubridate]{duration}, or \link[base]{difftime}.}

\item{epoch}{The epoch at which the data was sampled. Can be either a
\link[lubridate]{duration} or a string. If it is a string, it needs to be
either \code{"dominant.epoch"} (the default) for a guess based on the data, or a valid
\link[lubridate]{duration} string, e.g., \code{"1 day"} or \code{"10 sec"}.}

\item{sleep.onset}{The time of habitual sleep onset. Can be HMS, numeric, or NULL.
If NULL (the default), then the data is assumed to start at habitual sleep onset.
If \code{Time.vector} is HMS or POSIXct, \code{sleep.onset} must be HMS. Likewise, if
\code{Time.vector} is numeric, \code{sleep.onset} must be numeric.}
}
\value{
A numeric vector containing the nvRC data. The output has the same
length as \code{Time.vector}.
}
\description{
This function calculates the non-visual circadian response (nvRC). It takes into account
the assumed response dynamics of the non-visual system and the circadian rhythm
and processes the light exposure signal to quantify the effective circadian-weighted
input to the non-visual system (see Details).
}
\details{
The timeseries is assumed to be regular. Missing values in the
light data will be replaced by 0.
}
\examples{

dataset1 <-
  tibble::tibble(
    Id = rep("B", 60 * 48),
    Datetime = lubridate::as_datetime(0) + lubridate::minutes(0:(60*48-1)),
    Illuminance = c(rep(0, 60*8), rep(sample(1:1000, 16, replace = TRUE), each = 60),
                    rep(0, 60*8), rep(sample(1:1000, 16, replace = TRUE), each = 60)),
    MEDI = Illuminance * rep(sample(0.5:1.5, 48, replace = TRUE), each = 60)
  )
# Time.vector as POSIXct
dataset1.nvRC <- dataset1 \%>\%
  dplyr::mutate(
    nvRC = nvRC(MEDI, Illuminance, Datetime, sleep.onset = hms::as_hms("22:00:00"))
  )

# Time.vector as difftime
dataset2 <- dataset1 \%>\% 
  dplyr::mutate(Datetime = Datetime - lubridate::as_datetime(lubridate::dhours(22)))
dataset2.nvRC <- dataset2 \%>\%
  dplyr::mutate(
    nvRC = nvRC(MEDI, Illuminance, Datetime, sleep.onset = lubridate::dhours(0))
  )
  
}
\references{
Amundadottir, M.L. (2016). Light-driven model for identifying
indicators of non-visual health potential in the built environment
[Doctoral dissertation, EPFL]. EPFL infoscience.
\doi{10.5075/epfl-thesis-7146}
}
\seealso{
Other metrics: 
\code{\link{bright_dark_period}()},
\code{\link{centroidLE}()},
\code{\link{disparity_index}()},
\code{\link{dose}()},
\code{\link{duration_above_threshold}()},
\code{\link{exponential_moving_average}()},
\code{\link{frequency_crossing_threshold}()},
\code{\link{interdaily_stability}()},
\code{\link{intradaily_variability}()},
\code{\link{midpointCE}()},
\code{\link{nvRD}()},
\code{\link{nvRD_cumulative_response}()},
\code{\link{period_above_threshold}()},
\code{\link{pulses_above_threshold}()},
\code{\link{threshold_for_duration}()},
\code{\link{timing_above_threshold}()}
}
\concept{metrics}
