#' Asymmetric Third Order Rotatable Designs based on Third Order Designs given by Das and Narasimham (1962)
#'
#' @param v Number of input factors
#'
#' @return Asymmetric Third Order Rotatable Designs (ATORDs) for a given v.
#'@description Generates asymmetric third order rotatable designs based on third order designs given by Das and Narasimham (1962) for a given number of input factors, v (3<=v<=9) with coded levels of the factors.
#' @export
#'@note The user is given with a set of mixed level response surface designs and after entering the serial number of any of the designs from the displayed set, the chosen design will be generated and displayed.
#'@details
#'The user is given two options for design generation. Method 1 generates asymmetric third order rotatable designs based on the third order designs given by Das and Narasimham (1962) for a given number of input factors, while method 2 generates asymmetric third order rotatable designs in a smaller number of runs as compared to the design generated by method 1, however, the variances of estimated parameters of the same order may not remain the same in case of design generated by method 2. After entering the serial number of any of the methods from the displayed set, the design from the chosen method will be generated and displayed.
#' @examples
#' if(interactive()){
#'library(MixedLevelRSDs)
#'ATORDs_I(3)}
#' @references
#' 1) J.S. Mehta and M.N. Das (1968). "Asymmetric rotatable designs and orthogonal transformations".
#'
#' 2) M. N. Das, V. Narasimham (1962). < doi:10.1214/AOMS/1177704374>. "Construction of Rotatable Designs Through Balanced Incomplete Block Designs".
#'
#' 3) M. Hemavathi, Shashi Shekhar, Eldho Varghese, Seema Jaggi, Bikas Sinha & Nripes Kumar Mandal (2022)<doi: 10.1080/03610926.2021.1944213>. "Theoretical developments in response surface designs: an informative review and further thoughts".


ATORDs_I<-function(v){
  #library(TORDs)
  if(v==3){
    print("Please, select the levels for which design to be generated: 1)19,19,9 2)21,21,9 ",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){
      B<-matrix(c(0.866025404,0.500000000,0,-0.5,0.866025404,0,0,0,1),nrow=3,byrow=T)
    }
    if(levels==2){
      B<-matrix(c(0.6,0.8,0,-0.8,0.6,0,0,0,1),nrow=3,byrow=T)
    }
  }
  if(v==4){
    print("Please, select the levels for which design to be generated: 1)21,21,9,9 2)51,51,51,9 3)25,11,29,9 4)23,23,23,9 5)15,13,23,9 6)11,9,23,25 7)25,25,25,25 8)25,25,21,21", quote=F)
    cat("\n")
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){
      B<-matrix(c(0.6,	0.8,	0,	0, -0.8,	0.6,	0,	0, 0,	0,	1,	0, 0,	0, 0, 1),nrow=4,byrow=T)
    }
    if(levels==2){
      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0 ,  0.285714286,    0.428571429,	0.857142857,	0,    0.857142857,	0.285714286,	-0.428571429,	0,   0,	0,	0,	1 ),nrow=4,byrow=T)
    }
    if(levels==3){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	 0,  0.666666667,	0.707106781,	0.23570226,	0,  0.666666667,	-0.707106781,	0.23570226,	0,  0,	0,	0,	1	),nrow=4,byrow=T)
    }
    if(levels==4){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0 -0.666666667,	0.666666667,	0.333333333,	0 ,   0.666666667,	0.333333333,	0.666666667,	0,
                  0,	0,	0,	1),nrow=4,byrow=T)
    }
    if(levels==5){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,   0.577350269,
                  -0.707106781,	0.40824829,	0,  0.577350269,	0,	-0.816496581,	0,  0,	0,	0,	1	),nrow=4,byrow=T)
    }
    if(levels==6){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,   0.5,	-0.707106781,	0.40824829,	0.288675135 ,  0.5,	0,	-0.816496581,	0.288675135, 0.5,	0,	0,	-0.866025404	),nrow=4,byrow=T)
    }
    if(levels==7){
      B<-matrix(c(0.866025404,	0.500000000,	0,	0,
                  -0.5,	0.866025404,	0,	0,
                  0,	0,	0.6,	0.8,
                  0,	0,	-0.8,	0.6	),nrow=4,byrow=T)
    }
    if(levels==8){
      B<-matrix(c(0.866025404,	0.500000000,	0,	0,
                  -0.5,	0.866025404,	0,	0,
                  0,	0,	0.316227766,	0.632455532,
                  0,	0,	0.632455532,	-0.316227766),nrow=4,byrow=T)
    }
  }
  ###################
  if(v==5){
    print("Please, select the levels for which design to be generated: 1)17,17,7,7,7 2) 15,15,7,7,7 3)11,9,15,25,7 4) 23,9,27,7,7 5)  23,23,23,7,7 6) 11,9,23,7,7 7)  17,17,17,17,7 8) 17,17,15,15,7 9) 23,23,23,17,17 10)23,9,27,15,15 11)11,9,23,17,17 12) 11,9,23,15,15 13) 39,39,39,17,17 14) 39,39,39,15,15 15) 23,23,23,15,15  16) 8,18,9,23,25",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){

      B<-matrix(c(0.6,	 0.8,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,
                  0,	0,	1,	0,	0,
                  0,	0,	0,	1,	0,
                  0,	0,	0,	0,	1
      ),nrow=5,byrow=T)

    }
    if(levels==2){


      B<-matrix(c(0.316227766,	0.632455532,	0,	0,	0,
                  0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	1,	0,	0,
                  0,	0,	0,	1,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==3){

      B<-matrix(c(0.5	, 0.707106781,	0.40824829,	0.288675135,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,
                  0.5,	0,	0,	-0.866025404,	0,
                  0,	0,	0,	0,	1
      ),nrow=5,byrow=T)

    }
    if(levels==4){


      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	1,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==5){


      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	1,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==6){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	1,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==7){


      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,
                  0,	0,	0.6,	0.8,	0,
                  0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==8){

      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,
                  0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	1),nrow=5,byrow=T)

    }
    if(levels==9){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	0.866025404,	0.500000000,
                  0,	0,	0,	-0.5,	0.866025404),nrow=5,byrow=T)

    }
    if(levels==10){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,
                  0,	0,	0,	0.632455532,	-0.316227766),nrow=5,byrow=T)

    }
    if(levels==11){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0.866025404,	0.500000000,
                  0,	0,	0,	-0.5,	0.866025404),nrow=5,byrow=T)

    }
    if(levels==12){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,
                  0,	0,	0,	0.632455532,	-0.316227766),nrow=5,byrow=T)

    }
    if(levels==13){

      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,
                  0,	0,	0,	0.866025404,	0.500000000,
                  0,	0,	0,	-0.5,	0.866025404),nrow=5,byrow=T)

    }
    if(levels==14){

      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,
                  0,	0,	0,	0.632455532,	-0.316227766),nrow=5,byrow=T)

    }
    if(levels==15){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,
                  0,	0,	0,	0.632455532,	-0.316227766),nrow=5,byrow=T)

    }
    if(levels==16){

      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,
                  0.447213595,	-0.894427191,	0,	0,	0),nrow=5,byrow=T)

    }
  }
  if(v==6){
    print("Please, select the levels for which design to be generated: 1)25,25,25,25,11,11 2)  25,25,23,23,11,11 3)33,33,33,25,25,11 4)33,33,33,23,23,11 5)  17,13,33,25,25,11 6) 17,13,33,23,23,11  7)  53,53,53,25,25,11 8) 53,53,53,23,23,11 9) 33,13,39,25,25,11 10)33,13,39,23,23,11 11)25,25,11,11,11,11 12) 23,23,11,11,11,11 13) 21,41,13,33,37,11 14) 17,13,33,37,11,11 15) 33,33,33,11,11,11 16) 17,13,33,11,11,11 17) 53,53,53,11,11,11 18) 33,13,39,11,11,11",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){

      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,
                  0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==2){

      B<-matrix(c(0.6, 	 0.8,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)
    }
    if(levels==3){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==4){


      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==5){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==6){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==7){
      B<-matrix(c(0.428571429,	-0.857142857,  0.285714286,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==8){

      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857, 0, 	0 ,	0 ,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==9){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==10){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==11){
      B<-matrix(c(0.6,	0.8,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==12){

      B<-matrix(c(0.316227766,	0.632455532,	0,	0,	0,	0,
                  0.632455532,	-0.316227766,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==13){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)


    }
    if(levels==14){
      B<-matrix(c(0.5	, 0.707106781,	0.40824829,	0.288675135,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==15){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)

    }
    if(levels==16){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)


    }
    if(levels==17){

      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)


    }
    if(levels==18){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	1),nrow=6,byrow=T)
    }
  }
  if(v==7){
    print("Please, select the levels for which design to be generated: 1)13,13,5,5,5,5,5 2)9,9,5,5,5,5,5 3)33,33,33,5,5,5,5 4)17,5,21,5,5,5,5 5) 17,17,17,5,5,5,5 6) 7,5,17,5,5,5,5 7) 7,5,17,19,5,5,5 8) 9,23,5,17,19,5,5 9) 11,27,23,19,17,5,5 10)13,13,13,13,5,5,5 11)13,13,9,9,5,5,5 12) 17,17,17,13,13,5,5 13) 17,17,17,9,9,5,5 14) 7,5,17,13,13,5,5 15) 7,5,17,9,9,5,5 16) 33,33,33,13,13,5,5 17) 33,33,33,9,9,5,5 18) 17,5,21,13,13,5,5 19)17,5,21,9,9,5,5 20) 33,33,33,17,5,21,5 21) 33,33,33,7,5,17,5 22) 33,33,33, 17,17,17,5 23) 17,17,17,17,5,21,5 24) 17,17,17, 7,5,17,5 25)17,5,21,7,5,17,5 26) 17,5,21,17,5,21,5 27) 33,33,33,33,33,33,5 28) 7,5,17,7,5,17,5 29) 17,17,17,17,17,17,5 30) 7,5,17,19,13,13,5 31) 7,5,17,19,9,9,5",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){

      B<-matrix(c(0.6,	0.8,	0,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==2){

      B<-matrix(c(0.316227766,	0.632455532,	0,	0,	0,	0,	0,
                  0.632455532,	-0.316227766,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1 ),nrow=7,byrow=T)
    }
    if(levels==3){

      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==4){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }


    if(levels==5){


      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667, 	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==6){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==7){

      B<-matrix(c(0.5	, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }

    if(levels==8){

      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==9){

      B<-matrix(c(0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	-0.816496581,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,
                  0.40824829,	0.182574186,	-0.894427191,	0,	0,	0,	0,
                  0.40824829,	-0.912870929,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==10){

      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }

    if(levels==11){

      B<-matrix(c(0.6	,  0.8,	0,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,	0,
                  0,	0,	0.316227766,	0.632455532,	0,	0,	0,
                  0,	0,	0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==12){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==13){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==14){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==15){


      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==16){

      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==17){


      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==18){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==19){


      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==20){

      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==21){

      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==22){


      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==23){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,
                  0,	0,	0,   0.666666667,	0.707106781,	0.23570226,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==24){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	 0,	0,	0,	0,
                  -0.666666667,	 0.666666667,	 0.333333333,	 0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	1	),nrow=7,byrow=T)
    }
    if(levels==25){

      B<-matrix(c(0.333333333,	0,	-0.942809042, 	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269, -0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)
    }
    if(levels==26){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	  0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042, 0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,
                  0,	0,	0,	0.666666667, -0.707106781,	0.23570226,	0,
                  0,	0,	0,	0 ,	0 ,	0,	1),nrow=7,byrow=T)}

    if(levels==27){

      B<-matrix(c(0.428571429,	-0.857142857, 0.285714286,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,
                  0,	0,	0,	0.428571429,	-0.857142857,	0.285714286,	0,
                  0,	0,	0,	0.285714286,	0.428571429,	0.857142857,	0,
                  0,	0,	0,	0.857142857,	0.285714286,	-0.428571429,	0,
                  0,	0,	0,	0,	0,	0,	1 ),nrow=7,byrow=T)}
    if(levels==28){

      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,
                  0.577350269,	 0,	-0.816496581, 	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,   0.707106781,	 0.40824829,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)}
    if(levels==29){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667, 	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)}
    if(levels==30){

      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	-0.707106781,	 0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	0,	-0.816496581,	 0.288675135,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,
                  0,	0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)}
    if(levels==31){

      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	-0.707106781, 	0.40824829,	0.288675135,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,
                  0,	0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	0,	1),nrow=7,byrow=T)}


  }
  if(v==8){
    print("Please, select the levels for which design to be generated: 1) 17,17,5,5,5,5,5,5 2) 11,11,5,5,5,5,5,5 3) 45,45,45,5,5,5,5,5 4) 23,7,27,5,5,5,5,5 5) 23,23,23,5,5,5,5,5 6) 9,7,23,5,5,5,5,5 7) 9,7,23,29,5,5,5 8) 11,35,7,23,29,5,5,5 9) 17,37,35,29,23,7,5,5 10) 13,29, 37,35,29,23,7,5 11) 17,17,11,11,5,5,5,5 12) 17,17,17,17,5,5,5,5 13) 23,23,23,17,17,5,5,5 14) 23,23,23,11,11,5,5,5 15) 9,7,23,17,17,5,5,5 16) 9,7,23,11,11,5,5,5 17) 45,45,45,17,17,5,5,5 18) 45,45,45,11,11,5,5,5 19) 23,23,23,17,17,5,5,5 20) 23,23,23,11,11,5,5,5 21) 45,45,45,23,7,27,5,5  22) 45,45,45,45,45,45,5,5 23) 45,45,45,9,7,23,5,5 24) 45,45,45,23,23,23,5,5 25) 23,23,23,23,7,27,5,5 26) 23,23,23,9,7,23,5,5 27) 23,7,27,9,7,23,5,5 28) 23,7,27, 23,7,27,5,5 29) 9,7,23,9,7,23,5,5 30) 23,23,23,23,23,23,5,5 31) 9,7,23,29,17,17,5,5  32) 9,7,23,29,11,11,5,5 33) 9,7,23,29,45,45,45,5 34) 9,7,23,29,23,7,27,5 35) 9,7,23,29,23,23,23,5 36)  9,7,23,29,9,7,23,5 37)  11,35,7,23,29,17,17,5  38) 11,35,7,23,29,11,11,5",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){
      B<-matrix(c(0.6,	0.8,	0,	0,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}


    if(levels==2){

      B<-matrix(c(0.316227766,	0.632455532,	0,	0,	0,	0,	0,	0,
                  0.632455532,	-0.316227766, 	0,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==3){

      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	 0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==4){
      B<-matrix(c(0.333333333,	0,	-0.942809042, 	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==5){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==6){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==7){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==8){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,
                  0.447213595, -0.894427191,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==9){
      B<-matrix(c(0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	-0.816496581,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	-0.894427191,	0,	0,	0,	0,	0,
                  0.40824829,	-0.912870929, 	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==10){
      B<-matrix(c(0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,
                  -0.707106781,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	-0.816496581,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	-0.894427191,	0,	0,	0,	0,
                  0.377964473,	0.15430335,	-0.912870929,	0,	0,	0,	0,	0,
                  0.377964473,	-0.9258201,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==11){
      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,	0,	0,
                  0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,
                  0,	0,	0.632455532,	-0.316227766,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==12){

      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,	0,	0,
                  0,	0,	0.6,	0.8,	0,	0,	0,	0,
                  0,	0,	-0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}


    if(levels==13){

      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==14){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==15){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==16){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==17){
      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==18){
      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==19){

      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==20){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==21){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==22){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.428571429,	-0.857142857,	0.285714286,	0,	0,
                  0,	0,	0,	0.285714286,	0.428571429,	0.857142857,	0,	0,
                  0,	0,	0,	0.857142857,	0.285714286,	-0.428571429,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==23){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==24){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==25){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0.666666667, -0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==26){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.33333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==27){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==28){
      B<-matrix(c(0.333333333,	0,	-0.942809042, 	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}

    if(levels==29){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==30){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}
    if(levels==31){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==32){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	1	),nrow=8,byrow=T)}
    if(levels==33){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.428571429,	-0.857142857,	0.285714286,	0,
                  0,	0,	0,	0,	0.285714286,	0.428571429,	0.857142857,	0,
                  0,	0,	0,	0,	0.857142857,	0.285714286,	-0.428571429,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}

    if(levels==34){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.333333333,	0,	-0.942809042,	0,
                  0,	0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,
                  0,	0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}
    if(levels==35){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,
                  0,	0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,
                  0,	0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}
    if(levels==36){
      B<-matrix(c(0.5,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,
                  0,	0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,
                  0,	0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	0,	1),nrow=8,byrow=T)}
    if(levels==37){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,
                  0.447213595, -0.894427191,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}
    if(levels==38){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,
                  0.447213595, -0.894427191,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	0,	0,	1
      ),nrow=8,byrow=T)}




  }
  if(v==9){
    print("Please, select the levels for which design to be generated: 1) 21,21,9,9,9,9,9,9,9 2) 19,19,9,9,9,9,9,9,9 3) 49,49,49,9,9,9,9,9,9 4) 31,11,35, 9,9,9,9,9,9 5) 31,31,31, 9,9,9,9,9,9 6) 17,11,31, 9,9,9,9,9,9 7) 19,11,31,37, 9,9,9,9,9 8) 23,45,11,31,37,5,5,5,5 9) 29,45,45,37,31,11,9,9,9 10) 27,45,45,45,37,31,11,9,9 11) 27,45,45,45,37,31,11,9,9 12) 21,21,21,21,9,9,9,9,9 13) 21,21,19,19,9,9,9,9,9 14) 31,31,31,21,21,9,9,9,9 15) 31,31,31,19,19,9,9,9,9 16) 17,11,31,19,19,9,9,9,9 17) 17,11,31,21,21,9,9,9,9 18) 49,49,49,21,21,9,9,9,9 19) 49,49,49,19,19,9,9,9,9 20) 31,11,35,21,21,9,9,9,9 21) 31,11,35,19,19,9,9,9,9 22) 49,49,49,31,11,35,9,9,9 23) 49,49,49, 49,49,49,9,9,9 24) 49,49,49, 17,11,31,9,9,9 25) 49,49,49, 31,31,31,9,9,9 26) 31,31,31,31,11,35,9,9,9  27) 31,31,31,17,11,31,9,9,9 28) 31, 11, 35,17,11,31,9,9,9 29) 31,11,35, 31,11,35,9,9,9 30) 31,11,35, 31,11,35,9,9,9 31) 31, 31, 31, 31, 31, 31,9,9,9 32) 19, 11, 31,37,21,21,9,9,9 33) 19, 11, 31,37,19,19,9,9,9 34) 19, 11,31,37,49,49,49,9,9 35) 19,11,31,37,31,11,35,9,9 36)  19,11,31,37,31,31,31,9,9 37)  19,11,31,37,17,11,31,9,9 38) 23,45,11,31,37,21,21,9,9 39) 23,45,11,31,37,19,19,9,9 40) 23,45,11,31,37,49,49,49,9 41) 23,45,11,31,37,17,11,31,9 42) 23,45,11,31,37,31,31,31,9  43) 19,11,31,37,19,11,31,37,9 44) 29,45,45,37,31,11,21,21,9 45) 29,45,45,37,31,11,19,19,9 46) 19,11,31,37, 21,21,21,21,9 47)  19,11,31,37, 21,21,19,19,9",quote=F)
    print("Please enter the serial number among the combinations:",quote=F)
    levels = readline()
    if(levels==1){
      B<-matrix(c(0.6,	0.8,	0,	0,	0,	0,	0,	0,	0,
                  -0.8,	0.6,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==2){
      B<-matrix(c(0.316227766,	0.632455532,	0,	0,	0,	0,	0,	0,	0,
                  0.632455532,	-0.316227766,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	1,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==3){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==4){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==5){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==6){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	1,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==7){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==8){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==9){
      B<-matrix(c(0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,0.223606798,	0.288675135,	-0.816496581,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	-0.894427191,	0,	0,	0,	0,	0,	0,
                  0.40824829,	-0.912870929,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==10){
      B<-matrix(c(0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,
                  -0.707106781,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	-0.816496581,	0,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,	0,
                  0.377964473,	0.15430335,	0.182574186,	-0.894427191,	0,	0,	0,	0,	0,
                  0.377964473,	0.15430335,	-0.912870929, 0,	0,	0,	0,	0,	0,
                  0.377964473,	-0.9258201,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==11){
      B<-matrix(c(0.353553391,	0.133630621,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,
                  0.353553391,	0.133630621,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,
                  0.353553391,	0.133630621,	0.15430335,	0.182574186,	0.223606798,	0.288675135,	-0.816496581,	0,	0,
                  0.353553391,	0.133630621,	0.15430335,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,
                  0.353553391,	0.133630621,	0.15430335,	0.182574186,	-0.894427191,	0,	0,	0,	0,
                  0.353553391,	0.133630621,	0.15430335,	-0.912870929,	0,	0,	0,	0,	0,
                  0.353553391,	0.133630621,	-0.9258201,	0,	0,	0,	0,	0,	0,
                  0.353553391,	-0.935414347,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==12){
      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0.6,	0.8,	0,	0,	0,	0,	0,
                  0,	0,	-0.8,	0.6,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==13){
      B<-matrix(c(0.866025404,	0.500000000,	0,	0,	0,	0,	0,	0,	0,
                  -0.5,	0.866025404,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,	0,
                  0,	0,	0.632455532,	-0.316227766	,0,	0,	0,	0,	0,
                  0,	0,	0,	0,	1,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==14){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.866025404,	0.500000000,  0,	0,	0,	0,
                  0,	0,	0,	-0.5,	0.866025404,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==15){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==16){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==17){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==18){
      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==19){
      B<-matrix(c(0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,	0,	0,	0,
                  0.285714286,	0.428571429,	0.857142857,	0,	0,	0,	0,	0,	0,
                  0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==20){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	 0,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.6,	0.8,	0,	0,	0,	0,
                  0,	0,	0,	-0.8,	0.6,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==21){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	 0,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.316227766,	0.632455532,	0,	0,	0,	0,
                  0,	0,	0,	0.632455532,	-0.316227766,	 0,	0,	0,	0,
                  0,	0,	0,	0,	0,	1,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==22){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==23){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	 -0.193548387,	0,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	 0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.428571429,	-0.857142857,	0.285714286,	0,	0,	0,
                  0,	0,	0,	0.285714286,	0.428571429,	0.857142857,	0,	0,	0,
                  0,	0,	0,	0.857142857,	0.285714286,	-0.428571429,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==24){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==25){
      B<-matrix(c(0.677419355,	0.193548387,	0.709677419,	0,	0,	0,	0,	0,	0,
                  0.451612903,	-0.870967742,	-0.193548387,	0,	0,	0,	0,	0,	0,
                  0.580645161,	0.451612903,	-0.677419355,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,	0,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==26){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==27){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==28){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==29){
      B<-matrix(c(0.333333333,	0,	-0.942809042,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,	0,
                  0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==30){
      B<-matrix(c(0.577350269,	0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,	0,	0,	0,
                  0.577350269,	0,	-0.816496581,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,	0,
                  0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==31){
      B<-matrix(c(0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,	0,	0,	0,
                  -0.666666667,	0.666666667,	0.333333333,	0,	0,	0,	0,	0,	0,
                  0.666666667,	0.333333333,	0.666666667,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,	0,	0,
                  0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,	0,	0,
                  0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==32){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.6,	0.8,	0,	0,	0,
                  0,	0,	0,	0,	-0.8,	0.6,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==33){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0, 0.316227766,	0.632455532,	0,		0,	0,
                  0,	0,	0,	0, 0.632455532,	-0.316227766,	0,		0,	0,
                  0,	0,	0,	0,	0,	0,	1,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==34){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.428571429,	-0.857142857,	0.285714286,	0,	0,
                  0,	0,	0,	0,	0.285714286,	0.428571429,	0.857142857,	0,	0,
                  0,	0,	0,	0,	0.857142857,	0.285714286,	-0.428571429,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==35){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.333333333,	0,	-0.942809042,	0,	0,
                  0,	0,	0,	0,	0.666666667,	0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0,	0.666666667,	-0.707106781,	0.23570226,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==36){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,	0,
                  0,	0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,	0,
                  0,	0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==37){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,	0,
                  0,	0,	0,	0,	0.577350269,	0,	-0.816496581,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==38){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.6,	0.8,	0,	0,
                  0,	0,	0,	0,	0,	-0.8,	0.6,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1),nrow=9,byrow=T)}
    if(levels==39){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.316227766,	0.632455532,	0,	0,
                  0,	0,	0,	0,	0,	0.632455532,	-0.316227766,	0,	0,
                  0,	0,	0,	0,	0,	0,	0,	1,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==40){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.677419355,	0.193548387,	0.709677419,	0,
                  0,	0,	0,	0,	0,	0.451612903,	-0.870967742,	-0.193548387,	0,
                  0,	0,	0,	0,	0,	0.580645161,	0.451612903,	-0.677419355,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==41){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.577350269,	0.707106781,	0.40824829,	0,
                  0,	0,	0,	0,	0,	0.577350269,	-0.707106781,	0.40824829,	0,
                  0,	0,	0,	0,	0,	0.577350269,	0,	-0.816496581,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==42){
      B<-matrix(c(0.447213595,	0.223606798,	0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,
                  0.447213595,	0.223606798,	0,	0,	-0.866025404,	0,	0,	0,	0,
                  0.447213595,	-0.894427191,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0.333333333,	0.666666667,	-0.666666667,	0,
                  0,	0,	0,	0,	0,	-0.666666667,	0.666666667,	0.333333333,	0,
                  0,	0,	0,	0,	0,	0.666666667,	0.333333333,	0.666666667,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==43){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.5,	0.707106781,	0.40824829,	0.288675135,	0,
                  0,	0,	0,	0,	0.5,	-0.707106781,	0.40824829,	0.288675135,	0,
                  0,	0,	0,	0,	0.5,	0,	-0.816496581,	0.288675135,	0,
                  0,	0,	0,	0,	0.5,	0,	0,	-0.866025404,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==44){
      B<-matrix(c(0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,0.223606798,	0.288675135,	-0.816496581,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	-0.894427191,	0,	0,	0,	0,	0,	0,
                  0.40824829,	-0.912870929,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	0,	0,	0,	-0.8,	0.6 ,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==45){
      B<-matrix(c(0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	0.288675135,	0.40824829,	-0.707106781,	0,	0,	0,
                  0.40824829,	0.182574186,0.223606798,	0.288675135,	-0.816496581,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	0.223606798,	-0.866025404,	0,	0,	0,	0,	0,
                  0.40824829,	0.182574186,	-0.894427191,	0,	0,	0,	0,	0,	0,
                  0.40824829,	-0.912870929,	0,	0,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==46){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.866025404,	0.500000000,	0,	0,	0,
                  0,	0,	0,	0,	-0.5,	0.866025404,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0.6,	0.8,	0,
                  0,	0,	0,	0,	0,	0,	-0.8,	0.6,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}
    if(levels==47){
      B<-matrix(c(0.5, 0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	-0.707106781,	0.40824829,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	-0.816496581,	0.288675135,	0,	0,	0,	0,	0,
                  0.5,	0,	0,	-0.866025404,	0,	0,	0,	0,	0,
                  0,	0,	0,	0,	0.866025404,	0.500000000,	0,	0,	0,
                  0,	0,	0,	0,	-0.5,	0.866025404,	0,	0,	0,
                  0,	0,	0,	0,	0,	0,	0.316227766,	0.632455532,	0,
                  0,	0,	0,	0,	0,	0,	0.632455532,	-0.316227766,	0,
                  0,	0,	0,	0,	0,	0,	0,	0,	1	),nrow=9,byrow=T)}

  }
  if(v==3){
    print("Please choose option number: 1) method 1 2) method 2",quote=F)
    type<-readline()
    if(type==1){
    z<-suppressMessages(TORDs::NSTORDs(v)$NSTORD)%*%B
    }
    if(type==2){
      m3<-matrix(c(
        1	,	1	,	0	,
        1	,	-1	,	0	,
        -1	,	1	,	0	,
        -1	,	-1	,	0	,
        1	,	0	,	1	,
        1	,	0	,	-1	,
        -1	,	0	,	1	,
        -1	,	0	,	-1	,
        0	,	1	,	1	,
        0	,	1	,	-1	,
        0	,	-1	,	1	,
        0	,	-1	,	-1	,
        1.4522	,	0	,	0	,
        -1.4522	,	0	,	0	,
        0	,	1.4522	,	0	,
        0	,	-1.4522	,	0	,
        0	,	0	,	1.4522	,
        0	,	0	,	-1.4522	,
        0.7937	,	0.7937	,	0.7937	,
        0.7937	,	0.7937	,	-0.7937	,
        0.7937	,	-0.7937	,	0.7937	,
        0.7937	,	-0.7937	,	-0.7937	,
        -0.7937	,	0.7937	,	0.7937	,
        -0.7937	,	0.7937	,	-0.7937	,
        -0.7937	,	-0.7937	,	0.7937	,
        -0.7937	,	-0.7937	,	-0.7937
      ),nrow=26,byrow=T)
     z=m3%*%B
    }
  }
  if(v>3){
    print("Please choose option number: 1) method 1 2) method 2",quote=F)
    type<-readline()
    if(type==1){
      z<-suppressMessages(TORDs::NSTORDs(v)$NSTORD)%*%B
    }
    if(type==2){
      if(v==4){
        m4<-matrix(c(	0	,	0	,	1	,	1	,0	,	0	,	1	,	-1	, 0	,	0	,	-1	,	1	,0	,	0	,	-1	,	-1	,
                      1	,	0	,	1	,	0	,1	,	0	,	-1	,	0	,-1	,	0	,	1	,	0	,-1	,	0	,	-1	,	0	,1	,	1	,	0	,	0	,
                      1	,	-1	,	0	,	0	,-1	,	1	,	0	,	0	,
                      -1	,	-1	,	0	,	0	,
                      0	,	1	,	0	,	1	,
                      0	,	1	,	0	,	-1	,
                      0	,	-1	,	0	,	1	,
                      0	,	-1	,	0	,	-1	,
                      1	,	1	,	-1	,	0	,
                      -1	,	1	,	-1	,	0	,
                      -1	,	1	,	1	,	0	,
                      -1	,	-1	,	1	,	0	,
                      1	,	-1	,	-1	,	0	,
                      1	,	-1	,	1	,	0	,
                      1	,	1	,	1	,	0	,
                      -1	,	-1	,	-1	,	0	,
                      0	,	1	,	1	,	-1	,
                      0	,	-1	,	1	,	-1	,
                      0	,	-1	,	1	,	1	,
                      0	,	-1	,	-1	,	1	,
                      0	,	1	,	-1	,	-1	,
                      0	,	1	,	-1	,	1	,
                      0	,	1	,	1	,	1	,
                      0	,	-1	,	-1	,	-1	,
                      1	,	1	,	0	,	-1	,
                      -1	,	1	,	0	,	-1	,
                      -1	,	1	,	0	,	1	,
                      -1	,	-1	,	0	,	1	,
                      1	,	-1	,	0	,	-1	,
                      1	,	-1	,	0	,	1	,
                      1	,	1	,	0	,	1	,
                      -1	,	-1	,	0	,	-1	,
                      1	,	0	,	1	,	-1	,
                      -1	,	0	,	1	,	-1	,
                      -1	,	0	,	1	,	1	,
                      -1	,	0	,	-1	,	1	,
                      1	,	0	,	-1	,	-1	,
                      1	,	0	,	-1	,	1	,
                      1	,	0	,	1	,	1	,
                      -1	,	0	,	-1	,	-1	,
                      2	,	0	,	0	,	0	,
                      -2	,	0	,	0	,	0	,
                      0	,	2	,	0	,	0	,
                      0	,	-2	,	0	,	0	,
                      0	,	0	,	2	,	0	,
                      0	,	0	,	-2	,	0	,
                      0	,	0	,	0	,	2	,
                      0	,	0	,	0	,	-2
        ),nrow=56,byrow=T)

        z<-m4%*%B
      }
      if(v==5){
        m5<-matrix(c(
          1	,	1	,	1	,	0	,	0	,
          1	,	1	,	-1	,	0	,	0	,
          1	,	-1	,	1	,	0	,	0	,
          1	,	-1	,	-1	,	0	,	0	,
          -1	,	1	,	1	,	0	,	0	,
          -1	,	1	,	-1	,	0	,	0	,
          -1	,	-1	,	1	,	0	,	0	,
          -1	,	-1	,	-1	,	0	,	0	,
          1	,	1	,	0	,	1	,	0	,
          1	,	1	,	0	,	-1	,	0	,
          1	,	-1	,	0	,	1	,	0	,
          1	,	-1	,	0	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	0	,
          -1	,	-1	,	0	,	1	,	0	,
          -1	,	-1	,	0	,	-1	,	0	,
          1	,	1	,	0	,	0	,	1	,
          1	,	1	,	0	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	1	,
          1	,	-1	,	0	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	1	,	1	,	0	,
          1	,	0	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	1	,	0	,
          -1	,	0	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	1	,
          1	,	0	,	1	,	0	,	-1	,
          1	,	0	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	0	,	-1	,
          -1	,	0	,	1	,	0	,	1	,
          -1	,	0	,	1	,	0	,	-1	,
          -1	,	0	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	0	,	-1	,
          1	,	0	,	0	,	1	,	1	,
          1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	0	,	-1	,	-1	,
          0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	1	,	0	,	1	,
          0	,	1	,	1	,	0	,	-1	,
          0	,	1	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	0	,	-1	,
          0	,	-1	,	1	,	0	,	1	,
          0	,	-1	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,
          0	,	1	,	0	,	1	,	1	,
          0	,	1	,	0	,	1	,	-1	,
          0	,	1	,	0	,	-1	,	1	,
          0	,	1	,	0	,	-1	,	-1	,
          0	,	-1	,	0	,	1	,	1	,
          0	,	-1	,	0	,	1	,	-1	,
          0	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	0	,	-1	,	-1	,
          0	,	0	,	1	,	1	,	1	,
          0	,	0	,	1	,	1	,	-1	,
          0	,	0	,	1	,	-1	,	1	,
          0	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	-1	,	1	,	1	,
          0	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	-1	,	-1	,	-1	,
          1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	0	,	0	,	0	,
          1	,	0	,	1	,	0	,	0	,
          1	,	0	,	-1	,	0	,	0	,
          -1	,	0	,	1	,	0	,	0	,
          -1	,	0	,	-1	,	0	,	0	,
          1	,	0	,	0	,	1	,	0	,
          1	,	0	,	0	,	-1	,	0	,
          -1	,	0	,	0	,	1	,	0	,
          -1	,	0	,	0	,	-1	,	0	,
          1	,	0	,	0	,	0	,	1	,
          1	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	0	,	0	,	1	,
          -1	,	0	,	0	,	0	,	-1	,
          0	,	1	,	1	,	0	,	0	,
          0	,	1	,	-1	,	0	,	0	,
          0	,	-1	,	1	,	0	,	0	,
          0	,	-1	,	-1	,	0	,	0	,
          0	,	1	,	0	,	1	,	0	,
          0	,	1	,	0	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	0	,
          0	,	-1	,	0	,	-1	,	0	,
          0	,	1	,	0	,	0	,	1	,
          0	,	1	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	0	,	1	,
          0	,	-1	,	0	,	0	,	-1	,
          0	,	0	,	1	,	1	,	0	,
          0	,	0	,	1	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,
          0	,	0	,	-1	,	-1	,	0	,
          0	,	0	,	1	,	0	,	1	,
          0	,	0	,	1	,	0	,	-1	,
          0	,	0	,	-1	,	0	,	1	,
          0	,	0	,	-1	,	0	,	-1	,
          0	,	0	,	0	,	1	,	1	,
          0	,	0	,	0	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,
          2.23606798	,	0	,	0	,	0	,	0	,
          -2.23606798	,	0	,	0	,	0	,	0	,
          0	,	2.23606798	,	0	,	0	,	0	,
          0	,	-2.23606798	,	0	,	0	,	0	,
          0	,	0	,	2.23606798	,	0	,	0	,
          0	,	0	,	-2.23606798	,	0	,	0	,
          0	,	0	,	0	,	2.23606798	,	0	,
          0	,	0	,	0	,	-2.23606798	,	0	,
          0	,	0	,	0	,	0	,	2.23606798	,
          0	,	0	,	0	,	0	,	-2.23606798

        ),nrow=130,byrow=T)

        z<-m5%*%B
      }
      if(v==6){
        m6<-matrix(c(
          0	,	0	,	0	,	1	,	1	,	0	,
          0	,	0	,	0	,	1	,	-1	,	0	,
          0	,	0	,	0	,	-1	,	1	,	0	,
          0	,	0	,	0	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	0	,	0	,	1	,
          0	,	-1	,	0	,	0	,	0	,	1	,
          0	,	1	,	0	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	0	,	0	,	-1	,
          1	,	0	,	1	,	0	,	0	,	0	,
          1	,	0	,	-1	,	0	,	0	,	0	,
          -1	,	0	,	1	,	0	,	0	,	0	,
          -1	,	0	,	-1	,	0	,	0	,	0	,
          1	,	1	,	0	,	-1	,	0	,	0	,
          -1	,	1	,	0	,	-1	,	0	,	0	,
          -1	,	1	,	0	,	1	,	0	,	0	,
          -1	,	-1	,	0	,	1	,	0	,	0	,
          1	,	-1	,	0	,	-1	,	0	,	0	,
          1	,	-1	,	0	,	1	,	0	,	0	,
          1	,	1	,	0	,	1	,	0	,	0	,
          -1	,	-1	,	0	,	-1	,	0	,	0	,
          0	,	1	,	1	,	0	,	-1	,	0	,
          0	,	-1	,	1	,	0	,	-1	,	0	,
          0	,	-1	,	1	,	0	,	1	,	0	,
          0	,	-1	,	-1	,	0	,	1	,	0	,
          0	,	1	,	-1	,	0	,	-1	,	0	,
          0	,	1	,	-1	,	0	,	1	,	0	,
          0	,	1	,	1	,	0	,	1	,	0	,
          0	,	-1	,	-1	,	0	,	-1	,	0	,
          0	,	0	,	1	,	1	,	0	,	-1	,
          0	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	0	,	-1	,	1	,	0	,	1	,
          0	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	0	,	1	,	-1	,	0	,	1	,
          0	,	0	,	1	,	1	,	0	,	1	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	0	,	0	,	1	,	1	,
          -1	,	0	,	0	,	0	,	-1	,	1	,
          1	,	0	,	0	,	0	,	-1	,	1	,
          1	,	0	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	0	,	0	,	1	,	-1	,
          1	,	0	,	0	,	0	,	1	,	-1	,
          1	,	0	,	0	,	0	,	1	,	1	,
          -1	,	0	,	0	,	0	,	-1	,	-1	,
          0	,	0	,	1	,	0	,	1	,	-1	,
          0	,	0	,	-1	,	0	,	1	,	-1	,
          0	,	0	,	-1	,	0	,	1	,	1	,
          0	,	0	,	-1	,	0	,	-1	,	1	,
          0	,	0	,	1	,	0	,	-1	,	-1	,
          0	,	0	,	1	,	0	,	-1	,	1	,
          0	,	0	,	1	,	0	,	1	,	1	,
          0	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	0	,	0	,	1	,	0	,	-1	,
          -1	,	0	,	0	,	1	,	0	,	-1	,
          -1	,	0	,	0	,	1	,	0	,	1	,
          -1	,	0	,	0	,	-1	,	0	,	1	,
          1	,	0	,	0	,	-1	,	0	,	-1	,
          1	,	0	,	0	,	-1	,	0	,	1	,
          1	,	0	,	0	,	1	,	0	,	1	,
          -1	,	0	,	0	,	-1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	0	,
          0	,	-1	,	1	,	1	,	0	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	0	,
          0	,	1	,	-1	,	1	,	0	,	0	,
          0	,	1	,	1	,	1	,	0	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	0	,
          1	,	1	,	1	,	0	,	0	,	1	,
          1	,	1	,	1	,	0	,	0	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	1	,
          1	,	1	,	-1	,	0	,	0	,	-1	,
          1	,	-1	,	1	,	0	,	0	,	1	,
          1	,	-1	,	1	,	0	,	0	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	1	,
          1	,	-1	,	-1	,	0	,	0	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	1	,
          -1	,	1	,	1	,	0	,	0	,	-1	,
          -1	,	1	,	-1	,	0	,	0	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	-1	,
          -1	,	-1	,	1	,	0	,	0	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	1	,
          -1	,	-1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	-1	,	1	,	0	,
          1	,	0	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	1	,	1	,	0	,
          1	,	0	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	1	,	1	,	1	,
          0	,	1	,	0	,	1	,	1	,	-1	,
          0	,	1	,	0	,	1	,	-1	,	1	,
          0	,	1	,	0	,	1	,	-1	,	-1	,
          0	,	1	,	0	,	-1	,	1	,	1	,
          0	,	1	,	0	,	-1	,	1	,	-1	,
          0	,	1	,	0	,	-1	,	-1	,	1	,
          0	,	1	,	0	,	-1	,	-1	,	-1	,
          0	,	-1	,	0	,	1	,	1	,	1	,
          0	,	-1	,	0	,	1	,	1	,	-1	,
          0	,	-1	,	0	,	1	,	-1	,	1	,
          0	,	-1	,	0	,	1	,	-1	,	-1	,
          0	,	-1	,	0	,	-1	,	1	,	1	,
          0	,	-1	,	0	,	-1	,	1	,	-1	,
          0	,	-1	,	0	,	-1	,	-1	,	1	,
          0	,	-1	,	0	,	-1	,	-1	,	-1	,
          2.44948974	,	0	,	0	,	0	,	0	,	0	,
          -2.44948974	,	0	,	0	,	0	,	0	,	0	,
          0	,	2.44948974	,	0	,	0	,	0	,	0	,
          0	,	-2.44948974	,	0	,	0	,	0	,	0	,
          0	,	0	,	2.44948974	,	0	,	0	,	0	,
          0	,	0	,	-2.44948974	,	0	,	0	,	0	,
          0	,	0	,	0	,	2.44948974	,	0	,	0	,
          0	,	0	,	0	,	-2.44948974	,	0	,	0	,
          0	,	0	,	0	,	0	,	2.44948974	,	0	,
          0	,	0	,	0	,	0	,	-2.44948974	,	0	,
          0	,	0	,	0	,	0	,	0	,	2.44948974	,
          0	,	0	,	0	,	0	,	0	,	-2.44948974
        ),nrow=136,byrow=T)

        z<-m6%*%B
      }
      if(v==7){
        m7<-matrix(c(
          1	,	1	,	0	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	0	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	0	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	0	,	1	,	0	,	0	,	0	,
          1	,	-1	,	0	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	0	,	1	,	0	,	0	,	0	,
          1	,	1	,	0	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	0	,	-1	,	0	,	0	,	0	,
          0	,	1	,	1	,	0	,	-1	,	0	,	0	,
          0	,	-1	,	1	,	0	,	-1	,	0	,	0	,
          0	,	-1	,	1	,	0	,	1	,	0	,	0	,
          0	,	-1	,	-1	,	0	,	1	,	0	,	0	,
          0	,	1	,	-1	,	0	,	-1	,	0	,	0	,
          0	,	1	,	-1	,	0	,	1	,	0	,	0	,
          0	,	1	,	1	,	0	,	1	,	0	,	0	,
          0	,	-1	,	-1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	1	,	1	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,	1	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	0	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	0	,
          0	,	0	,	1	,	-1	,	0	,	1	,	0	,
          0	,	0	,	1	,	1	,	0	,	1	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	0	,
          0	,	0	,	0	,	1	,	1	,	0	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	0	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	0	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	0	,	1	,
          0	,	0	,	0	,	1	,	1	,	0	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	0	,	0	,	1	,	1	,	0	,
          -1	,	0	,	0	,	0	,	-1	,	1	,	0	,
          1	,	0	,	0	,	0	,	-1	,	1	,	0	,
          1	,	0	,	0	,	0	,	-1	,	-1	,	0	,
          -1	,	0	,	0	,	0	,	1	,	-1	,	0	,
          1	,	0	,	0	,	0	,	1	,	-1	,	0	,
          1	,	0	,	0	,	0	,	1	,	1	,	0	,
          -1	,	0	,	0	,	0	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	0	,	0	,	1	,	1	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	1	,
          0	,	1	,	0	,	0	,	0	,	-1	,	1	,
          0	,	1	,	0	,	0	,	0	,	-1	,	-1	,
          0	,	-1	,	0	,	0	,	0	,	1	,	-1	,
          0	,	1	,	0	,	0	,	0	,	1	,	-1	,
          0	,	1	,	0	,	0	,	0	,	1	,	1	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	0	,	0	,	0	,	1	,
          1	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          1	,	0	,	1	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	0	,	0	,	0	,	1	,
          -1	,	0	,	1	,	0	,	0	,	0	,	1	,
          1	,	0	,	1	,	0	,	0	,	0	,	1	,
          -1	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          0	,	0	,	1	,	0	,	1	,	1	,	1	,
          0	,	0	,	1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	1	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	1	,	0	,	-1	,	-1	,	-1	,
          0	,	0	,	-1	,	0	,	1	,	1	,	1	,
          0	,	0	,	-1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	-1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	-1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	-1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	-1	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	-1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,
          1	,	0	,	0	,	1	,	0	,	1	,	1	,
          1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	0	,	0	,	1	,	0	,	1	,	1	,
          -1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          -1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	1	,	0	,	0	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,
          1	,	1	,	1	,	0	,	0	,	1	,	0	,
          1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          0	,	1	,	1	,	1	,	0	,	0	,	1	,
          0	,	1	,	1	,	1	,	0	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	0	,	0	,	1	,
          0	,	1	,	1	,	-1	,	0	,	0	,	-1	,
          0	,	1	,	-1	,	1	,	0	,	0	,	1	,
          0	,	1	,	-1	,	1	,	0	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	0	,	0	,	1	,
          0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	0	,	0	,	1	,
          0	,	-1	,	1	,	1	,	0	,	0	,	-1	,
          0	,	-1	,	1	,	-1	,	0	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,
          0	,	-1	,	-1	,	1	,	0	,	0	,	1	,
          0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,
          0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,
          0	,	1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          2.64575131	,	0	,	0	,	0	,	0	,	0	,	0	,
          -2.64575131	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	2.64575131	,	0	,	0	,	0	,	0	,	0	,
          0	,	-2.64575131	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	2.64575131	,	0	,	0	,	0	,	0	,
          0	,	0	,	-2.64575131	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	2.64575131	,	0	,	0	,	0	,
          0	,	0	,	0	,	-2.64575131	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	2.64575131	,	0	,	0	,
          0	,	0	,	0	,	0	,	-2.64575131	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	2.64575131	,	0	,
          0	,	0	,	0	,	0	,	0	,	-2.64575131	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	2.64575131	,
          0	,	0	,	0	,	0	,	0	,	0	,	-2.64575131
        ),nrow=182,byrow=T)

        z<-m7%*%B
      }
      if(v==8){
        m8<-matrix(c(

          -1	,	0	,	1	,	0	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	1	,	1	,	0	,	1	,	1	,	1	,
          0	,	0	,	-1	,	1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	1	,	1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	1	,	-1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	-1	,	1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	1	,	1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	1	,	1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	1	,	1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	-1	,
          0	,	0	,	1	,	-1	,	0	,	1	,	1	,	1	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	1	,	1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	1	,	-1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	-1	,
          0	,	0	,	1	,	-1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	-1	,	1	,	0	,	1	,	1	,	1	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	1	,	1	,
          0	,	0	,	1	,	1	,	0	,	-1	,	-1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	-1	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	1	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	1	,
          0	,	0	,	1	,	1	,	0	,	-1	,	1	,	-1	,
          0	,	-1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	1	,	1	,	1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          0	,	1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	1	,	1	,	1	,	1	,	0	,
          0	,	-1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          0	,	1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          1	,	1	,	0	,	0	,	1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          1	,	1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          1	,	1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	-1	,	1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	1	,	1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	1	,	1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,
          -1	,	1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	1	,	1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          1	,	1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          1	,	1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	-1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	1	,	1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          1	,	1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          1	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	1	,	1	,	1	,	0	,	0	,	1	,	0	,
          -1	,	1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	1	,	1	,	0	,	0	,	1	,	0	,
          1	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	1	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	0	,
          -1	,	-1	,	1	,	-1	,	0	,	0	,	1	,	0	,
          -1	,	-1	,	-1	,	1	,	0	,	0	,	1	,	0	,
          1	,	1	,	-1	,	1	,	0	,	0	,	-1	,	0	,
          0	,	1	,	0	,	0	,	0	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	0	,	0	,	1	,	-1	,	0	,
          0	,	-1	,	0	,	0	,	0	,	1	,	1	,	0	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	1	,	0	,
          0	,	1	,	0	,	0	,	0	,	-1	,	-1	,	0	,
          0	,	1	,	0	,	0	,	0	,	-1	,	1	,	0	,
          0	,	1	,	0	,	0	,	0	,	1	,	1	,	0	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	0	,
          0	,	0	,	1	,	0	,	1	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	0	,	1	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	0	,	1	,	0	,	1	,	0	,
          0	,	0	,	-1	,	0	,	-1	,	0	,	1	,	0	,
          0	,	0	,	1	,	0	,	-1	,	0	,	-1	,	0	,
          0	,	0	,	1	,	0	,	-1	,	0	,	1	,	0	,
          0	,	0	,	1	,	0	,	1	,	0	,	1	,	0	,
          0	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	0	,
          0	,	1	,	0	,	1	,	0	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	1	,	0	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	1	,	0	,	0	,	0	,	1	,
          0	,	-1	,	0	,	-1	,	0	,	0	,	0	,	1	,
          0	,	1	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          0	,	1	,	0	,	-1	,	0	,	0	,	0	,	1	,
          0	,	1	,	0	,	1	,	0	,	0	,	0	,	1	,
          0	,	-1	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	0	,	0	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	0	,	0	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	0	,	0	,	1	,	0	,	0	,	0	,
          1	,	-1	,	0	,	0	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	0	,	0	,	1	,	0	,	0	,	0	,
          1	,	1	,	0	,	0	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	0	,	0	,	-1	,	0	,	0	,	0	,
          1	,	0	,	1	,	0	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	0	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	1	,	0	,	0	,	0	,	0	,	1	,
          -1	,	0	,	-1	,	0	,	0	,	0	,	0	,	1	,
          1	,	0	,	-1	,	0	,	0	,	0	,	0	,	-1	,
          1	,	0	,	-1	,	0	,	0	,	0	,	0	,	1	,
          1	,	0	,	1	,	0	,	0	,	0	,	0	,	1	,
          -1	,	0	,	-1	,	0	,	0	,	0	,	0	,	-1	,
          0	,	0	,	1	,	1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	-1	,	1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	-1	,	1	,	0	,	1	,	0	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	1	,	0	,	0	,
          0	,	0	,	1	,	-1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	1	,	-1	,	0	,	1	,	0	,	0	,
          0	,	0	,	1	,	1	,	0	,	1	,	0	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	-1	,	0	,	0	,
          1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,
          -1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,
          -1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,
          1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,
          1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,
          1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,
          -1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,
          0	,	0	,	0	,	0	,	1	,	1	,	0	,	-1	,
          0	,	0	,	0	,	0	,	-1	,	1	,	0	,	-1	,
          0	,	0	,	0	,	0	,	-1	,	1	,	0	,	1	,
          0	,	0	,	0	,	0	,	-1	,	-1	,	0	,	1	,
          0	,	0	,	0	,	0	,	1	,	-1	,	0	,	-1	,
          0	,	0	,	0	,	0	,	1	,	-1	,	0	,	1	,
          0	,	0	,	0	,	0	,	1	,	1	,	0	,	1	,
          0	,	0	,	0	,	0	,	-1	,	-1	,	0	,	-1	,
          2.828427	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          -2.82843	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	2.828427	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	-2.82843	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	2.828427	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	-2.82843	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	2.828427	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	-2.82843	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	2.828427	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	-2.82843	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	2.828427	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	-2.82843	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	2.828427	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	-2.82843	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	2.828427	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	-2.82843

        ),nrow=336,byrow=T)

        z<-m8%*%B
      }
      if(v==9){
        m9<-matrix(c(
          1	,	1	,	-1	,	0	,	0	,	0	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	0	,	0	,	0	,
          -1	,	1	,	1	,	0	,	0	,	0	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	0	,	0	,	0	,
          1	,	-1	,	1	,	0	,	0	,	0	,	0	,	0	,	0	,
          1	,	1	,	1	,	0	,	0	,	0	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	1	,	1	,	-1	,	0	,	0	,	0	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          0	,	0	,	0	,	-1	,	1	,	1	,	0	,	0	,	0	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          0	,	0	,	0	,	1	,	-1	,	1	,	0	,	0	,	0	,
          0	,	0	,	0	,	1	,	1	,	1	,	0	,	0	,	0	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	1	,	1	,	-1	,
          0	,	0	,	0	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	0	,	0	,	0	,	-1	,	1	,	1	,
          0	,	0	,	0	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	0	,	0	,	0	,	1	,	-1	,	1	,
          0	,	0	,	0	,	0	,	0	,	0	,	1	,	1	,	1	,
          0	,	0	,	0	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,	0	,
          -1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,	0	,
          -1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,	0	,
          -1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,	0	,
          1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,	0	,
          1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,	0	,
          1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,	0	,
          -1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,	0	,
          0	,	1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,
          0	,	-1	,	0	,	0	,	1	,	0	,	0	,	-1	,	0	,
          0	,	-1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,
          0	,	-1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,
          0	,	1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,
          0	,	1	,	0	,	0	,	-1	,	0	,	0	,	1	,	0	,
          0	,	1	,	0	,	0	,	1	,	0	,	0	,	1	,	0	,
          0	,	-1	,	0	,	0	,	-1	,	0	,	0	,	-1	,	0	,
          0	,	0	,	1	,	0	,	0	,	1	,	0	,	0	,	-1	,
          0	,	0	,	-1	,	0	,	0	,	1	,	0	,	0	,	-1	,
          0	,	0	,	-1	,	0	,	0	,	1	,	0	,	0	,	1	,
          0	,	0	,	-1	,	0	,	0	,	-1	,	0	,	0	,	1	,
          0	,	0	,	1	,	0	,	0	,	-1	,	0	,	0	,	-1	,
          0	,	0	,	1	,	0	,	0	,	-1	,	0	,	0	,	1	,
          0	,	0	,	1	,	0	,	0	,	1	,	0	,	0	,	1	,
          0	,	0	,	-1	,	0	,	0	,	-1	,	0	,	0	,	-1	,
          1	,	0	,	0	,	0	,	0	,	1	,	0	,	-1	,	0	,
          -1	,	0	,	0	,	0	,	0	,	1	,	0	,	-1	,	0	,
          -1	,	0	,	0	,	0	,	0	,	1	,	0	,	1	,	0	,
          -1	,	0	,	0	,	0	,	0	,	-1	,	0	,	1	,	0	,
          1	,	0	,	0	,	0	,	0	,	-1	,	0	,	-1	,	0	,
          1	,	0	,	0	,	0	,	0	,	-1	,	0	,	1	,	0	,
          1	,	0	,	0	,	0	,	0	,	1	,	0	,	1	,	0	,
          -1	,	0	,	0	,	0	,	0	,	-1	,	0	,	-1	,	0	,
          0	,	1	,	0	,	1	,	0	,	0	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	1	,	0	,	0	,	0	,	0	,	-1	,
          0	,	-1	,	0	,	1	,	0	,	0	,	0	,	0	,	1	,
          0	,	-1	,	0	,	-1	,	0	,	0	,	0	,	0	,	1	,
          0	,	1	,	0	,	-1	,	0	,	0	,	0	,	0	,	-1	,
          0	,	1	,	0	,	-1	,	0	,	0	,	0	,	0	,	1	,
          0	,	1	,	0	,	1	,	0	,	0	,	0	,	0	,	1	,
          0	,	-1	,	0	,	-1	,	0	,	0	,	0	,	0	,	-1	,
          0	,	0	,	1	,	0	,	1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	-1	,	0	,	1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	-1	,	0	,	1	,	0	,	1	,	0	,	0	,
          0	,	0	,	-1	,	0	,	-1	,	0	,	1	,	0	,	0	,
          0	,	0	,	1	,	0	,	-1	,	0	,	-1	,	0	,	0	,
          0	,	0	,	1	,	0	,	-1	,	0	,	1	,	0	,	0	,
          0	,	0	,	1	,	0	,	1	,	0	,	1	,	0	,	0	,
          0	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	0	,
          1	,	0	,	0	,	0	,	1	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	0	,	0	,	1	,	0	,	0	,	0	,	-1	,
          -1	,	0	,	0	,	0	,	1	,	0	,	0	,	0	,	1	,
          -1	,	0	,	0	,	0	,	-1	,	0	,	0	,	0	,	1	,
          1	,	0	,	0	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          1	,	0	,	0	,	0	,	-1	,	0	,	0	,	0	,	1	,
          1	,	0	,	0	,	0	,	1	,	0	,	0	,	0	,	1	,
          -1	,	0	,	0	,	0	,	-1	,	0	,	0	,	0	,	-1	,
          0	,	1	,	0	,	0	,	0	,	1	,	-1	,	0	,	0	,
          0	,	-1	,	0	,	0	,	0	,	1	,	-1	,	0	,	0	,
          0	,	-1	,	0	,	0	,	0	,	1	,	1	,	0	,	0	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	1	,	0	,	0	,
          0	,	1	,	0	,	0	,	0	,	-1	,	-1	,	0	,	0	,
          0	,	1	,	0	,	0	,	0	,	-1	,	1	,	0	,	0	,
          0	,	1	,	0	,	0	,	0	,	1	,	1	,	0	,	0	,
          0	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	0	,	0	,
          0	,	0	,	1	,	1	,	0	,	0	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,	0	,	0	,	-1	,	0	,
          0	,	0	,	-1	,	1	,	0	,	0	,	0	,	1	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	0	,	0	,	1	,	0	,
          0	,	0	,	1	,	-1	,	0	,	0	,	0	,	-1	,	0	,
          0	,	0	,	1	,	-1	,	0	,	0	,	0	,	1	,	0	,
          0	,	0	,	1	,	1	,	0	,	0	,	0	,	1	,	0	,
          0	,	0	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	0	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	1	,	1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	1	,	1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	1	,	1	,	1	,
          0	,	0	,	0	,	1	,	1	,	1	,	-1	,	-1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	-1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	-1	,	-1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	1	,	-1	,	-1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	1	,	-1	,	1	,	1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	-1	,	1	,	1	,	-1	,
          0	,	0	,	0	,	1	,	1	,	-1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	-1	,	1	,	-1	,	-1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	-1	,	-1	,	1	,	1	,
          0	,	0	,	0	,	-1	,	1	,	1	,	-1	,	-1	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          1	,	1	,	1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	1	,	1	,	1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	1	,	1	,
          1	,	1	,	1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          1	,	1	,	1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          1	,	1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	1	,	1	,	1	,
          1	,	1	,	1	,	0	,	0	,	0	,	1	,	1	,	1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          1	,	1	,	1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	1	,	1	,	1	,
          1	,	1	,	1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	1	,	1	,	1	,
          1	,	1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	-1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	-1	,
          1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	1	,	-1	,	-1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          1	,	-1	,	1	,	0	,	0	,	0	,	1	,	-1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	1	,	1	,	1	,
          -1	,	-1	,	-1	,	0	,	0	,	0	,	1	,	1	,	-1	,
          1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	-1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	1	,	-1	,	0	,	0	,	0	,	-1	,	1	,	1	,
          -1	,	1	,	1	,	0	,	0	,	0	,	-1	,	-1	,	1	,
          -1	,	-1	,	1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	1	,	1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	1	,	1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	1	,	1	,	1	,	0	,	0	,	0	,
          1	,	1	,	1	,	-1	,	-1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	-1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	-1	,	-1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	1	,	-1	,	-1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          1	,	-1	,	1	,	1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	-1	,	1	,	1	,	-1	,	0	,	0	,	0	,
          1	,	1	,	-1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	-1	,	1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	-1	,	-1	,	1	,	1	,	0	,	0	,	0	,
          -1	,	1	,	1	,	-1	,	-1	,	1	,	0	,	0	,	0	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	1	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	1	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	-1	,	-1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	1	,
          0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	-1	,	1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	1	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	-1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	1	,	1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	1	,	-1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	-1	,	-1	,
          0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	1	,	-1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	1	,	1	,	0	,	-1	,	1	,
          0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	1	,	0	,	1	,	1	,
          0	,	1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,
          0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,
          0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	1	,	1	,
          0	,	1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	1	,	1	,
          0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	-1	,
          0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	1	,
          0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	1	,
          -1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          -1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          -1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,	1	,
          1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          -1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          -1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,	1	,
          1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,	1	,
          -1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          -1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,	-1	,
          1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,	1	,
          1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,
          1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,
          1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          -1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,
          -1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,	1	,
          1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,	1	,
          -1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,	-1	,
          1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,	1	,
          -1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,
          1	,	1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,
          1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,
          -1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,
          -1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,
          1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,
          -1	,	-1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,
          -1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,
          -1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,
          1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,
          -1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,
          1	,	-1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,
          -1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,
          -1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,
          1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,
          -1	,	1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,
          1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,
          1	,	1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,
          -1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,
          1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,
          1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,
          1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,
          1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,
          -1	,	1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,
          1	,	1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,
          -1	,	-1	,	0	,	1	,	1	,	0	,	1	,	-1	,	0	,
          1	,	-1	,	0	,	1	,	1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,
          1	,	-1	,	0	,	1	,	-1	,	0	,	1	,	-1	,	0	,
          1	,	1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,
          1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	-1	,	0	,
          1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	-1	,	0	,
          -1	,	-1	,	0	,	-1	,	-1	,	0	,	1	,	-1	,	0	,
          1	,	-1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,
          -1	,	-1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	1	,	0	,	-1	,	-1	,	0	,
          -1	,	1	,	0	,	1	,	-1	,	0	,	1	,	1	,	0	,
          1	,	-1	,	0	,	1	,	1	,	0	,	-1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	1	,	0	,	1	,	1	,	0	,
          -1	,	-1	,	0	,	-1	,	1	,	0	,	1	,	-1	,	0	,
          1	,	1	,	0	,	-1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	-1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,
          -1	,	1	,	0	,	-1	,	-1	,	0	,	1	,	1	,	0	,
          -1	,	1	,	0	,	1	,	-1	,	0	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	1	,	1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	-1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	-1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	-1	,	1	,	1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	-1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	1	,	-1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	-1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	-1	,	1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	1	,	-1	,	1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	-1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	1	,	1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	-1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	1	,	1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	-1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	-1	,	1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	1	,	1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	-1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	1	,	1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	1	,	1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	1	,	1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	-1	,	1	,	1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	1	,	-1	,	1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	1	,	0	,	1	,	0	,	1	,
          0	,	1	,	1	,	1	,	-1	,	0	,	-1	,	0	,	-1	,
          0	,	1	,	-1	,	1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	-1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	-1	,	-1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	-1	,	-1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	1	,	0	,	-1	,	0	,	-1	,
          0	,	-1	,	1	,	1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	1	,	-1	,	1	,	1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	-1	,	-1	,	1	,	0	,	1	,	0	,	-1	,
          0	,	1	,	1	,	-1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	-1	,	1	,	-1	,	0	,	-1	,	0	,	1	,
          0	,	-1	,	1	,	-1	,	-1	,	0	,	1	,	0	,	1	,
          0	,	-1	,	1	,	1	,	-1	,	0	,	-1	,	0	,	1	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	1	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	1	,	-1	,	0	,
          1	,	0	,	-1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	-1	,	-1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	-1	,	1	,	1	,	0	,
          1	,	0	,	-1	,	0	,	1	,	1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	1	,	1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	-1	,	1	,	1	,	-1	,	0	,
          1	,	0	,	1	,	0	,	-1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	-1	,	-1	,	1	,	1	,	0	,
          -1	,	0	,	1	,	0	,	1	,	-1	,	-1	,	1	,	0	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,	1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,	1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,	1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,	1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,	-1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	1	,	0	,	1	,	0	,	-1	,	1	,
          1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,	1	,
          1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	-1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	-1	,
          1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	-1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	-1	,	-1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	1	,	1	,
          1	,	-1	,	0	,	1	,	0	,	1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	1	,	0	,	1	,	1	,
          -1	,	-1	,	0	,	-1	,	0	,	1	,	0	,	1	,	-1	,
          1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	1	,
          -1	,	1	,	0	,	-1	,	0	,	-1	,	0	,	1	,	1	,
          -1	,	1	,	0	,	1	,	0	,	-1	,	0	,	-1	,	1	,
          0	,	-1	,	-1	,	1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	1	,	1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	1	,	1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	1	,	1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	1	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	1	,	0	,	1	,	1	,	1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	1	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	1	,	1	,	1	,	0	,
          0	,	1	,	1	,	1	,	0	,	-1	,	-1	,	-1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	-1	,	1	,	-1	,	0	,
          0	,	1	,	-1	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	1	,	-1	,	-1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	1	,	-1	,	1	,	0	,	1	,	-1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	1	,	1	,	1	,	0	,
          0	,	-1	,	-1	,	-1	,	0	,	1	,	1	,	-1	,	0	,
          0	,	1	,	1	,	-1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	-1	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          0	,	-1	,	1	,	-1	,	0	,	-1	,	1	,	1	,	0	,
          0	,	-1	,	1	,	1	,	0	,	-1	,	-1	,	1	,	0	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,	1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,	-1	,
          -1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	1	,	1	,	0	,	0	,	-1	,	1	,
          1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,	1	,
          1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,	-1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	-1	,	-1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	-1	,	-1	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	1	,	1	,
          1	,	0	,	-1	,	1	,	1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	1	,	-1	,	1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	-1	,	-1	,	1	,	0	,	0	,	1	,	-1	,
          1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	-1	,	1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	0	,	1	,	-1	,	-1	,	0	,	0	,	1	,	1	,
          -1	,	0	,	1	,	1	,	-1	,	0	,	0	,	-1	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	1	,	-1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	1	,	1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	1	,	1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	1	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	-1	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	1	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	-1	,	1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	1	,	1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	1	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	1	,	1	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	1	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	-1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	1	,	-1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	-1	,	1	,	1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	-1	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	1	,	1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	1	,	1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	1	,	-1	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	1	,	1	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	1	,	1	,	1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	1	,	-1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	-1	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	1	,	1	,	0	,	1	,
          1	,	1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	1	,	1	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	-1	,
          1	,	-1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	-1	,
          -1	,	-1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	1	,	-1	,	0	,	-1	,
          -1	,	1	,	0	,	0	,	1	,	-1	,	1	,	0	,	1	,
          1	,	-1	,	0	,	0	,	1	,	1	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	1	,	1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	-1	,	1	,	1	,	0	,	-1	,
          1	,	1	,	0	,	0	,	-1	,	-1	,	-1	,	0	,	1	,
          -1	,	-1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	-1	,	-1	,	1	,	0	,	1	,
          -1	,	1	,	0	,	0	,	1	,	-1	,	-1	,	0	,	1	,
          3	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          -3	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	3	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	-3	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	3	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	-3	,	0	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	3	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	-3	,	0	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	3	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	-3	,	0	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	3	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	-3	,	0	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	3	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	-3	,	0	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	3	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	-3	,	0	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,	3	,
          0	,	0	,	0	,	0	,	0	,	0	,	0	,	0	,	-3
        ),nrow=882,byrow=T)


        z<-m9%*%B
      }
    }
  }
  z=round(z,digits = 5)
  z=format(z,nsmall=5)
  z=noquote(z)
  message("Asymmetric Third Order Rotatable Design")
  return(z)
  # list=list("Asymmetric Third Order Rotatable Design"=z)
  # return(list)
}

