% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MNM_fit.R
\name{MNM_fit}
\alias{MNM_fit}
\title{Fit a Multi-Species N-Mixture (MNM) Model in Nimble}
\usage{
MNM_fit(
  Y = NULL,
  AR = FALSE,
  Hurdle = FALSE,
  Xp = NULL,
  Xn = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{Y}{Array of observed count data:
\itemize{
\item Dimensions for standard MNM and hurdle models: \eqn{R \times T \times S}
\item Dimensions for MNM with AR or both AR and hurdle components: \eqn{R \times T \times S \times K}
\item \code{R}: Number of sites
\item \code{T}: Number of replicates
\item \code{S}: Number of species
\item \code{K}: Number of time periods (required if \code{AR = TRUE})
}}

\item{AR}{Logical. Indicates whether to include an autocorrelation component in the model. Defaults to \code{FALSE}.}

\item{Hurdle}{Logical. Indicates whether to include a hurdle component in the model. Defaults to \code{FALSE}.}

\item{Xp}{An array of covariates affecting detection probability, with dimensions (R, S, P1), where:
\itemize{
\item \code{R}: Number of sites
\item \code{S}: Number of species
\item \code{P1}: Number of detection-related covariates
See examples for implementation details.
}}

\item{Xn}{An array of covariates affecting abundance, with dimensions (R, S, P2), where:
\itemize{
\item \code{R}: Number of sites.
\item \code{S}: Number of species.
\item \code{P2}: Number of abundance-related covariates.
}}

\item{verbose}{Control the level of output displayed during function execution. Default is TRUE.}

\item{...}{Additional arguments for prior distributions. Supported priors include:
\itemize{
\item \code{prior_detection_probability}, \code{prior_precision}, \code{prior_mean}, \code{prior_mean_AR}, \code{prior_sd_AR}, \code{prior_hurdle}.
\item Supported distributions include: \code{dnorm}, \code{dexp}, \code{dgamma}, \code{dbeta}, \code{dunif}, \code{dlnorm}, \code{dbern}, \code{dpois}, \code{dbinom}, \code{dcat}, \code{dmnorm}, \code{dwish}, \code{dchisq}, \code{dinvgamma}, \code{dt}, \code{dweib}, \code{ddirch}, \code{dmulti}, \code{dmvt}. Refer to the \href{https://r-nimble.org}{Nimble documentation} for details.
}}
}
\value{
An MNM object that contains the following components:
\itemize{
\item summary: Nimble model summary (mean, standard deviation, standard error, quantiles, effective sample size and Rhat value for all monitored values)
\item n_parameters: Number of parameters in the model (for use in calculating information criteria)
\item data: Observed abundances
\item fitted_Y: Predicted values for Y. Posterior predictive checks can be performed by comparing fitted_Y with the observed data.
\item logLik: Log-likelihood of the observed data (Y) given the model parameters.
\item n_converged: Number of parameters with successful convergence (Rhat < 1.1).
}
}
\description{
Fits a Multi-Species N-Mixture (MNM) model to observed count data using Nimble, with options to include autocorrelation (AR) and/or hurdle components for advanced ecological modeling.
}
\details{
This function implements the Bayesian MNM model to estimate latent species abundances and inter-species correlations based on observed count data. Extensions include options for incorporating autocorrelation (AR) to account for temporal dependencies and a hurdle model to handle zero inflation in the data. The input data and covariates must conform to specific dimensional requirements as described below.

The MNM model (Mimnagh et al., 2022) builds upon Royle's (2004) N-mixture model by allowing simultaneous modeling of multiple species, enabling inference about inter-species relationships and correlations.
}
\note{
Ensure that the dimensions of \code{Y}, \code{Xp}, and \code{Xn} match the requirements specified above. Mismatched dimensions will result in errors during model fitting.#'
}
\examples{
# Example 1: Fit a standard MNM model
Y <- array(data = rpois(60, lambda = 5), dim = c(3, 5, 4))  # Simulated counts
Xp <- array(data = rnorm(60), dim = c(3, 4, 2))  # Detection covariates
Xn <- array(data = rnorm(60), dim = c(3, 4, 2))  # Abundance covariates

\donttest{model <- MNM_fit(Y = Y, AR = FALSE, Hurdle = FALSE, Xp = Xp, Xn = Xn)}
# nimble creates auxiliary functions that may be removed after model
# run is complete using rm(list = ls(pattern = "^str"))

# Example 2: Fit an MNM model with AR-1 component
Y <- array(data = rpois(180, lambda = 5), dim = c(3, 5, 4, 3))  # Simulated counts
Xp <- array(data = rnorm(180), dim = c(3, 4, 3, 2))  # Detection covariates
Xn <- array(data = rnorm(180), dim = c(3, 4, 3, 2))  # Abundance covariates

\donttest{model <- MNM_fit(Y = Y, AR = TRUE, Hurdle = FALSE, Xp = Xp, Xn = Xn)}

# Example 3: Fit an MNM model with user-specified prior distributions
Y <- array(data = rpois(60, lambda = 5), dim = c(3, 5, 4))  # Simulated counts
Xp <- array(data = rnorm(60), dim = c(3, 4, 2))  # Detection covariates
Xn <- array(data = rnorm(60), dim = c(3, 4, 2))  # Abundance covariates

\donttest{model <- MNM_fit(Y = Y, AR = FALSE, Hurdle = TRUE, Xp = Xp, Xn = Xn,
                          prior_detection_probability="dnorm(0.01,0.01)")}
# Access traceplots and density plots
\donttest{tracePlot(model, "N[3, 1]")}
\donttest{density(model, "N[3, 1]")}

}
\references{
\itemize{
\item Royle, J. A. (2004). N-mixture models for estimating population size from spatially replicated counts. Biometrics, 60(1), 108-115.
\item Mimnagh, N., Parnell, A., Prado, E., & Moral, R. D. A. (2022). Bayesian multi-species N-mixture models for unmarked animal communities. Environmental and Ecological Statistics, 29(4), 755-778.
}
}
\seealso{
\itemize{
\item \code{simulateData}: For generating example datasets compatible with this function.
\item \code{MNM}: For details on creation of covariate arrays Xp and Xn.
}
}
