
#define TOLERANCE 0.0000000001 // added tolerance for cast to int

#include <Rcpp.h>
#include <algorithm> 

using namespace Rcpp;

// for test purposes
// [[Rcpp::export(name = ".kernelSmoothing")]]
NumericVector kernelSmoothing(const NumericVector &Y, const NumericVector &K) {
  int n = Y.size();
  int b = (K.size() - 1) / 2;
  double est, sumK;
  NumericVector ret = NumericVector(n);
  
  for (int i = 0; i < n; ++i) {
    est = 0.0;
    sumK = 0.0;
    
    for (int j = std::max(i - b, 0), k = std::max(0, b - i); j <= std::min(i + b, n - 1); ++j, ++k) {
      est += Y[j] * K[k];
      sumK += K[k];
    }
    ret[i] = est / sumK;
  }
  
  return ret;
}

// [[Rcpp::export(name = ".kernelSmoothingEpanechnikov")]]
NumericVector kernelSmoothingEpanechnikov(const NumericVector &Y, const double h) {
  int n = Y.size();
  double nh = n * h;
  double nh2 = nh * nh;
  int L = nh + TOLERANCE;
  double L2 = L * L;
  double L2p1 = (L + 1) * (L + 1);
  NumericVector ret = NumericVector(n);
  long double cumsumConstant = 0;
  long double cumsumLinear = 0;
  long double cumsumQuadratic = 0;
  long double denominator = 0;
  
  // initialize
  for (int i = 0; i <= L; ++i) {
    cumsumConstant += Y[i];
    cumsumLinear -= i * Y[i];
    cumsumQuadratic += i * i * Y[i];
    denominator += 1 - i * i / nh2;
  }
  ret[0] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  
  int i = 1;
  for ( ; i <= L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L];
    cumsumLinear = cumsumLinear + cumsumConstant - L * Y[i + L];
    cumsumConstant = cumsumConstant + Y[i + L];
    denominator += 1 - i * i / nh2;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  for ( ; i < n - L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L] - L2p1 * Y[i - 1 - L];
    cumsumLinear = cumsumLinear + cumsumConstant - L * Y[i + L] - (L + 1) * Y[i - 1 - L];
    cumsumConstant = cumsumConstant + Y[i + L] - Y[i - 1 - L];
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  int j = L;
  for ( ; i < n; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant - L2p1 * Y[i - 1 - L];
    cumsumLinear = cumsumLinear + cumsumConstant - (L + 1) * Y[i - 1 - L];
    cumsumConstant = cumsumConstant - Y[i - 1 - L];
    denominator -= 1 - j * j / nh2;
    --j;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  return ret;
}

// [[Rcpp::export(name = ".kernelSmoothingEpanechnikovCVleft")]]
NumericVector kernelSmoothingEpanechnikovCVleft(const NumericVector &Y, const double h) {
  int n = Y.size();
  double nh = n * h;
  double nh2 = nh * nh;
  int L = nh + 0.5 + TOLERANCE;
  
  double Lm05 = L - 0.5;
  double Lp05 = L + 0.5;
  double L2 = Lm05 * Lm05;
  double L2p1 = Lp05 * Lp05;
  NumericVector ret = NumericVector(n);
  double cumsumConstant = 0;
  double cumsumLinear = 0;
  double cumsumQuadratic = 0;
  double denominator = 0;
  
  // initialize
  for (int i = 0; i < L; ++i) {
    cumsumConstant += Y[i];
    cumsumLinear -= (i + 0.5) * Y[i];
    cumsumQuadratic += (i + 0.5) * (i + 0.5) * Y[i];
    denominator += 1 - (i + 0.5) * (i + 0.5) / nh2;
  }
  ret[0] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  
  int i = 1;
  for ( ; i <= L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L - 1];
    cumsumLinear = cumsumLinear + cumsumConstant - Lm05 * Y[i + L - 1];
    cumsumConstant = cumsumConstant + Y[i + L - 1];
    denominator += 1 - (i - 0.5) * (i - 0.5) / nh2;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  for ( ; i <= n - L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L - 1] - L2p1 * Y[i - 1 - L];
    cumsumLinear = cumsumLinear + cumsumConstant - Lm05 * Y[i + L - 1] - Lp05 * Y[i - 1 - L];
    cumsumConstant = cumsumConstant + Y[i + L - 1] - Y[i - 1 - L];
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  int j = L;
  for ( ; i < n; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant - L2p1 * Y[i - 1 - L];
    cumsumLinear = cumsumLinear + cumsumConstant - Lp05 * Y[i - 1 - L];
    cumsumConstant = cumsumConstant - Y[i - 1 - L];
    denominator -= 1 - (j - 0.5) * (j - 0.5) / nh2;
    --j;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  return ret;
}

// [[Rcpp::export(name = ".kernelSmoothingEpanechnikovCVright")]]
NumericVector kernelSmoothingEpanechnikovCVright(const NumericVector &Y, const double h) {
  int n = Y.size();
  double nh = n * h;
  double nh2 = nh * nh;
  int L = nh + 0.5 + TOLERANCE;

  double Lm05 = L - 0.5;
  double Lp05 = L + 0.5;
  double L2 = Lm05 * Lm05;
  double L2p1 = Lp05 * Lp05;
  NumericVector ret = NumericVector(n);
  double cumsumConstant = 0;
  double cumsumLinear = 0;
  double cumsumQuadratic = 0;
  double denominator = 0;
  
  // initialize
  for (int i = 0; i <= L; ++i) {
    cumsumConstant += Y[i];
    cumsumLinear -= (i - 0.5) * Y[i];
    cumsumQuadratic += (i - 0.5) * (i - 0.5) * Y[i];
    denominator += 1 - (i - 0.5) * (i - 0.5) / nh2;
  }
  ret[0] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  
  int i = 1;
  for ( ; i < L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L];
    cumsumLinear = cumsumLinear + cumsumConstant - Lm05 * Y[i + L];
    cumsumConstant = cumsumConstant + Y[i + L];
    denominator += 1 - (i + 0.5) * (i + 0.5) / nh2;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  for ( ; i < n - L; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant + L2 * Y[i + L] - L2p1 * Y[i - L];
    cumsumLinear = cumsumLinear + cumsumConstant - Lm05 * Y[i + L] - Lp05 * Y[i - L];
    cumsumConstant = cumsumConstant + Y[i + L] - Y[i - L];
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  int j = L;
  for ( ; i < n; ++i) {
    cumsumQuadratic = cumsumQuadratic + 2 * cumsumLinear + cumsumConstant - L2p1 * Y[i - L];
    cumsumLinear = cumsumLinear + cumsumConstant - Lp05 * Y[i - L];
    cumsumConstant = cumsumConstant - Y[i - L];
    denominator -= 1 - (j - 0.5) * (j - 0.5) / nh2;
    --j;
    ret[i] = (cumsumConstant - cumsumQuadratic / nh2) / denominator;
  }
  
  return ret;
}

// [[Rcpp::export(name = ".CVkernelSmoothing")]]
double CVkernelSmoothing(const NumericVector &Y, const NumericVector &K) {
  int n = Y.size();
  int b = K.size();
  double cv = 0.0, est, sumK;
  
  for (int i = 0; i < n; ++i) {
    est = 0.0;
    sumK = 0.0;
    
    for (int j = i - 1, k = 0; k < std::min(b, i); --j, ++k) {
      est += Y[j] * K[k];
      sumK += K[k];
    }
    
    for (int j = i + 1, k = 0; k < std::min(b, n - i - 1); ++j, ++k) {
      est += Y[j] * K[k];
      sumK += K[k];
    }
    cv += (est / sumK - Y[i]) * (est / sumK - Y[i]);
  }
  
  return cv;
}

// [[Rcpp::export(name = ".kernelSmoothingVfold")]]
NumericVector kernelSmoothingVfold(const NumericVector &Y, const NumericVector &K, int V, int fold) {
  int n = Y.size();
  int b = (K.size() - 1) / 2;
  double est, sumK;
  NumericVector ret = NumericVector(1 + (n - fold) / V);
  --fold;
  int index = 0;
  
  for (int i = fold; i < n; i += V) {
    est = 0.0;
    sumK = 0.0;
    
    for (int j = std::max(i - b, 0), k = std::max(0, b - i); j <= std::min(i + b, n - 1); ++j, ++k) {
      // if (j % V != fold) {
      est += Y[j] * K[k];
      sumK += K[k];
      // }
    }
    ret[index++] = est / sumK;
  }
  
  return ret;
}

// [[Rcpp::export(name = ".kernelSmoothingVfoldMJ")]]
NumericVector kernelSmoothingVfoldMJ(const NumericVector &Y, const NumericVector &K, const int lengthMJ) {
  int n = Y.size();
  int b = (K.size() - 1) / 2;
  double est, sumK;
  NumericVector ret = NumericVector(lengthMJ);
  int index = 0;
  
  for (int i = 0; i < n; ++i) {
    if (!R_IsNA(Y[i])) {
      est = 0.0;
      sumK = 0.0;
      
      for (int j = std::max(i - b, 0), k = std::max(0, b - i); j <= std::min(i + b, n - 1); ++j, ++k) {
        if (!R_IsNA(Y[j])) {
          est += Y[j] * K[k];
          sumK += K[k];
        }
      }
      ret[index++] = est / sumK;
    }
  }
  
  return ret;
}
