% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BDportfolio_optim.R
\name{BDportfolio_optim}
\alias{BDportfolio_optim}
\title{Portfolio Optimization by Benders decomposition}
\usage{
BDportfolio_optim(dat, portfolio_return,  
risk=c("CVAR", "DCVAR","LSAD","MAD"), alpha=0.95,  
Aconstr=NULL, bconstr=NULL, LB=NULL, UB=NULL, maxiter=500,tol=1e-8)
}
\arguments{
\item{dat}{Time series of returns data; dat = cbind(rr, pk), where \eqn{rr} is an array (time series) of asset returns,
for \eqn{n} returns and \eqn{k} assets it is an array with \eqn{\dim(rr) = (n, k)},     
\eqn{pk} is a vector of length \eqn{n} containing probabilities of returns.}

\item{portfolio_return}{Target portfolio return.}

\item{risk}{Risk measure chosen for optimization; one of "CVAR", "DCVAR", "LSAD", "MAD", where 
"CVAR" -- denotes Conditional Value-at-Risk (CVaR),
"DCVAR" -- denotes deviation CVaR,
"LSAD" -- denotes Lower Semi Absolute Deviation,
"MAD" -- denotes Mean Absolute Deviation.}

\item{alpha}{Value of alpha quantile used to compute portfolio VaR and CVaR; used also as quantile value for risk measures CVAR and DCVAR.}

\item{Aconstr}{Matrix defining additional constraints,   \eqn{\dim (Aconstr) = (m,k)}, where 
\eqn{k} -- number of assets, \eqn{m} -- number of constraints.}

\item{bconstr}{Vector defining additional constraints, length (\eqn{bconstr}) \eqn{ = m}.}

\item{LB}{Vector of length k, lower bounds of portfolio weights \eqn{\theta}; warning: condition LB = NULL is equivalent to LB = rep(0, k) (lower bound zero).}

\item{UB}{Vector of length k, upper bounds for portfolio weights \eqn{\theta}.}

\item{maxiter}{Maximal number of iterations.}

\item{tol}{Accuracy of computations, stopping rule.}
}
\value{
BDportfolio_optim returns a list with items: 
\tabular{llll}{ 
\code{return_mean} \tab vector of asset returns  mean values. \cr 

\code{mu} \tab  realized portfolio return.\cr 

\code{theta} \tab  portfolio weights.\cr 

\code{CVaR} \tab  portfolio CVaR.\cr 

\code{VaR} \tab  portfolio VaR.\cr 

\code{MAD} \tab  portfolio MAD.\cr 

\code{risk} \tab  portfolio risk measured by the risk measure chosen for optimization.\cr 

\code{new_portfolio_return} \tab  modified target portfolio return; when the original target portfolio return \cr 

\code{ } \tab is to high for the problem, the optimization problem is solved for \cr 

\code{ } \tab new_portfolio_return as the target return. \cr 
}
}
\description{
BDportfolio_optim is a linear program  for financial portfolio optimization. 
Portfolio risk is measured by one of the risk measures from the list  c("CVAR", "DCVAR", "LSAD", "MAD"). 
Benders decomposition method is explored to enable optimization for very large returns samples (\eqn{\sim 10^6}). 

The optimization problem is:\cr  
\eqn{\min F({\theta^{T}} r)} \cr  
over \cr 
\eqn{\theta^{T} E(r)} \eqn{\ge} \eqn{portfolio\_return}, \cr 
\eqn{LB} \eqn{\le \theta \le} \eqn{UB}, \cr 
\eqn{Aconstr} \eqn{\theta \le} \eqn{bconstr}, \cr 
where \cr 
\eqn{F} is a measure of risk; \cr 
\eqn{r} is a time series of returns of assets; \cr 
\eqn{\theta} is a vector of portfolio weights. \cr
}
\examples{

library (Rsymphony)  
library(Rglpk) 
library(mvtnorm)
k = 3 
num =100
dat <-  cbind(rmvnorm (n=num, mean = rep(0,k), sigma=diag(k)), matrix(1/num,num,1)) 
# a data sample with num rows and (k+1) columns for k assets; 
port_ret = 0.05 # target portfolio return 
alpha_optim = 0.95 

# minimal constraints set: \\eqn{\\sum \\theta_{i} = 1} 
# has to be in two inequalities: \\eqn{1 - \\epsilon <= \\sum \\theta_{i} <= 1 + \\epsilon} 
a0 <- rep(1,k) 
Aconstr <- rbind(a0,-a0) 
bconstr <- c(1+1e-8, -1+1e-8) 

LB <- rep(0,k) 
UB <- rep(1,k) 

res <- BDportfolio_optim(dat, port_ret, "CVAR", alpha_optim, 
Aconstr, bconstr, LB, UB, maxiter=200, tol=1e-8) 

cat ( c("Benders decomposition portfolio:\\n\\n")) 
cat(c("weights \\n")) 
print(res$theta) 

cat(c("\\n mean = ", res$mu, " risk  = ", res$risk, 
"\\n CVaR = ", res$CVaR, " VaR = ", res$VaR, "\\n MAD = ", res$MAD, "\\n\\n")) 
 

}
\references{
Benders, J.F.,   Partitioning procedures for solving mixed-variables programming problems. Number. Math., 4 (1962),  238--252, reprinted in 
Computational Management Science 2 (2005), 3--19. DOI: 10.1007/s10287-004-0020-y. 

Konno, H., Piecewise linear risk function and portfolio optimization, Journal of the Operations Research Society of Japan, 33 (1990), 139--156. 

Konno, H., Yamazaki, H.,  Mean-absolute deviation portfolio optimization model and its application to Tokyo stock market. Management Science, 37 (1991), 519--531. 

Konno, H., Waki, H.,   Yuuki, A., Portfolio optimization under lower partial risk measures, Asia-Pacific Financial Markets, 9 (2002), 127--140. DOI: 10.1023/A:1022238119491. 

Kunzi-Bay, A.,  Mayer, J., Computational aspects of minimizing conditional value at risk. Computational Management Science, 3 (2006), 3--27. DOI: 10.1007/s10287-005-0042-0. 

Rockafellar, R.T., Uryasev, S.,  Optimization of conditional value-at-risk. Journal of Risk, 2 (2000), 21--41. DOI: 10.21314/JOR.2000.038.  

Rockafellar, R. T., Uryasev, S.,  Zabarankin, M.,  Generalized deviations in risk analysis. Finance and Stochastics, 10 (2006), 51--74. DOI: 10.1007/s00780-005-0165-8.
}
