#' The cpam class
#'
#' @description
#' The \code{cpam} class stores data and analysis results for a time series
#' omics experiment. This object is generated by the \code{\link{prepare_cpam}}
#' function and contains all necessary data and parameters for downstream analysis.
#'
#' @details
#' A \code{cpam} object is a list with the following components:
#' \describe{
#'   \item{exp_design}{A data frame containing the experimental design information, with columns for 'sample', 'time', and potentially other variables.}
#'   \item{count_matrix_raw}{The original count matrix before filtering.}
#'   \item{count_matrix_filtered}{The count matrix after filtering low-count genes/transcripts.}
#'   \item{target_to_keep}{A vector of transcript/gene IDs that passed the filtering criteria.}
#'   \item{data_long}{A long-format data frame containing all relevant information for each target and sample.}
#'   \item{t2g}{A transcript-to-gene mapping data frame if provided.}
#'   \item{regularize}{Logical; whether empirical Bayes regularization of dispersions was used.}
#'   \item{overdispersion.prior}{Median overdispersion.}
#'   \item{model_type}{String; the type of design used, either "case-only" or "case-control".}
#'   \item{condition_var}{String; the column name in exp_design for the condition variable (for case-control models).}
#'   \item{case_value}{The value of condition_var that indicates the "case" in case-control models.}
#'   \item{bootstrap}{Logical; whether bootstrap samples (inferential replicates) were used.}
#'   \item{nboot}{The number of bootstrap samples used, if applicable.}
#'   \item{filter}{A list containing the filtering function and its arguments used.}
#'   \item{gene_level}{Logical; whether the analysis was performed at the gene level.}
#'   \item{aggregate_to_gene}{Logical; whether p-values should be aggregated from transcript to gene level.}
#'   \item{times}{An ordered vector of unique time points in the experimental design.}
#'   \item{num_cores}{The number of cores used for parallel computation.}
#'   \item{fixed_effects}{The formula for fixed effects in the model.}
#'   \item{intercept_cc}{String; the intercept type for case-control models.}
#'   \item{bss}{A vector of basis function types used for modelling.}
#' }
#'
#' @section Methods:
#' Objects of class \code{cpam} have print and summary methods available.
#'
#' @seealso \code{\link{prepare_cpam}} for creating a cpam object.
#'
#' @examples
#'
#' # load gene-only example data
#' load(system.file("extdata", "exp_design_example.rda", package = "cpam"))
#' load(system.file("extdata", "count_matrix_example.rda", package = "cpam"))
#'
#' # Create a cpam object with the example data
#' cpo <- prepare_cpam(exp_design = exp_design_example,
#'                     count_matrix = count_matrix_example,
#'                     gene_level = TRUE)
#'
#' # Print the object structure
#' cpo
#'
#' # Get a summary of the cpam object
#' summary(cpo)
#'
#' @name cpam-class
#' @aliases cpam
NULL



#' @export
print.cpam <- function(x, ...){
  cli::cli_h1("cpam object")
  cli::cli_ul()
  cli::cli_li(cli::style_dim("{x$model_type} time series"))
  cli::cli_li(cli::style_dim("{length(x$exp_design$sample)} samples"))
  cli::cli_li(cli::style_dim("{length(unique(x$exp_design$time))} time points"))
  if(x$gene_level) cli::cli_li(cli::style_dim("Counts aggregated for gene-level inference"))
  if(x$bootstrap){
    cli::cli_li(cli::style_dim("Overdispersion estimated using {x$nboot} inferential replicates"))
    cli::cli_li(cli::style_dim("Counts rescaled by estimated overdispersion"))
  }
}

#' @export
summary.cpam <- function(object, ...) {
  print(object)
  cli::cat_line()
  if(!(is.null(object[["p_table"]]))){
    cli::cli_alert_success("p-values computed")
  } else {
    cli::cli_alert("use {.fun compute_p_values()} to compute p-values")
  }
  if(!(is.null(object[["changepoints"]]))){
    cli::cli_alert_success("changepoints estimated for {.val {nrow(object$changepoints)}} targets")
  } else {
    cli::cli_alert("use {.fun estimate_changepoints()} to estimate changepoints")
  }
  if(!(is.null(object[["shapes"]]))){
    cli::cli_alert_success("shapes selected for {.val {nrow(object$shapes)}} targets")
  } else {
    cli::cli_alert("use {.fun select_shape()} to select among candidate shapes")
  }
}
