% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_bnb.r
\name{sim_bnb}
\alias{sim_bnb}
\title{Simulate data from a BNB distribution}
\usage{
sim_bnb(
  n,
  mean1,
  mean2,
  ratio,
  dispersion,
  nsims = 1L,
  return_type = "list",
  max_zeros = 0.99
)
}
\arguments{
\item{n}{(integer: \verb{[2, Inf)})\cr
The number(s) of paired observations.}

\item{mean1}{(numeric: \verb{(0, Inf)})\cr
The mean(s) of sample 1 \eqn{(\mu_1)}.}

\item{mean2, ratio}{(numeric: \verb{(0, Inf)})\cr
Only specify one of these arguments.
\itemize{
\item \code{mean2}: The mean(s) of sample 2 \eqn{(\mu_2)}.
\item \code{ratio}: The ratio(s) of means for sample 2 with respect to sample 1
\eqn{\left( r = \frac{\mu_2}{\mu_1} \right)}.
}

\code{mean2 = ratio * mean1}}

\item{dispersion}{(numeric: \verb{(0, Inf)})\cr
The gamma distribution shape parameter(s) \eqn{(\theta)} which control
the dispersion and the correlation between sample 1 and 2.
See 'Details' and 'Examples'.}

\item{nsims}{(Scalar integer: \code{1L}; \verb{[1, Inf)})\cr
The expected number of simulated data sets. If \code{nsims > 1}, the data
is returned in a list-column of a depower simulation data frame.
\code{nsims} may be reduced depending on \code{max_zeros}.}

\item{return_type}{(string: \code{"list"}; \code{c("list", "data.frame")})\cr
The data structure of the simulated data. If \code{"list"} (default), a
list object is returned. If \code{"data.frame"} a data frame in tall format
is returned. The list object provides computational efficiency and the
data frame object is convenient for formulas. See 'Value'.}

\item{max_zeros}{(Scalar numeric: \code{0.99}; \verb{[0, 1]})\cr
The maximum proportion of zeros each group in a simulated dataset is
allowed to have. If the proportion of zeros is greater than this
value, the corresponding data is excluded from the set of simulations.
This is most likely to occur when the sample size is small and the
dispersion parameter is small.}
}
\value{
If \code{nsims = 1} and the number of unique parameter combinations is
one, the following objects are returned:
\itemize{
\item If \code{return_type = "list"}, a list:
\tabular{lll}{
Slot \tab Name \tab Description \cr
1 \tab \tab Simulated counts from sample 1. \cr
2 \tab \tab Simulated counts from sample 2.
}
\item If \code{return_type = "data.frame"}, a data frame:
\tabular{lll}{
Column \tab Name \tab Description \cr
1 \tab \code{item} \tab Subject/item indicator. \cr
2 \tab \code{condition} \tab Sample/condition indicator. \cr
3 \tab \code{value} \tab Simulated counts.
}
}

If \code{nsims > 1} or the number of unique parameter combinations is greater than
one, each object described above is returned in a list-column named \code{data} in
a depower simulation data frame:
\tabular{lll}{
Column \tab Name \tab Description \cr
1 \tab \code{n1} \tab Sample size of sample 1. \cr
2 \tab \code{n2} \tab Sample size of sample 2. \cr
3 \tab \code{mean1} \tab Mean for sample 1. \cr
4 \tab \code{mean2} \tab Mean for sample 2. \cr
5 \tab \code{ratio} \tab Ratio of means (sample 2 / sample 1). \cr
6 \tab \code{dispersion} \tab Gamma distribution shape parameter (dispersion). \cr
7 \tab \code{nsims} \tab Number of valid simulation iterations. \cr
8 \tab \code{distribution} \tab Distribution sampled from. \cr
9 \tab \code{data} \tab List-column of simulated data.
}
}
\description{
Simulate data from the bivariate negative binomial (BNB) distribution. The
BNB distribution is used to simulate count data where the event counts are
jointly dependent (correlated). For independent data, see
\code{\link[=sim_nb]{sim_nb()}}.
}
\details{
The negative binomial distribution may be defined using a gamma-Poisson
mixture distribution. In this case, the Poisson parameter \eqn{\lambda}
is a random variable with gamma distribution. Equivalence between different
parameterizations are demonstrated below:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Define constants and their relationships
n <- 10000
dispersion <- 8
mu <- 4
p <- dispersion / (dispersion + mu)
q <- mu / (mu + dispersion)
variance <- mu + (mu^2 / dispersion)
rate <- p / (1 - p)
scale <- (1 - p) / p

# alternative formula for mu
mu_alt <- (dispersion * (1 - p)) / p
stopifnot(isTRUE(all.equal(mu, mu_alt)))

set.seed(20240321)

# Using built-in rnbinom with dispersion and mean
w <- rnbinom(n = n, size = dispersion, mu = mu)

# Using gamma-Poisson mixture with gamma rate parameter
x <- rpois(
  n = n,
  lambda = rgamma(n = n, shape = dispersion, rate = rate)
)

# Using gamma-Poisson mixture with gamma scale parameter
y <- rpois(
  n = n,
  lambda = rgamma(n = n, shape = dispersion, scale = scale)
)

# Using gamma-Poisson mixture with multiplicative mean and
# gamma scale parameter
z <- rpois(
  n = n,
  lambda = mu * rgamma(n = n, shape = dispersion, scale = 1/dispersion)
)

# Compare CDFs
par(mar=c(4,4,1,1))
plot(
  x = sort(w),
  y = (1:n)/n,
  xlim = range(c(w,x,y,z)),
  ylim = c(0,1),
  col = 'green',
  lwd = 4,
  type = 'l',
  main = 'CDF'
)
lines(x = sort(x), y = (1:n)/n, col = 'red', lwd = 2)
lines(x = sort(y), y = (1:n)/n, col = 'yellow', lwd = 1.5)
lines(x = sort(z), y = (1:n)/n, col = 'black')
}\if{html}{\out{</div>}}

\if{html}{\out{<div style="display: flex; justify-content: center; padding-top: 10px; padding-bottom: 10px;">
  <img style="max-width: 400px; width: 100\%; height: auto;" src="figures/gamma_poisson_mixture-1.png" alt="Gamma-Poisson Mixture CDF" />
</div>
<br>}}
\if{latex}{
  \out{\begin{center}}
  \figure{gamma_poisson_mixture-1.png}{options: width=3in}
  \out{\end{center}}
}

The BNB distribution is implemented by compounding two conditionally
independent Poisson random variables \eqn{X_1 \mid G = g \sim \text{Poisson}(\mu g)}
and \eqn{X_2 \mid G = g \sim \text{Poisson}(r \mu g)} with a gamma random
variable \eqn{G \sim \text{Gamma}(\theta, \theta^{-1})}. The probability mass
function for the joint distribution of \eqn{X_1, X_2} is

\deqn{
P(X_1 = x_1, X_2 = x_2) = \frac{\Gamma(x_1 + x_2 + \theta)}{(\mu + r \mu + \theta)^{x_1 + x_2 + \theta}}
  \frac{\mu^{x_1}}{x_1!} \frac{(r \mu)^{x_2}}{x_2!}
  \frac{\theta^{\theta}}{\Gamma(\theta)}
}

where \eqn{x_1,x_2 \in \mathbb{N}^{\geq 0}} are specific values of the count
outcomes, \eqn{\theta \in \mathbb{R}^{> 0}} is the \code{dispersion} parameter
which controls the dispersion and level of correlation between the two
samples (otherwise known as the shape parameter of the gamma distribution),
\eqn{\mu \in \mathbb{R}^{> 0}} is the mean parameter, and
\eqn{r = \frac{\mu_2}{\mu_1} \in \mathbb{R}^{> 0}} is the \code{ratio} parameter
representing the multiplicative change in the mean of the second sample
relative to the first sample. \eqn{G} denotes the random subject effect and
the gamma distribution scale parameter is assumed to be the inverse of the
dispersion parameter (\eqn{\theta^{-1}}) for identifiability.

Correlation decreases from 1 to 0 as the dispersion parameter increases from
0 to infinity. For a given dispersion, increasing means also increases the
correlation. See 'Examples' for a demonstration.

See 'Details' in \code{\link[=sim_nb]{sim_nb()}} for additional information on the
negative binomial distribution.
}
\examples{
#----------------------------------------------------------------------------
# sim_bnb() examples
#----------------------------------------------------------------------------
library(depower)

# Paired two-sample data returned in a data frame
sim_bnb(
  n = 10,
  mean1 = 10,
  ratio = 1.6,
  dispersion = 3,
  nsims = 1,
  return_type = "data.frame"
)

# Paired two-sample data returned in a list
sim_bnb(
  n = 10,
  mean1 = 10,
  ratio = 1.6,
  dispersion = 3,
  nsims = 1,
  return_type = "list"
)

# Two simulations of paired two-sample data
# returned as a list of data frames
sim_bnb(
  n = 10,
  mean1 = 10,
  ratio = 1.6,
  dispersion = 3,
  nsims = 2,
  return_type = "data.frame"
)

# Two simulations of Paired two-sample data
# returned as a list of lists
sim_bnb(
  n = 10,
  mean1 = 10,
  ratio = 1.6,
  dispersion = 3,
  nsims = 2,
  return_type = "list"
)

#----------------------------------------------------------------------------
# Visualization of the BNB distribution as dispersion varies.
#----------------------------------------------------------------------------
set.seed(1234)
data <- lapply(
  X = c(1, 10, 100, 1000),
  FUN = function(x) {
    d <- sim_bnb(
      n = 1000,
      mean1 = 10,
      ratio = 1.5,
      dispersion = x,
      nsims = 1,
      return_type = "data.frame"
    )
    cor <- cor(
      x = d[d$condition == "1", ]$value,
      y = d[d$condition == "2", ]$value
    )
    cbind(dispersion = x, correlation = cor, d)
  }
)

data <- do.call(what = "rbind", args = data)

ggplot2::ggplot(
  data = data,
  mapping = ggplot2::aes(x = value, fill = condition)
) +
  ggplot2::facet_wrap(
    facets = ggplot2::vars(.data$dispersion),
    ncol = 2,
    labeller = ggplot2::labeller(.rows = ggplot2::label_both)
  ) +
  ggplot2::geom_density(alpha = 0.3) +
  ggplot2::coord_cartesian(xlim = c(0, 60)) +
  ggplot2::geom_text(
    mapping = ggplot2::aes(
      x = 30,
      y = 0.12,
      label = paste0("Correlation: ", round(correlation, 2))
    ),
    check_overlap = TRUE
  ) +
  ggplot2::labs(
    x = "Value",
    y = "Density",
    fill = "Condition",
    caption = "Mean1=10, Mean2=15, ratio=1.5"
  )

}
\references{
\insertRef{yu_2020}{depower}

\insertRef{rettiganti_2012}{depower}

\insertRef{aban_2009}{depower}
}
\seealso{
\code{\link[=sim_nb]{sim_nb()}}
}
