% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transformers.R
\name{epoxy_transform}
\alias{epoxy_transform}
\alias{epoxy_transform_get}
\alias{epoxy_transform_set}
\title{epoxy Transformers}
\usage{
epoxy_transform(..., engine = NULL, syntax = lifecycle::deprecated())

epoxy_transform_get(engine = c("md", "html", "latex"), inline = FALSE)

epoxy_transform_set(..., engine = NULL, syntax = lifecycle::deprecated())
}
\arguments{
\item{...}{Transformer functions, e.g.
\link{epoxy_transform_bold} or the name of an \pkg{epoxy}
transform function, e.g. \code{"bold"}, or a call to a transform function, e.g.
\code{\link[=epoxy_transform_bold]{epoxy_transform_bold()}}. \code{epoxy_transform()} chains the transformer
functions together, applying the transformers in order from first to last.

For example, \code{epoxy_transform("bold", "collapse")} results in replaced
strings that are emboldened \emph{and then} collapsed, e.g. \verb{**a** and **b**}.
On the other hand, \code{epoxy_transform("collapse", "bold")}  will collapse the
vector \emph{and then} embolden the entire string.

In \code{epoxy_transform_apply()}, the \code{...} are passed to the underlying call
the underlying function call.

In \code{epoxy_transform_collapse()}, the \code{...} are ignored.}

\item{engine}{One of \code{"markdown"} (or \code{"md"}), \code{"html"}, or \code{"latex"}. The
default is chosen based on the engine of the chunk where the transform
function is called, or according to the option \code{epoxy.engine}. Caution:
invalid options are silently ignored, falling back to \code{"markdown"}.}

\item{syntax}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{engine} instead.}

\item{inline}{In \code{epoxy_transform_get()}, whether to return the
session-specific inline formatting functions for
\code{\link[=epoxy_transform_inline]{epoxy_transform_inline()}}.}
}
\value{
A function of \code{text} and \code{envir} suitable for the \code{.transformer} argument of
\code{\link[glue:glue]{glue::glue()}}.
}
\description{
These transformers provide additional automatic formatting for the template
strings. They are designed to be used with the \code{.transformer} chunk option of
in \code{epoxy} chunks. You can use \code{epoxy_transform()} to chain several
transformers together. \code{epoxy_transform()} and individual \pkg{epoxy}
transform functions can be used in \code{epoxy}, \code{epoxy_html} and \code{epoxy_latex}
chunks and will choose the correct engine for each.
}
\section{Functions}{
\itemize{
\item \code{epoxy_transform()}: Construct a chained transformer using \pkg{epoxy}
transformers for use as a glue transformer. The resulting transformers can
be passed to the \code{.transformer} argument of \code{\link[=epoxy]{epoxy()}} or \code{\link[glue:glue]{glue::glue()}}.

\item \code{epoxy_transform_get()}: Get the default epoxy \code{.transformer} for all
epoxy engines or for a subset of engines.

\item \code{epoxy_transform_set()}: Set the default epoxy \code{.transformer} for all
epoxy engines or for a subset of engines.

}}
\section{Output-specific transformations}{

The \code{epoxy_transform_} functions will attempt to use the correct engine for
transforming the replacement text for markdown, HTML and LaTeX. This choice
is driven by the chunk engine where the transformer function is used. The
\code{epoxy} engine corresponds to markdown, \code{epoxy_html} to HTML, and
\code{epoxy_latex} to LaTeX.

Automatic engine selection only works when the epoxy transform functions are
used with epoxy knitr engines and during the knitr rendering process. When
used outside of this context, you can choose the desired engine by setting
the \code{engine} to one of \code{"markdown"}, \code{"html"} or \code{"latex"}.
}

\section{Session-wide settings}{


To change the transformer used by \code{\link[=epoxy]{epoxy()}} and the HTML and LaTeX variants, use \code{epoxy_transform_set()}.
This function takes the same values as \code{\link[=epoxy_transform]{epoxy_transform()}}, but makes them the default transformer for any \code{\link[=epoxy]{epoxy()}} calls that do not specify a transformer.
By default, the setting is made for all engines, but you can specify a single engine with the \code{engine} argument.

Here's a small example that applies the \link[=epoxy_transform_bold]{bold} and \link[=epoxy_transform_collapse]{collapse} transformers to all epoxy chunks:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{epoxy_transform_set("bold", "collapse")
}\if{html}{\out{</div>}}

Most often, you'll want to to update the default transformer to customize the formatting functions used by the \link[=epoxy_transform_inline]{inline transformer}.
You can use \code{epoxy_transform_set()} to change settings of existing formatting functions or to add new one.
Pass the new function to an argument with the dot-prefixed name.

In the next example I'm setting the \code{.dollar} transformation to use "K" and "M" to abbreviate large numbers.
I'm also adding my own transformation that truncates long strings to fit in 8 characters.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{epoxy_transform_set(
	.dollar = scales::label_dollar(
		accuracy = 0.01,
		scale_cut = scales::cut_short_scale()
	),
	.trunc8 = function(x) glue::glue_collapse(x, width = 8)
)

epoxy("\{.dollar 12345678\}")
#> $12.34M
epoxy("\{.trunc8 12345678\}")
#> 12345...
}\if{html}{\out{</div>}}

Note that the \code{engine} argument can be used even with inline tranformations, e.g. to apply a change only for HTML you can use \code{engine = "html"}.

To unset the session defaults, you have two options:
\enumerate{
\item Unset everything by passing \code{NULL} to \code{epoxy_transform_set()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{epoxy_transform_set(NULL)
}\if{html}{\out{</div>}}
\item Unset a single inline transformation by passing \code{rlang::zap()} to the named argument:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{epoxy_transform_set(.dollar = rlang::zap())
}\if{html}{\out{</div>}}
}

Or you can provide new values to overwrite the current settings.
And as before, you can unset session defaults for a specific \code{engine}.
}

\examples{
epoxy("{.strong {.and letters[1:3]}}")
epoxy("{.and {.strong letters[1:3]}}")

# If you used the development version of epoxy, the above is equivalent to:
epoxy("{letters[1:3]&}", .transformer = epoxy_transform("bold", "collapse"))
epoxy("{letters[1:3]&}", .transformer = epoxy_transform("collapse", "bold"))

# In an epoxy_html chunk...
epoxy_html("{{.strong {{.or letters[1:3]}} }}")

# Or in an epoxy_latex chunk...
epoxy_latex("<<.and <<.strong letters[1:3]>> >>")

# ---- Other Transformers ----

# Format numbers with an inline transformation
amount <- 123.4234234
epoxy("{.number amount}")
epoxy(
  "{.number amount}",
  .transformer = epoxy_transform_inline(
    .number = scales::label_number(accuracy = 0.01)
  )
)

# Apply _any_ function to all replacements
epoxy(
  "{amount} is the same as {amount}",
  .transformer = epoxy_transform_apply(round, digits = 0)
)

epoxy(
  "{amount} is the same as {amount}",
  .transformer = epoxy_transform(
    epoxy_transform_apply(~ .x * 100),
    epoxy_transform_apply(round, digits = 2),
    epoxy_transform_apply(~ paste0(.x, "\%"))
  )
)
}
\seealso{
Other epoxy's glue transformers: 
\code{\link{epoxy_transform_html}()},
\code{\link{epoxy_transform_inline}()}
}
\concept{epoxy's glue transformers}
