% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factorH-package.R
\docType{package}
\name{factorH}
\alias{factorH}
\alias{factorH-package}
\title{factorH: Multifactor rank-based ANOVA utilities}
\description{
Multifactor nonparametric analysis of variance based on ranks. Builds on the Kruskal-Wallis H test and its 2x2 Scheirer-Ray-Hare extension to handle any factorial designs. Provides effect sizes, Dunn-Bonferroni pairwise-comparison matrices, and simple-effects analyses. Tailored for psychology and the social sciences, with beginner-friendly R syntax and outputs that can be dropped into journal reports. Includes helpers to export tab-separated results and compact tables of descriptive statistics (to APA-style reports).
}
\details{
\subsection{What this package does (and why)}{

\emph{factorH} provides a \strong{simple, single-call workflow} for \strong{multifactor
nonparametric, rank-based ANOVA} and publication-ready outputs:
\itemize{
\item ANOVA-like table based on \code{ranks} (rooted in \strong{Kruskal-Wallis H} and
the 2x2 \strong{Scheirer-Ray-Hare} extension),
\item \strong{effect sizes} computed directly from H
\item \strong{Dunn–Bonferroni} post hoc \strong{comparison matrices}
\item \code{simple-effects} post hocs (pairwise comparisons \strong{within levels} of
conditioning factors),
\item compact \strong{descriptive tables} and a \strong{TSV writer} for quick
formatting in Excel or a manuscript.
}

Why? Popular GUI stats tools do not offer a ready-made, \strong{user-friendly
multifactor rank-based} pipeline that mirrors standard H / SRH analyses
in a way that is easy for beginners. factorH aims to fill that gap with
\strong{clear, R-like formula syntax} and a \code{one-command} report function.

The package is intentionally small: most users will only ever need:
\itemize{
\item srh.kway.full(…) to compute everything
\item write.srh.kway.full.tsv(…) to export the results into a single
tab-separated file.
}
}

\subsection{Formula syntax at a glance}{

All high-level functions use standard R \strong{model formulas}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{response ~ factorA + factorB + factorC
}\if{html}{\out{</div>}}

lists \strong{main effects} - Interactions are handled internally. You do
\code{not} need to write A:B or A*B. The \code{response} (left of ~) must be
\code{numeric} (e.g., a Likert score coded as 1..5 stored as numeric).

Examples below use the included dataset mimicry.

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(factorH)
data(mimicry, package = "factorH")
str(mimicry)
}\if{html}{\out{</div>}}

Predictors should be factors. If not, functions will coerce them.

\strong{What is allowed?}

\if{html}{\out{<div class="sourceCode">}}\preformatted{# One factor (KW-style):  
  liking ~ condition

# Two factors (SRH-style):  
  liking ~ gender + condition

# Three or more factors (k-way):  
  liking ~ gender + condition + age_cat
}\if{html}{\out{</div>}}

You do \code{not} need to write gender:condition or gender*condition. The
package will build all needed interactions internally when relevant.
}

\subsection{Numeric response (Likert note)}{

The response must be \code{numeric}. For Likert-type items (e.g., 1 =
strongly disagree … 5 = strongly agree), keep them numeric; rank-based
tests are robust for such \code{ordinal-like} data.

If your Likert is accidentally a \code{factor} or \code{character}, coerce safely:

\if{html}{\out{<div class="sourceCode">}}\preformatted{# if stored as character "1","2",...:
mimicry$liking <- as.numeric(mimicry$liking)
# if stored as factor with labels "1","2",...:
mimicry$liking <- as.numeric(as.character(mimicry$liking))
}\if{html}{\out{</div>}}
}

\subsection{The one-call pipeline}{

The main function srh.kway.full() runs:
\enumerate{
\item ANOVA-like table on ranks
\item descriptive summary
\item post hoc matrices (Dunn–Bonferroni; P.adj)
\item simple-effects post hocs (within-family Bonferroni).
}

For 2 factors:

\if{html}{\out{<div class="sourceCode">}}\preformatted{res2 <- srh.kway.full(liking ~ gender + condition, data = mimicry)
names(res2)
res2$anova
head(res2$summary)
names(res2$posthoc_cells)
names(res2$posthoc_simple)[1:4]
}\if{html}{\out{</div>}}

For 3 factors:

\if{html}{\out{<div class="sourceCode">}}\preformatted{res3 <- srh.kway.full(liking ~ gender + condition + age_cat, data =
mimicry)
res3$anova
}\if{html}{\out{</div>}}

\strong{Export to a tab-separated file}

\if{html}{\out{<div class="sourceCode">}}\preformatted{f <- tempfile(fileext = ".tsv")
write.srh.kway.full.tsv(res3, file = f, dec = ".") # decimal dot
file.exists(f)
}\if{html}{\out{</div>}}

If you need \code{comma} as decimal mark:

\if{html}{\out{<div class="sourceCode">}}\preformatted{f2 <- tempfile(fileext = ".tsv")
write.srh.kway.full.tsv(res3, file = f2, dec = ",") # decimal comma
file.exists(f2)
}\if{html}{\out{</div>}}

The TSV contains clearly separated sections:
## SRH: EFFECTS TABLE, ## SUMMARY STATS, ## POSTHOC CELLS, ## SIMPLE
EFFECTS, ## META.
}

\subsection{What is in the example dataset?}{

mimicry is a real study on the \strong{chameleon effect (Trzmielewska, Duras,
Juchacz & Rak, 2025)}: how \code{mimicry} vs other \strong{movement conditions}
affect \code{liking} of an interlocutor. Potential moderators include
\code{gender} and \code{age} (with dichotomized age_cat, and a 3-level age_cat2).
This makes it a natural playground for \code{multifactor} rank-based
analyses.

\if{html}{\out{<div class="sourceCode">}}\preformatted{table(mimicry\$condition)
table(mimicry\$gender)
table(mimicry\$age_cat)
}\if{html}{\out{</div>}}
}

\subsection{What the functions compute (high level)}{
\itemize{
\item \strong{srh.kway()}: rank-based k-way ANOVA table using Type II SS on
ranks; p-values are tie-corrected; H is reported with and without the
correction factor; effect sizes from unadjusted H.
\item \strong{srh.effsize()}: 2-way SRH table with effect sizes (eta2H, eps2H)
computed from H.
\item \strong{nonpar.datatable()}: compact descriptive tables with \strong{global
ranks} (means of ranks per cell), medians, quartiles, IQR, etc., for
all main effects and interactions.
\item \strong{srh.posthocs()}: Dunn–Bonferroni \strong{pairwise matrices} (P.adj) for
\strong{all effects} (main and interactions).
\item \strong{srh.simple.posthoc() / srh.simple.posthocs()}: \code{simple-effects}
pairwise comparisons \strong{within levels} of conditioning factors
(SPSS-like “within” scope by default).
\item \strong{srh.kway.full()}: orchestrates all of the above.
\item \strong{write.srh.kway.full.tsv()}: exports everything into one TSV (with
dot or comma decimal mark).
}

That is it. For most users, the intro ends here: \strong{use srh.kway.full()}
and export with \strong{write.srh.kway.full.tsv()}.
}
}
\author{
\strong{Maintainer}: Tomasz Rak \email{tomasz.rak@upjp2.edu.pl}

Authors:
\itemize{
  \item Szymon Wrzesniowski \email{szymon.wrzesniowski@upjp2.edu.pl}
}

}
\keyword{package}
