% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/position-.R, R/position-dodge.R,
%   R/position-dodge2.R, R/position-identity.R, R/position-jitter.R,
%   R/position-jitterdodge.R, R/position-nudge.R, R/position-stack.R
\docType{data}
\name{Position}
\alias{Position}
\alias{PositionDodge}
\alias{PositionDodge2}
\alias{PositionIdentity}
\alias{PositionJitter}
\alias{PositionJitterdodge}
\alias{PositionNudge}
\alias{PositionStack}
\alias{PositionFill}
\title{Positions}
\description{
All \verb{position_*()} functions (like \code{position_dodge()}) return a
\verb{Position*} object (like \code{PositionDodge}). The \verb{Position*}
object is responsible for adjusting the position of overlapping geoms.
}
\details{
The way that the \verb{position_*} functions work is slightly different from
the \verb{geom_*} and \verb{stat_*} functions, because a \verb{position_*}
function actually "instantiates" the \verb{Position*} object by creating a
descendant, and returns that. The object is chaperoned by the \link{Layer} class.

To create a new type of Position object, you typically will want to override
one or more of the following:
\itemize{
\item The \code{required_aes} and \code{default_aes} fields.
\item The \code{setup_params()} and \code{setup_data()} methods.
\item One of the \code{compute_layer()} or \code{compute_panel()} methods.
}
}
\section{Fields}{

\describe{
\item{\code{required_aes}}{A character vector naming aesthetics that are necessary
to compute the position adjustment.}

\item{\code{default_aes}}{A \link[=aes]{mapping} of default values for aesthetics.}

\item{\code{use_defaults}}{\strong{Description}

A function method for completing the layer data by filling in default
position aesthetics that are not present. These can come from two sources:
either from the layer parameters as static, unmapped aesthetics or from
the \code{default_aes} field.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$use_defaults(data, params)
}\if{html}{\out{</div>}}

\strong{Arguments}
\describe{
\item{\code{data}}{A data frame of the layer's data}
\item{\code{params}}{A list of fixed aesthetic parameters}
}

\strong{Value}

A data frame with completed layer data}

\item{\code{setup_params}}{\strong{Description}

A function method for modifying or checking the parameters based on the
data. The default method returns an empty list.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$setup_params(data)
}\if{html}{\out{</div>}}

\strong{Arguments}
\describe{
\item{\code{data}}{A data frame with the layer's data.}
}

\strong{Value}

A list of parameters}

\item{\code{setup_data}}{\strong{Description}

A function method for modifying or checking the data. The default method
checks for the presence of required aesthetics.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$setup_data(data, params)
}\if{html}{\out{</div>}}

\strong{Arguments}
\describe{
\item{\code{data}}{A data frame with the layer's data.}
\item{\code{params}}{A list of parameters coming from the \code{setup_params()}
method}
}

\strong{Value}

A data frame with layer data}

\item{\code{compute_layer}}{\strong{Description}

A function method orchestrating the position adjust of the entire layer.
The default method splits the data and passes on adjustment tasks to the
panel-level \code{compute_panel()}. In addition, it finds the correct scales
in the layout object to pass to the panel computation.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$compute_layer(data, params, layout)
}\if{html}{\out{</div>}}

\strong{Arguments}
\describe{
\item{\code{data}}{A data frame with the layer's data.}
\item{\code{params}}{A list of parameters coming from the \code{setup_params()}
method}
\item{\code{layout}}{A \verb{<Layout>} ggproto object.}
}

\strong{Value}

A data frame with layer data}

\item{\code{compute_panel}}{\strong{Description}

A function method executing the position adjustment at the panel level.
The default method is not implemented.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$compute_panel(data, params, scales)
}\if{html}{\out{</div>}}

\strong{Arguments}
\describe{
\item{\code{data}}{A data frame with the layer's data.}
\item{\code{params}}{A list of parameters coming from the \code{setup_params()}
method}
\item{\code{scales}}{A list of pre-trained \code{x} and \code{y} scales. Note that the
position scales are not finalised at this point and reflect the initial
data range before computing stats.}
}

\strong{Value}

A data frame with layer data}

\item{\code{aesthetics}}{\strong{Description}

A function method for listing out custom position aesthetics for this
position adjustment.

\strong{Usage}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{Position$aesthetics()
}\if{html}{\out{</div>}}

\strong{Value}

A character vector of aesthetic names.}
}}

\section{Convention}{


The object name that a new class is assigned to is typically the same as the
class name. Position class name are in UpperCamelCase and start with the
\verb{Position*} prefix, like \code{PositionNew}.

A constructor functions is usually paired with a Position class. The
constructor copies the position class and populates parameters. The
constructor function name is formatted by taking the Position class name and
formatting it with snake_case, so that \code{PositionNew} becomes \code{position_new()}.
}

\examples{
# Extending the class
PositionRank <- ggproto(
  "PositionRank", Position,
  # Fields
  required_aes = c("x", "y"),
  # Methods
  setup_params = function(self, data) list(width = self$width),
  compute_panel = function(data, params, scales) {
    width <- params$width
    if (is.null(width)) {
      width <- resolution(data$x, zero = FALSE, TRUE) * 0.4
    }
    rank   <- stats::ave(data$y, data$group, FUN = rank)
    rank   <- scales::rescale(rank, to = c(-width, width) / 2)
    data$x <- data$x + rank
    data
  }
)

# Building a constructor
position_rank <- function(width = NULL) {
  ggproto(NULL, PositionRank, width = width)
}

# Use new position in plot
ggplot(mpg, aes(drv, displ)) +
  geom_point(position = position_rank(width = 0.5))
}
\seealso{
The \href{https://ggplot2-book.org/extensions#new-positions}{new positions section} of the online ggplot2 book.

Other Layer components: 
\code{\link{Geom}},
\code{\link{Layer-class}},
\code{\link{Stat}}
}
\concept{Layer components}
\keyword{datasets}
\keyword{internal}
