% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hasselayout.R
\encoding{UTF-8}
\name{hasselayout}
\alias{hasselayout}
\title{Hasse diagram of the layout structure}
\usage{
hasselayout(
  datadesign,
  randomfacsid = NULL,
  showLS = "Y",
  showpartialLS = "Y",
  showdfLS = "Y",
  check.confound.df = "Y",
  maxlevels.df = "Y",
  table.out = "N",
  pdf = "N",
  example = "example",
  outdir = NULL,
  hasse.font = "sans",
  produceBWPlot = "N",
  structural.colour = "grey",
  structural.width = 2,
  partial.colour = "orange",
  partial.width = 1.5,
  objects.colour = "mediumblue",
  df.colour = "red",
  larger.fontlabelmultiplier = 1,
  middle.fontlabelmultiplier = 1,
  smaller.fontlabelmultiplier = 1
)
}
\arguments{
\item{datadesign}{A data frame, list or environment (or object coercible by \code{\link[base]{as.data.frame}} to a data frame) containing the factors in the experimental design. The data frame should \strong{only} include the factors/columns that the user wants to include in the Hasse diagram. All factors are treated as categorical. Moreover, the first two letters of factor names are used for interactions between factors so it is advised that these be unique.}

\item{randomfacsid}{An optional vector specifying whether the factors are defined as fixed (entry = 0) or random (entry = 1). The default choice is NULL and the function automatically sets all entries to 0. The length of the vector should be equal to the number of factors in the design, i.e., the length of the vector should be equal to the number of columns of the argument \code{datadesign}.}

\item{showLS}{logical. If "N" then generation of the Hasse diagram is suppressed. The default is "Y".}

\item{showpartialLS}{logical. If "N" then the partial crossing between structural objects (using dotted connecting lines) is not illustrated on the Hasse diagram of the layout structure. The default is "Y".}

\item{showdfLS}{logical. If "N" then the structural object label is not displayed on the Hasse diagram of the layout structure. The default is "Y".}

\item{check.confound.df}{logical. If "N" then the check for confounded degrees of freedom is not performed. The default is "Y".}

\item{maxlevels.df}{logical. If "N" then the potential maximum number of levels of a generalised factor is removed from the structural object label on the Hasse diagram of the layout structure. The default is "Y".}

\item{table.out}{logical. If "Y" then a table that shows the relationships between the structural objects in the layout structure is printed. The default is "N".}

\item{pdf}{logical. If "Y" then a pdf file containing the Hasse diagram of the layout structure is generated. The default is "N", i.e., a pdf file is not generated.}

\item{example}{File name for the pdf output file containing the Hasse diagram. The default is set to "example".}

\item{outdir}{Location of the pdf output file if \code{pdf="Y"}. The default is set to \code{NULL} and in this case the pdf output file containing the Hasse diagram output is stored to a temporary file. To specify a permanent location this argument needs be specified.}

\item{hasse.font}{The name of the font family used for all text included in the Hasse diagram. Standard and safe font families to choose are "sans", "serif", and "mono". If the design's factor labels contain Unicode characters, or for consistency with Hasse diagrams of restricted layout structures using \link[hassediagrams]{hasserls}, a Unicode friendly font family should be selected. For more details on Unicode friendly family options see the Details section in the \link[hassediagrams]{hasserls} documentation. The default is "sans".}

\item{produceBWPlot}{logical. If "Y" then the Hasse diagram will be generated in black and white format. The default is set to "N", i.e., a coloured version of the plot is produced.}

\item{structural.colour}{The colour of the structural lines that connect structural objects on the Hasse diagram. The default colour is "grey".}

\item{structural.width}{The width of the structural lines on the Hasse diagram. The default width is 2.}

\item{partial.colour}{The colour of the partial crossing dotted lines of the connecting objects on the Hasse diagram. The default colour is "orange".}

\item{partial.width}{The width of the partial crossing dotted lines on the Hasse diagram. The default width is 1.5.}

\item{objects.colour}{The colour of the labels of the structural objects on the Hasse diagram. The default colour is "mediumblue".}

\item{df.colour}{The colour of the degrees of the freedom labels on the Hasse diagram. The default colour is "red".}

\item{larger.fontlabelmultiplier}{The large font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram where there are four or less objects at that level in the diagram. The default is 1.}

\item{middle.fontlabelmultiplier}{The medium font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram involving a factor that is equivalent to a generalised factor. The default is 1.}

\item{smaller.fontlabelmultiplier}{The small font multiplier is the multiplier for the font used for the labels of objects on the Hasse diagram where there are five or more objects at that level of the diagram. The default is 1.}
}
\value{
The function \code{\link[hassediagrams]{hasselayout}} returns:

1. The Hasse diagram of the layout structure (if \code{showLS="Y"}).

2. The layout structure table shows the relationships between the structural objects in the layout structure (if \code{table.out="Y"}). 
The individual entries in the table consist of blanks on the main diagonal and 0's, (0)'s or 1's elsewhere. 
If the factor (or generalised factor) corresponding to the ith row and the factor (or generalised factor) corresponding to the 
jth column are fully crossed, then a 0 is entered in the (i,j)th entry in the table. If these factors (or generalised factors) are 
partially crossed, or the ith row factor (or generalised factor) only has one level and nests the jth column factor (or generalised factor), 
then the (i,j)th entry is (0). If the ith row factor (or generalised factor) is nested within the jth column factor (or generalised factor), 
then a 1 is entered in the (i,j)th entry. If two factors (or generalised factors) are equivalent, then they share a single row and column in 
the table, where the row and column headers include both factor (or generalised factor) names, separated by an "=" sign.

3. If there are confounded degrees of freedom, a table of the structural objects and a description of the associated degrees of freedom is printed.
}
\description{
Returns a Hasse diagram of the layout structure of an experimental design
}
\details{
The hasselayout function generates the Hasse diagram of the layout structure of the experimental design, as described in Bate and Chatfield (2016a). The diagram consists of a set of structural objects, corresponding to the factors and generalised factors, and the relationships between the structural objects (either crossed, nested, partially crossed or equivalent), as defined by the structure of the experimental design.

The function requires a dataframe containing only the factors corresponding to the experimental factors that define the experimental design (i.e., no response should be included). 

In the dataframe the levels of the factors must be uniquely identified and have a physical meaning, otherwise the function will not correctly identify the nesting/crossing of the factors. For example, consider an experiment consisting of Factor A (with k levels) that nests Factor B (with q levels per level of Factor A). The levels of Factor B should be labelled 1 to k x q and not 1 to q (repeated k times). 

Where present, two partially crossed factors are illustrated on the diagram with a dotted line connecting them. This feature can be excluded using the \code{showpartialLS} option.

The maximum number of possible levels of each generalised factor, along with the actual number present in the design and the "skeleton ANOVA" degrees of freedom, can be included in the structural object label on the Hasse diagram.

Using the randomfacsid argument the factors that correspond to random effects can be highlighted by underlining them on the Hasse diagram. The vector should be equal to the number of factors in the design and consist of fixed (entry = 0) or random (entry = 1) values.

The \code{\link[hassediagrams]{hasselayout}} function evaluates the design in order to identify if there are any confounded degrees of freedom across the design. It is not recommended to perform this evaluation for large designs due to the potential high computational cost. This can be controlled using the check.confound.df = "N" option.
}
\examples{
\donttest{
## Examples using the package build-in data concrete, dental, human, analytical.

## A fractional factorial design for investigating asphalt concrete production
hasselayout(datadesign=concrete, larger.fontlabelmultiplier=1.6, 
            smaller.fontlabelmultiplier=1.3, table.out="Y")

## A crossover design for a dental study
hasselayout(datadesign=dental, randomfacsid = c(0,1,0,0,0), 
            larger.fontlabelmultiplier = 1.6)

## A block design for an experiment assessing human-computer interaction
hasselayout(datadesign=human, randomfacsid = c(1,1,0,0,0,0,1), 
            larger.fontlabelmultiplier=1.4)

## A cross-nested design for an analytical method investigation
hasselayout(datadesign=analytical, randomfacsid = c(0,0,1,1,1,0,0,0), 
            showpartialLS="N", check.confound.df="N", 
            larger.fontlabelmultiplier=1, 
            smaller.fontlabelmultiplier=1.6)


## Examples using data from the dae package (conditionally loaded)

if (requireNamespace("dae", quietly = TRUE)) {
  
  ## Data for a balanced incomplete block experiment, Joshi (1987)
  
  data(BIBDWheat.dat, package = "dae")
  # remove the response from the dataset
  BIBDWheat <- BIBDWheat.dat[, -4]
  hasselayout(datadesign=BIBDWheat, example = "BIBDWheat")
  
  
  ## Data for an un-replicated 2^4 factorial experiment to investigate a chemical process
  ## from Table 10.6 of Box, Hunter and Hunter (1978)
  
  data(Fac4Proc.dat, package = "dae")
  # remove the response from the dataset
  Fac4Proc <- Fac4Proc.dat[, -6]
  hasselayout(datadesign=Fac4Proc, example = "Fac4Proc", showpartialLS="N", 
              smaller.fontlabelmultiplier=2)
  
 
 ## Data for an experiment with rows and columns from p.144 of 
 ## Williams, Matheson and Harwood (2002)

 data(Casuarina.dat, package = "dae")
 # remove the response from the dataset
 Casuarina <- Casuarina.dat[, -7]
 # create unique factor level labels
 Casuarina$Row <- paste(Casuarina$Reps, Casuarina$Rows)
 Casuarina$Col <- paste(Casuarina$Reps, Casuarina$Columns)
 Casuarina <- Casuarina[, -c(2,3)]
 hasselayout(datadesign=Casuarina, randomfacsid=c(1,0,1,0,0,0), 
             example="Casuarina", check.confound.df="N", 
             showpartialLS="N")
  
} else {
  message("Examples using data from the 'dae' package 
           require 'dae' to be installed.")
}
}


}
\references{
Bate, S.T. and Chatfield, M.J. (2016a), Identifying the structure of the experimental design. Journal of Quality Technology, 48, 343-364.

Bate, S.T. and Chatfield, M.J. (2016b), Using the structure of the experimental design and the randomization to construct a mixed model. Journal of Quality Technology, 48, 365-387.

Box, G.E.P., Hunter, J.S., and Hunter, W.G., (1978), Statistics for Experimenters. Wiley.

Joshi, D.D. (1987), Linear Estimation and Design of Experiments. Wiley Eastern, New Delhi.

Williams, E.R., Matheson, A.C. and Harwood, C.E. (2002), Experimental design and analysis for tree improvement. 2nd edition. CSIRO, Melbourne, Australia.
}
\author{
Damianos Michaelides, Simon Bate, and Marion Chatfield
}
