% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adjacency.R
\name{graph_from_adjacency_matrix}
\alias{graph_from_adjacency_matrix}
\alias{from_adjacency}
\title{Create graphs from adjacency matrices}
\usage{
graph_from_adjacency_matrix(
  adjmatrix,
  mode = c("directed", "undirected", "max", "min", "upper", "lower", "plus"),
  weighted = NULL,
  diag = TRUE,
  add.colnames = NULL,
  add.rownames = NA
)

from_adjacency(...)
}
\arguments{
\item{adjmatrix}{A square adjacency matrix. From igraph version 0.5.1 this
can be a sparse matrix created with the \code{Matrix} package.}

\item{mode}{Character scalar, specifies how igraph should interpret the
supplied matrix. See also the \code{weighted} argument, the interpretation
depends on that too. Possible values are: \code{directed},
\code{undirected}, \code{upper}, \code{lower}, \code{max}, \code{min},
\code{plus}. See details below.}

\item{weighted}{This argument specifies whether to create a weighted graph
from an adjacency matrix. If it is \code{NULL} then an unweighted graph is
created and the elements of the adjacency matrix gives the number of edges
between the vertices. If it is a character constant then for every non-zero
matrix entry an edge is created and the value of the entry is added as an
edge attribute named by the \code{weighted} argument. If it is \code{TRUE}
then a weighted graph is created and the name of the edge attribute will be
\code{weight}. See also details below.}

\item{diag}{Logical scalar, whether to include the diagonal of the matrix in
the calculation. If this is \code{FALSE} then the diagonal is zerod out
first.}

\item{add.colnames}{Character scalar, whether to add the column names as
vertex attributes. If it is \code{NULL} (the default) then, if
present, column names are added as vertex attribute \sQuote{name}. If
\code{NA} or \code{FALSE} then they will not be added.  If a character constant,
then it gives the name of the vertex attribute to add.}

\item{add.rownames}{Character scalar, whether to add the row names as vertex
attributes. Possible values the same as the previous argument. By default
row names are not added. If \sQuote{\code{add.rownames}} and
\sQuote{\code{add.colnames}} specify the same vertex attribute, then the
former is ignored.}

\item{...}{Passed to \code{graph_from_adjacency_matrix()}.}
}
\value{
An igraph graph object.
}
\description{
\code{graph_from_adjacency_matrix()} is a flexible function for creating \code{igraph}
graphs from adjacency matrices.
}
\details{
The order of the vertices are preserved, i.e. the vertex corresponding to
the first row will be vertex 0 in the graph, etc.

\code{graph_from_adjacency_matrix()} operates in two main modes, depending on the
\code{weighted} argument.

If this argument is \code{NULL} then an unweighted graph is created and an
element of the adjacency matrix gives the number of edges to create between
the two corresponding vertices.  The details depend on the value of the
\code{mode} argument:
\describe{
\item{"directed"}{
The graph will be directed and a matrix element gives
the number of edges between two vertices.
}
\item{"undirected"}{
This is exactly the same as \code{max}, for convenience.
Note that it is \emph{not} checked whether the matrix is symmetric.
}
\item{"max"}{
An undirected graph will be created and \code{max(A(i,j), A(j,i))}
gives the number of edges.
}
\item{"upper"}{
An undirected graph will be created, only the upper right triangle
(including the diagonal) is used for the number of edges.
}
\item{"lower"}{
An undirected graph will be created, only the lower left triangle
(including the diagonal) is used for creating the edges.
}
\item{"min"}{
An undirected graph will be created with \code{min(A(i,j), A(j,i))} edges
between vertex \code{i} and \code{j}.
}
\item{"plus"}{
An undirected graph will be created with \code{A(i,j)+A(j,i)} edges
between vertex \code{i} and \code{j}.
}
}

If the \code{weighted} argument is not \code{NULL} then the elements of the
matrix give the weights of the edges (if they are not zero).  The details
depend on the value of the \code{mode} argument:
\describe{
\item{"directed"}{
The graph will be directed and a matrix element gives the edge weights.
}
\item{"undirected"}{
First we check that the matrix is symmetric. It is an error if not.
Then only the upper triangle is used to create a weighted undirected graph.
}
\item{"max"}{
An undirected graph will be created and \code{max(A(i,j), A(j,i))}
gives the edge weights.
}
\item{"upper"}{
An undirected graph will be created, only the upper right triangle
(including the diagonal) is used (for the edge weights).
}
\item{"lower"}{
An undirected graph will be created, only the lower left triangle
(including the diagonal) is used for creating the edges.
}
\item{"min"}{
An undirected graph will be created, \code{min(A(i,j), A(j,i))}
gives the edge weights.
}
\item{"plus"}{
An undirected graph will be created, \code{A(i,j)+A(j,i)}
gives the edge weights.
}
}
}
\examples{

g1 <- sample(
  x = 0:1, size = 100, replace = TRUE,
  prob = c(0.9, 0.1)
) \%>\%
  matrix(ncol = 10) \%>\%
  graph_from_adjacency_matrix()

g2 <- sample(
  x = 0:5, size = 100, replace = TRUE,
  prob = c(0.9, 0.02, 0.02, 0.02, 0.02, 0.02)
) \%>\%
  matrix(ncol = 10) \%>\%
  graph_from_adjacency_matrix(weighted = TRUE)
E(g2)$weight

## various modes for weighted graphs, with some tests
non_zero_sort <- function(x) sort(x[x != 0])
adj_matrix <- matrix(runif(100), 10)
adj_matrix[adj_matrix < 0.5] <- 0
g3 <- graph_from_adjacency_matrix(
  (adj_matrix + t(adj_matrix)) / 2,
  weighted = TRUE,
  mode = "undirected"
)

g4 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "max"
)
expected_g4_weights <- non_zero_sort(
  pmax(adj_matrix, t(adj_matrix))[upper.tri(adj_matrix, diag = TRUE)]
)
actual_g4_weights <- sort(E(g4)$weight)
all(expected_g4_weights == actual_g4_weights)

g5 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "min"
)
expected_g5_weights <- non_zero_sort(
  pmin(adj_matrix, t(adj_matrix))[upper.tri(adj_matrix, diag = TRUE)]
)
actual_g5_weights <- sort(E(g5)$weight)
all(expected_g5_weights == actual_g5_weights)

g6 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "upper"
)
expected_g6_weights <- non_zero_sort(adj_matrix[upper.tri(adj_matrix, diag = TRUE)])
actual_g6_weights <- sort(E(g6)$weight)
all(expected_g6_weights == actual_g6_weights)

g7 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "lower"
)
expected_g7_weights <- non_zero_sort(adj_matrix[lower.tri(adj_matrix, diag = TRUE)])
actual_g7_weights <- sort(E(g7)$weight)
all(expected_g7_weights == actual_g7_weights)

g8 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "plus"
)
halve_diag <- function(x) {
  diag(x) <- diag(x) / 2
  x
}
expected_g8_weights <- non_zero_sort(
  halve_diag(adj_matrix + t(adj_matrix))[lower.tri(adj_matrix, diag = TRUE)]
)
actual_g8_weights <- sort(E(g8)$weight)
all(expected_g8_weights == actual_g8_weights)

g9 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  mode = "plus",
  diag = FALSE
)
zero_diag <- function(x) {
  diag(x) <- 0
}
expected_g9_weights <- non_zero_sort((zero_diag(adj_matrix + t(adj_matrix)))[lower.tri(adj_matrix)])
actual_g9_weights <- sort(E(g9)$weight)
all(expected_g9_weights == actual_g9_weights)

## row/column names
rownames(adj_matrix) <- sample(letters, nrow(adj_matrix))
colnames(adj_matrix) <- seq(ncol(adj_matrix))
g10 <- graph_from_adjacency_matrix(
  adj_matrix,
  weighted = TRUE,
  add.rownames = "code"
)
summary(g10)

}
\seealso{
\code{\link[=make_graph]{make_graph()}} and \code{\link[=graph_from_literal]{graph_from_literal()}} for other ways to
create graphs.
}
\author{
Gabor Csardi \email{csardi.gabor@gmail.com}
}
\concept{adjacency}
\keyword{graphs}
