#' An S4 class to represent the lambda distribution
#' 
#' The \code{ecld} class serves as an object-oriented interface for the lambda distribution,
#' which is just the exponential power distribution in GSL and Wolfram. 
#'
#' @slot call the match.call slot
#' @slot lambda numeric
#' @slot sigma numeric
#' @slot mu  numeric
#'
#' @keywords ecld
#' 
#' @author Stephen H. Lihn
#' 
#' @section Details:
#'   The lambda distribution is just the exponential power distribution in GSL and Wolfram, 
#'   with a different definition in the exponent of the stretched exponential function.
#' 
#'   The distribution is symmetric. Its PDF is \deqn{
#'     P\left(x; \lambda, \sigma, \mu\right)
#'     \equiv\, \frac{1}{\lambda \Gamma\left(\frac{2}{\lambda}\right) \sigma} 
#'     e^{-{\left|\frac{x-\mu}{\sigma}\right|}^{\frac{2}{\lambda}}}.
#'   }
#'   where 
#'   \eqn{\lambda} is the shape parameter, 
#'   \eqn{\sigma} is the scale parameter, 
#'   \eqn{\mu} is the location parameter.
#'   \cr
#'   This functional form is not unfamiliar and has appeared under several other names, such as
#'   generalized normal distribution and power exponential distribution, etc..
#'   
#' @references
#'   This distribution is the same as \emph{gnorm} and is implemented from it since V0.6. 
#'   See \url{https://cran.r-project.org/package=gnorm}.
#'   \cr
#'   For lambda distribution and option pricing model, see 
#'   Stephen Lihn (2015). 
#'   \emph{The Special Elliptic Option Pricing Model and Volatility Smile}.
#'   SSRN: \url{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=2707810}.
#'   \cr
#'   Closed form solutions are derived in 
#'   Stephen Lihn (2016). \emph{Closed Form Solution and Term Structure for SPX Options}.
#'   SSRN: \url{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=2805769}
#'   and \cr 
#'   Stephen Lihn (2017). \emph{From Volatility Smile to Risk Neutral Probability and 
#'   Closed Form Solution of Local Volatility Function}.
#'   SSRN: \url{https://papers.ssrn.com/sol3/papers.cfm?abstract_id=2906522}
#'
#' @exportClass ecld
setClass("ecld",
         representation(call = "call",
                        lambda = "numeric",
                        sigma = "numeric",
                        mu = "numeric"),
          prototype(call = call("ecld"),
                    lambda = NaN,
                    sigma = NaN,
                    mu = NaN)
)
### <---------------------------------------------------------------------->
