% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mable-po-model.r
\name{mable.po}
\alias{mable.po}
\title{Mable fit of proportional odds rate regression model}
\usage{
mable.po(
  formula,
  data,
  M,
  g = NULL,
  tau,
  x0 = NULL,
  controls = mable.ctrl(),
  progress = TRUE
)
}
\arguments{
\item{formula}{regression formula. Response must be \code{cbind}.  See 'Details'.}

\item{data}{a data frame containing variables in \code{formula}.}

\item{M}{a positive integer or a vector \code{(m0, m1)}. 
If \code{M = m} or \code{m0 = m1 = m},
  then \code{m} is a preselected degree. If \code{m0<m1} it specifies the set of
  consective candidate model degrees \code{m0:m1} for searching an optimal degree,
  where \code{m1-m0>3}.}

\item{g}{an initial guess of \eqn{d}-vector of regression coefficients.  See 'Details'.}

\item{tau}{right endpoint of support \eqn{[0, \tau]} must be greater than or equal to the
maximum observed time}

\item{x0}{a data frame specifying working baseline covariates on the right-hand-side of \code{formula}. See 'Details'.}

\item{controls}{Object of class \code{mable.ctrl()} specifying iteration limit
and other control options. Default is \code{\link{mable.ctrl}}.}

\item{progress}{if \code{TRUE} a text progressbar is displayed}
}
\value{
A list with components
\itemize{
  \item \code{m} the selected/preselected optimal degree \code{m}
  \item \code{p} the estimate of \eqn{p = (p_0, \dots, p_m)}, the coefficients of
            Bernstein polynomial of degree \code{m}
  \item \code{coefficients} the estimated regression coefficients of the PO model
  \item \code{SE} the standard errors of the estimated regression coefficients
  \item \code{z} the z-scores of the estimated regression coefficients
  \item \code{mloglik} the maximum log-likelihood at an optimal degree \code{m}
  \item \code{tau.n} maximum observed time \eqn{\tau_n}
  \item \code{tau} right endpoint of support \eqn{[0, \tau]}
  \item \code{x0} the working baseline covariates
  \item \code{egx0} the value of \eqn{e^{\gamma'x_0}}
  \item \code{convergence} an integer code, 1 indicates either the EM-like iteration 
    for finding maximum likelihood reached the maximum iteration for at least one \code{m}
    or the search of an optimal degree using change-point method reached the maximum
    candidate degree, 2 indicates both occured, and 0 indicates a successful completion.
  \item \code{delta} the final \code{delta} if \code{m0 = m1} or the final \code{pval} of
     the change-point for searching the optimal degree \code{m};
 }
 and, if \code{m0<m1},
\itemize{
  \item \code{M} the vector \code{(m0, m1)}, where \code{m1} is the last candidate degree
       when the search stoped
  \item \code{lk} log-likelihoods evaluated at \eqn{m \in \{m_0,\ldots, m_1\}}
  \item \code{lr} likelihood ratios for change-points evaluated at 
       \eqn{m \in \{m_0+1, \ldots, m_1\}}
  \item \code{pval} the p-values of the change-point tests for choosing optimal model degree
  \item \code{chpts} the change-points chosen with the given candidate model degrees
}
}
\description{
Maximum approximate Bernstein/Beta likelihood estimation in general 
    proportional odds regression model based on interal censored event time data.
}
\details{
Consider PO model with covariate for interval-censored failure time data:
\eqn{[1-S(t|x)]/S(t|x) = \exp[\gamma'(x-x_0)][1-S(t|x_0)]/S(t|x_0)}, 
 where \eqn{x_0} satisfies \eqn{\gamma'(x-x_0)\ge 0}, where \eqn{x} and \eqn{x_0} may
contain dummy variables and interaction terms.  The working baseline \code{x0} in arguments
contains only the values of terms excluding dummy variables and interaction terms 
in the right-hand-side of \code{formula}. Thus \code{g} is the initial guess of 
the coefficients \eqn{\gamma} of \eqn{x-x_0} and could be longer than \code{x0}.
  Let \eqn{f(t|x)} and \eqn{F(t|x) = 1-S(t|x)} be the density and cumulative distribution
functions of the event time given \eqn{X = x}, respectively.
Then \eqn{f(t|x_0)} on \eqn{[0, \tau]} can be approximated by
\eqn{f_m(t|x_0, p) = \tau^{-1}\sum_{i=0}^m p_i\beta_{mi}(t/\tau)},
where \eqn{p_i \ge 0}, \eqn{i = 0, \ldots, m}, \eqn{\sum_{i=0}^mp_i = 1},
\eqn{\beta_{mi}(u)} is the beta denity with shapes \eqn{i+1} and \eqn{m-i+1}, and
\eqn{\tau} is the right endpoint of support interval of the baseline density. 
 We can approximate  \eqn{S(t|x_0)} on \eqn{[0,\tau]} by
\eqn{S_m(t|x_0; p) = \sum_{i=0}^{m} p_i \bar B_{mi}(t/\tau)}, where
\eqn{\bar B_{mi}(u)}, \eqn{i = 0, \ldots, m}, is the beta survival function with shapes
 \eqn{i+1} and \eqn{m-i+1}.

Response variable should be of the form \code{cbind(l,u)}, where \code{(l,u)} is the interval
containing the event time. Data is uncensored if \code{l = u}, right censored
if \code{u = Inf} or \code{u = NA}, and  left censored if \code{l = 0}.
The associated covariate contains \eqn{d} columns. The baseline \code{x0} should chosen so
that \eqn{\gamma'(x-x_0)} is nonnegative for all the observed \eqn{x} and
all \eqn{\gamma} in a neighborhood of its true value.

A missing initial value of \code{g} is imputed by \code{ic_sp()} of package \code{icenReg} 
with \code{model="po"}.
 The search for optimal degree \code{m} stops if either \code{m1} is reached or the test
 for change-point results in a p-value \code{pval} smaller than \code{sig.level}.
This process takes longer than \code{\link{maple.po}} to select an optimal degree.
}
\examples{
\donttest{
# Veteran's Administration Lung Cancer Data 
require(survival)
require(icenReg)
require(mable)
l<-veteran$time->u
u[veteran$status==0]<-Inf
veteran1<-data.frame(l=l, u=u, karno=veteran$karno, celltype=veteran$celltype, 
           trt=veteran$trt, age=veteran$age, prior=veteran$prior>0) 
fit.sp<-ic_sp(cbind(l,u) ~ karno+celltype, data = veteran1,  model="po") 
x0<-data.frame(karno=100, celltype="squamous")
tau<-2000
res<-mable.po(cbind(l,u) ~ karno+celltype, data = veteran1, M=c(1,35),                               
     g=-fit.sp$coefficients, x0=x0, tau=tau)                          
op<-par(mfrow=c(2,2))
plot(res, which = "likelihood")
plot(res, which = "change-point")
plot(res, y=data.frame(karno=20, celltype="squamous"), which="survival", 
      add=FALSE, type="l", xlab="Days", 
      main=expression(paste("Survival: ", bold(x)==0)))
plot(res, y=data.frame(karno=80, celltype="smallcell"), which="survival", 
      add=FALSE, type="l", xlab="Days", 
      main=expression(paste("Survival: ", bold(x)==bold(x)[0])))
par(op)

}
}
\references{
Guan, Z. Maximum Likelihood Estimation in Proportional Odds Regression Model 
           Based on Interval-Censored Event-time Data
}
\seealso{
\code{\link{maple.ph}}
}
\author{
Zhong Guan <zguan@iu.edu>
}
\concept{Generalized proportional odds model}
\concept{interval censoring}
\keyword{distribution}
\keyword{models}
\keyword{nonparametric}
\keyword{regression}
\keyword{smooth}
\keyword{survival}
