#' Create the path elements for MeteoGalicia API
#'
#' Path vectors for MeteoGalicia API to use with httr2
#'
#' @param api_options Option list as generated by \link{\code{meteogalicia_options}}
#'
#' @noRd
.create_meteogalicia_path <- function(api_options) {

  # we need the resolution to create the corresponding path
  resolution <- api_options$resolution

  temp_path <- switch(
    resolution,
    "instant" = c('mgrss', 'observacion', 'ultimos10minEstacionsMeteo.action'),
    "current_day" = c('mgrss', 'observacion', 'ultimosHorariosEstacions.action'),
    "daily" = c('mgrss', 'observacion', 'datosDiariosEstacionsMeteo.action'),
    "monthly" = c('mgrss', 'observacion', 'datosMensuaisEstacionsMeteo.action'),
    FALSE
  )

  # not recognised resolution, we abort
  if (isFALSE(temp_path)) {
    cli::cli_abort(c(
      "{.arg {api_options$resolution}} is not a valid temporal resolution for MeteoGalicia. Please see meteogalicia_options help for more information"
    ))
  }

  return(temp_path)
}

#' Create the query element for MeteoGalicia API
#'
#' Query string for MeteoGalicia API to use with httr2
#'
#' MeteoGalicia allows to include queries in the link, to select dates, stations and hours in current. This
#' is more flexible than the paths in AEMET and allows to retrieve the stations desired directly in an easy
#' way.
#'
#' @param api_options Option list as generated by \link{\code{meteogalicia_options}}
#'
#' @noRd
.create_meteogalicia_query <- function(api_options) {

  # In case of dates supplied and in the corresponding resolutions, we need to transform the dates to the
  # character string specific format (dd-mm-yyyy) for the meteogalicia query
  # We will use a stamp function:
  meteogalicia_stamp <- lubridate::stamp("25/12/2001", orders = "d0mY", quiet = TRUE)

  # All possible queries, later we remove based in resolution
  query_list <- list(
    idEst = glue::glue_collapse(api_options$stations, sep = ","),
    dataIni = I(meteogalicia_stamp(api_options$start_date)),
    dataFin = I(meteogalicia_stamp(api_options$end_date)),
    numHoras = 24
  )
  
  # remove the stations, as it is the common part for any resolution
  if (rlang::is_null(api_options$stations)) {
    query_list[["idEst"]] <- NULL
  }
  # now the specifics for each resolution:
  #   - instant, nothing, only the stations if any
  #   - current day, stations if any and numHoras=24
  #   - daily, stations if any, start date and end date
  #   - monthly, stations if any, start date and end date
  if (api_options$resolution == 'instant') {
    query_list[c("dataIni", "dataFin", "numHoras")] <- NULL
  }
  if (api_options$resolution == 'current_day') {
    query_list[c("dataIni", "dataFin")] <- NULL
  }
  if (api_options$resolution %in% c('daily', 'monthly')) {
    query_list[c("numHoras")] <- NULL
  }

  return(query_list)
}

#' Create request for Meteogalicia API
#'
#' Create the request for meteogalicia based on path and query
#'
#' This function creates the request, deal with errors and retries if needed
#' and access the data of the response.
#'
#' @param path Character vector with the path as obtained by
#'   \link{\code{.create_meteogalicia_path}}
#' @param query List of name value pairs to query dates, needed for the request
#'   on some API endpoints
#'
#' @return a tibble converted from the response json.
#'
#' @noRd
.create_meteogalicia_request <- function(path, query = NULL) {

  meteogalicia_request <- httr2::request("https://servizos.meteogalicia.gal") |>
    httr2::req_url_path_append(path) |>
    httr2::req_url_query(!!!query) |>
    httr2::req_user_agent(
      "meteospain R package (https://emf.creaf.cat/software/meteospain/)"
    ) |>
    httr2::req_error(
      body = \(resp) {
        message <- httr2::resp_status(resp)
        
        # more verbose known errors
        if (httr2::resp_status(resp) %in% c(404L)) {
          message <- c(
            message,
            "This usually happens when unknown station ids are supplied."
          )
        }

        if (httr2::resp_status(resp) %in% c(500L)) {
          message <- c(
            message,
            httr2::resp_body_html(resp) |>
              rvest::html_element("body") |>
              rvest::html_text()
          )
        }

        return(message)
      }
    ) |>
    httr2::req_retry(
      max_tries = 3,
      retry_on_failure = TRUE,
      is_transient = \(resp) {
        httr2::resp_status(resp) %in% c(429, 503)
      },
      backoff = \(resp) {
        60
      },
      after = \(resp) {
        NA
      }
    )

  res <- httr2::req_perform(meteogalicia_request) |>
    httr2::resp_body_json(flatten = TRUE, simplifyDataFrame = TRUE) |>
    dplyr::as_tibble()
  
  if (nrow(res) < 1) {
    cli::cli_abort(c(
      "MeteoGalicia API returned no data:\n",
      i = "This usually happens when there is no data for the dates supplied."
    ))
  }

  res
}


#' Get info for the meteogalicia stations
#'
#' Get info for the meteogalicia stations
#'
#' @noRd

.get_info_meteogalicia <- function() {

  # GET ---------------------------------------------------------------------------------------------------
  # path
  path_resolution <- c(
    'mgrss', 'observacion', 'listaEstacionsMeteo.action'
  )
  # cache
  cache_ref <- rlang::hash(path_resolution)

  # get data from cache or from API if new
  .get_cached_result(cache_ref, {
    .create_meteogalicia_request(path_resolution) |>
      unnest_safe("listaEstacionsMeteo") |>
      dplyr::mutate(service = "meteogalicia") |>
      .info_table_checker() |>
      dplyr::select(
        "service", station_id = "idEstacion", station_name = "estacion", station_province = "provincia",
        "altitude", "lat", "lon"
      ) |>
      dplyr::mutate(
        station_id = as.character(.data$station_id),
        altitude = units::set_units(.data$altitude, "m")
      ) |>
      sf::st_as_sf(coords = c('lon', 'lat'), crs = 4326)
  })
}

#' Get data from MeteoGalicia
#'
#' Get data from MeteoGalicia service
#'
#' MeteoGalicia API, based on RSS, is very well organized. It allows to supply dates and stations as
#' query parameters, which gives a lot of flexibility and makes my life easier. All my love to MeteoGalicia!!
#'
#' @param api_options Option list as generated by \link{\code{meteogalicia_options}}
#'
#' @noRd
.get_data_meteogalicia <- function(api_options) {

  # api path
  path_resolution <- .create_meteogalicia_path(api_options)
  # get api query
  query_resolution <- .create_meteogalicia_query(api_options)

  # cache (in this case with path and query to get the date also)
  cache_ref <- rlang::hash(c(path_resolution, query_resolution))

  # if resolution less than daily, remove the cache
  if (api_options$resolution %in% c("instant", "current_day")) {
    apis_cache$remove(cache_ref)
  }

  data_meteogalicia <- .get_cached_result(cache_ref, {
    
    # choosing the unnesting strategy based on resolution
    # Now, resolutions have differences, in the component names of the list returned and also in variables
    # returned. So we create specific functions for each resolution and use a common pipe (see aemet.helpers
    # for a more complete rationale)
    resolution_specific_unnesting <- switch(
      api_options$resolution,
      'instant' = .meteogalicia_instant_unnesting,
      'current_day' = .meteogalicia_current_day_unnesting,
      'daily' = .meteogalicia_daily_unnesting,
      'monthly' = .meteogalicia_monthly_unnesting
    )
    resolution_specific_carpentry <- switch(
      api_options$resolution,
      'instant' = .meteogalicia_instant_carpentry,
      'current_day' = .meteogalicia_current_day_carpentry,
      'daily' = .meteogalicia_daily_carpentry,
      'monthly' = .meteogalicia_monthly_carpentry
    )
    resolution_specific_joinvars <- c('service', 'station_id', 'station_name')
    if (api_options$resolution %in% c('daily', 'monthly')) {
      resolution_specific_joinvars <- c(resolution_specific_joinvars, 'station_province')
    }

    res <- .create_meteogalicia_request(path_resolution, query_resolution) |>
      resolution_specific_unnesting() |>
      # final unnest, common to all resolutions
      unnest_safe("listaMedidas") |>
      # remove the non valid data (0 == no validated data, 3 = wrong data, 9 = data not registered)
      dplyr::filter(!.data$lnCodigoValidacion %in% c(0, 3, 9)) |>
      # remove unwanted variables
      dplyr::select(-"lnCodigoValidacion", -"nomeParametro", -"unidade") |>
      # now, some stations can have errors in the sense of duplicated precipitation values.
      # We get the first record
      tidyr::pivot_wider(
        names_from = "codigoParametro", values_from = "valor", values_fn = dplyr::first
      ) |>
      # resolution-specific transformations
      resolution_specific_carpentry() |>
      dplyr::arrange(.data$timestamp, .data$station_id) |>
      dplyr::left_join(.get_info_meteogalicia(), by = resolution_specific_joinvars) |>
      # reorder variables to be consistent among all services
      relocate_vars() |>
      sf::st_as_sf()

    # Copyright message -------------------------------------------------------------------------------------
    cli::cli_inform(c(
      i = copyright_style("A informaci\u00F3n divulgada a trav\u00E9s deste servidor ofr\u00E9cese gratuitamente aos cidad\u00E1ns para que poida ser"),
      copyright_style("utilizada libremente por eles, co \u00FAnico compromiso de mencionar expresamente a MeteoGalicia e \u00E1"),
      copyright_style("Conseller\u00EDa de Medio Ambiente, Territorio e Vivenda da Xunta de Galicia como fonte da mesma cada vez"),
      copyright_style("que as utilice para os usos distintos do particular e privado."),
      legal_note_style("https://www.meteogalicia.gal/aviso-legal")
    ))

    res
  })

  return(data_meteogalicia)
}


# resolution_specific_unnesting --------------------------------------------------------------------------
.meteogalicia_instant_unnesting <- function(response_content) {
  return(response_content$listUltimos10min)
}

.meteogalicia_current_day_unnesting <- function(response_content) {
  res <- response_content$listHorarios |>
    unnest_safe("listaInstantes")

  return(res)
}

.meteogalicia_daily_unnesting <- function(response_content) {
  res <- response_content$listDatosDiarios |>
    unnest_safe("listaEstacions")

  return(res)
}

.meteogalicia_monthly_unnesting <- function(response_content) {
  res <- response_content$listDatosMensuais |>
    unnest_safe("listaEstacions")

  return(res)
}


# resolution_specific_carpentry -------------------------------------------------------------------------

.meteogalicia_instant_carpentry <- function(data) {
  data |>
    # When querying stations, it can happen that some stations lack some variables, making the further
    # select step to fail. We create missing variables and populate them with NAs to avoid this error
    .create_missing_vars(
      var_names = c(
        # temperatures
        "TA_AVG_1.5m", "TS_AVG_-0.1m", "TS_AVG_-0.5m",
        # humidities
        "HR_AVG_1.5m",
        # precipitation
        "PP_SUM_1.5m",
        # winds
        "DV_AVG_2m", "VV_AVG_2m", "VV_RACHA_2m", "DV_CONDICION_2m",
        # radiations
        "RS_AVG_1.5m", "HSOL_SUM_1.5m", "RREF_AVG_1.5m", "BIO_AVG_1.5m",
        # pressures
        "PR_AVG_1.5m", "PRED_AVG_1.5m",
        # soil moisture
        "HS_CV_AVG_-0.2m"
      )
    ) |>
    dplyr::select(
      timestamp = "instanteLecturaUTC", station_id = "idEstacion", station_name = "estacion",
      # temperatures
      temperature = "TA_AVG_1.5m",
      temperature_soil_100 = "TS_AVG_-0.1m", temperature_soil_500 = "TS_AVG_-0.5m",
      # humidities
      relative_humidity = "HR_AVG_1.5m",
      # precipitation
      precipitation = "PP_SUM_1.5m",
      # winds
      wind_direction = "DV_AVG_2m", wind_speed = "VV_AVG_2m",
      max_wind_speed = "VV_RACHA_2m", max_wind_direction = "DV_CONDICION_2m",
      # radiations
      global_solar_radiation = "RS_AVG_1.5m",
      insolation = "HSOL_SUM_1.5m",
      reflected_radiation = "RREF_AVG_1.5m", uv_radiation = "BIO_AVG_1.5m",
      # pressures
      atmospheric_pressure = "PR_AVG_1.5m", atmospheric_pressure_reduced = "PRED_AVG_1.5m",
      # soil moisture
      soil_moisture = "HS_CV_AVG_-0.2m"
    ) |>
    dplyr::mutate(
      timestamp = lubridate::as_datetime(.data$timestamp),
      service = 'meteogalicia',
      station_id = as.character(.data$station_id),
      temperature = units::set_units(.data$temperature, "degree_C"),
      temperature_soil_100 = units::set_units(.data$temperature_soil_100, "degree_C"),
      temperature_soil_500 = units::set_units(.data$temperature_soil_500, "degree_C"),
      relative_humidity = units::set_units(.data$relative_humidity, "%"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      wind_direction = units::set_units(.data$wind_direction, "degree"),
      wind_speed = units::set_units(.data$wind_speed, "m/s"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      max_wind_direction = units::set_units(.data$max_wind_direction, "degree"),
      global_solar_radiation = units::set_units(.data$global_solar_radiation, "W/m^2"),
      insolation = units::set_units(.data$insolation, "hours"),
      reflected_radiation = units::set_units(.data$reflected_radiation, "W/m^2"),
      uv_radiation = units::set_units(.data$uv_radiation, "W/m^2"),
      atmospheric_pressure = units::set_units(.data$atmospheric_pressure, "hPa"),
      atmospheric_pressure_reduced = units::set_units(.data$atmospheric_pressure_reduced, "hPa"),
      soil_moisture = units::set_units(.data$soil_moisture, "m^3/m^3")
    )
}
.meteogalicia_current_day_carpentry <- function(data) {
  data |>
    # When querying stations, it can happen that some stations lack some variables, making the further
    # select step to fail. We create missing variables and populate them with NAs to avoid this error
    .create_missing_vars(
      var_names = c(
        # temperatures
        "TA_AVG_1.5m", "TA_MIN_1.5m", "TA_MAX_1.5m",
        # humidities
        "HR_AVG_1.5m",
        # Precipitations
        "PP_SUM_1.5m", "IP_MAX_1.5m",
        # winds
        "DV_AVG_2m", "VV_AVG_2m", "VV_RACHA_2m", "DV_CONDICION_2m",
        # radiations
        "HSOL_SUM_1.5m"
      )
    ) |>
    dplyr::select(
      timestamp = "instanteLecturaUTC", station_id = "idEstacion", station_name = "estacion",
      # temperatures
      temperature = "TA_AVG_1.5m",
      min_temperature = "TA_MIN_1.5m", max_temperature = "TA_MAX_1.5m",
      # humidities
      relative_humidity = "HR_AVG_1.5m",
      # Precipitations
      precipitation = "PP_SUM_1.5m", max_precipitation_hour = "IP_MAX_1.5m",
      # winds
      wind_direction = "DV_AVG_2m", wind_speed = "VV_AVG_2m",
      max_wind_speed = "VV_RACHA_2m", max_wind_direction = "DV_CONDICION_2m",
      # radiations
      insolation = "HSOL_SUM_1.5m"
    ) |>
    dplyr::mutate(
      timestamp = lubridate::as_datetime(.data$timestamp),
      service = 'meteogalicia',
      station_id = as.character(.data$station_id),
      temperature = units::set_units(.data$temperature, "degree_C"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      relative_humidity = units::set_units(.data$relative_humidity, "%"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      max_precipitation_hour = units::set_units(.data$max_precipitation_hour, "L/m^2/h"),
      wind_direction = units::set_units(.data$wind_direction, "degree"),
      wind_speed = units::set_units(.data$wind_speed, "m/s"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      max_wind_direction = units::set_units(.data$max_wind_direction, "degree"),
      insolation = units::set_units(.data$insolation, "hours")
    )
}
.meteogalicia_daily_carpentry <- function(data) {
  data |>
    # When querying stations, it can happen that some stations lack some variables, making the further
    # select step to fail. We create missing variables and populate them with NAs to avoid this error
    .create_missing_vars(
      var_names = c(
        # temperatures
        "TA_AVG_1.5m", "TA_MIN_1.5m", "TA_MAX_1.5m", "TS_AVG_-0.1m",
        # humidities
        "HR_AVG_1.5m", "HR_MIN_1.5m", "HR_MAX_1.5m",
        # precipitations
        "PP_SUM_1.5m", "ET0_SUM_1.5m",
        # winds
        "DVP_MODA_2m", "VV_AVG_2m", "VV_MAX_2m", "DV_CONDICION_2m",
        # radiations
        "HSOL_SUM_1.5m", "IUVX_MAX_1.5m", "INS_RATIO_1.5m", "IRD_SUM_1.5m",
        # pressures
        "PR_AVG_1.5m", "PRED_AVG_1.5m",
        # soil moisture
        "HS_CV_AVG_-0.2m"
      )
    ) |>
    dplyr::select(
      timestamp = "data",
      station_id = "idEstacion", station_name = "estacion", station_province = "provincia",
      # temperatures
      mean_temperature = "TA_AVG_1.5m", min_temperature = "TA_MIN_1.5m",
      max_temperature = "TA_MAX_1.5m", mean_soil_temperature = "TS_AVG_-0.1m",
      # humidities
      mean_relative_humidity = "HR_AVG_1.5m",
      min_relative_humidity = "HR_MIN_1.5m", max_relative_humidity = "HR_MAX_1.5m",
      # precipitations
      precipitation = "PP_SUM_1.5m", reference_evapotranspiration = "ET0_SUM_1.5m",
      # winds
      mean_wind_direction = "DVP_MODA_2m", mean_wind_speed = "VV_AVG_2m",
      max_wind_speed = "VV_MAX_2m", max_wind_direction = "DV_CONDICION_2m",
      # radiations
      insolation = "HSOL_SUM_1.5m", uv_radiation_index = "IUVX_MAX_1.5m",
      insolation_ratio = "INS_RATIO_1.5m", global_solar_irradiation = "IRD_SUM_1.5m",
      # pressures
      mean_atmospheric_pressure = "PR_AVG_1.5m",
      atmospheric_pressure_reduced = "PRED_AVG_1.5m",
      # soil moisture
      soil_moisture = "HS_CV_AVG_-0.2m"
    ) |>
    dplyr::mutate(
      timestamp = lubridate::as_datetime(.data$timestamp),
      service = 'meteogalicia',
      station_id = as.character(.data$station_id),
      mean_temperature = units::set_units(.data$mean_temperature, "degree_C"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      mean_soil_temperature = units::set_units(.data$mean_soil_temperature, "degree_C"),
      mean_relative_humidity = units::set_units(.data$mean_relative_humidity, "%"),
      min_relative_humidity = units::set_units(.data$min_relative_humidity, "%"),
      max_relative_humidity = units::set_units(.data$max_relative_humidity, "%"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      reference_evapotranspiration = units::set_units(.data$reference_evapotranspiration, "L/m^2"),
      mean_wind_direction = units::set_units(.data$mean_wind_direction, "degree"),
      mean_wind_speed = units::set_units(.data$mean_wind_speed, "m/s"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      max_wind_direction = units::set_units(.data$max_wind_direction, "degree"),
      insolation = units::set_units(.data$insolation, "hours"),
      uv_radiation_index = units::set_units(.data$uv_radiation_index, ""),
      insolation_ratio = units::set_units(.data$insolation_ratio, "%"),
      global_solar_irradiation = units::set_units(.data$global_solar_irradiation, "kJ/m^2") / 10,
      mean_atmospheric_pressure = units::set_units(.data$mean_atmospheric_pressure, "hPa"),
      atmospheric_pressure_reduced = units::set_units(.data$atmospheric_pressure_reduced, "hPa"),
      soil_moisture = units::set_units(.data$soil_moisture, "m^3/m^3")
    )
}
.meteogalicia_monthly_carpentry <- function(data) {
  data |>
    # When querying stations, it can happen that some stations lack some variables, making the further
    # select step to fail. We create missing variables and populate them with NAs to avoid this error
    .create_missing_vars(
      var_names = c(
        # temperatures
        "TA_AVG_1.5m", "TA_MIN_1.5m", "TA_MAX_1.5m", "TS_AVG_-0.1m",
        "TA_AVGMIN_1.5m", "TA_AVGMAX_1.5m",
        # humidities
        "HR_AVG_1.5m", "HR_AVGMAX_1.5m", "HR_AVGMIN_1.5m",
        # precipitations
        "PP_SUM_1.5m", "PP_MAX_1.5m", "NDPP_RECUENTO_1.5m", "NDPP1_RECUENTO_1.5m",
        "NDPP10_RECUENTO_1.5m", "NDPP30_RECUENTO_1.5m", "NDPP1_RECUENTO_1.5m",
        # winds
        "DVP_MODA_2m", "VV_AVG_2m", "VV_MAX_2m", "DV_CONDICION_2m",
        # radiations
        "HSOL_SUM_1.5m", "INS_AVG_1.5m", "IRD_AVG_1.5m",
        # pressures
        "PR_AVG_1.5m", "PRED_AVG_1.5m"
      )
    ) |>
    dplyr::select(
      timestamp = "data",
      station_id = "idEstacion", station_name = "estacion", station_province = "provincia",
      # temperatures
      mean_temperature = "TA_AVG_1.5m", min_temperature = "TA_MIN_1.5m",
      max_temperature = "TA_MAX_1.5m", mean_soil_temperature = "TS_AVG_-0.1m",
      min_temperature_mean = "TA_AVGMIN_1.5m", max_temperature_mean = "TA_AVGMAX_1.5m",
      # humidities
      mean_relative_humidity = "HR_AVG_1.5m",
      max_relative_humidity_mean = "HR_AVGMAX_1.5m",
      min_relative_humidity_mean = "HR_AVGMIN_1.5m",
      # precipitations
      precipitation = "PP_SUM_1.5m", max_precipitation_24h = "PP_MAX_1.5m",
      rain_days_0 = "NDPP_RECUENTO_1.5m", rain_days_01 = "NDPP1_RECUENTO_1.5m",
      rain_days_10 = "NDPP10_RECUENTO_1.5m", rain_days_30 = "NDPP30_RECUENTO_1.5m",
      rain_days_60 = "NDPP1_RECUENTO_1.5m",
      # winds
      mean_wind_direction = "DVP_MODA_2m", mean_wind_speed = "VV_AVG_2m",
      max_wind_speed = "VV_MAX_2m", max_wind_direction = "DV_CONDICION_2m",
      # radiations
      insolation = "HSOL_SUM_1.5m", insolation_ratio = "INS_AVG_1.5m",
      global_solar_irradiation = "IRD_AVG_1.5m",
      # pressures
      mean_atmospheric_pressure = "PR_AVG_1.5m",
      atmospheric_pressure_reduced = "PRED_AVG_1.5m"
    ) |>
    dplyr::mutate(
      timestamp = lubridate::as_datetime(.data$timestamp),
      service = 'meteogalicia',
      station_id = as.character(.data$station_id),
      mean_temperature = units::set_units(.data$mean_temperature, "degree_C"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      mean_soil_temperature = units::set_units(.data$mean_soil_temperature, "degree_C"),
      min_temperature_mean = units::set_units(.data$min_temperature_mean, "degree_C"),
      max_temperature_mean = units::set_units(.data$max_temperature_mean, "degree_C"),
      mean_relative_humidity = units::set_units(.data$mean_relative_humidity, "%"),
      max_relative_humidity_mean = units::set_units(.data$max_relative_humidity_mean, "%"),
      min_relative_humidity_mean = units::set_units(.data$min_relative_humidity_mean, "%"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      max_precipitation_24h = units::set_units(.data$max_precipitation_24h, "L/m^2"),
      rain_days_0 = units::set_units(.data$rain_days_0, "days"),
      rain_days_01 = units::set_units(.data$rain_days_01, "days"),
      rain_days_10 = units::set_units(.data$rain_days_10, "days"),
      rain_days_30 = units::set_units(.data$rain_days_30, "days"),
      rain_days_60 = units::set_units(.data$rain_days_60, "days"),
      mean_wind_direction = units::set_units(.data$mean_wind_direction, "degree"),
      mean_wind_speed = units::set_units(.data$mean_wind_speed, "m/s"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      max_wind_direction = units::set_units(.data$max_wind_direction, "degree"),
      insolation = units::set_units(.data$insolation, "hours"),
      insolation_ratio = units::set_units(.data$insolation_ratio, "%"),
      global_solar_irradiation = units::set_units(.data$global_solar_irradiation, "kJ/m^2") / 10,
      mean_atmospheric_pressure = units::set_units(.data$mean_atmospheric_pressure, "hPa"),
      atmospheric_pressure_reduced = units::set_units(.data$atmospheric_pressure_reduced, "hPa")
    )
}

# info table checker ------------------------------------------------------------------------------------

.info_table_checker <- function(data) {

  mandatory_names <- c("service", "idEstacion", "estacion", "provincia", "altitude", "lat", "lon")
  names_ok <- mandatory_names %in% names(data)

  if (!all(names_ok)) {
    cli::cli_abort(c(
      x = "Oops, something went wrong and some info about stations is missing:",
      mandatory_names[names_ok]
    ))
  }

  return(data)
}
