% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pathway_model.R
\name{pathway_model}
\alias{pathway_model}
\title{Pathway model}
\usage{
pathway_model(
  ntrade_data,
  IDs_col,
  values_col,
  expression,
  parameters,
  niter = 100
)
}
\arguments{
\item{ntrade_data}{A data frame with the quantity of potentially infested commodities
imported from third countries where the pest is present (\eqn{N_{trade}}).
It can be calculated using the \code{\link[=ntrade]{ntrade()}} function.}

\item{IDs_col}{A string specifying the column name in \code{ntrade_data} with the
country or region IDs of interest. See details on
\strong{IDs - country or region identification codes}.}

\item{values_col}{A string specifying the column name in \code{ntrade_data} with
the \eqn{N_{trade}} values (quantity of potentially infested commodity imports)
to be used in the pathway model.}

\item{expression}{A string of characters representing the equation for the pathway model.
This expression must not include \eqn{N_{trade}}, since it is added multiplicatively to the
entered equation by default. The resulting equation will be of the form:
\deqn{NPFP = N_{trade_i} \cdot \; ``expression"}}

\item{parameters}{A named list specifying the distributions for each parameter
used in \code{expression}. Each element of the list must be another list containing:
\itemize{
\item \code{dist}: A string indicating the distribution name (e.g., "norm", "beta").
\item Additional arguments required by the specified distribution (e.g., \code{mean},
\code{sd} for "norm").
}
See details on \strong{Parameter distributions} for a list of available distributions
and examples on how to specify them.}

\item{niter}{The number of iterations to generate random samples from the distributions.
The default is 100 iterations.}
}
\value{
A data frame with the statistics (mean, SD, minimum, first quartile,
median, third quartile, and maximum) resulting from the iterations of the \eqn{NPFP}
for each country/region and for the total (i.e., the results for the set of all
countries/regions).
}
\description{
Estimates the number of potential founder populations (\eqn{NPFP}) of a pest
in different regions, using \eqn{N_{trade}} data combined with additional user-defined
parameters.
}
\details{
\subsection{IDs - country or region identification codes:}{

The use of ISO 3166 (alpha-2) codes
(\href{https://www.iso.org/iso-3166-country-codes.html}{ISO 3166 Maintenance Agency}),
or NUTS codes in the case of European countries
\href{https://ec.europa.eu/eurostat/web/nuts}{Nomenclature of territorial units for statistics},
as country or region identifiers (\code{IDs_col}) is recommended
for subsequent compatibility with other functions of the  \link{qPRAentry} package.
}

\subsection{Parameter distributions:}{

The following distributions are supported. For details on their parameters, refer
to the linked R documentation:

\tabular{lll}{
\strong{Distribution} \tab \verb{ } \tab\strong{Documentation} \cr
"beta"    \tab \verb{ } \tab \code{\link[=rbeta]{rbeta()}} (Beta distribution) \cr
"binom"   \tab \verb{ } \tab \code{\link[=rbinom]{rbinom()}} (Binomial distribution) \cr
"cauchy"  \tab \verb{ } \tab \code{\link[=rcauchy]{rcauchy()}} (Cauchy distribution) \cr
"chisq"   \tab \verb{ } \tab \code{\link[=rchisq]{rchisq()}} (Chi-squared distribution) \cr
"exp"     \tab \verb{ } \tab \code{\link[=rexp]{rexp()}} (Exponential distribution) \cr
"f"       \tab \verb{ } \tab \code{\link[=rf]{rf()}} (F distribution) \cr
"gamma"   \tab \verb{ } \tab \code{\link[=rgamma]{rgamma()}} (Gamma distribution) \cr
"geom"    \tab \verb{ } \tab \code{\link[=rgeom]{rgeom()}} (Geometric distribution) \cr
"lnorm"   \tab \verb{ } \tab \code{\link[=rlnorm]{rlnorm()}} (Log-normal distribution) \cr
"nbinom"  \tab \verb{ } \tab \code{\link[=rnbinom]{rnbinom()}} (Negative Binomial distribution) \cr
"norm"    \tab \verb{ } \tab \code{\link[=rnorm]{rnorm()}} (Normal distribution) \cr
"pois"    \tab \verb{ } \tab \code{\link[=rpois]{rpois()}} (Poisson distribution) \cr
"t"       \tab \verb{ } \tab \code{\link[=rt]{rt()}} (Student's t distribution) \cr
"unif"    \tab \verb{ } \tab \code{\link[=runif]{runif()}} (Uniform distribution) \cr
"weibull" \tab \verb{ } \tab \code{\link[=rweibull]{rweibull()}} (Weibull distribution) \cr
}

For example, to specify a normal distribution with mean 0 and standard deviation 1:\cr
\code{list(dist = "norm", mean = 0, sd = 1)}

Ensure that all parameters required by the chosen distribution are included.
}
}
\examples{
## Example using Northern American countries and ntrade simulated data
data("datatrade_NorthAm")
# Extract country IDs and simulate ntrade data
IDs <- datatrade_NorthAm$internal_production$reporter
df <- data.frame(IDs = IDs,
                 ntrade_values = abs(rnorm(length(IDs), 10000, 2000)))
# Expression for the pathway model using 3 parameters
eq <- "(1/P1)*P2*P3"
# Distribution for each parameter
parameters <- list(
  P1 = list(dist = "beta", shape1 = 0.5, shape2 = 1),
  P2 = list(dist = "gamma", shape = 1.5, scale = 100),
  P3 = list(dist = "lnorm", mean = 5, sd = 2)
)
# Run pathway_model()
res_pathway <- pathway_model(ntrade_data = df,
                             IDs_col = "IDs",
                             values_col = "ntrade_values",
                             expression = eq,
                             parameters = parameters,
                             niter = 100)
head(res_pathway)
# summary of the total for all countries
res_pathway[res_pathway$IDs == "Total",]
# plot
plot_countries(res_pathway, "IDs", "Median")

}
\seealso{
\code{\link[=ntrade]{ntrade()}}
}
