% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\encoding{UTF-8}
\name{knn_euclid}
\alias{knn_euclid}
\title{Euclidean Nearest Neighbours}
\usage{
knn_euclid(
  X,
  k = 1L,
  Y = NULL,
  algorithm = "auto",
  max_leaf_size = 0L,
  squared = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{X}{the "database"; a matrix of shape \eqn{n\times d}}

\item{k}{requested number of nearest neighbours
(should be rather small)}

\item{Y}{the "query points"; \code{NULL} or a matrix of shape \eqn{m\times d};
note that setting \code{Y=X}, contrary to \code{NULL},
will include the query points themselves amongst their own neighbours}

\item{algorithm}{\code{"auto"}, \code{"kd_tree"} or \code{"brute"};
K-d trees can be used for \code{d} between 2 and 20 only;
\code{"auto"} selects \code{"kd_tree"} in low-dimensional spaces}

\item{max_leaf_size}{maximal number of points in the K-d tree leaves;
smaller leaves use more memory, yet are not necessarily faster;
use \code{0} to select the default value, currently set to 32}

\item{squared}{whether the output \code{nn.dist} should be based on
the squared Euclidean distance}

\item{verbose}{whether to print diagnostic messages}
}
\value{
A list with two elements, \code{nn.index} and \code{nn.dist}, is returned.

\code{nn.dist} and \code{nn.index} have shape \eqn{n\times k}
or \eqn{m\times k}, depending whether \code{Y} is given.

\code{nn.index[i,j]} is the index (between \eqn{1} and \eqn{n})
of the \eqn{j}-th nearest neighbour of \eqn{i}.

\code{nn.dist[i,j]} gives the weight of the edge \code{{i, nn.index[i,j]}},
i.e., the distance between the \eqn{i}-th point and its \eqn{j}-th
nearest neighbour, \eqn{j=1,\dots,k}.
\code{nn.dist[i,]} is sorted nondecreasingly for all \eqn{i}.
}
\description{
If \code{Y} is \code{NULL}, then the function determines the first \code{k}
nearest neighbours of each point in \code{X} with respect
to the Euclidean distance. It is assumed that each query point is
not its own neighbour.

Otherwise, for each point in \code{Y}, this function determines the \code{k}
nearest points thereto from \code{X}.
}
\details{
The implemented algorithms, see the \code{algorithm} parameter, assume
that \eqn{k} is rather small, say, \eqn{k \leq 20}.

Our implementation of K-d trees (Bentley, 1975) has been quite optimised;
amongst others, it has good locality of reference (at the cost of making
a copy of the input dataset), features the sliding
midpoint (midrange) rule suggested by Maneewongvatana and Mound (1999),
node pruning strategies inspired by some ideas from (Sample et al., 2001),
and a couple of further tuneups proposed by the current author.
Still, it is well-known that K-d trees
perform well only in spaces of low intrinsic dimensionality.  Thus,
due to the so-called curse of dimensionality, for high \code{d},
the brute-force algorithm is recommended.

The number of threads used is controlled via the \code{OMP_NUM_THREADS}
environment variable or via the \code{\link{omp_set_num_threads}} function
at runtime. For best speed, consider building the package
from sources using, e.g., \code{-O3 -march=native} compiler flags.
}
\examples{
library("datasets")
data("iris")
X <- jitter(as.matrix(iris[1:2]))  # some data
neighbours <- knn_euclid(X, 1)  # 1-NNs of each point
plot(X, asp=1, las=1)
segments(X[,1], X[,2], X[neighbours$nn.index,1], X[neighbours$nn.index,2])

knn_euclid(X, 5, matrix(c(6, 4), nrow=1))  # five closest points to (6, 4)


}
\references{
J.L. Bentley, Multidimensional binary search trees used for associative
searching, \emph{Communications of the ACM} 18(9), 509–517, 1975,
\doi{10.1145/361002.361007}.

S. Maneewongvatana, D.M. Mount, It's okay to be skinny, if your friends
are fat, \emph{4th CGC Workshop on Computational Geometry}, 1999.

N. Sample, M. Haines, M. Arnold, T. Purcell, Optimizing search
strategies in K-d Trees, \emph{5th WSES/IEEE Conf. on Circuits, Systems,
Communications & Computers} (CSCC'01), 2001.
}
\seealso{
The official online manual of \pkg{quitefastmst} at \url{https://quitefastmst.gagolewski.com/}

\code{\link{mst_euclid}}
}
\author{
\href{https://www.gagolewski.com/}{Marek Gagolewski}
}
