% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_rate.bg.R
\name{calc_rate.bg}
\alias{calc_rate.bg}
\title{Calculate background oxygen uptake or input rates}
\usage{
calc_rate.bg(x, time = NULL, oxygen = NULL, plot = TRUE, ...)
}
\arguments{
\item{x}{\code{data.frame} or \code{inspect} object. This is the data to extract
background rate(s) from.}

\item{time}{integer. Defaults to 1. This specifies the column number of the
time data.}

\item{oxygen}{integer value or vector. This specifies the column number(s) of
the oxygen data. Multiple columns of oxygen can be specified. If NULL,
function assumes oxygen data are in \emph{all} columns of the data frame except
the \code{time} column.}

\item{plot}{logical. Defaults to TRUE. Plots the data. See Details.}

\item{...}{Allows additional plotting controls to be passed, such as \code{pos},
\code{legend = FALSE}, and \code{quiet = TRUE}.}
}
\value{
Output is a \code{list} object of class \code{calc_rate.bg} containing original
data, linear models, summary information, and the primary output of
interest \verb{$rate.bg}, which contains a rate for each oxygen column present
in the input data. There is also \verb{$rate.bg.mean} containing the mean of all
background rates. Note, this is not used in \code{adjust_rate}, where the
\code{method} input there determines how \verb{$rate.bg} is applied, but can easily
be extracted and applied as an adjustment value if desired.
}
\description{
This function calculates the rate of change of oxygen over time from "blank"
or control respirometry experiments, to allow for background adjustments of
experimental data. It accepts background oxygen~time data as data frames and
\code{inspect} objects. The data must be in the same time and oxygen units as the
data from which the rate which will be adjusted was extracted. Multiple
columns of background oxygen measurements can be entered as long as they
share the same time data. In this case the function returns rates for all
columns, and also calculates a mean rate.
}
\details{
The main difference between \code{calc_rate.bg} and \code{calc_rate}, is that this
function allows a rate to be determined from the same region of multiple
oxygen data columns, whereas \code{calc_rate} allows multiple rates to be
determined from different regions of a single dataset.
\subsection{Units}{

There are no units involved in \code{calc_rate.bg}. This is a deliberate decision.
The units of oxygen concentration and time will be specified later in
\code{\link[=convert_rate]{convert_rate()}} when rates are converted to specific output units. It is
important however, the background time~oxygen data is in the same time and
oxygen units as the data used to determine the rate which will be adjusted.
}

\subsection{Subsetting data regions}{

\code{calc_rate.bg} does not have internal subsetting of data regions. If you need
to subset the data to specific regions you don't want to use, see
\code{\link[=subset_data]{subset_data()}}, which allows for easy passing (or piping) of subsets to
\code{calc_rate.bg}.
}

\subsection{Background respiration vs background input of oxygen}{

Most users will be using this function to account for background oxygen
consumption rates from microbial activity that need to be quantified and
their effects removed from experimental specimen rates. However, there are
some experiments where oxygen \emph{input} rates may be of interest, for example
in open tank or open arena respirometry where the input of oxygen from the
water surface has been calculated or quantified. There are also cases in
closed respirometry where there may be an input of oxygen via leaks or oxygen
production from photosynthesis which need to be quantified. \code{calc_rate.bg} is
readily capable of quantifying production rates as well as consumption, and
these can also be used for adjustments in \code{\link[=adjust_rate]{adjust_rate()}}.
}

\subsection{Plot}{

A plot is produced (provided \code{plot = TRUE}) showing all examined columns of
oxygen against time (bottom blue axis) and row index (top red axis), with the
rate and linear model coefficients. Single rates can be plotted by changing
the \code{pos} input either in the main function call, or by plotting the output,
e.g. \code{plot(object, pos = 2)}. Console output messages can be suppressed using
\code{quiet = TRUE}. If equations obscure the plot they can be suppressed using
\code{legend = FALSE}.
}

\subsection{S3 Generic Functions}{

Saved output objects can be used in the generic S3 functions \code{print()},
\code{summary()}, and \code{mean()}.
\itemize{
\item \code{print()}: prints all background rates, plus the mean background rate.
\item \code{summary()}: prints summary table of all results and metadata, or those
specified by the \code{pos} input. e.g. \code{summary(x, pos = 1:5)}. The summary can
be exported as a separate dataframe by passing \code{export = TRUE}.
\item \code{mean()}: calculates the mean of all rates, or those specified by the \code{pos}
input. e.g. \code{mean(x, pos = 1:5)} The mean can be exported as a separate value
by passing \code{export = TRUE}.
}
}

\subsection{More}{

For additional help, documentation, vignettes, and more visit the \code{respR}
website at \url{https://januarharianto.github.io/respR/}
}
}
\examples{
\donttest{
# Inspect and calculate background rate from two columns
inspect(urchins.rd, time = 1, oxygen = 18:19) \%>\%
  calc_rate.bg()

# Same example but enter as a data frame, save as an object and use
# in adjust_rate
bg_rate <- calc_rate.bg(urchins.rd,
                        time = 1,
                        oxygen = 18:19,
                        plot = FALSE)

inspect(urchins.rd, 1, 2, plot = FALSE) \%>\%
  calc_rate(from = 10, to = 30, by = "time", plot = FALSE) \%>\%
  adjust_rate(by = bg_rate)

# Subset single column data first before calculating background rate
subset_data(background_con.rd, from = 5000, to = 20000, by = "time") \%>\%
  calc_rate.bg()
  }
}
