% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rma.exact.R
\name{rma.exact}
\alias{rma.exact}
\title{Compute an exact confidence interval for the grand mean in a normal-normal random effects meta-analysis.}
\usage{
rma.exact(yi, vi, c0 = 1, level = 0.05, mu.bounds = NULL,
  tau2.bounds = NULL, resolution = 100, resolution.mu = resolution,
  resolution.tau2 = resolution, Z = NULL, B = 3000, mu.alpha = 0.995,
  tau2.alpha = 0.995, plot = TRUE, test.stat = NULL, ...)
}
\arguments{
\item{yi}{a vector containing the primary study measurements}

\item{vi}{a vector of the same length as yi containing the variances of the of the primary study measurements contained in yi}

\item{c0}{a vector containing the mixing parameters for the test statistics}

\item{level}{the level of the confidence interval; set to NULL to plot a confidence region, otherwise rma.exact.fast is called using the specified level}

\item{mu.bounds}{upper and lower bounds for the range of population effect values for constructing the confidence region; if NULL, value will be calculated from mu.alpha}

\item{tau2.bounds}{upper and lower bounds for the range of population variance values for constructing the confidence region; if NULL, value will be calculated from tau2.alpha}

\item{resolution}{resolution of the population mean and variance values within the bounding box}

\item{resolution.mu}{resolution of the population mean values within the bounding box}

\item{resolution.tau2}{resolution of the population variance values within the bounding box}

\item{Z}{a matrix of length(yi) rows with each row consisting of standard normal samples to be used in the monte carlo estimation of the null distribution of the test statistic; if NULL, B values
will be sampled per row}

\item{B}{the number of monte carlo replicates per primary study observation to be used}

\item{mu.alpha}{the level of the exact CI for constructing the bounds on the population mean dimension of the bounding box}

\item{tau2.alpha}{the level of the exact CI for constructing the bounds on the population variance dimension of the bounding box}

\item{plot}{whether to plot the confidence region (if level is not NULL) or its boundary (if level is NULL)}

\item{test.stat}{(currently for internal use)}

\item{...}{(currently for internal use)}
}
\value{
if "level" is not NULL, a named vector of CI endpoints; otherwise, an object of class  RMA.Exact
}
\description{
Compute an exact confidence interval for the grand mean in a normal-normal random effects meta-analysis.
}
\details{
Computes an exact (up to monte carlo error), unconditional, non-randomized CI for the grand mean in a random effects meta-analysis assuming a normal-normal model for the primary study observations. This function implements the algorithm described in:

Michael, Thornton, Xie, and Tian (2017). Exact Inference on the Random Effects Model for Meta-Analyses with Few Studies. (Submitted.)

If the parameter "level" is not NULL (the default value is .05), this function passes the call down to rma.exact.fast, which computes a CI at the specified level. If "level" is set to NULL, an entire 2-dimensional grid of p-values is estimated. In the latter case, an RMA.Exact object is returned, which may be passed to a plot routine to plot contours of confidence regions.
}
\examples{
set.seed(1)

K <- 5
c0 <- 1
mu0 <- 0
tau2 <- 12.5
vi <- (seq(1, 5, length=K))^2
yi <- rnorm(K)*sqrt(vi+tau2)+mu0
rma.exact(yi=yi,vi=vi)

## plotting a conifdence region and printing CIs with an RMA.Exact object
rma0 <- rma.exact(yi=yi,vi=vi,level=NULL)
plot(rma0)
confint(rma0)

\donttest{
## confidence region with multiple c0 values using an RMA.Exact object
c0 <- c(0,.25,1)
tau2 <- 12.5
vi <- (seq(1, 5, length=K))^2
yi=rnorm(K)*sqrt(vi+tau2)+mu0
rma0 <- rma.exact(yi=yi,vi=vi,c0=c0,level=NULL)
plot(rma0)
confint(rma0)

## setting tau2.bounds and other parameters to non-default values using an RMA.Exact object
Z <- matrix(rnorm(K*5e3),nrow=K)
B <- ncol(Z)
resolution <- 3e2
rma0 <- rma.exact(yi=yi,vi=vi,level=NULL,Z=Z,resolution=resolution,c0=c0,
tau2.bounds=c(1,120),resolution.tau2=1e3,resolution.mu=1e2)
plot(rma0)

c0 <- 1:4
rma0 <- rma.exact(yi=yi,vi=vi,level=NULL,Z=Z,resolution=resolution,c0=c0,
tau2.bounds=c(1,450),resolution.tau2=1e3,resolution.mu=1e2)
plot(rma0)
confint(rma0,level=c(.05))
}
}
\seealso{
\code{\link{rma.exact.fast}} for computing confidence intervals at specified levels, \code{\link{plot.RMA.Exact}}, \code{\link{confint.RMA.Exact}}
}
