% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/conditionalJS.R
\name{jsCallOncePerFlush}
\alias{jsCallOncePerFlush}
\alias{conditionalJS}
\title{Run JS when condition is met}
\usage{
jsCallOncePerFlush(session)

conditionalJS(ui, condition, jsCall, once = TRUE, ns = shiny::NS(NULL))
}
\arguments{
\item{session}{Shiny session object.}

\item{ui}{A `shiny.tag` element to which the JS callback should be attached.}

\item{condition}{A JavaScript expression that will be evaluated repeatedly.
When the evaluated `condition` is true, `jsCall`'s true (`jsCall$true`) callback is run,
when false -  `jsCall$false` is executed in application browser.}

\item{jsCall}{A list of two `htmltools::JS` elements named 'true' and 'false'
storing JS expressions.
The 'true' object is evaluated when `condition` is true, 'false' otherwise.
In order to skip true/false callback assign it to NULL (or skip).
Use `this` object in the expressions to refer to the `ui` object.
See \link{jsCalls} for possible actions.}

\item{once}{Should the JS action be called only when condition state changes?}

\item{ns}{The \link[shiny]{NS} object of the current module, if any.}
}
\description{
`conditionalJS` is an extended version of \link[shiny]{conditionalPanel}.
The function allows to run selected or custom JS action when the provided
condition is true or false.

To see the possible JS actions check \link{jsCalls}.

Optionally call `jsCallOncePerFlush` in server to assure the call is run once
per application flush cycle (see. https://github.com/rstudio/shiny/issues/3668).
This prevents i.e. running animation multiple times when
`runAnimation(once = FALSE)` is used.
}
\examples{
if (interactive()) {
  library(shiny)

  ui <- fluidPage(
    tags$style(".boldme {font-weight: bold;}"),
    sliderInput("value", "Value", min = 1, max = 10, value = 1),
    textOutput("slid_val"),
    conditionalJS(
      tags$button("Show me when slider value at least 3"),
      "input.value >= 3",
      jsCalls$show()
    ),
    hr(),
    conditionalJS(
      tags$button("Show me when value less than 3"),
      "input.value >= 3",
      jsCalls$show(when = FALSE)
    ),
    hr(),
    conditionalJS(
      tags$button("I'm disabled when value at least 4"),
      "input.value >= 4",
      jsCalls$disable()
    ),
    hr(),
    conditionalJS(
      tags$button("I'm disabled when value less than 4"),
      "input.value >= 4",
      jsCalls$disable(when = FALSE)
    ),
    hr(),
    conditionalJS(
      tags$button("I have class 'boldme' when value at least 5"),
      "input.value >= 5",
      jsCalls$attachClass("boldme")
    ),
    hr(),
    conditionalJS(
      tags$button("I change color when value at least 6"),
      "input.value >= 6",
      jsCalls$custom(
        true = "$(this).css('background-color', 'red');",
        false = "$(this).css('background-color', 'green');"
      )
    ),
    hr(),
    conditionalJS(
      tags$button("I change border when value at least 7"),
      "input.value >= 7",
      jsCalls$css(
        border = "dashed"
      )
    ),
    hr(),
    conditionalJS(
      tags$button("I'm disabled permanently when value at least 8"),
      "input.value >= 8",
      jsCalls$disable()["true"] # remove false condition
    ),
    hr(),
    conditionalJS(
      tags$button("I bounce when value at least 9"),
      "input.value >= 9",
      jsCalls$custom(true = runAnimation()),
      once = FALSE
    )
  )

  server <- function(input, output, session) {
    output$slid_val <- renderText({
      input$value
    })
    jsCallOncePerFlush(session)
  }

  shinyApp(ui, server)
}

if (interactive()) {
  library(shiny)
  library(shinyGizmo)

  ui <- fluidPage(
    textInput("name", "Name"),
    conditionalJS(
      actionButton("value", "Type name to enable the button"),
      "input.name != ''",
      jsCalls$disable(when = FALSE)
    )
  )

  server <- function(input, output, session) {}

  shinyApp(ui, server)
}

}
