% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core.R
\name{euclidean_embedding}
\alias{euclidean_embedding}
\title{Main topolow algorithm implementation}
\usage{
euclidean_embedding(
  dissimilarity_matrix,
  ndim,
  mapping_max_iter = 1000,
  k0,
  cooling_rate,
  c_repulsion,
  relative_epsilon = 1e-04,
  convergence_counter = 5,
  initial_positions = NULL,
  write_positions_to_csv = FALSE,
  output_dir,
  verbose = FALSE
)
}
\arguments{
\item{dissimilarity_matrix}{Matrix. A square, symmetric dissimilarity matrix. Can contain
NA values for missing measurements and character strings with < or > prefixes for
thresholded measurements.}

\item{ndim}{Integer. Number of dimensions for the embedding space.}

\item{mapping_max_iter}{Integer. Maximum number of map optimization iterations.}

\item{k0}{Numeric. Initial spring constant controlling spring forces.}

\item{cooling_rate}{Numeric. Rate of spring constant decay per iteration (0 < cooling_rate < 1).}

\item{c_repulsion}{Numeric. Repulsion constant controlling repulsive forces.}

\item{relative_epsilon}{Numeric. Convergence threshold for relative change in error.
Default is 1e-4.}

\item{convergence_counter}{Integer. Number of iterations below threshold before declaring
convergence. Default is 5.}

\item{initial_positions}{Matrix or NULL. Optional starting coordinates. If NULL,
random initialization is used. Matrix should have nrow = nrow(dissimilarity_matrix)
and ncol = ndim.}

\item{write_positions_to_csv}{Logical. Whether to save point positions to a CSV file.
Default is FALSE.}

\item{output_dir}{Character. Directory to save the CSV file. Required if
\code{write_positions_to_csv} is TRUE.}

\item{verbose}{Logical. Whether to print progress messages. Default is FALSE.}
}
\value{
A \code{list} object of class \code{topolow}. This list contains the results of the
optimization and includes the following components:
\itemize{
\item \code{positions}: A \code{matrix} of the optimized point coordinates in the N-dimensional space.
\item \code{est_distances}: A \code{matrix} of the Euclidean distances between points in the final optimized configuration.
\item \code{mae}: The final Mean Absolute Error between the target dissimilarities and the estimated distances.
\item \code{iter}: The total number of iterations performed before the algorithm terminated.
\item \code{parameters}: A \code{list} containing the input parameters used for the optimization run.
\item \code{convergence}: A \code{list} containing the final convergence status, including a logical \code{achieved} flag and the final \code{error} value.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

topolow (topological stochastic pairwise reconstruction for Euclidean embedding) optimizes
point positions in an N-dimensional space to match a target dissimilarity matrix.
The algorithm uses a physics-inspired approach with spring and repulsive forces
to find optimal point configurations while handling missing and thresholded measurements.
}
\details{
The algorithm iteratively updates point positions using:
\itemize{
\item Spring forces between points with measured dissimilarities.
\item Repulsive forces between points without measurements.
\item Conditional forces for thresholded measurements (< or >).
\item An adaptive spring constant that decays over iterations.
\item Convergence monitoring based on relative error change.
\item Automatic matrix reordering to optimize convergence.
Consider if downstream analyses depend on specific point ordering: The order of points in
the output is adjusted to put high-dissimilarity points in the opposing ends.
}

This function replaces the deprecated \code{\link[=create_topolow_map]{create_topolow_map()}}. The core algorithm
is identical, but includes performance improvements and enhanced validation.
}
\examples{
# Create a simple dissimilarity matrix
dist_mat <- matrix(c(0, 2, 3, 2, 0, 4, 3, 4, 0), nrow=3)

# Run topolow in 2D
result <- euclidean_embedding(
  dissimilarity_matrix = dist_mat,
  ndim = 2,
  mapping_max_iter = 100,
  k0 = 1.0,
  cooling_rate = 0.001,
  c_repulsion = 0.01,
  verbose = FALSE
)

# View results
head(result$positions)
print(result$mae)

# Example with thresholded measurements
thresh_mat <- matrix(c(0, ">2", 3, ">2", 0, "<5", 3, "<5", 0), nrow=3)
result_thresh <- euclidean_embedding(
  dissimilarity_matrix = thresh_mat,
  ndim = 2,
  mapping_max_iter = 50,
  k0 = 0.5,
  cooling_rate = 0.01,
  c_repulsion = 0.001
)

}
\seealso{
\code{\link[=create_topolow_map]{create_topolow_map()}} for the deprecated predecessor function.
}
