#' Compute the product moment.
#' 
#' It computes the kappa-th order product moment for a truncated multivariate 
#'     normal distribution.
#' 
#' \code{prodmnt} is a S3 generic function of the class 
#'     \code{\link{mtrunmnt}}. Using the multivariate Gaussian quadrature, it 
#'     computes the product moment 
#'     \eqn{E(\prod_{i=1}^n Y_i^{k_i}) | a_i < Y_i < b_i, i = 1, \ldots, n)}, 
#'     where \eqn{Y \sim N(\mu, ZDZ' + E)}. 
#'     
#' @param Obj mtrunmnt object created by the \code{\link{mtrunmnt}}.
#' 
#' @param kappa Vector of orders of length equal to mu.   
#' 
#' @seealso \code{\link{mtrunmnt}}, \code{\link[MomTrunc]{momentsTMD}}.
#' 
#' @return a numeric value.    
#'
#' @export
#'  
#' @examples 
#' ###  A simple example ####
#' 
#' set.seed(123)
#' sigma2e <- 1
#' sigma2a <- 2
#' n <- 5
#' mu <- seq(-1,1, length.out = n)
#' y <- mu + rnorm(1, sd = sqrt(sigma2a)) + rnorm(n, sd = sqrt(sigma2e))
#' a  <- rep(-Inf, n)
#' b  <- rep(Inf, n)
#' a[y >= 0] <- 0
#' b[y <  0] <- 0
#' obj <- mtrunmnt(mu, lower = a, upper = b, Sigmae = sigma2e, D = sigma2a) 
#' prodmnt(obj, c(2,2,0,0,0))
#' 

prodmnt <- function(Obj, kappa){
  UseMethod("prodmnt")
}

#' @export
prodmnt.mtrunmnt <- function(Obj, kappa){
  pmoment <- getExpect(kappa, Obj$lower, Obj$upper, Obj$mu, Obj$sigmae, Obj$probab, Obj$Z, Obj$D, Obj$weights, Obj$nodes)
  return(pmoment)   
}

