% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dpqr-ugomquant.R
\name{UGOM}
\alias{UGOM}
\alias{dUGOM}
\alias{pUGOM}
\alias{qUGOM}
\alias{rUGOM}
\title{The unit-Gompertz distribution - quantile parameterization}
\usage{
dUGOM(x, mu, sigma, tau = 0.5, log = FALSE)

pUGOM(q, mu, sigma, tau = 0.5, lower.tail = TRUE, log.p = FALSE)

qUGOM(p, mu, sigma, tau = 0.5, lower.tail = TRUE, log.p = FALSE)

rUGOM(n, mu, sigma, tau = 0.5)

UGOM(mu.link = "logit", sigma.link = "log")
}
\arguments{
\item{x, q}{vector of quantiles on the (0,1) interval.}

\item{mu}{vector of quantile parameter values.}

\item{sigma}{vector of shape parameter values.}

\item{tau}{the \eqn{\tau}-th fixed quantile in [d-p-q-r]-UGOM function.}

\item{log, log.p}{logical; If TRUE, probabilities p are given as log(p).}

\item{lower.tail}{logical; If TRUE, (default), \eqn{P(X \leq{x})} are returned, otherwise \eqn{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{the number of observations. If \code{length(n) > 1}, the length is taken to be the number required.}

\item{mu.link}{the mu link function with default logit.}

\item{sigma.link}{the sigma link function with default logit.}
}
\value{
\code{UGOM()} return a gamlss.family object which can be used to fit a unit-Gompertz distribution by gamlss() function.
}
\description{
The function \code{UGOM()} define the unit-Gompertz distribution for a \code{gamlss.family} object to be used in GAMLSS fitting. \code{UGOM()} has the \eqn{\tau}-th quantile equal to the parameter mu and sigma as the shape parameter. The functions \code{dUGOM}, \code{pUGOM}, \code{qUGOM} and \code{rUGOM} define the density, distribution function, quantile function and random generation for unit-Gompertz distribution.
}
\details{
Probability density function
\deqn{f\left( {x\mid \mu ,\sigma ,\tau } \right)=\left ( \frac{\log \left( \tau \right) }{1-\mu ^{-\sigma }} \right ) \sigma x^{-\left( 1+\sigma \right) }\exp \left[ \left ( \frac{\log \left( \tau \right) }{1-\mu ^{-\sigma }} \right ) \left( 1-x^{-\sigma }\right) \right] }

Cumulative distribution function
\deqn{F\left({x\mid \mu ,\sigma ,\tau } \right) = \exp \left[ \left ( \frac{\log \left( \tau \right) }{1-\mu ^{-\sigma }} \right ) \left( 1-x^{-\sigma }\right) \right]}

Mean
\deqn{E(X)=\left( \frac{\log \left( \tau \right) }{1-\mu ^{-\sigma }}\right) ^{\frac{1}{\theta }}\exp \left(\frac{\log \left( \tau \right) }{1-\mu ^{-\sigma }}\right)\Gamma \left( \frac{\sigma -1}{\sigma },\frac{\log \left( \tau \right) }{  1-\mu ^{-\sigma }}\right)}

where \eqn{0 < (x, \mu)<1}, \eqn{\mu} is, for a fixed and known value of \eqn{\tau}, the \eqn{\tau}-th quantile, \eqn{\sigma} is the shape parameter and \eqn{\Gamma(a, b)} is the upper incomplete gamma function.
}
\note{
Note that for \code{UGOM()}, mu is the \eqn{\tau}-th quantile and sigma a shape parameter. The \code{\link[gamlss]{gamlss}} function is used for parameters estimation.
}
\examples{

set.seed(123)
x <- rUGOM(n = 1000, mu = 0.50, sigma = 1.69, tau = 0.50)
R <- range(x)
S <- seq(from = R[1], to = R[2], length.out = 1000)

hist(x, prob = TRUE, main = 'unit-Gompertz')
lines(S, dUGOM(x = S, mu = 0.50, sigma = 1.69, tau = 0.50), col = 2)

plot(ecdf(x))
lines(S, pUGOM(q = S, mu = 0.50, sigma = 1.69, tau = 0.50), col = 2)

plot(quantile(x, probs = S), type = "l")
lines(qUGOM(p = S, mu = 0.50, sigma = 1.69, tau = 0.50), col = 2)

library(gamlss)
set.seed(123)
data <- data.frame(y =  rUGOM(n = 100, mu = 0.5, sigma = 2.0, tau = 0.5))

tau <- 0.50
fit <- gamlss(y ~ 1, data = data, family = UGOM)

set.seed(123)
n <- 100
x <- rbinom(n, size = 1, prob = 0.5)
eta <- 0.5 + 1 * x;
mu <- 1 / (1 + exp(-eta));
sigma <- 1.5;
y <- rUGOM(n, mu, sigma, tau = 0.5)
data <- data.frame(y, x)

tau <- 0.50
fit <- gamlss(y ~ x, data = data, family = UGOM(mu.link = "logit", sigma.link = "log"))
}
\references{
Hastie, T. J. and Tibshirani, R. J. (1990). \emph{Generalized Additive Models}. Chapman and Hall, London.

Mazucheli, J., Alve, B. (2021). The Unit-Gompertz quantile regression model for bounded responses. \emph{preprint}, \bold{0}(0), 1-20.

Mazucheli, J., Menezes, A. F. and Dey S. (2019). Unit-Gompertz distribution with applications. \emph{Statistica}, \bold{79}(1), 25--43.

Rigby, R. A. and  Stasinopoulos, D. M. (2005). Generalized additive models for location, scale and shape (with discussion). \emph{Applied. Statistics}, \bold{54}(3), 507--554.

Rigby, R. A., Stasinopoulos, D. M.,  Heller, G. Z. and De Bastiani, F. (2019). \emph{Distributions for modeling location, scale, and shape: Using GAMLSS in R}. Chapman and Hall/CRC.

Stasinopoulos, D. M. and Rigby, R. A. (2007) Generalized additive models for location scale and shape (GAMLSS) in R. \emph{Journal of Statistical Software}, \bold{23}(7), 1--45.

Stasinopoulos, D. M., Rigby, R. A., Heller, G., Voudouris, V. and De Bastiani F. (2017) \emph{Flexible Regression and Smoothing: Using GAMLSS in R}, Chapman and Hall/CRC.
}
\author{
Josmar Mazucheli \email{jmazucheli@gmail.com}

Bruna Alves \email{pg402900@uem.br}
}
