% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/phasePortraitBw.R
\name{phasePortraitBw}
\alias{phasePortraitBw}
\title{Create two-color phase portraits of complex functions}
\usage{
phasePortraitBw(
  FUN,
  moreArgs = NULL,
  xlim,
  ylim,
  invertFlip = FALSE,
  res = 150,
  blockSizePx = 2250000,
  tempDir = NULL,
  nCores = max(1, parallel::detectCores() - 1),
  bwType = "ma",
  pi2Div = 18,
  logBase = exp(2 * pi/pi2Div),
  argOffset = 0,
  bwCols = c("black", "gray95", "gray"),
  asp = 1,
  deleteTempFiles = TRUE,
  noScreenDevice = FALSE,
  autoDereg = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{FUN}{The function to be visualized. There are two possibilities to
provide it, a quoted character string, or a function object. The quoted
character string must contain an expression that can be interpreted by R as
a function of a complex number \code{z} (like e.g. "sin(z)", "(z^2 -
1i)/(tan(z))", "1/4*z^2 - 10*z/(z^4+4)"). See the documentation of
\code{\link{phasePortrait}} for a complete presentation of all options.}

\item{moreArgs}{A named list of other arguments to FUN. The names must match
the names of the arguments in FUN's definition.}

\item{xlim}{The x limits (x1, x2) of the plot as a two-element numeric
vector. Follows exactly the same definition as in
\code{\link{plot.default}}. Here, \code{xlim} has to be interpreted as the
plot limits on the real axis.}

\item{ylim}{The y limits of the plot (y1, y2) to be used in the same way as
\code{xlim}. Evidently, \code{ylim} indicates the plot limits on the
imaginary axis.}

\item{invertFlip}{If \code{TRUE}, the function is mapped over a z plane,
which has been transformed to \code{1/z * exp(1i*pi)}. This is the
projection required to plot the north Riemann hemisphere in the way
proposed by \insertCite{wegert_visualcpx_2012;textual}{viscomplexr}, p. 41.
Defaults to \code{FALSE}. If this option is chosen, the numbers at the axis
ticks have another meaning than in the normal case. Along the real axis,
they represent the real part of \code{1/z}, and along the imaginary axis,
they represent the imaginary part of \code{1/z}. Thus, if you want
annotation, you should choose appropriate axis labels like \code{xlab =
Re(1/z)}, and \code{ylab = Im(1/z)}.}

\item{res}{Desired resolution of the plot in dots per inch (dpi). Default is
150 dpi. All other things being equal, \code{res} has a strong influence on
computing times (double \code{res} means fourfold number of pixels to
compute). A good approach could be to make a plot with low resolution (e.g.
the default 150 dpi) first, adjust whatever required, and plot into a
graphics file with high resolution after that.}

\item{blockSizePx}{Number of pixels and corresponding complex values to be
processed at the same time (see Details). Default is 2250000. This value
gave good performance on older systems as well as on a high-end gaming
machine, but some tweaking for your individual system might even improve
things.}

\item{tempDir}{\code{NULL} or a character string, specifying the name of the
directory where the temporary files written by \code{phasePortrait} are
stored. Default is \code{NULL}, which makes \code{phasePortrait} use the
current R session's temporary directory. Note that if you specify another
directory, it will be created if it does not exist already. Even though the
temporary files are deleted after completing a phase portrait (unless the
user specifies \code{deleteTempFiles = FALSE}, see below), the directory
will remain alive even if has been created by \code{phasePortrait}.}

\item{nCores}{Number of processor cores to be used in the parallel computing
tasks. Defaults to the maximum number of cores available minus 1. Any
number between 1 (serial computation) and the maximum number of cores
available as indicated by \code{parallel::detectCores()} is accepted. If
\code{nCores} is set to a value greater than the available number of cores,
the function will use one core less than available.}

\item{bwType}{One of the three options for plotting, "m", "a", and "ma", to
be provided as a character string. Defaults to "ma". This parameter has a
comparable role to the parameter \code{pType} in
\code{\link{phasePortrait}}. Option "m" produces a plot that colors modulus
zones only. In more detail, for each input number's modulus, the logarithm
with base \code{logBase} (see below) is calculated and cut down to the next
lower integer value. If this is an even number, the first color given in
\code{bwCols} (see below) is taken. In case of an odd number, the second
color is used. Option "a" produces a plot that exclusively colors argument
(phase angle) zones. To that end, the full angle (2*pi) is divided into
\code{p2Div} (see below) zones, which are numbered from 0 to pi2Div - 1
with increasing angle. Such an integer number is attributed to the complex
number of interest according to the zone it falls into. Even and odd zone
numbers are mapped to the first and the second color in \code{bwCols},
respectively. For normal purposes, the input parameter \code{pi2Div} should
be an even number in order to avoid the first and the last zone having the
same color. With option "ma", a chessboard-like alternation of colors is
displayed over the tiles formed by the intersecting modulus and argument
zones (both determined separately as with the options "m" and "a").}

\item{pi2Div}{Angle distance for the argument reference zones added for
\code{pType = "pma"} or \code{pType = "pa"}. The value has to be given as
an integer (reasonably) fraction of 2*pi (i.e. 360 degrees). Unlike with
\code{\link{phasePortrait}}, the default is 18; thus, reference zones are
delineated by default in distances of 2*pi/18, i.e. (20 degrees), starting
with 0 if not defined otherwise with the parameter \code{argOffset}. While
the default of \code{pi2Div} is 9 with \code{\link{phasePortrait}} for good
reasons (see there), setting \code{pi2Div} to an odd number is usually not
a good choice with two-color phase portraits, because the first and the
last phase angle zone would get the same color. However, as \code{pi2Div}
here defaults to double the value as with \code{\link{phasePortrait}}, both
plot types can be nicely compared even when using their specific defaults
of \code{pi2Div}.}

\item{logBase}{Modulus ratio between the edges of the modulus zones in
\code{bwType} \code{"m"} and \code{"ma"}. As recommended by
\insertCite{wegert_visualcpx_2012;textual}{viscomplexr}, the default
setting is \code{logBase = exp(2*pi/pi2Div)}. This relation between the
parameters \code{logBase} and \code{pi2Div} ensures an analogue scaling of
the modulus and argument reference zones (see Details section in the
documentation of \code{\link{phasePortrait}}). Conveniently, for the
default \code{pi2Div = 18}, we obtain \code{logBase == 1.4177...}, which is
very close to the square root of 2. Thus, when crossing two modulus zones,
the modulus at the higher edge of the second zone is almost exactly two
times the value at the lower edge of the first zone.}

\item{argOffset}{The (complex number) argument in radians counterclockwise,
at which the argument (phase angle) reference zones are fixed, i.e. the
lower angle of the first zone. Default is 0.}

\item{bwCols}{Color definition for the plot provided as a character vector of
length 3. Each element of the vector must be either a color name R
recognizes, or a hexadecimal color string like e.g. "#00FF11". The first
and the second color make the appearance of two-color phase portraits (see
\code{bwType} above for details), while the third color is reserved for
special cases, where the input value cannot sufficiently evaluated (NaNs,
partly Inf). Defaults to c("black", "gray95", "gray"), which leads to an
alternation of black and very light gray zones or tiles, and uses a neutral
gray in special cases.}

\item{asp}{Aspect ratio y/x as defined in \code{\link{plot.window}}. Default
is 1, ensuring an accurate representation of distances between points on
the screen.}

\item{deleteTempFiles}{If TRUE (default), all temporary files are deleted
after the plot is completed. Set it on FALSE only, if you know exactly what
you are doing - the temporary files can occupy large amounts of hard disk
space (see details).}

\item{noScreenDevice}{Suppresses any graphical output if TRUE. This is only
intended for test purposes and makes probably only sense together with
\code{deleteTempFiles == FALSE}. For dimensioning purposes,
\code{phasePortraitBw} will use a 1 x 1 inch pseudo graphics device in this
case. The default for this parameter is \code{FALSE}, and you should change
it only if you really know what you are doing.}

\item{autoDereg}{if TRUE, automatically register sequential backend after the
plot is completed. Default is FALSE, because registering a parallel backend
can be time consuming. Thus, if you want to make several phase portraits in
succession, you should set \code{autoDereg} only for the last one, or
simply type \code{foreach::registerDoSEQ} after you are done. In any case,
you don't want to have an unused parallel backend lying about.}

\item{verbose}{if TRUE (default), \code{phasePortraitBw} will continuously
write progress messages to the console. This is convenient for normal
purposes, as calculating larger phase portraits in higher resolution may
take several minutes. The setting \code{verbose = FALSE}, will suppress any
output to the console.}

\item{...}{All parameters accepted by the \code{\link{plot.default}}
function.}
}
\description{
\code{phasePortraitBw} allows for creating two-color phase portraits of
complex functions based on a polar chessboard grid (cf.
\insertCite{wegert_visualcpx_2012;textual}{viscomplexr}, p. 35). Compared to
the full phase portraits that can be made with \code{\link{phasePortrait}},
two-color portraits omit information. Especially in combination with full
phase portraits they can be, however, very helpful tools for interpretation.
Besides, two-color phase portraits have a special aesthetic appeal which is
worth exploring for itself. In its parameters and its mode of operation,
\code{phasePortraitBw} is very similar to \code{\link{phasePortrait}}.
}
\details{
This function is intended to be used inside the framework of R base graphics.
It plots into the active open graphics device where it will display the phase
plot of a user defined function as a raster image. If no graphics device is
open when called, the function will plot into the default graphics device.
This principle allows to utilize the full functionality of R base graphics.
All graphics parameters (\code{\link{par}}) can be freely set and the
function \code{phasePortrait} accepts all parameters that can be passed to
the \code{\link{plot.default}} function. This allows all kinds of plots -
from scientific representations with annotated axes and auxiliary lines,
notation, etc. to poster-like artistic pictures. The general mode of operation,
including the usage of parallel processing is exactly the same as with
\code{\link{phasePortrait}}, see details section there.
}
\examples{
# Map the complex plane on itself

# x11(width = 8, height = 8)     # Screen device commented out
                                 # due to CRAN test requirements.
                                 # Use it when trying this example
phasePortraitBw("z", xlim = c(-2, 2), ylim = c(-2, 2),
                xlab = "real", ylab = "imaginary",
                verbose = FALSE, # Suppress progress messages
                nCores = 2)      # Max. two cores allowed on CRAN
                                 # not a limit for your own use
\dontshow{
# R CMD check: make sure any open connections are closed afterward
foreach::registerDoSEQ()
doParallel::stopImplicitCluster()
}



# Sinus with default colors and default bwType ("ma")
\donttest{
# x11(width = 8, height = 8)       # Screen device commented out
                                   # due to CRAN test requirements.
                                   # Use it when trying this example
phasePortraitBw("sin(z)",
                xlim = c(-pi, pi),
                ylim = c(-pi, pi),
                verbose = FALSE,
                nCores = 2)        # Max. two cores allowed on CRAN
                                   # not a limit for your own use
  \dontshow{
  # R CMD check: make sure any open connections are closed afterward
  foreach::registerDoSEQ()
  doParallel::stopImplicitCluster()
  }
}



# Sinus with custom colors and bwType "a"
\donttest{
# x11(width = 8, height = 8)       # Screen device commented out
                                   # due to CRAN test requirements.
                                   # Use it when trying this example
phasePortraitBw("sin(z)",
                xlim = c(-pi, pi),
                ylim = c(-pi, pi),
                bwType = "a",
                bwCols = c("darkgreen", "green", "gray"),
                verbose = FALSE,
                nCores = 2)        # Max. two cores allowed on CRAN
                                   # not a limit for your own use
  \dontshow{
  # R CMD check: make sure any open connections are closed afterward
  foreach::registerDoSEQ()
  doParallel::stopImplicitCluster()
  }
}



# Sinus with custom colors and bwType "m"
\donttest{
# x11(width = 8, height = 8)       # Screen device commented out
                                   # due to CRAN test requirements.
                                   # Use it when trying this example
phasePortraitBw("sin(z)",
                xlim = c(-pi, pi),
                ylim = c(-pi, pi),
                bwType = "m",
                bwCols = c("darkblue", "skyblue", "gray"),
                verbose = FALSE,
                nCores = 2)        # Max. two cores allowed on CRAN
                                   # not a limit for your own use
  \dontshow{
  # R CMD check: make sure any open connections are closed afterward
  foreach::registerDoSEQ()
  doParallel::stopImplicitCluster()
  }
}



# Map the complex plane on itself, show all bwType options
\donttest{
# x11(width = 8, height = 8)       # Screen device commented out
                                   # due to CRAN test requirements.
                                   # Use it when trying this example
op <- par(mfrow = c(2, 2), mar = c(4.1, 4.1, 1.1, 1.1))
for(bwType in c("ma", "a", "m")) {
  phasePortraitBw("z", xlim = c(-2, 2), ylim = c(-2, 2),
                  bwType = bwType,
                  xlab = "real", ylab = "imaginary",
                  verbose = FALSE, # Suppress progress messages
                  nCores = 2)      # Max. two cores allowed on CRAN
                                   # not a limit for your own use
}
# Add normal phase portrait for comparison
phasePortrait("z", xlim = c(-2, 2), ylim = c(-2, 2),
              xlab = "real", ylab = "imaginary",
              verbose = FALSE,
              pi2Div = 18,         # Use same angular division as default
                                   # in phasePortraitBw
              nCores = 2)
par(op)

  \dontshow{
  # R CMD check: make sure any open connections are closed afterward
  foreach::registerDoSEQ()
  doParallel::stopImplicitCluster()
  }
}


# A rational function, show all bwType options
\donttest{
# x11(width = 8, height = 8)       # Screen device commented out
                                   # due to CRAN test requirements.
                                   # Use it when trying this example
funString <- "(z + 1.4i - 1.4)^2/(z^3 + 2)"
op <- par(mfrow = c(2, 2), mar = c(4.1, 4.1, 1.1, 1.1))
for(bwType in c("ma", "a", "m")) {
  phasePortraitBw(funString, xlim = c(-2, 2), ylim = c(-2, 2),
                  bwType = bwType,
                  xlab = "real", ylab = "imaginary",
                  verbose = FALSE, # Suppress progress messages
                  nCores = 2)      # Max. two cores allowed on CRAN
                                   # not a limit for your own use
}
# Add normal phase portrait for comparison
phasePortrait(funString, xlim = c(-2, 2), ylim = c(-2, 2),
              xlab = "real", ylab = "imaginary",
              verbose = FALSE,
              pi2Div = 18,         # Use same angular division as default
                                   # in phasePortraitBw
              nCores = 2)
par(op)

  \dontshow{
  # R CMD check: make sure any open connections are closed afterward
  foreach::registerDoSEQ()
  doParallel::stopImplicitCluster()
  }
}


}
\references{
\insertAllCited{}
}
