#ifndef __errlib_h
#define __errlib_h

#include <errno.h>
#include <setjmp.h>
/* The Package Variable should be a constant variable so that a single
   Error function could handle errors from multiple packages. 
   Error functions are officially defined as never returning.
   They MUST either abort the program, or longjmp outside of the calling
   function.  This is done because it makes all the error handling in 
   libraries much MUCH easier, because you can nest functions in your 
   library and just assume that everything is proceeding as planned.
   An implementation of exceptions is handled which allows for slightly
   easier handling of the setjmp longjmp problem.  The implementation of
   exceptions is my own but is based on something I read in Byte magazine
   a very long time ago. */

/* If the error message is longer than 1023 characters, this code should
   abort.  I don't know of any way of providing printf style functionality
   into a string without possibly overflowing the end. */

/* If you want to store the error message that you recieve inside an
   exception handler, you will need to save it in a malloced buffer, 
   because it will be destroyed on the next call to the error handling
   functions */

typedef void (*ErrorFunction)(char *package,char *errid,char *errfmt,...);

/* This error function prints out the package and the msg format and
   then aborts. This should NOT be the default error function for libraries,
   since that would force people to always reset the error function and
   the LongJumpPrintErrorFunction provides almost exactly the same 
   functionality in the case where the user doesn't handle the error. */
void AbortErrorFunction(char *package,char *errid,char *errfmt, ...);

/* This error function prints out a warning and then longjumps back to 
   an outside handler.  If there is no outside handler, it prints out 
   a message about unhandled exception and aborts */
void LongJmpPrintErrorFunction(char *package,char *errid,char *errfmt, ...);

/* This error function just builds up the exception structure and then
   longjmps to a handler.  If there isn't any handler, it will print out
   a message about an unhandled exception and abort */
void LongJmpErrorFunction(char *package,char *errid,char *errfmt, ...);

/* Definitions for exception handling. */

typedef struct __exceptionStruct {
  struct __exceptionStruct *cur;
  struct __exceptionStruct *next;/* For consistancy checking */
  jmp_buf towhere;
} exceptionStruct;

/* You can examine these values, but you should not modify them.  They are
   guarenteed to be non-NULL after a call to throw. */

extern char *exception_progname;
extern char *exception_package;
extern char *exception_eid; /* Error Id, made a char * so that changes in error
			    id's will be checked at link time */
extern char *exception_errmsg;

void errlib_setprogname(char *progname);

void throwerr(char *package,char *eid,char *buf);

#define WITH_HANDLING { exceptionStruct __excstr;\
			  __excstr.cur = &__excstr;\
			  __excstr.next = __errlib_top;\
			  __errlib_top = &__excstr;\
			  if (setjmp(__excstr.towhere)==0) {
#define CLEANUP_HANDLING __errlib_top = __errlib_top->next; 
#define HANDLE CLEANUP_HANDLING } else {
#define END_HANDLING CLEANUP_HANDLING }}
#define BEGIN_MATCH if (0) {
#define EXCP_MATCH(package,eid) } else if ((package)==exception_package &&\
					   (eid)==exception_eid) {
#define XMATCH(pname,eid) EXCP_MATCH(pname ## _packagever,pname ## _E ## eid)
#define RERAISE() { CLEANUP_HANDLING;\
                  throwerr(exception_package,exception_eid,exception_errmsg);}
#define PROTECT_UNWIND RERAISE(); }}
#define END_MATCH } else RERAISE();
/* Simple Handle */
#define SHANDLE1(do,pname,eid,handle) \
WITH_HANDLING { \
  do; \
} HANDLE { \
  BEGIN_MATCH; \
  XMATCH(pname,eid) { \
    handle; \
  } \
  END_MATCH; \
} END_HANDLING; 

/* Define this for machines which don't have it by default */
#ifdef sparc
char *strerror(int errno);
#endif

extern char *errlib_packagever,*errlib_EOverflowBuffer;

extern char *internal_Einternal; 
/* This is for some sort of internal error that shouldn't have occurred, and
   probably means something has stomped on memory or something like that */

/* DO NOT MODIFY THIS VARIABLE UNLESS YOU REALLY REALLY 
   UNDERSTAND WHAT YOU ARE DOING, I.E. YOU HAVE READ THE CODE THAT MODIFIES
   THIS AND YOU UNDERSTAND WHAT THAT CODE IS TRYING TO DO! */
extern exceptionStruct *__errlib_top;

#endif /*ndef __errlib_h*/
